﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include "VibrationDecoder.h"
#include <algorithm>
#include <limits>
#include <nn/fs.h>
#include <nn/nn_Assert.h>
#include <nn/os/os_Result.h>


namespace nns
{


/**
 * @brief Table of allowed frequencies for the decoder
 *
 * The values are such that `g_freqs[i] = 10.0 * pow(2.0, i/32.0);`
 */
const NN_ALIGNAS(64) float g_freqs[256] =
{
      10.000000f,   10.218972f,   10.442738f,   10.671405f,   10.905077f,   11.143867f,   11.387886f,   11.637249f,
      11.892071f,   12.152473f,   12.418577f,   12.690510f,   12.968396f,   13.252367f,   13.542556f,   13.839100f,
      14.142136f,   14.451808f,   14.768262f,   15.091644f,   15.422108f,   15.759809f,   16.104904f,   16.457556f,
      16.817928f,   17.186193f,   17.562521f,   17.947090f,   18.340080f,   18.741676f,   19.152065f,   19.571442f,
      20.000000f,   20.437944f,   20.885475f,   21.342810f,   21.810154f,   22.287735f,   22.775772f,   23.274498f,
      23.784142f,   24.304947f,   24.837154f,   25.381020f,   25.936792f,   26.504734f,   27.085112f,   27.678200f,
      28.284271f,   28.903616f,   29.536524f,   30.183289f,   30.844215f,   31.519619f,   32.209808f,   32.915112f,
      33.635857f,   34.372387f,   35.125042f,   35.894180f,   36.680161f,   37.483353f,   38.304131f,   39.142883f,
      40.000000f,   40.875889f,   41.770950f,   42.685619f,   43.620308f,   44.575470f,   45.551544f,   46.548996f,
      47.568283f,   48.609894f,   49.674309f,   50.762039f,   51.873585f,   53.009468f,   54.170223f,   55.356400f,
      56.568542f,   57.807232f,   59.073048f,   60.366577f,   61.688431f,   63.039238f,   64.419617f,   65.830223f,
      67.271713f,   68.744774f,   70.250084f,   71.788361f,   73.360321f,   74.966705f,   76.608261f,   78.285767f,
      80.000000f,   81.751778f,   83.541901f,   85.371239f,   87.240616f,   89.150940f,   91.103088f,   93.097992f,
      95.136566f,   97.219788f,   99.348618f,  101.524078f,  103.747169f,  106.018936f,  108.340446f,  110.712799f,
     113.137085f,  115.614464f,  118.146095f,  120.733154f,  123.376862f,  126.078476f,  128.839233f,  131.660446f,
     134.543427f,  137.489548f,  140.500168f,  143.576721f,  146.720642f,  149.933411f,  153.216522f,  156.571533f,
     160.000000f,  163.503555f,  167.083801f,  170.742477f,  174.481232f,  178.301880f,  182.206177f,  186.195984f,
     190.273132f,  194.439575f,  198.697235f,  203.048157f,  207.494339f,  212.037872f,  216.680893f,  221.425598f,
     226.274170f,  231.228928f,  236.292191f,  241.466309f,  246.753723f,  252.156952f,  257.678467f,  263.320892f,
     269.086853f,  274.979095f,  281.000336f,  287.153442f,  293.441284f,  299.866821f,  306.433044f,  313.143066f,
     320.000000f,  327.007111f,  334.167603f,  341.484955f,  348.962463f,  356.603760f,  364.412354f,  372.391968f,
     380.546265f,  388.879150f,  397.394470f,  406.096313f,  414.988678f,  424.075745f,  433.361786f,  442.851196f,
     452.548340f,  462.457855f,  472.584381f,  482.932617f,  493.507446f,  504.313904f,  515.356934f,  526.641785f,
     538.173706f,  549.958191f,  562.000671f,  574.306885f,  586.882568f,  599.733643f,  612.866089f,  626.286133f,
     640.000000f,  654.014221f,  668.335205f,  682.969910f,  697.924927f,  713.207520f,  728.824707f,  744.783936f,
     761.092529f,  777.758301f,  794.788940f,  812.192627f,  829.977356f,  848.151489f,  866.723572f,  885.702393f,
     905.096680f,  924.915710f,  945.168762f,  965.865234f,  987.014893f, 1008.627808f, 1030.713867f, 1053.283569f,
    1076.347412f, 1099.916382f, 1124.001343f, 1148.613770f, 1173.765137f, 1199.467285f, 1225.732178f, 1252.572266f,
    1280.000000f, 1308.028442f, 1336.670410f, 1365.939819f, 1395.849854f, 1426.415039f, 1457.649414f, 1489.567871f,
    1522.185059f, 1555.516602f, 1589.577881f, 1624.385254f, 1659.954712f, 1696.302979f, 1733.447144f, 1771.404785f,
    1810.193359f, 1849.831421f, 1890.337524f, 1931.730469f, 1974.029785f, 2017.255615f, 2061.427734f, 2106.567139f,
    2152.694824f, 2199.832764f, 2248.002686f, 2297.227539f, 2347.530273f, 2398.934570f, 2451.464355f, 2505.144531f
};


/*
 * Sine table such that each entry is 32767 * sin(x) with x in [0; 2π]
 */
static NN_ALIGNAS(64) const int16_t g_sin[257] =
{
         0,   804,  1607,  2410,  3211,  4011,  4807,  5601,  6392,  7179,  7961,  8739,  9511, 10278, 11038, 11792,
     12539, 13278, 14009, 14732, 15446, 16150, 16845, 17530, 18204, 18867, 19519, 20159, 20787, 21402, 22004, 22594,
     23169, 23731, 24278, 24811, 25329, 25831, 26318, 26789, 27244, 27683, 28105, 28510, 28897, 29268, 29621, 29955,
     30272, 30571, 30851, 31113, 31356, 31580, 31785, 31970, 32137, 32284, 32412, 32520, 32609, 32678, 32727, 32757,
     32767, 32757, 32727, 32678, 32609, 32520, 32412, 32284, 32137, 31970, 31785, 31580, 31356, 31113, 30851, 30571,
     30272, 29955, 29621, 29268, 28897, 28510, 28105, 27683, 27244, 26789, 26318, 25831, 25329, 24811, 24278, 23731,
     23169, 22594, 22004, 21402, 20787, 20159, 19519, 18867, 18204, 17530, 16845, 16150, 15446, 14732, 14009, 13278,
     12539, 11792, 11038, 10278,  9511,  8739,  7961,  7179,  6392,  5601,  4807,  4011,  3211,  2410,  1607,   804,
         0,  -804, -1607, -2410, -3211, -4011, -4807, -5601, -6392, -7179, -7961, -8739, -9511,-10278,-11038,-11792,
    -12539,-13278,-14009,-14732,-15446,-16150,-16845,-17530,-18204,-18867,-19519,-20159,-20787,-21402,-22004,-22594,
    -23169,-23731,-24278,-24811,-25329,-25831,-26318,-26789,-27244,-27683,-28105,-28510,-28897,-29268,-29621,-29955,
    -30272,-30571,-30851,-31113,-31356,-31580,-31785,-31970,-32137,-32284,-32412,-32520,-32609,-32678,-32727,-32757,
    -32767,-32757,-32727,-32678,-32609,-32520,-32412,-32284,-32137,-31970,-31785,-31580,-31356,-31113,-30851,-30571,
    -30272,-29955,-29621,-29268,-28897,-28510,-28105,-27683,-27244,-26789,-26318,-25831,-25329,-24811,-24278,-23731,
    -23169,-22594,-22004,-21402,-20787,-20159,-19519,-18867,-18204,-17530,-16845,-16150,-15446,-14732,-14009,-13278,
    -12539,-11792,-11038,-10278, -9511, -8739, -7961, -7179, -6392, -5601, -4807, -4011, -3211, -2410, -1607,  -804,
         0
};


template<typename T>
NN_FORCEINLINE T Round(T v, int shift)
{
    T rounding = 1 << (shift - 1);
    return (v + rounding) >> shift;
}


/**
 * @brief Sine implementation using the above table and linear interpolation
 */
NN_FORCEINLINE int VibrationDecoder::Sin(uint32_t phase) NN_NOEXCEPT
{
    uint32_t index = phase >> PhasePrecision;
    int sin1 = g_sin[index];
    int sin2 = g_sin[index + 1];
    int ratio = (phase >> (PhasePrecision - 16)) & 0xFFFF;
    return sin1 + (((sin2 - sin1) * ratio) >> 16);
}


/**
 * @brief Initializes the decoder
 *
 * The bands frequencies are set to 160 Hz and 320 Hz, respectively. Amplitudes are both set to 0.
 *
 * @param [in] samplingRate The sampling rate of the waveform to be generated, must be a multiple of 200 Hz
 */
nn::Result VibrationDecoder::Init(uint32_t samplingRate) NN_NOEXCEPT
{
    if (samplingRate % 200 != 0)
    {
        return nn::os::ResultInvalidParameter() ;
    }

    m_states[0].phase     = 0;
    m_states[0].frequency = 160 << FrequencyPrecision;
    m_states[0].amplitude = 0;
    m_states[1].phase     = 0;
    m_states[1].frequency = 320 << FrequencyPrecision;
    m_states[1].amplitude = 0;

    m_phaseRate   = uint32_t((256ull << PhasePrecision) / samplingRate); // 2π/samplingRate with 24 bit precision
    m_chunkLength = samplingRate / 200; // 5 ms

    return nn::ResultSuccess();
}


/**
 * @brief Generates a 5 ms chunk of audio from a vibration value
 */
size_t VibrationDecoder::GenerateChunk(const nn::hid::VibrationValue& vib, int16_t* dest) NN_NOEXCEPT
{
    NN_ASSERT(10.0f <= vib.frequencyLow  && vib.frequencyLow  <= 2505.144531f);
    NN_ASSERT(10.0f <= vib.frequencyHigh && vib.frequencyHigh <= 2505.144531f);
    NN_ASSERT(0.0f <= vib.amplitudeLow   && vib.amplitudeLow  <= 1.0f);
    NN_ASSERT(0.0f <= vib.amplitudeHigh  && vib.amplitudeHigh <= 1.0f);

    // Convert input values to fixed point
    int32_t  freq1 = static_cast<int32_t>(vib.frequencyLow  * (1 << FrequencyPrecision));
    int32_t  freq2 = static_cast<int32_t>(vib.frequencyHigh * (1 << FrequencyPrecision));
    uint32_t am1   = static_cast<uint32_t>(vib.amplitudeLow  * 255.0f);
    uint32_t am2   = static_cast<uint32_t>(vib.amplitudeHigh * 255.0f);

    return DoGenerateChunk(freq1, am1, freq2, am2, dest);
}


/**
 * @brief Generates a 5 ms chunk of audio from raw vibration bytes values
 */
size_t VibrationDecoder::GenerateChunk(uint32_t fm1, uint32_t am1, uint32_t fm2, uint32_t am2, int16_t* dest) NN_NOEXCEPT
{
    NN_ASSERT(0 <= fm1 && fm1 <= 255);
    NN_ASSERT(0 <= am1 && am1 <= 255);
    NN_ASSERT(0 <= fm2 && fm2 <= 255);
    NN_ASSERT(0 <= am2 && am2 <= 255);

    // Convert input values to fixed point
    int32_t freq1 = static_cast<int32_t>(g_freqs[fm1] * (1 << FrequencyPrecision));
    int32_t freq2 = static_cast<int32_t>(g_freqs[fm2] * (1 << FrequencyPrecision));

    return DoGenerateChunk(freq1, am1, freq2, am2, dest);
}


/**
 * @brief Generates a 5 ms chunk of audio
 *
 * @param [in]  freq1 The frequency for the lower  band, in Hz as a 12.20 fixed point
 * @param [in]  am1   The amplitude for the lower  band, in range [0; 255]
 * @param [in]  freq2 The frequency for the higher band, in Hz as a 12.20 fixed point
 * @param [in]  am2   The amplitude for the higher band, in range [0; 255]
 * @param [out] dest  Pointer to the destination buffer
 */
size_t VibrationDecoder::DoGenerateChunk(uint32_t freq1, uint32_t am1, uint32_t freq2, uint32_t am2, int16_t* dest) NN_NOEXCEPT
{
    // Retrieve previous phases
    uint32_t phase1 = m_states[0].phase;
    uint32_t phase2 = m_states[1].phase;

    // Compute frequency increments (as 12.20 fixed point)
    uint32_t f1     = m_states[0].frequency;
    uint32_t f2     = m_states[1].frequency;
    int32_t  f1Inc = int32_t(freq1 - f1) / int32_t(m_chunkLength);
    int32_t  f2Inc = int32_t(freq2 - f2) / int32_t(m_chunkLength);

    // Compute amplitude increments
    int a1    = m_states[0].amplitude << AmplitudePrecision;
    int a2    = m_states[1].amplitude << AmplitudePrecision;
    int a1Inc = (int(am1 << AmplitudePrecision) - a1) / int(m_chunkLength);
    int a2Inc = (int(am2 << AmplitudePrecision) - a2) / int(m_chunkLength);

    for (size_t i=0; i<m_chunkLength; ++i)
    {
        f1 += f1Inc;
        f2 += f2Inc;

        // Compute phase increments
        uint32_t phase1Inc = uint32_t((uint64_t(f1) * m_phaseRate) >> FrequencyPrecision);
        uint32_t phase2Inc = uint32_t((uint64_t(f2) * m_phaseRate) >> FrequencyPrecision);

        // Update amplitude
        a1 += a1Inc;
        a2 += a2Inc;

        // Compute current value
        int value1 = Sin(phase1);
        int value2 = Sin(phase2);
        int value  = ((Round(a1, AmplitudePrecision) * value1) + (Round(a2, AmplitudePrecision) * value2)) / 256;
        value = std::min(std::max(-32768, value), 32767);
        dest[i] = static_cast<int16_t>(value);

        // Update phase
        phase1 += phase1Inc;
        phase2 += phase2Inc;
    }

    // Save current state
    m_states[0].phase     = phase1;
    m_states[0].frequency = freq1;
    m_states[0].amplitude = am1;
    m_states[1].phase     = phase2;
    m_states[1].frequency = freq2;
    m_states[1].amplitude = am2;

    return m_chunkLength;
}


} // namespace nns
