﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{GfxShaderCompile.cpp,PageSampleGfxShaderCompile}
 *
 * @brief
 *  シェーダーのランタイムコンパイルのサンプルプログラム
 */

/**
 * @page PageSampleGfxShaderCompile GfxShaderCompile
 * @tableofcontents
 *
 * @brief
 *  シェーダーのランタイムコンパイルのサンプルプログラムの解説です。
 *
 * @section PageSampleGfxShaderCompile_SectionBrief 概要
 *  gfx を使用して、シェーダーを実行時にソースからコンパイルするサンプルです。
 *
 * @section PageSampleGfxShaderCompile_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/GfxShaderCompile
 *  Samples/Sources/Applications/GfxShaderCompile @endlink 以下にあります。
 *
 * @section PageSampleGfxShaderCompile_SectionNecessaryEnvironment 必要な環境
 *  画面表示が利用可能である必要があります。
 *
 * @section PageSampleGfxShaderCompile_SectionHowToOperate 操作方法
 *  特にありません。
 *
 * @section PageSampleGfxShaderCompile_SectionPrecaution 注意事項
 *  特にありません。
 *
 * @section PageSampleGfxShaderCompile_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleGfxShaderCompile_SectionDetail 解説
 *  このサンプルプログラムは、シェーダーをソースからランタイムコンパイルし、
 *  コンパイルしたシェーダーを使ってシンプルな三角形をレンダリングします。
 *
 * このサンプルプログラムの処理の流れは以下の通りです。
 *
 * - ディスプレイを取得・レイヤーを初期化
 * - デバイスを初期化
 * - メモリープールを初期化
 * - スワップチェーンを初期化
 * - キューを初期化
 * - コマンドバッファーを初期化
 * - ビューポートを初期化
 * - 頂点シェーダーを初期化
 * - ピクセルシェーダーを初期化
 * - ラスタライザーステートを初期化
 * - ブレンドステートを初期化
 * - 深度ステンシルステートを初期化
 * - 頂点ステートを初期化
 * - 頂点バッファーを初期化
 * - インデクスバッファーを初期化
 * - 深度バッファーを初期化
 * - 深度バッファービューを初期化
 * - 定数バッファーを初期化
 * - テクスチャーを初期化
 * - テクスチャービューを初期化
 * - サンプラーを初期化
 * - デスクリプタープールを初期化
 * - フェンスを初期化
 * - コマンドリストを作成
 * - ループ開始
 * - コマンドリストを実行
 * - ディスプレイへプレゼンテーション
 * - ループ開始に戻る
 * - 各種オブジェクトを破棄
 */

#include <algorithm>
#include <cstdio>
#include <cstring>
#include <stdint.h> // This is C++11, not supported by Cafe: #include <cstdint>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/init.h>
#include <nn/vi.h>
#include <nn/gfx.h>
#include <nn/fs.h>

#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN )
    #include <cstdlib>
#endif
#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN ) && defined( NN_BUILD_APISET_NX )
    #include <nv/nv_MemoryManagement.h>
#endif
#if defined( NN_BUILD_APISET_CAFE )
    #include <cafe/gx2.h>
    #include <SimpleShader.h>
#endif

#if NN_GFX_IS_TARGET_NVN
    #include <nvn/nvn.h>
    #include <nvn/nvn_FuncPtrInline.h>
    #if defined( NN_BUILD_TARGET_PLATFORM_OS_NN )
        #include <nvnTool/nvnTool_GlslcInterface.h>
    #endif
#endif

namespace {

const bool IsDescriptorPoolMode = true;
const bool IsPipelineMode = false;

const int ScanBufferCount = 2;

const int RenderTargetWidth = 1280;
const int RenderTargetHeight = 720;

const char* const VsGlsl =
    "#version 430\n"
    "out gl_PerVertex\n"
    "{\n"
    "    vec4 gl_Position;\n"
    "};\n"
    "in vec4 i_Position;\n"
    "in vec2 i_TexCoord;\n"
    "out vec2 texCoord;\n"
    "void main()\n"
    "{\n"
    "    gl_Position = i_Position;\n"
    "    texCoord = i_TexCoord.xy;\n"
    "}\n";

const char* const PsGlsl =
    "#version 430\n"
    "layout( location = 0 ) out vec4 o_Color;\n"
    "in vec2 texCoord;\n"
    "uniform sampler2D tex;\n"
    "layout( std140 ) uniform Mat\n"
    "{\n"
    "    vec4 color;"
    "};\n"
    "void main()\n"
    "{\n"
    "    vec4 tex_color = texture( tex, texCoord );\n"
    "    o_Color = color + tex_color;\n"
    "}\n";

const float Vertices[] =
{
    -0.5f, -0.5f, 0.0f, -0.5f, -0.5f,
    0.0f, 0.5f, 0.0f, 0.5f, 0.5f,
    0.5f, -0.5f, 0.0f, 0.5f, -0.5f
};

const int Indices[] =
{
    0, 1, 2
};

int g_BufferDescriptorBaseIndex = 0;
int g_TextureDescriptorBaseIndex = 0;
int g_SamplerDescriptorBaseIndex = 0;

//-----------------------------------------------------------------------------
// メモリー

const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;

nn::util::BytePtr g_pMemoryHeap( NULL );
nn::util::BytePtr g_pMemory( NULL );

enum MemoryPoolType
{
    MemoryPoolType_CpuCached_GpuCached,
    MemoryPoolType_CpuUncached_GpuCached,
    MemoryPoolType_CpuInvisible_GpuCached_Compressible,

    MemoryPoolType_End
};

const size_t MemoryPoolSize[ MemoryPoolType_End ] =
{
    16 * 1024 * 1024,
    16 * 1024 * 1024,
    20 * 1024 * 1024
};

void* g_pPoolMemory[ MemoryPoolType_End ] = {};
ptrdiff_t g_MemoryPoolOffset[ MemoryPoolType_End ] = {};

// ディスプレイを初期化
nn::vi::Display* g_pDisplay;
nn::vi::Layer* g_pLayer;
void InitializeLayer()
{
    nn::Result result = nn::vi::OpenDefaultDisplay( &g_pDisplay );
    NN_ASSERT( result.IsSuccess() );
    NN_UNUSED( result );

    result = nn::vi::CreateLayer( &g_pLayer, g_pDisplay );
    NN_ASSERT( result.IsSuccess() );

    result = nn::vi::SetLayerScalingMode( g_pLayer, nn::vi::ScalingMode_FitToLayer );
    NN_ASSERT( result.IsSuccess() );
}

// デバイスを初期化
nn::gfx::Device g_Device;
void InitializeDevice()
{
    nn::gfx::Device::InfoType info;
    info.SetDefault();
    info.SetDebugMode( nn::gfx::DebugMode_Enable );
    info.SetApiVersion( nn::gfx::ApiMajorVersion, nn::gfx::ApiMinorVersion );
    g_Device.Initialize( info );
}

nn::gfx::MemoryPool g_MemoryPool[ MemoryPoolType_End ];
void InitializeMemoryPool()
{
    const int MemoryPoolProperty[ MemoryPoolType_End ] =
    {
        nn::gfx::MemoryPoolProperty_CpuCached | nn::gfx::MemoryPoolProperty_GpuCached,
        nn::gfx::MemoryPoolProperty_CpuUncached | nn::gfx::MemoryPoolProperty_GpuCached,
        nn::gfx::MemoryPoolProperty_CpuInvisible | nn::gfx::MemoryPoolProperty_GpuCached
            | nn::gfx::MemoryPoolProperty_Compressible
    };

    nn::gfx::MemoryPool::InfoType info;
    for( int idxMemoryPool = 0; idxMemoryPool < MemoryPoolType_End; ++idxMemoryPool )
    {
        info.SetDefault();
        info.SetMemoryPoolProperty( MemoryPoolProperty[ idxMemoryPool ] );
        size_t alignment = nn::gfx::MemoryPool::GetPoolMemoryAlignment( &g_Device, info );
        size_t granularity = nn::gfx::MemoryPool::GetPoolMemorySizeGranularity( &g_Device, info );
        g_pPoolMemory[ idxMemoryPool ] = malloc( MemoryPoolSize[ idxMemoryPool ] );
        void* pPoolMemoryAligned = nn::util::BytePtr(
            g_pPoolMemory[ idxMemoryPool ] ).AlignUp( alignment ).Get();
        size_t memoryPoolSizeAligned = nn::util::align_down( MemoryPoolSize[ idxMemoryPool ], granularity );
        info.SetPoolMemory( pPoolMemoryAligned, memoryPoolSizeAligned );
        g_MemoryPool[ idxMemoryPool ].Initialize( &g_Device, info );
        g_MemoryPoolOffset[ idxMemoryPool ] = 0;
    }
}

// スワップチェーンを初期化
nn::gfx::SwapChain g_SwapChain;
void InitializeSwapChain()
{
    nn::gfx::SwapChain::InfoType info;

    info.SetDefault();
    info.SetLayer( g_pLayer );
    info.SetWidth( RenderTargetWidth );
    info.SetHeight( RenderTargetHeight );
    info.SetFormat( nn::gfx::ImageFormat_R8_G8_B8_A8_UnormSrgb );
    info.SetBufferCount( ScanBufferCount );
    if( NN_STATIC_CONDITION( nn::gfx::SwapChain::IsMemoryPoolRequired ) )
    {
        size_t size = g_SwapChain.CalculateScanBufferSize( &g_Device, info );
        ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuInvisible_GpuCached_Compressible ];
        offset = nn::util::align_up( offset, nn::gfx::SwapChain::GetScanBufferAlignment( &g_Device, info ) );
        g_SwapChain.Initialize( &g_Device, info,
            &g_MemoryPool[ MemoryPoolType_CpuInvisible_GpuCached_Compressible ], offset, size );
        offset += size;
    }
    else
    {
        g_SwapChain.Initialize( &g_Device, info, NULL, 0, 0 );
    }
}

// キューを初期化
nn::gfx::Queue g_Queue;
void InitializeQueue()
{
    nn::gfx::Queue::InfoType info;
    info.SetDefault();
    info.SetCapability( nn::gfx::QueueCapability_Graphics );
    g_Queue.Initialize( &g_Device, info );
}

// コマンドバッファーを初期化
nn::gfx::CommandBuffer g_QueueCommandBuffer[ ScanBufferCount ];
void InitializeQueueCommandBuffer()
{
    nn::gfx::CommandBuffer::InfoType info;
    info.SetDefault();
    info.SetQueueCapability( nn::gfx::QueueCapability_Graphics );
    info.SetCommandBufferType( nn::gfx::CommandBufferType_Direct );
    for( int idxCommandBuffer = 0; idxCommandBuffer < ScanBufferCount; ++idxCommandBuffer )
    {
        g_QueueCommandBuffer[ idxCommandBuffer ].Initialize( &g_Device, info );
    }
}

nn::gfx::CommandBuffer g_CommandBuffer;
void InitializeCommandBuffer()
{
    nn::gfx::CommandBuffer::InfoType info;
    info.SetDefault();
    info.SetQueueCapability( nn::gfx::QueueCapability_Graphics );
    info.SetCommandBufferType( nn::gfx::CommandBufferType_Nested );
    g_CommandBuffer.Initialize( &g_Device, info );
}

// ビューポートシザーを初期化
nn::gfx::ViewportScissorState g_ViewportScissor;
void InitializeViewport()
{
    nn::gfx::ViewportScissorState::InfoType info;
    info.SetDefault();
    info.SetScissorEnabled( true );
    nn::gfx::ViewportStateInfo viewportInfo;
    {
        viewportInfo.SetDefault();
        viewportInfo.SetWidth( static_cast< float >( RenderTargetWidth ) );
        viewportInfo.SetHeight( static_cast< float >( RenderTargetHeight ) );
    }
    nn::gfx::ScissorStateInfo scissorInfo;
    {
        scissorInfo.SetDefault();
        scissorInfo.SetWidth( RenderTargetWidth );
        scissorInfo.SetHeight( RenderTargetHeight );
    }
    info.SetViewportStateInfoArray( &viewportInfo, 1 );
    info.SetScissorStateInfoArray( &scissorInfo, 1 );
    // ビューポートの数がひとつのときは追加要求メモリーなし
    g_ViewportScissor.Initialize( &g_Device, info );
}

// 頂点シェーダーを初期化
nn::gfx::Shader g_VertexShader;
void InitializeVertexShader()
{
    nn::gfx::Shader::InfoType info;
    info.SetDefault();
    info.SetSeparationEnabled( true );
#ifdef CAFE
    NN_UNUSED( VsGlsl );
    info.SetShaderCodePtr( nn::gfx::ShaderStage_Vertex, &SimpleShader_VS );
    info.SetCodeType( nn::gfx::ShaderCodeType_Binary );
#else
    nn::gfx::ShaderCode code;
    code.codeSize = static_cast< uint32_t >( strlen( VsGlsl ) );
    code.pCode = VsGlsl;
    info.SetShaderCodePtr( nn::gfx::ShaderStage_Vertex, &code );
    info.SetSourceFormat( nn::gfx::ShaderSourceFormat_Glsl );
    info.SetCodeType( nn::gfx::ShaderCodeType_Source );
#endif
    nn::gfx::ShaderInitializeResult result = g_VertexShader.Initialize( &g_Device, info );
    NN_ASSERT( result == nn::gfx::ShaderInitializeResult_Success );
    NN_UNUSED( result );
}
// ピクセルシェーダーを初期化
nn::gfx::Shader g_PixelShader;
int g_SlotTex;
int g_SlotMat;
void InitializePixelShader()
{
    nn::gfx::Shader::InfoType info;
    info.SetDefault();
    info.SetSeparationEnabled( true );
#ifdef CAFE
    NN_UNUSED( PsGlsl );
    info.SetShaderCodePtr( nn::gfx::ShaderStage_Pixel, &SimpleShader_PS );
    info.SetCodeType( nn::gfx::ShaderCodeType_Binary );
#else
    nn::gfx::ShaderCode code;
    code.codeSize = static_cast< uint32_t >( strlen( PsGlsl ) );
    code.pCode = PsGlsl;
    info.SetShaderCodePtr( nn::gfx::ShaderStage_Pixel, &code );
    info.SetSourceFormat( nn::gfx::ShaderSourceFormat_Glsl );
    info.SetCodeType( nn::gfx::ShaderCodeType_Source );
#endif
    nn::gfx::ShaderInitializeResult result = g_PixelShader.Initialize( &g_Device, info );
    NN_ASSERT( result == nn::gfx::ShaderInitializeResult_Success );
    NN_UNUSED( result );

    g_SlotMat = g_PixelShader.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel,
        nn::gfx::ShaderInterfaceType_ConstantBuffer, "Mat" );
    g_SlotTex = g_PixelShader.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel,
        nn::gfx::ShaderInterfaceType_Sampler, "tex" );
}

// パイプラインを初期化
nn::gfx::Pipeline g_Pipeline;
nn::gfx::Shader g_Shader;
void InitializePipeline()
{
    {
        nn::gfx::Shader::InfoType info;
        info.SetDefault();
        info.SetSeparationEnabled( false );
#if defined( CAFE )
        NN_UNUSED( VsGlsl );
        NN_UNUSED( PsGlsl );
        info.SetShaderCodePtr( nn::gfx::ShaderStage_Vertex, &SimpleShader_VS );
        info.SetShaderCodePtr( nn::gfx::ShaderStage_Pixel, &SimpleShader_PS );
        info.SetCodeType( nn::gfx::ShaderCodeType_Binary );
#else
        nn::gfx::ShaderCode vsCode;
        vsCode.codeSize = static_cast< uint32_t >( strlen( VsGlsl ) );
        vsCode.pCode = VsGlsl;
        nn::gfx::ShaderCode psCode;
        psCode.codeSize = static_cast< uint32_t >( strlen( PsGlsl ) );
        psCode.pCode = PsGlsl;
        info.SetShaderCodePtr( nn::gfx::ShaderStage_Vertex, &vsCode );
        info.SetShaderCodePtr( nn::gfx::ShaderStage_Pixel, &psCode );
        info.SetSourceFormat( nn::gfx::ShaderSourceFormat_Glsl );
        info.SetCodeType( nn::gfx::ShaderCodeType_Source );
#endif
        nn::gfx::ShaderInitializeResult result = g_Shader.Initialize( &g_Device, info );
        NN_ASSERT( result == nn::gfx::ShaderInitializeResult_Success );
        NN_UNUSED( result );
        g_SlotMat = g_Shader.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel,
            nn::gfx::ShaderInterfaceType_ConstantBuffer, "Mat" );
        g_SlotTex = g_Shader.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel,
            nn::gfx::ShaderInterfaceType_Sampler, "tex" );
    }

    nn::gfx::RasterizerState::InfoType rasterizerStateInfo;
    rasterizerStateInfo.SetDefault();
    rasterizerStateInfo.SetCullMode( nn::gfx::CullMode_None );
    rasterizerStateInfo.SetPrimitiveTopologyType( nn::gfx::PrimitiveTopologyType_Triangle );
    rasterizerStateInfo.SetScissorEnabled( true );
    rasterizerStateInfo.SetDepthClipEnabled( false );

    nn::gfx::BlendState::InfoType blendStateInfo;
    blendStateInfo.SetDefault();
    nn::gfx::BlendTargetStateInfo blendTargetStateInfo;
    {
        blendTargetStateInfo.SetDefault();
    };
    blendStateInfo.SetBlendTargetStateInfoArray( &blendTargetStateInfo, 1 );

    nn::gfx::DepthStencilState::InfoType depthStencilStateInfo;
    depthStencilStateInfo.SetDefault();
    depthStencilStateInfo.SetDepthTestEnabled( true );
    depthStencilStateInfo.SetDepthWriteEnabled( true );

    nn::gfx::VertexState::InfoType vertexStateInfo;
    vertexStateInfo.SetDefault();
    ptrdiff_t stride = sizeof( float ) * 5;
    nn::gfx::VertexAttributeStateInfo vertexAttributeStateInfo[ 2 ];
    {
        vertexAttributeStateInfo[ 0 ].SetDefault();
        vertexAttributeStateInfo[ 0 ].SetNamePtr( "i_Position" );
        vertexAttributeStateInfo[ 0 ].SetBufferIndex( 0 );
        vertexAttributeStateInfo[ 0 ].SetFormat( nn::gfx::AttributeFormat_32_32_32_Float );
        vertexAttributeStateInfo[ 0 ].SetOffset( 0 );
    }
    {
        vertexAttributeStateInfo[ 1 ].SetDefault();
        vertexAttributeStateInfo[ 1 ].SetNamePtr( "i_TexCoord" );
        vertexAttributeStateInfo[ 1 ].SetBufferIndex( 0 );
        vertexAttributeStateInfo[ 1 ].SetFormat( nn::gfx::AttributeFormat_32_32_Float );
        vertexAttributeStateInfo[ 1 ].SetOffset( sizeof( float ) * 3 );
    }
    nn::gfx::VertexBufferStateInfo vertexBufferStateInfo;
    {
        vertexBufferStateInfo.SetDefault();
        vertexBufferStateInfo.SetStride( stride );
    }
    vertexStateInfo.SetVertexAttributeStateInfoArray( vertexAttributeStateInfo, 2 );
    vertexStateInfo.SetVertexBufferStateInfoArray( &vertexBufferStateInfo, 1 );

    nn::gfx::RenderTargetStateInfo renderTargetStateInfo;
    renderTargetStateInfo.SetDefault();

    nn::gfx::Pipeline::InfoType info;
    info.SetDefault();
    info.SetShaderPtr( &g_Shader );
    info.SetRasterizerStateInfo( &rasterizerStateInfo );
    info.SetBlendStateInfo( &blendStateInfo );
    info.SetDepthStencilStateInfo( &depthStencilStateInfo );
    info.SetVertexStateInfo( &vertexStateInfo );
    info.SetRenderTargetStateInfo( &renderTargetStateInfo );
    size_t dataSize = nn::gfx::Pipeline::GetRequiredMemorySize( info );
    g_pMemory.AlignUp( nn::gfx::Pipeline::RequiredMemoryInfo_Alignment );
    g_Pipeline.SetMemory( g_pMemory.Get(), dataSize );
    g_pMemory.Advance( dataSize );
    g_Pipeline.Initialize( &g_Device, info );
}

// Initialize the rasterizer state
nn::gfx::RasterizerState g_RasterizerState;
static void InitializeRasterizerState()
{
    nn::gfx::RasterizerState::InfoType info;
    info.SetDefault();
    info.SetCullMode( nn::gfx::CullMode_None );
    info.SetPrimitiveTopologyType( nn::gfx::PrimitiveTopologyType_Triangle );
    info.SetScissorEnabled( true );
    info.SetDepthClipEnabled( false );
    g_RasterizerState.Initialize( &g_Device, info );
}

// Initialize the blend state
nn::gfx::BlendState g_BlendState;
static void InitializeBlendState()
{
    nn::gfx::BlendState::InfoType info;
    info.SetDefault();
    nn::gfx::BlendTargetStateInfo targetInfo;
    {
        targetInfo.SetDefault();
    };
    info.SetBlendTargetStateInfoArray( &targetInfo, 1 );
    size_t size = nn::gfx::BlendState::GetRequiredMemorySize( info );
    g_pMemory.AlignUp( nn::gfx::BlendState::RequiredMemoryInfo_Alignment );
    g_BlendState.SetMemory( g_pMemory.Get(), size );
    g_pMemory.Advance( size );
    g_BlendState.Initialize( &g_Device, info );
}

// Initialize the depth stencil state
nn::gfx::DepthStencilState g_DepthStencilState;
static void InitializeDepthStencilState()
{
    nn::gfx::DepthStencilState::InfoType info;
    info.SetDefault();
    info.SetDepthTestEnabled( true );
    info.SetDepthWriteEnabled( true );
    g_DepthStencilState.Initialize( &g_Device, info );
}

// Initialize the vertex state
nn::gfx::VertexState g_VertexState;
static void InitializeVertexState()
{
    nn::gfx::VertexState::InfoType info;
    info.SetDefault();
    ptrdiff_t stride = sizeof( float ) * 5;
    nn::gfx::VertexAttributeStateInfo attribs[ 2 ];
    {
        attribs[ 0 ].SetDefault();
        attribs[ 0 ].SetNamePtr( "i_Position" );
        attribs[ 0 ].SetBufferIndex( 0 );
        attribs[ 0 ].SetFormat( nn::gfx::AttributeFormat_32_32_32_Float );
        attribs[ 0 ].SetOffset( 0 );
    }
    {
        attribs[ 1 ].SetDefault();
        attribs[ 1 ].SetNamePtr( "i_TexCoord" );
        attribs[ 1 ].SetBufferIndex( 0 );
        attribs[ 1 ].SetFormat( nn::gfx::AttributeFormat_32_32_Float );
        attribs[ 1 ].SetOffset( sizeof( float ) * 3 );
    }
    nn::gfx::VertexBufferStateInfo buffer;
    {
        buffer.SetDefault();
        buffer.SetStride( stride );
    }
    info.SetVertexAttributeStateInfoArray( attribs, 2 );
    info.SetVertexBufferStateInfoArray( &buffer, 1 );
    size_t size = nn::gfx::VertexState::GetRequiredMemorySize( info );
    g_pMemory.AlignUp( nn::gfx::VertexState::RequiredMemoryInfo_Alignment );
    g_VertexState.SetMemory( g_pMemory.Get(), size );
    g_pMemory.Advance( size );
    g_VertexState.Initialize( &g_Device, info, &g_VertexShader );
}

// 頂点用のバッファーを初期化
nn::gfx::Buffer g_VertexBuffer;
void InitializeVertexBuffer()
{
    nn::gfx::Buffer::InfoType info;
    info.SetDefault();
    info.SetSize( sizeof( Vertices ) );
    info.SetGpuAccessFlags( nn::gfx::GpuAccess_VertexBuffer );
    if( NN_STATIC_CONDITION( nn::gfx::Buffer::IsMemoryPoolRequired ) )
    {
        ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuUncached_GpuCached ];
        offset = nn::util::align_up( offset, nn::gfx::Buffer::GetBufferAlignment( &g_Device, info ) );
        g_VertexBuffer.Initialize( &g_Device, info,
            &g_MemoryPool[ MemoryPoolType_CpuUncached_GpuCached ], offset, info.GetSize() );
        offset += info.GetSize();
    }
    else
    {
        g_VertexBuffer.Initialize( &g_Device, info, NULL, 0, 0 );
    }

    void* pMapped = g_VertexBuffer.Map();
    memcpy( pMapped, Vertices, info.GetSize() );
    g_VertexBuffer.FlushMappedRange( 0, info.GetSize() );
    g_VertexBuffer.Unmap();
}

// インデクス用のバッファーを初期化
nn::gfx::Buffer g_IndexBuffer;
void InitializeIndexBuffer()
{
    nn::gfx::Buffer::InfoType info;
    info.SetDefault();
    info.SetSize( sizeof( Indices ) );
    info.SetGpuAccessFlags( nn::gfx::GpuAccess_IndexBuffer );
    if( NN_STATIC_CONDITION( nn::gfx::Buffer::IsMemoryPoolRequired ) )
    {
        ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuUncached_GpuCached ];
        offset = nn::util::align_up( offset, nn::gfx::Buffer::GetBufferAlignment( &g_Device, info ) );
        g_IndexBuffer.Initialize( &g_Device, info,
            &g_MemoryPool[ MemoryPoolType_CpuUncached_GpuCached ], offset, info.GetSize() );
        offset += info.GetSize();
    }
    else
    {
        g_IndexBuffer.Initialize( &g_Device, info, NULL, 0, 0 );
    }

    void* pMapped = g_IndexBuffer.Map();
    memcpy( pMapped, Indices, info.GetSize() );
    g_IndexBuffer.Unmap();
}

// 深度ステンシルバッファー用のテクスチャーを初期化
nn::gfx::Texture g_DepthStencilBuffer;
void InitializeDepthBuffer()
{
    nn::gfx::Texture::InfoType info;
    info.SetDefault();
    info.SetWidth( RenderTargetWidth );
    info.SetHeight( RenderTargetHeight );
    info.SetGpuAccessFlags( nn::gfx::GpuAccess_DepthStencil );
    info.SetImageStorageDimension( nn::gfx::ImageStorageDimension_2d );
    info.SetImageFormat( nn::gfx::ImageFormat_D16_Unorm );
    info.SetMipCount( 1 );
    if( NN_STATIC_CONDITION( nn::gfx::Texture::IsMemoryPoolRequired ) )
    {
        ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuInvisible_GpuCached_Compressible ];
        offset = nn::util::align_up( offset, nn::gfx::Texture::CalculateMipDataAlignment( &g_Device, info ) );
        size_t size = nn::gfx::Texture::CalculateMipDataSize( &g_Device, info );
        g_DepthStencilBuffer.Initialize( &g_Device, info,
            &g_MemoryPool[ MemoryPoolType_CpuInvisible_GpuCached_Compressible ], offset, size );
        offset += size;
    }
    else
    {
        g_DepthStencilBuffer.Initialize( &g_Device, info, NULL, 0, 0 );
    }
}
// 深度ステンシルビューを初期化
nn::gfx::DepthStencilView g_DepthStencilView;
void InitializeDepthBufferView()
{
    nn::gfx::DepthStencilView::InfoType info;
    info.SetDefault();
    info.SetImageDimension( nn::gfx::ImageDimension_2d );
    info.SetTexturePtr( &g_DepthStencilBuffer );
    g_DepthStencilView.Initialize( &g_Device, info );
}

// 定数バッファー用のバッファを初期化
nn::gfx::Buffer g_ConstantBuffer;
void InitializeConstantBuffer()
{
    nn::gfx::Buffer::InfoType info;
    info.SetDefault();
    info.SetSize( sizeof( float ) * 4 );
    info.SetGpuAccessFlags( nn::gfx::GpuAccess_ConstantBuffer );
    if( NN_STATIC_CONDITION( nn::gfx::Buffer::IsMemoryPoolRequired ) )
    {
        ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuUncached_GpuCached ];
        offset = nn::util::align_up( offset, nn::gfx::Buffer::GetBufferAlignment( &g_Device, info ) );
        g_ConstantBuffer.Initialize( &g_Device, info,
            &g_MemoryPool[ MemoryPoolType_CpuUncached_GpuCached ], offset, info.GetSize() );
        offset += info.GetSize();
    }
    else
    {
        g_ConstantBuffer.Initialize( &g_Device, info, NULL, 0, 0 );
    }
}

// テクスチャーを初期化
nn::gfx::Texture g_Texture;
void InitializeLinearTexture()
{
    nn::gfx::Texture::InfoType info;
    info.SetDefault();
    info.SetGpuAccessFlags( nn::gfx::GpuAccess_Texture );
    info.SetWidth( 256 );
    info.SetHeight( 256 );
    info.SetImageStorageDimension( nn::gfx::ImageStorageDimension_2d );
    info.SetImageFormat( nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm );
    info.SetMipCount( 1 );
    info.SetTileMode( nn::gfx::TileMode_Linear );

    size_t alignment = nn::gfx::Texture::CalculateMipDataAlignment( &g_Device, info );
    size_t size = nn::gfx::Texture::CalculateMipDataSize( &g_Device, info );
    size_t pitch = nn::gfx::Texture::GetRowPitch( &g_Device, info );
    ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuCached_GpuCached ];
    offset = nn::util::align_up( offset, alignment );

    nn::gfx::MemoryPool& memoryPool = g_MemoryPool[ MemoryPoolType_CpuCached_GpuCached ];
    uint8_t* pixels = nn::util::BytePtr( memoryPool.Map(), offset ).Get< uint8_t >();
    for ( int y = 0; y < info.GetHeight(); ++y )
    {
        uint8_t* row = pixels + y * pitch;
        for ( int x = 0; x < info.GetWidth(); ++x )
        {
            uint8_t* pixel = row + x * 4;
            pixel[ 0 ] = static_cast< uint8_t >( x );
            pixel[ 1 ] = static_cast< uint8_t >( y );
            pixel[ 2 ] = 0x00;
            pixel[ 3 ] = 0xFF;
        }
    }
    memoryPool.FlushMappedRange( offset, size );
    memoryPool.Unmap();

    g_Texture.Initialize( &g_Device, info, &memoryPool, offset, size );
    offset += size;
};

// テクスチャービューを初期化
nn::gfx::TextureView g_TextureView;
void InitializeLinearTextureView()
{
    nn::gfx::TextureView::InfoType info;
    info.SetDefault();
    info.SetImageDimension( nn::gfx::ImageDimension_2d );
    info.SetImageFormat( nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm );
    info.SetTexturePtr( &g_Texture );
    info.SetChannelMapping( nn::gfx::ChannelMapping_Red, nn::gfx::ChannelMapping_Green,
        nn::gfx::ChannelMapping_Blue, nn::gfx::ChannelMapping_Alpha );
    g_TextureView.Initialize( &g_Device, info );
}

// サンプラーを初期化
nn::gfx::Sampler g_Sampler;
void InitializeSampler()
{
    nn::gfx::Sampler::InfoType info;
    info.SetDefault();
    info.SetFilterMode( nn::gfx::FilterMode_MinLinear_MagLinear_MipPoint );
    info.SetAddressU( nn::gfx::TextureAddressMode_Mirror );
    info.SetAddressV( nn::gfx::TextureAddressMode_Mirror );
    info.SetAddressW( nn::gfx::TextureAddressMode_Mirror );
    g_Sampler.Initialize( &g_Device, info );
}

nn::gfx::DescriptorPool g_BufferDescriptorPool;
void InitializeBufferDescriptorPool()
{
    nn::gfx::DescriptorPool::InfoType info;
    info.SetDefault();
    info.SetDescriptorPoolType( nn::gfx::DescriptorPoolType_BufferView );
    info.SetSlotCount( g_BufferDescriptorBaseIndex + 1 );
    size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize( &g_Device, info );
    ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuUncached_GpuCached ];
    offset = nn::util::align_up( offset, nn::gfx::DescriptorPool::GetDescriptorPoolAlignment( &g_Device, info ) );
    g_BufferDescriptorPool.Initialize( &g_Device, info,
        &g_MemoryPool[ MemoryPoolType_CpuUncached_GpuCached ], offset, size );
    offset += size;
}

nn::gfx::DescriptorPool g_SamplerDescriptorPool;
void InitializeSamplerDescriptorPool()
{
    nn::gfx::DescriptorPool::InfoType info;
    info.SetDefault();
    info.SetDescriptorPoolType( nn::gfx::DescriptorPoolType_Sampler );
    info.SetSlotCount( g_SamplerDescriptorBaseIndex + 1 );
    size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize( &g_Device, info );
    ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuUncached_GpuCached ];
    offset = nn::util::align_up( offset, nn::gfx::DescriptorPool::GetDescriptorPoolAlignment( &g_Device, info ) );
    g_SamplerDescriptorPool.Initialize( &g_Device, info,
        &g_MemoryPool[ MemoryPoolType_CpuUncached_GpuCached ], offset, size );
    offset += size;
}

nn::gfx::DescriptorPool g_TextureDescriptorPool;
void InitializeTextureDescriptorPool()
{
    nn::gfx::DescriptorPool::InfoType info;
    info.SetDefault();
    info.SetDescriptorPoolType( nn::gfx::DescriptorPoolType_TextureView );
    info.SetSlotCount( g_TextureDescriptorBaseIndex + 2 );
    size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize( &g_Device, info );
    ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuUncached_GpuCached ];
    offset = nn::util::align_up( offset, nn::gfx::DescriptorPool::GetDescriptorPoolAlignment( &g_Device, info ) );
    g_TextureDescriptorPool.Initialize( &g_Device, info,
        &g_MemoryPool[ MemoryPoolType_CpuUncached_GpuCached ], offset, size );
    offset += size;
}

nn::gfx::Fence g_Fence;
void InitializeFence()
{
    nn::gfx::Fence::InfoType info;
    info.SetDefault();
    g_Fence.Initialize( &g_Device, info );
}

nn::gfx::Semaphore g_DisplaySemaphore;
void InitializeSemaphore()
{
    nn::gfx::Semaphore::InfoType info;
    info.SetDefault();
    g_DisplaySemaphore.Initialize( &g_Device, info );
}

void InitializeGfxObjects()
{
    // メモリー
    g_pMemoryHeap.Reset( malloc( 1024 * 1024 ) );
    g_pMemory = g_pMemoryHeap;

    InitializeDevice();

#if NN_GFX_IS_TARGET_NVN
    nn::gfx::Device::DataType& deviceData = nn::gfx::AccessorToData( g_Device );
    nvnDeviceGetInteger( deviceData.pNvnDevice,
        NVN_DEVICE_INFO_RESERVED_TEXTURE_DESCRIPTORS, &g_TextureDescriptorBaseIndex );
    nvnDeviceGetInteger( deviceData.pNvnDevice,
        NVN_DEVICE_INFO_RESERVED_SAMPLER_DESCRIPTORS, &g_SamplerDescriptorBaseIndex );
#endif

    InitializeMemoryPool();

    InitializeSwapChain();
    InitializeQueue();

    InitializeQueueCommandBuffer();
    InitializeCommandBuffer();
    InitializeViewport();
    InitializeVertexShader();
    InitializePixelShader();

    if ( NN_STATIC_CONDITION( IsPipelineMode ) )
    {
        InitializePipeline();
    }
    else
    {
        InitializeRasterizerState();
        InitializeBlendState();
        InitializeDepthStencilState();
        InitializeVertexState();
    }

    InitializeVertexBuffer();
    InitializeIndexBuffer();
    InitializeDepthBuffer();
    InitializeDepthBufferView();
    InitializeConstantBuffer();

    InitializeLinearTexture();
    InitializeLinearTextureView();
    InitializeSampler();

    if( NN_STATIC_CONDITION( IsDescriptorPoolMode ) )
    {
        InitializeBufferDescriptorPool();
        InitializeSamplerDescriptorPool();
        InitializeTextureDescriptorPool();
    }

    InitializeFence();
    InitializeSemaphore();

    NN_ASSERT( g_pMemoryHeap.Distance( g_pMemory.Get() ) < 1024 * 1024 );
    for( int idxMemoryPool = 0; idxMemoryPool < MemoryPoolType_End; ++idxMemoryPool )
    {
        NN_ASSERT( static_cast< size_t >( g_MemoryPoolOffset[ idxMemoryPool ] )
            < MemoryPoolSize[ idxMemoryPool ] );
    }
}

void OutOfCommandMemoryEventCallback(
    nn::gfx::CommandBuffer* pCommandBuffer, const nn::gfx::OutOfMemoryEventArg& arg )
{
    NN_ASSERT( arg.minRequiredSize <= 1024 );
    NN_UNUSED( arg );
    ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuUncached_GpuCached ];
    offset = nn::util::align_up( offset, nn::gfx::CommandBuffer::GetCommandMemoryAlignment( &g_Device ) );
    pCommandBuffer->AddCommandMemory(
        &g_MemoryPool[ MemoryPoolType_CpuUncached_GpuCached ], offset, 1024 );
    offset += 1024;
}

void OutOfControlMemoryEventCallback(
    nn::gfx::CommandBuffer* pCommandBuffer, const nn::gfx::OutOfMemoryEventArg& arg )
{
    NN_ASSERT( arg.minRequiredSize <= 256 );
    NN_UNUSED( arg );
    g_pMemory.AlignUp( nn::gfx::CommandBuffer::GetControlMemoryAlignment( &g_Device ) );
    pCommandBuffer->AddControlMemory( g_pMemory.Get(), 256 );
    g_pMemory.Advance( 256 );
}

void MakeCommand()
{
    nn::gfx::GpuAddress gpuAddress;

    g_CommandBuffer.Reset();
    {
        ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuUncached_GpuCached ];
        offset = nn::util::align_up( offset, nn::gfx::CommandBuffer::GetCommandMemoryAlignment( &g_Device ) );
        g_CommandBuffer.AddCommandMemory(
            &g_MemoryPool[ MemoryPoolType_CpuUncached_GpuCached ], offset, 1024 );
        offset += 1024;
        g_pMemory.AlignUp( nn::gfx::CommandBuffer::GetControlMemoryAlignment( &g_Device ) );
        g_CommandBuffer.AddControlMemory( g_pMemory.Get(), 256 );
        g_pMemory.Advance( 256 );
        g_CommandBuffer.SetOutOfCommandMemoryEventCallback( OutOfCommandMemoryEventCallback );
        g_CommandBuffer.SetOutOfControlMemoryEventCallback( OutOfControlMemoryEventCallback );
    }
    g_CommandBuffer.Begin();
    {
        g_VertexBuffer.GetGpuAddress( &gpuAddress );
        g_CommandBuffer.SetVertexBuffer( 0, gpuAddress, sizeof( float ) * 5, sizeof( Vertices ) );

        g_CommandBuffer.InvalidateMemory( nn::gfx::GpuAccess_ShaderCode | nn::gfx::GpuAccess_Descriptor );

        if( NN_STATIC_CONDITION( IsDescriptorPoolMode ) )
        {
            g_CommandBuffer.SetDescriptorPool( &g_BufferDescriptorPool );
            g_CommandBuffer.SetDescriptorPool( &g_TextureDescriptorPool );
            g_CommandBuffer.SetDescriptorPool( &g_SamplerDescriptorPool );
        }

#if NN_GFX_IS_TARGET_GX
        GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_BLOCK );
#endif

        g_CommandBuffer.SetViewportScissorState( &g_ViewportScissor );

        if ( NN_STATIC_CONDITION( IsPipelineMode ) )
        {
            g_CommandBuffer.SetPipeline( &g_Pipeline );
        }
        else
        {
            g_CommandBuffer.SetShader( &g_VertexShader, nn::gfx::ShaderStageBit_Vertex );
            g_CommandBuffer.SetShader( &g_PixelShader, nn::gfx::ShaderStageBit_Pixel );
            g_CommandBuffer.SetRasterizerState( &g_RasterizerState );
            g_CommandBuffer.SetBlendState( &g_BlendState );
            g_CommandBuffer.SetDepthStencilState( &g_DepthStencilState );
            g_CommandBuffer.SetVertexState( &g_VertexState );
        }

#if defined( CAFE )
        GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_BLOCK );
#endif

        g_CommandBuffer.SetBufferStateTransition( &g_ConstantBuffer,
            nn::gfx::BufferState_DataTransfer, 0, nn::gfx::BufferState_ConstantBuffer, nn::gfx::PipelineStageBit_PixelShader );
        g_CommandBuffer.SetBufferStateTransition( &g_VertexBuffer,
            nn::gfx::BufferState_DataTransfer, 0, nn::gfx::BufferState_VertexBuffer, 0 );
        g_CommandBuffer.SetBufferStateTransition( &g_IndexBuffer,
            nn::gfx::BufferState_DataTransfer, 0, nn::gfx::BufferState_IndexBuffer, 0 );
        g_CommandBuffer.SetTextureStateTransition( &g_Texture, NULL,
            nn::gfx::TextureState_DataTransfer, 0, nn::gfx::TextureState_ShaderRead, nn::gfx::PipelineStageBit_PixelShader );
        g_CommandBuffer.SetTextureStateTransition( &g_DepthStencilBuffer, NULL,
            nn::gfx::TextureState_DataTransfer, 0, nn::gfx::TextureState_DepthRead | nn::gfx::TextureState_DepthWrite, 0 );

        if( NN_STATIC_CONDITION( IsDescriptorPoolMode ) )
        {
            nn::gfx::DescriptorSlot constantBufferDescriptor;
            nn::gfx::DescriptorSlot textureDescriptor;
            nn::gfx::DescriptorSlot samplerDescriptor;
            g_BufferDescriptorPool.GetDescriptorSlot( &constantBufferDescriptor, g_BufferDescriptorBaseIndex );
            g_TextureDescriptorPool.GetDescriptorSlot( &textureDescriptor, g_TextureDescriptorBaseIndex );
            g_SamplerDescriptorPool.GetDescriptorSlot( &samplerDescriptor, g_SamplerDescriptorBaseIndex );
            textureDescriptor.Offset( nn::gfx::DescriptorPool::GetDescriptorSlotIncrementSize(
                &g_Device, nn::gfx::DescriptorPoolType_TextureView ) );
            g_CommandBuffer.SetConstantBuffer( g_SlotMat,
                nn::gfx::ShaderStage_Pixel, constantBufferDescriptor );
            g_CommandBuffer.SetTextureAndSampler( g_SlotTex,
                nn::gfx::ShaderStage_Pixel, textureDescriptor, samplerDescriptor );
        }
        else
        {
            g_ConstantBuffer.GetGpuAddress( &gpuAddress );
            g_CommandBuffer.SetConstantBuffer( g_SlotMat,
                nn::gfx::ShaderStage_Pixel, gpuAddress, sizeof( float ) * 4 );
            g_CommandBuffer.SetTextureAndSampler( g_SlotTex,
                nn::gfx::ShaderStage_Pixel, &g_TextureView, &g_Sampler );
        }

        g_IndexBuffer.GetGpuAddress( &gpuAddress );
        g_CommandBuffer.DrawIndexed( nn::gfx::PrimitiveTopology_TriangleList,
            nn::gfx::IndexFormat_Uint32, gpuAddress, 3, 0 );
    }
    g_CommandBuffer.End();

    nn::gfx::ColorTargetView* pScanBufferViews[ ScanBufferCount ];
    g_SwapChain.GetScanBufferViews( pScanBufferViews, ScanBufferCount );
    for( int idxCommandBuffer = 0; idxCommandBuffer < ScanBufferCount; ++idxCommandBuffer )
    {
        nn::gfx::CommandBuffer& commandBuffer = g_QueueCommandBuffer[ idxCommandBuffer ];
        nn::gfx::ColorTargetView* pScanBuffer = pScanBufferViews[ idxCommandBuffer ];
        commandBuffer.Reset();
        ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuUncached_GpuCached ];
        offset = nn::util::align_up( offset,
            nn::gfx::CommandBuffer::GetCommandMemoryAlignment( &g_Device ) );
        commandBuffer.AddCommandMemory(
            &g_MemoryPool[ MemoryPoolType_CpuUncached_GpuCached ], offset, 1024 );
        offset += 1024;
        g_pMemory.AlignUp( nn::gfx::CommandBuffer::GetControlMemoryAlignment( &g_Device ) );
        commandBuffer.AddControlMemory( g_pMemory.Get(), 256 );
        g_pMemory.Advance( 256 );
        commandBuffer.Begin();
        {
            commandBuffer.ClearColor( pScanBuffer, 0.3f, 0.1f, 0.1f, 1.0f, NULL );
            commandBuffer.ClearDepthStencil( &g_DepthStencilView,
                1.0f, 0, nn::gfx::DepthStencilClearMode_DepthStencil, NULL );
            commandBuffer.SetRenderTargets( 1, &pScanBuffer, &g_DepthStencilView );
            commandBuffer.CallCommandBuffer( &g_CommandBuffer );
        }
        commandBuffer.End();
    }

} // NOLINT

void FinalizeGfxObjects()
{
    // 各オブジェクトを破棄
    g_DisplaySemaphore.Finalize( &g_Device );
    g_Fence.Finalize( &g_Device );
    if( NN_STATIC_CONDITION( IsDescriptorPoolMode ) )
    {
        g_BufferDescriptorPool.Finalize( &g_Device );
        g_TextureDescriptorPool.Finalize( &g_Device );
        g_SamplerDescriptorPool.Finalize( &g_Device );
    }
    g_Sampler.Finalize( &g_Device );
    g_TextureView.Finalize( &g_Device );
    g_Texture.Finalize( &g_Device );
    g_ConstantBuffer.Finalize( &g_Device );
    g_DepthStencilView.Finalize( &g_Device );
    g_DepthStencilBuffer.Finalize( &g_Device );
    g_IndexBuffer.Finalize( &g_Device );
    g_VertexBuffer.Finalize( &g_Device );
    if( NN_STATIC_CONDITION( IsPipelineMode ) )
    {
        g_Pipeline.Finalize( &g_Device );
        g_Shader.Finalize( &g_Device );
    }
    else
    {
        g_RasterizerState.Finalize( &g_Device );
        g_BlendState.Finalize( &g_Device );
        g_DepthStencilState.Finalize( &g_Device );
        g_VertexState.Finalize( &g_Device );
    }

    g_PixelShader.Finalize( &g_Device );
    g_VertexShader.Finalize( &g_Device );
    g_ViewportScissor.Finalize( &g_Device );
    g_CommandBuffer.Finalize( &g_Device );
    for( int idxCommandBuffer = 0; idxCommandBuffer < ScanBufferCount; ++idxCommandBuffer )
    {
        g_QueueCommandBuffer[ idxCommandBuffer ].Finalize( &g_Device );
    }
    g_Queue.Finalize( &g_Device );
    g_SwapChain.Finalize( &g_Device );
    for( int idxMemoryPool = 0; idxMemoryPool < MemoryPoolType_End; ++idxMemoryPool )
    {
        g_MemoryPool[ idxMemoryPool ].Finalize( &g_Device );
        free( g_pPoolMemory[ idxMemoryPool ] );
    }
    g_Device.Finalize();

    free( g_pMemoryHeap.Get() );
}

};

//
//  Main Function
//  メイン関数です。
//

#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN ) && defined( NN_BUILD_APISET_NX )
void* Allocate( size_t size, size_t alignment, void* )
{
    return aligned_alloc( alignment, nn::util::align_up(size , alignment));
}
void Free( void* addr, void* )
{
    free( addr );
}
void* Reallocate( void* addr, size_t newSize, void* )
{
    return realloc( addr, newSize );
}
#endif

extern "C" void nnMain()
{
#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN ) && defined( NN_BUILD_APISET_NX )
    // グラフィックスシステムのためのメモリー周りの初期化を行います。
    {
        nv::SetGraphicsAllocator( Allocate, Free, Reallocate, NULL );
        nv::InitializeGraphics( malloc( GraphicsSystemMemorySize ), GraphicsSystemMemorySize );
    }
    // グラフィックス開発者向けツールおよびデバッグレイヤのためのメモリーアロケーターを設定します。
    nv::SetGraphicsDevtoolsAllocator( Allocate, Free, Reallocate, NULL );
    // GLSLC のためのメモリーアロケーターを設定します。
    glslcSetAllocator( Allocate, Free, Reallocate, NULL );
#endif

    nn::vi::Initialize();
    InitializeLayer();

    // ライブラリーを初期化
    nn::gfx::Initialize();
    InitializeGfxObjects();

    if( NN_STATIC_CONDITION( IsDescriptorPoolMode ) )
    {
        g_BufferDescriptorPool.BeginUpdate();
        {
            nn::gfx::GpuAddress gpuAddress;
            g_ConstantBuffer.GetGpuAddress( &gpuAddress );
            g_BufferDescriptorPool.SetBufferView( g_BufferDescriptorBaseIndex, gpuAddress, sizeof( float ) * 4 );
        }
        g_BufferDescriptorPool.EndUpdate();

        g_TextureDescriptorPool.BeginUpdate();
        {
            g_TextureDescriptorPool.SetTextureView( g_TextureDescriptorBaseIndex + 1, &g_TextureView );
        }
        g_TextureDescriptorPool.EndUpdate();

        g_SamplerDescriptorPool.BeginUpdate();
        {
            g_SamplerDescriptorPool.SetSampler( g_SamplerDescriptorBaseIndex, &g_Sampler );
        }
        g_SamplerDescriptorPool.EndUpdate();
    }

    MakeCommand();

    // 毎フレームのレンダリング
    for( int frame = 0; frame < 60 * 10; ++frame )
    {
        // 定数バッファーを更新
        float* pConstantBuffer = g_ConstantBuffer.Map< float >();
        {
            pConstantBuffer[ 0 ] =  ( ( frame & 0x0100 ) - ( frame & 0xFF ) *
                ( ( ( frame & 0x0100 ) >> 7 ) - 1 ) ) * ( 1.0f / 256.0f );
            pConstantBuffer[ 1 ] = 0.4f;
            pConstantBuffer[ 2 ] = 0.4f;
            pConstantBuffer[ 3 ] = 1.0f;
        }
#ifdef CAFE
        GX2EndianSwap( pConstantBuffer, sizeof( float ) * 4 );
#endif
        g_ConstantBuffer.Unmap();

        // スキャンバッファーの取得
        int nextScanBufferIndex = 0;
        nn::gfx::AcquireScanBufferResult acquireResult = g_SwapChain.AcquireNextScanBufferIndex(
            &nextScanBufferIndex, &g_DisplaySemaphore, NULL );
        NN_ASSERT( acquireResult == nn::gfx::AcquireScanBufferResult_Success );
        NN_UNUSED( acquireResult );
        g_Queue.SyncSemaphore( &g_DisplaySemaphore );

        // コマンドの実行
        g_Queue.ExecuteCommand( &g_QueueCommandBuffer[ nextScanBufferIndex ], &g_Fence );

        // 結果の表示
        g_Queue.Present( &g_SwapChain, 1 );

        g_Queue.Flush();

        // 単純さのため同じフレームのコマンドを待機しています。
        // GPU リソースをダブルバッファー化して CPU と GPU を並列に動かすとより効率的になります。
        g_Fence.Sync( nn::TimeSpan::FromSeconds( 1 ) );
    }

    g_Queue.Sync();

    FinalizeGfxObjects();
    nn::gfx::Finalize();

    nn::vi::DestroyLayer( g_pLayer );
    nn::vi::CloseDisplay( g_pDisplay );
    nn::vi::Finalize();
}
