﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstdlib>
#include <cstring>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>

#include <nn/os.h>
#include <nn/fs.h>
#include <nn/init.h>
#include <nn/vi.h>
#include <nn/gfx.h>
#include <nn/image/image_JpegDecoder.h>

#if NN_GFX_IS_TARGET_NVN
    #include <nvn/nvn.h>
    #include <nvn/nvn_FuncPtrInline.h>
#endif

#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN ) && defined( NN_BUILD_APISET_NX )
    #include <nv/nv_MemoryManagement.h>
#endif


namespace {

const int RenderWidth = 1280;
const int RenderHeight = 720;

// 頂点
// (x, y, z; u, v) を頂点数分並べる
const float Vertices[] =
{
    -1.0f, -1.0f, 0.0f, 0.0f, 0.0f,
    -1.0f, +1.0f, 0.0f, 0.0f, 1.0f,
    +1.0f, -1.0f, 0.0f, 1.0f, 0.0f,
    +1.0f, +1.0f, 0.0f, 1.0f, 1.0f,
};

// インデクス
// (i0, i1, i2) をポリゴン数分並べる
const int Indices[] =
{
    0, 2, 1,
    2, 3, 1,
};

int g_BufferDescriptorBaseIndex = 0;
int g_TextureDescriptorBaseIndex = 0;
int g_SamplerDescriptorBaseIndex = 0;

//-----------------------------------------------------------------------------
// メモリ

const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;

nn::util::BytePtr g_pMemoryHeap( NULL );
nn::util::BytePtr g_pMemory( NULL );

enum MemoryPoolType
{
    MemoryPoolType_CpuUncached_GpuCached,
    MemoryPoolType_CpuInvisible_GpuCached_Compressible,

    MemoryPoolType_End
};

const size_t MemoryPoolSize[ MemoryPoolType_End ] =
{
    64 * 1024 * 1024,
    64 * 1024 * 1024
};

void* g_pPoolMemory[ MemoryPoolType_End ] = {};
ptrdiff_t g_MemoryPoolOffset[ MemoryPoolType_End ] = {};

size_t g_MaxScratchMemory = 0;
ptrdiff_t g_ScratchMemoryOffset = 0;

//-----------------------------------------------------------------------------
// 画像ファイルのパス
const char* g_ImageSourcePath = "Contents:/Image.jpg";
// テクスチャの幅の最小単位[px]。読み込んだ画像の幅はこの値の整数倍までパディングが詰められる。
const int g_ImageLineAlignment = 1;

uint8_t* g_pImageData = NULL;
size_t g_ImageDataSize = 0;
nn::image::Dimension g_ImageDimension = {};
int g_ImageStride = 0;

//-----------------------------------------------------------------------------

void* ReadResource( const char* filename )
{
    nn::Result result;
    nn::fs::FileHandle hFile;

    int64_t fileSize = 0;
    result = nn::fs::OpenFile( &hFile, filename, nn::fs::OpenMode_Read );
    NN_ASSERT( result.IsSuccess() );

    result = nn::fs::GetFileSize( &fileSize, hFile );
    NN_ASSERT( result.IsSuccess() );

    nn::util::BinaryFileHeader fileHeader;
    size_t readSize;
    result = nn::fs::ReadFile( &readSize, hFile, 0, &fileHeader, sizeof( nn::util::BinaryFileHeader ) );
    NN_ASSERT( result.IsSuccess() );
    NN_ASSERT( readSize == sizeof( nn::util::BinaryFileHeader ) );
    size_t alignment = fileHeader.GetAlignment();

    g_pMemory.AlignUp( alignment );
    void* pBuffer = g_pMemory.Get();
    result = nn::fs::ReadFile( &readSize, hFile, 0, pBuffer, static_cast< size_t >( fileSize ) );
    NN_ASSERT( result.IsSuccess() );
    NN_ASSERT( readSize == static_cast< size_t >( fileSize ) );
    g_pMemory.Advance( static_cast< ptrdiff_t >( fileSize ) );

    nn::fs::CloseFile( hFile );

    return pBuffer;
}

void InitializeImage()
{
    // ファイルから読み込み
    NN_LOG("Load JpegImage\n");
    void* pCompressed = NULL;
    size_t compressedSize = 0;
    {
        nn::Result result;
        nn::fs::FileHandle file = {};
        result = nn::fs::OpenFile(&file, g_ImageSourcePath, nn::fs::OpenMode_Read);
        NN_ASSERT(result.IsSuccess());
        int64_t fileSize = 0;
        result = nn::fs::GetFileSize(&fileSize, file);
        NN_ASSERT(result.IsSuccess());
        compressedSize = static_cast<size_t>(fileSize);
        pCompressed = malloc(compressedSize);
        NN_ASSERT_NOT_NULL(pCompressed);
        result = nn::fs::ReadFile(file, 0, pCompressed, compressedSize);
        NN_ASSERT(result.IsSuccess());
        nn::fs::CloseFile(file);
    }
    // 画像を展開
    {
        nn::image::JpegStatus result;
        nn::image::JpegDecoder decoder;
        decoder.SetImageData(pCompressed, compressedSize);
        result = decoder.Analyze();
        NN_ASSERT(result == nn::image::JpegStatus_Ok);
        size_t workSize = decoder.GetAnalyzedWorkBufferSize();
        void* workBuffer = malloc(workSize);
        NN_ASSERT_NOT_NULL(workBuffer);
        g_ImageDimension = decoder.GetAnalyzedDimension();
        g_ImageStride = 4 * g_ImageLineAlignment * ((g_ImageDimension.width + g_ImageLineAlignment - 1) / g_ImageLineAlignment);
        g_ImageDataSize = g_ImageStride * g_ImageDimension.height;
        g_pImageData = reinterpret_cast<uint8_t*>(malloc(g_ImageDataSize));
        memset(g_pImageData, 0, g_ImageDataSize);
        result = decoder.Decode(g_pImageData, g_ImageDataSize, g_ImageLineAlignment, workBuffer, workSize);
        NN_ASSERT(result == nn::image::JpegStatus_Ok);
        free(workBuffer);
        workBuffer = NULL;
        workSize = 0;
    }
    free(pCompressed);
    NN_LOG("Image  W%d x H%d\n", g_ImageDimension.width, g_ImageDimension.height);
}
void FinalizeImage()
{
    free(g_pImageData);
    g_pImageData = NULL;
    g_ImageDataSize = 0;
}

// レイヤを初期化
nn::vi::Display* g_pDisplay;
nn::vi::Layer* g_pLayer;
void InitializeLayer()
{
    nn::Result result = nn::vi::OpenDefaultDisplay( &g_pDisplay );
    NN_ASSERT( result.IsSuccess() );
    NN_UNUSED( result );

    result = nn::vi::CreateLayer( &g_pLayer, g_pDisplay );
    NN_ASSERT( result.IsSuccess() );

    result = nn::vi::SetLayerScalingMode( g_pLayer, nn::vi::ScalingMode_FitToLayer );
    NN_ASSERT( result.IsSuccess() );
}

// デバイスを初期化
nn::gfx::Device g_Device;
void InitializeDevice()
{
    nn::gfx::Device::InfoType info;
    info.SetDefault();
    info.SetApiVersion( nn::gfx::ApiMajorVersion, nn::gfx::ApiMinorVersion );
    g_Device.Initialize( info );
}

nn::gfx::MemoryPool g_MemoryPool[ MemoryPoolType_End ];
void InitializeMemoryPool()
{
    const int MemoryPoolProperty[ MemoryPoolType_End ] =
    {
        nn::gfx::MemoryPoolProperty_CpuUncached | nn::gfx::MemoryPoolProperty_GpuCached,
        nn::gfx::MemoryPoolProperty_CpuInvisible | nn::gfx::MemoryPoolProperty_GpuCached
            | nn::gfx::MemoryPoolProperty_Compressible
    };

    nn::gfx::MemoryPool::InfoType info;
    for( int idxMemoryPool = 0; idxMemoryPool < MemoryPoolType_End; ++idxMemoryPool )
    {
        info.SetDefault();
        info.SetMemoryPoolProperty( MemoryPoolProperty[ idxMemoryPool ] );
        size_t alignment = nn::gfx::MemoryPool::GetPoolMemoryAlignment( &g_Device, info );
        size_t granularity = nn::gfx::MemoryPool::GetPoolMemorySizeGranularity( &g_Device, info );
        g_pPoolMemory[ idxMemoryPool ] = malloc( MemoryPoolSize[ idxMemoryPool ] );
        void* pPoolMemoryAligned = nn::util::BytePtr(
            g_pPoolMemory[ idxMemoryPool ] ).AlignUp( alignment ).Get();
        size_t memoryPoolSizeAligned = nn::util::align_down( MemoryPoolSize[ idxMemoryPool ], granularity );
        info.SetPoolMemory( pPoolMemoryAligned, memoryPoolSizeAligned );
        g_MemoryPool[ idxMemoryPool ].Initialize( &g_Device, info );
        g_MemoryPoolOffset[ idxMemoryPool ] = 0;
    }
}

// スワップチェーンを初期化
nn::gfx::SwapChain g_SwapChain;
void InitializeSwapChain()
{
    nn::gfx::SwapChain::InfoType info;

    info.SetDefault();
    info.SetLayer( g_pLayer );
    info.SetWidth( RenderWidth );
    info.SetHeight( RenderHeight );
    info.SetFormat( nn::gfx::ImageFormat_R8_G8_B8_A8_UnormSrgb );
    info.SetBufferCount( 2 );
    if( NN_STATIC_CONDITION( nn::gfx::SwapChain::IsMemoryPoolRequired ) )
    {
        ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuInvisible_GpuCached_Compressible ];
        size_t size = g_SwapChain.CalculateScanBufferSize( &g_Device, info );
        offset = nn::util::align_up( offset, nn::gfx::SwapChain::GetScanBufferAlignment( &g_Device, info ) );
        g_SwapChain.Initialize( &g_Device, info,
            &g_MemoryPool[ MemoryPoolType_CpuInvisible_GpuCached_Compressible ], offset, size );
        offset += size;
    }
    else
    {
        g_SwapChain.Initialize( &g_Device, info, NULL, 0, 0 );
    }
}

// キューを初期化
nn::gfx::Queue g_Queue;
void InitializeQueue()
{
    nn::gfx::Queue::InfoType info;
    info.SetDefault();
    info.SetCapability( nn::gfx::QueueCapability_Graphics );
    g_Queue.Initialize( &g_Device, info );
}

// コマンドバッファを初期化
nn::gfx::CommandBuffer g_CommandBuffer;
void InitializeCommandBuffer()
{
    nn::gfx::CommandBuffer::InfoType info;
    info.SetDefault();
    info.SetQueueCapability( nn::gfx::QueueCapability_Graphics );
    info.SetCommandBufferType( nn::gfx::CommandBufferType_Direct );
    g_CommandBuffer.Initialize( &g_Device, info );
}

// ビューポートシザーを初期化
nn::gfx::ViewportScissorState g_ViewportScissor;
void InitializeViewport()
{
    nn::gfx::ViewportScissorState::InfoType info;
    info.SetDefault();
    info.SetScissorEnabled( true );
    nn::gfx::ViewportStateInfo viewportInfo;
    {
        viewportInfo.SetDefault();
        viewportInfo.SetWidth( static_cast< float >( RenderWidth ) );
        viewportInfo.SetHeight( static_cast< float >( RenderHeight ) );
    }
    nn::gfx::ScissorStateInfo scissorInfo;
    {
        scissorInfo.SetDefault();
        scissorInfo.SetWidth( RenderWidth );
        scissorInfo.SetHeight( RenderHeight );
    }
    info.SetViewportStateInfoArray( &viewportInfo, 1 );
    info.SetScissorStateInfoArray( &scissorInfo, 1 );
    g_ViewportScissor.Initialize( &g_Device, info );
}

// Initialize the rasterizer state
nn::gfx::RasterizerState g_RasterizerState;
static void InitializeRasterizerState()
{
    nn::gfx::RasterizerState::InfoType info;
    info.SetDefault();
    info.SetCullMode( nn::gfx::CullMode_None );
    info.SetPrimitiveTopologyType( nn::gfx::PrimitiveTopologyType_Triangle );
    info.SetScissorEnabled( true );
    info.SetDepthClipEnabled( false );
    g_RasterizerState.Initialize( &g_Device, info );
}

// Initialize the blend state
nn::gfx::BlendState g_BlendState;
static void InitializeBlendState()
{
    nn::gfx::BlendState::InfoType info;
    info.SetDefault();
    nn::gfx::BlendTargetStateInfo targetInfo;
    {
        targetInfo.SetDefault();
    };
    info.SetBlendTargetStateInfoArray( &targetInfo, 1 );
    size_t size = nn::gfx::BlendState::GetRequiredMemorySize( info );
    g_pMemory.AlignUp( nn::gfx::BlendState::RequiredMemoryInfo_Alignment );
    g_BlendState.SetMemory( g_pMemory.Get(), size );
    g_pMemory.Advance( size );
    g_BlendState.Initialize( &g_Device, info );
}

// Initialize the depth stencil state
nn::gfx::DepthStencilState g_DepthStencilState;
static void InitializeDepthStencilState()
{
    nn::gfx::DepthStencilState::InfoType info;
    info.SetDefault();
    info.SetDepthTestEnabled( false );
    info.SetDepthWriteEnabled( false );
    g_DepthStencilState.Initialize( &g_Device, info );
}

// Initialize the vertex state
nn::gfx::VertexState g_VertexState;
static void InitializeVertexState()
{
    nn::gfx::VertexState::InfoType info;
    info.SetDefault();
    uint16_t stride = sizeof( float ) * 5;
    nn::gfx::VertexAttributeStateInfo attribs[ 2 ];
    {
        // i_Position
        attribs[ 0 ].SetDefault();
        attribs[ 0 ].SetBufferIndex( 0 );
        attribs[ 0 ].SetFormat( nn::gfx::AttributeFormat_32_32_32_Float );
        attribs[ 0 ].SetOffset( 0 );
        attribs[ 0 ].SetShaderSlot( 0 );
    }
    {
        // i_TexCoord
        attribs[ 1 ].SetDefault();
        attribs[ 1 ].SetBufferIndex( 0 );
        attribs[ 1 ].SetFormat( nn::gfx::AttributeFormat_32_32_Float );
        attribs[ 1 ].SetOffset( sizeof( float ) * 3 );
        attribs[ 1 ].SetShaderSlot( 1 );
    }
    nn::gfx::VertexBufferStateInfo buffer;
    {
        buffer.SetDefault();
        buffer.SetStride( stride );
    }
    info.SetVertexAttributeStateInfoArray( attribs, 2 );
    info.SetVertexBufferStateInfoArray( &buffer, 1 );
    size_t size = nn::gfx::VertexState::GetRequiredMemorySize( info );
    g_pMemory.AlignUp( nn::gfx::VertexState::RequiredMemoryInfo_Alignment );
    g_VertexState.SetMemory( g_pMemory.Get(), size );
    g_pMemory.Advance( size );
    g_VertexState.Initialize( &g_Device, info, NULL );
}

// 頂点用のバッファを初期化
nn::gfx::Buffer g_VertexBuffer;
void InitializeVertexBuffer()
{
    nn::gfx::Buffer::InfoType info;
    info.SetDefault();
    info.SetSize( sizeof( Vertices ) );
    info.SetGpuAccessFlags( nn::gfx::GpuAccess_VertexBuffer );
    if( NN_STATIC_CONDITION( nn::gfx::Buffer::IsMemoryPoolRequired ) )
    {
        ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuUncached_GpuCached ];
        offset = nn::util::align_up( offset, nn::gfx::Buffer::GetBufferAlignment( &g_Device, info ) );
        g_VertexBuffer.Initialize( &g_Device, info,
            &g_MemoryPool[ MemoryPoolType_CpuUncached_GpuCached ], offset, info.GetSize() );
        offset += info.GetSize();
    }
    else
    {
        g_VertexBuffer.Initialize( &g_Device, info, NULL, 0, 0 );
    }

    void* pMapped = g_VertexBuffer.Map();
    memcpy( pMapped, Vertices, info.GetSize() );
    g_VertexBuffer.Unmap();
}

// インデクス用のバッファを初期化
nn::gfx::Buffer g_IndexBuffer;
void InitializeIndexBuffer()
{
    nn::gfx::Buffer::InfoType info;
    info.SetDefault();
    info.SetSize( sizeof( Indices ) );
    info.SetGpuAccessFlags( nn::gfx::GpuAccess_IndexBuffer );
    if( NN_STATIC_CONDITION( nn::gfx::Buffer::IsMemoryPoolRequired ) )
    {
        ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuUncached_GpuCached ];
        offset = nn::util::align_up( offset, nn::gfx::Buffer::GetBufferAlignment( &g_Device, info ) );
        g_IndexBuffer.Initialize( &g_Device, info,
            &g_MemoryPool[ MemoryPoolType_CpuUncached_GpuCached ], offset, info.GetSize() );
        offset += info.GetSize();
    }
    else
    {
        g_IndexBuffer.Initialize( &g_Device, info, NULL, 0, 0 );
    }

    void* pMapped = g_IndexBuffer.Map();
    memcpy( pMapped, Indices, info.GetSize() );
    g_IndexBuffer.Unmap();
}

// 定数バッファ用のバッファを初期化
nn::gfx::Buffer g_ConstantBuffer;
void InitializeConstantBuffer()
{
    nn::gfx::Buffer::InfoType info;
    info.SetDefault();
    info.SetSize( sizeof( float ) * 4 );
    info.SetGpuAccessFlags( nn::gfx::GpuAccess_ConstantBuffer );
    if( NN_STATIC_CONDITION( nn::gfx::Buffer::IsMemoryPoolRequired ) )
    {
        ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuUncached_GpuCached ];
        offset = nn::util::align_up( offset, nn::gfx::Buffer::GetBufferAlignment( &g_Device, info ) );
        g_ConstantBuffer.Initialize( &g_Device, info,
            &g_MemoryPool[ MemoryPoolType_CpuUncached_GpuCached ], offset, info.GetSize() );
        offset += info.GetSize();
    }
    else
    {
        g_ConstantBuffer.Initialize( &g_Device, info, NULL, 0, 0 );
    }
}

// サンプラを初期化
nn::gfx::Sampler g_Sampler;
void InitializeSampler()
{
    nn::gfx::Sampler::InfoType info;
    info.SetDefault();
    info.SetFilterMode( nn::gfx::FilterMode_MinLinear_MagLinear_MipPoint );
    info.SetAddressU( nn::gfx::TextureAddressMode_Mirror );
    info.SetAddressV( nn::gfx::TextureAddressMode_Mirror );
    info.SetAddressW( nn::gfx::TextureAddressMode_Mirror );
    g_Sampler.Initialize( &g_Device, info );
}

nn::gfx::DescriptorPool g_BufferDescriptorPool;
void InitializeBufferDescriptorPool()
{
    nn::gfx::DescriptorPool::InfoType info;
    info.SetDefault();
    info.SetDescriptorPoolType( nn::gfx::DescriptorPoolType_BufferView );
    info.SetSlotCount( g_BufferDescriptorBaseIndex + 1 );
    size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize( &g_Device, info );
    ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuUncached_GpuCached ];
    offset = nn::util::align_up( offset, nn::gfx::DescriptorPool::GetDescriptorPoolAlignment( &g_Device, info ) );
    g_BufferDescriptorPool.Initialize( &g_Device, info,
        &g_MemoryPool[ MemoryPoolType_CpuUncached_GpuCached ], offset, size );
    offset += size;
}

nn::gfx::DescriptorPool g_SamplerDescriptorPool;
void InitializeSamplerDescriptorPool()
{
    nn::gfx::DescriptorPool::InfoType info;
    info.SetDefault();
    info.SetDescriptorPoolType( nn::gfx::DescriptorPoolType_Sampler );
    info.SetSlotCount( g_SamplerDescriptorBaseIndex + 1 );
    size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize( &g_Device, info );
    ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuUncached_GpuCached ];
    offset = nn::util::align_up( offset, nn::gfx::DescriptorPool::GetDescriptorPoolAlignment( &g_Device, info ) );
    g_SamplerDescriptorPool.Initialize( &g_Device, info,
        &g_MemoryPool[ MemoryPoolType_CpuUncached_GpuCached ], offset, size );
    offset += size;
}

nn::gfx::DescriptorPool g_TextureDescriptorPool;
void InitializeTextureDescriptorPool()
{
    nn::gfx::DescriptorPool::InfoType info;
    info.SetDefault();
    info.SetDescriptorPoolType( nn::gfx::DescriptorPoolType_TextureView );
    info.SetSlotCount( g_TextureDescriptorBaseIndex + 1 );
    size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize( &g_Device, info );
    ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuUncached_GpuCached ];
    offset = nn::util::align_up( offset, nn::gfx::DescriptorPool::GetDescriptorPoolAlignment( &g_Device, info ) );
    g_TextureDescriptorPool.Initialize( &g_Device, info,
        &g_MemoryPool[ MemoryPoolType_CpuUncached_GpuCached ], offset, size );
    offset += size;
}

nn::gfx::Texture g_Texture;
void InitializeTexture()
{
    nn::gfx::Texture::InfoType info;

    // Create temporary linear buffer
    nn::gfx::Buffer tempBuffer;
    size_t tempBufferSize = static_cast<size_t>(g_ImageStride * g_ImageDimension.height);
    {
        nn::gfx::MemoryPool& pool = g_MemoryPool[ MemoryPoolType_CpuUncached_GpuCached ];
        ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuUncached_GpuCached ];
        nn::gfx::Buffer::InfoType bufferInfo;
        bufferInfo.SetDefault();
        bufferInfo.SetGpuAccessFlags( nn::gfx::GpuAccess_Read );
        bufferInfo.SetSize( tempBufferSize );
        size_t alignment = nn::gfx::Buffer::GetBufferAlignment(&g_Device, bufferInfo );
        offset = nn::util::align_up(offset, alignment);
        tempBuffer.Initialize( &g_Device, bufferInfo, &pool, offset, tempBufferSize );
        offset += tempBufferSize;
    }

    // Configure the optimal texture
    info.SetDefault();
    info.SetGpuAccessFlags( nn::gfx::GpuAccess_Texture );
    info.SetWidth( g_ImageStride / 4 );
    info.SetHeight( g_ImageDimension.height );
    info.SetImageStorageDimension( nn::gfx::ImageStorageDimension_2d );
    info.SetImageFormat( nn::gfx::ImageFormat_R8_G8_B8_A8_UnormSrgb);
    info.SetMipCount( 1 );
    info.SetTileMode( nn::gfx::TileMode_Linear );

    // Allocate space for the optimal texture
    {
        nn::gfx::MemoryPool& pool = g_MemoryPool[ MemoryPoolType_CpuInvisible_GpuCached_Compressible ];
        ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuInvisible_GpuCached_Compressible ];
        offset = nn::util::align_up( offset,
            nn::gfx::Texture::CalculateMipDataAlignment( &g_Device, info ) );

        size_t size = nn::gfx::Texture::CalculateMipDataSize( &g_Device, info );
        g_Texture.Initialize( &g_Device, info, &pool, offset, size );
        offset += size;
    }

    // Generate the linear texture w/ CPU writes
    uint8_t* pixels = tempBuffer.Map< uint8_t >();
    for(int y = 0; y < g_ImageDimension.height; y++)
    {
        int revY = g_ImageDimension.height - 1 - y;
        memcpy(pixels + y * g_ImageStride, g_pImageData + revY * g_ImageStride, g_ImageStride);
    }
    tempBuffer.Unmap();

    // Use GPU to tile texture
    {
        const size_t CommandMemorySize = 1024;
        const size_t ControlMemorySize = 256;
        g_CommandBuffer.Reset();
        nn::gfx::MemoryPool& pool = g_MemoryPool[ MemoryPoolType_CpuUncached_GpuCached ];
        ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuUncached_GpuCached ];
        offset = nn::util::align_up( offset,
            nn::gfx::CommandBuffer::GetCommandMemoryAlignment( &g_Device ) );
        g_CommandBuffer.AddCommandMemory( &pool, offset, CommandMemorySize );
        offset += CommandMemorySize;
        g_pMemory.AlignUp( nn::gfx::CommandBuffer::GetControlMemoryAlignment( &g_Device ) );
        g_CommandBuffer.AddControlMemory( g_pMemory.Get(), ControlMemorySize );
        g_pMemory.Advance( ControlMemorySize );
        g_CommandBuffer.Begin();

        nn::gfx::TextureCopyRegion dstRegion;
        dstRegion.SetDefault();
        dstRegion.SetWidth( g_ImageStride / 4 );
        dstRegion.SetHeight( g_ImageDimension.height );
        dstRegion.SetDepth( 1 );
        g_CommandBuffer.CopyBufferToImage( &g_Texture, dstRegion, &tempBuffer, 0 );

        g_CommandBuffer.End();
        g_Queue.ExecuteCommand( &g_CommandBuffer, NULL );
        g_Queue.Sync();
    }

    tempBuffer.Finalize( &g_Device );
}

// テクスチャビューを初期化
nn::gfx::TextureView g_TextureView;
void InitializeTextureBufferView()
{
    nn::gfx::TextureView::InfoType info;
    info.SetDefault();
    info.SetImageDimension( nn::gfx::ImageDimension_2d );
    info.SetImageFormat( nn::gfx::ImageFormat_R8_G8_B8_A8_UnormSrgb);
    info.SetTexturePtr( &g_Texture );
    info.SetChannelMapping( nn::gfx::ChannelMapping_Red, nn::gfx::ChannelMapping_Green,
        nn::gfx::ChannelMapping_Blue, nn::gfx::ChannelMapping_Alpha );
    g_TextureView.Initialize( &g_Device, info );
}

nn::gfx::ResShaderFile* g_pResShaderFile;
void InitializeResShaderFile()
{
    void* pResource = ReadResource( "Contents:/GfxJpegViewerShader.bnsh" );
    g_pResShaderFile = nn::gfx::ResShaderFile::ResCast( pResource );
    nn::gfx::ResShaderContainer* pContainer = g_pResShaderFile->GetShaderContainer();
    NN_ASSERT_NOT_NULL( pContainer );
    pContainer->Initialize( &g_Device );

    for( int idxVariation = 0, variationCount = pContainer->GetShaderVariationCount();
        idxVariation < variationCount; ++idxVariation )
    {
        nn::gfx::ResShaderProgram* pProgram = pContainer->GetResShaderProgram( idxVariation );
        nn::gfx::ShaderInitializeResult result = pProgram->Initialize( &g_Device );
        NN_ASSERT_EQUAL( result, nn::gfx::ShaderInitializeResult_Success );
        NN_UNUSED( result );
    }

#if NN_GFX_IS_TARGET_NVN
    g_MaxScratchMemory = nn::gfx::NvnGetMaxRecommendedScratchMemorySize( &g_Device, &g_pResShaderFile, 1 );
    int scratchMemoryAlignment;
    nvnDeviceGetInteger( g_Device.ToData()->pNvnDevice,
        NVN_DEVICE_INFO_SHADER_SCRATCH_MEMORY_ALIGNMENT, &scratchMemoryAlignment );
    ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuInvisible_GpuCached_Compressible ];
    offset = nn::util::align_up( offset, scratchMemoryAlignment );
    g_ScratchMemoryOffset = offset;
    offset += g_MaxScratchMemory;
#endif
    NN_UNUSED( g_MaxScratchMemory );
    NN_UNUSED( g_ScratchMemoryOffset );
}

void FinalizeResShaderFile()
{
    auto pContainer = g_pResShaderFile->GetShaderContainer();
    NN_ASSERT_NOT_NULL( pContainer );

    for( int idxVariation = 0, variationCount = pContainer->GetShaderVariationCount();
        idxVariation < variationCount; ++idxVariation )
    {
        nn::gfx::ResShaderProgram* pProgram = pContainer->GetResShaderProgram( idxVariation );
        pProgram->Finalize( &g_Device );
    }

    pContainer->Finalize( &g_Device );
}

void InitializeGfxObjects()
{
    // メモリ
    g_pMemoryHeap.Reset( malloc( 1024 * 1024 ) );
    g_pMemory = g_pMemoryHeap;

    // 画像の読み込み
    InitializeImage();

    InitializeDevice();

#if NN_GFX_IS_TARGET_NVN
    nn::gfx::Device::DataType& deviceData = nn::gfx::AccessorToData( g_Device );
    nvnDeviceGetInteger( deviceData.pNvnDevice,
        NVN_DEVICE_INFO_RESERVED_TEXTURE_DESCRIPTORS, &g_TextureDescriptorBaseIndex );
    nvnDeviceGetInteger( deviceData.pNvnDevice,
        NVN_DEVICE_INFO_RESERVED_SAMPLER_DESCRIPTORS, &g_SamplerDescriptorBaseIndex );
#endif

    InitializeMemoryPool();

    InitializeSwapChain();
    InitializeQueue();

    InitializeCommandBuffer();
    InitializeViewport();

    InitializeRasterizerState();
    InitializeBlendState();
    InitializeDepthStencilState();
    InitializeVertexState();

    InitializeVertexBuffer();
    InitializeIndexBuffer();
    InitializeConstantBuffer();
    InitializeSampler();

    InitializeBufferDescriptorPool();
    InitializeSamplerDescriptorPool();
    InitializeTextureDescriptorPool();

    InitializeTexture();
    InitializeTextureBufferView();
    InitializeResShaderFile();

    NN_ASSERT( g_pMemoryHeap.Distance( g_pMemory.Get() ) < 1024 * 1024 );
    for( int idxMemoryPool = 0; idxMemoryPool < MemoryPoolType_End; ++idxMemoryPool )
    {
        NN_ASSERT( static_cast< size_t >( g_MemoryPoolOffset[ idxMemoryPool ] )
            < MemoryPoolSize[ idxMemoryPool ] );
    }
}

void MakeCommand()
{
    nn::gfx::GpuAddress gpuAddress;

    g_CommandBuffer.Reset();
    ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuUncached_GpuCached ];
    offset = nn::util::align_up( offset, nn::gfx::CommandBuffer::GetCommandMemoryAlignment( &g_Device ) );
    // ワンタイムのコマンドバッファには nn::gfx::MemoryPoolProperty_CpuUncached | nn::gfx::MemoryPoolProperty_GpuUncached のメモリプールを使うことがより適しています
    g_CommandBuffer.AddCommandMemory(
        &g_MemoryPool[ MemoryPoolType_CpuUncached_GpuCached ], offset, 1024 * 1024 );
    g_pMemory.AlignUp( 256 );
    g_CommandBuffer.AddControlMemory( g_pMemory.Get(), 256 );
    g_CommandBuffer.Begin();
    {
        nn::gfx::ColorTargetView* pTarget = g_SwapChain.AcquireNextScanBufferView();

        g_VertexBuffer.GetGpuAddress( &gpuAddress );
        g_CommandBuffer.SetVertexBuffer( 0, gpuAddress, sizeof( float ) * 5, sizeof( Vertices ) );

        g_CommandBuffer.InvalidateMemory( nn::gfx::GpuAccess_ShaderCode | nn::gfx::GpuAccess_Descriptor );

        g_CommandBuffer.SetDescriptorPool( &g_BufferDescriptorPool );
        g_CommandBuffer.SetDescriptorPool( &g_TextureDescriptorPool );
        g_CommandBuffer.SetDescriptorPool( &g_SamplerDescriptorPool );

        g_CommandBuffer.ClearColor( pTarget, 0.3f, 0.1f, 0.1f, 1.0f, NULL );

#if NN_GFX_IS_TARGET_NVN
        nvnCommandBufferSetShaderScratchMemory( g_CommandBuffer.ToData()->pNvnCommandBuffer,
            g_MemoryPool[ MemoryPoolType_CpuInvisible_GpuCached_Compressible ].ToData()->pNvnMemoryPool,
            g_ScratchMemoryOffset, g_MaxScratchMemory );
#endif

        nn::gfx::ResShaderContainer* pContainer = g_pResShaderFile->GetShaderContainer();
        int variation = 0;
        nn::gfx::ResShaderProgram* pProgram = pContainer->GetResShaderProgram( variation );
        nn::gfx::Shader* pShader = pProgram->GetShader();
        g_CommandBuffer.SetShader( pShader, nn::gfx::ShaderStageBit_All );

        g_CommandBuffer.SetRenderTargets( 1, &pTarget, NULL );

        g_CommandBuffer.SetViewportScissorState( &g_ViewportScissor );

        g_CommandBuffer.SetRasterizerState( &g_RasterizerState );
        g_CommandBuffer.SetBlendState( &g_BlendState );
        g_CommandBuffer.SetDepthStencilState( &g_DepthStencilState );
        g_CommandBuffer.SetVertexState( &g_VertexState );

        g_CommandBuffer.InvalidateMemory( nn::gfx::GpuAccess_Texture | nn::gfx::GpuAccess_IndexBuffer
            | nn::gfx::GpuAccess_ConstantBuffer | nn::gfx::GpuAccess_VertexBuffer );

        nn::gfx::DescriptorSlot constantBufferDescriptor;
        nn::gfx::DescriptorSlot textureDescriptor;
        nn::gfx::DescriptorSlot samplerDescriptor;
        g_BufferDescriptorPool.GetDescriptorSlot( &constantBufferDescriptor, g_BufferDescriptorBaseIndex );
        g_TextureDescriptorPool.GetDescriptorSlot( &textureDescriptor, g_TextureDescriptorBaseIndex );
        g_SamplerDescriptorPool.GetDescriptorSlot( &samplerDescriptor, g_SamplerDescriptorBaseIndex );

        int slotMat = pShader->GetInterfaceSlot( nn::gfx::ShaderStage_Pixel,
            nn::gfx::ShaderInterfaceType_ConstantBuffer, "Mat" );
        int slotTex = pShader->GetInterfaceSlot( nn::gfx::ShaderStage_Pixel,
            nn::gfx::ShaderInterfaceType_Sampler, "tex" );
        if( slotMat >= 0 )
        {
            g_CommandBuffer.SetConstantBuffer( slotMat,
                nn::gfx::ShaderStage_Pixel, constantBufferDescriptor );
        }
        if( slotTex >= 0 )
        {
            g_CommandBuffer.SetTextureAndSampler( slotTex,
                nn::gfx::ShaderStage_Pixel, textureDescriptor, samplerDescriptor );
        }

        g_IndexBuffer.GetGpuAddress( &gpuAddress );
        g_CommandBuffer.DrawIndexed( nn::gfx::PrimitiveTopology_TriangleList,
            nn::gfx::IndexFormat_Uint32, gpuAddress, sizeof(Indices) / sizeof(Indices[0]), 0 );
    }
    g_CommandBuffer.End();
}

void FinalizeGfxObjects()
{
    // 各オブジェクトを破棄
    FinalizeResShaderFile();

    g_Sampler.Finalize( &g_Device );
    g_TextureView.Finalize( &g_Device );
    g_Texture.Finalize( &g_Device );
    g_ConstantBuffer.Finalize( &g_Device );
    g_IndexBuffer.Finalize( &g_Device );
    g_VertexBuffer.Finalize( &g_Device );

    g_RasterizerState.Finalize( &g_Device );
    g_BlendState.Finalize( &g_Device );
    g_DepthStencilState.Finalize( &g_Device );
    g_VertexState.Finalize( &g_Device );

    g_BufferDescriptorPool.Finalize( &g_Device );
    g_TextureDescriptorPool.Finalize( &g_Device );
    g_SamplerDescriptorPool.Finalize( &g_Device );

    g_ViewportScissor.Finalize( &g_Device );
    g_CommandBuffer.Finalize( &g_Device );
    g_SwapChain.Finalize( &g_Device );
    g_Queue.Finalize( &g_Device );
    for( int idxMemoryPool = 0; idxMemoryPool < MemoryPoolType_End; ++idxMemoryPool )
    {
        g_MemoryPool[ idxMemoryPool ].Finalize( &g_Device );
        free( g_pPoolMemory[ idxMemoryPool ] );
    }
    g_Device.Finalize();

    // 画像を解放
    FinalizeImage();

    free( g_pMemoryHeap.Get() );
}

#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN ) && defined( NN_BUILD_APISET_NX )
void* Allocate( size_t size, size_t alignment, void* )
{
    return aligned_alloc( alignment, nn::util::align_up(size , alignment));
}
void Free( void* addr, void* )
{
    free( addr );
}
void* Reallocate( void* addr, size_t newSize, void* )
{
    return realloc( addr, newSize );
}
#endif

}// anonymous namespace

#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN )
//-----------------------------------------------------------------------------
// nninitStartup() is invoked before calling nnMain().
//
extern "C" void nninitStartup()
{
    const size_t MallocMemorySize = 256 * 1024 * 1024;
    nn::Result result = nn::os::SetMemoryHeapSize( 256 * 1024 * 1024 );
    uintptr_t address;
    result = nn::os::AllocateMemoryBlock( &address, MallocMemorySize );
    NN_ASSERT( result.IsSuccess() );
    nn::init::InitializeAllocator( reinterpret_cast<void*>( address ), MallocMemorySize );
}
#endif

//
//  Main Function
//  メイン関数です。
//

extern "C" void nnMain()
{
    nn::Result result;

    size_t cacheSize = 0;
    result = nn::fs::QueryMountRomCacheSize(&cacheSize);
    NN_ASSERT( result.IsSuccess() );

    void* mountRomCacheBuffer = malloc(cacheSize);
    NN_ASSERT_NOT_NULL(mountRomCacheBuffer);

    result = nn::fs::MountRom( "Contents", mountRomCacheBuffer, cacheSize );
    NN_ABORT_UNLESS_RESULT_SUCCESS( result );

#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN ) && defined( NN_BUILD_APISET_NX )
    // グラフィックスシステムのためのメモリ周りの初期化を行います。
    {
        nv::SetGraphicsAllocator( Allocate, Free, Reallocate, NULL );
        nv::InitializeGraphics( malloc( GraphicsSystemMemorySize ), GraphicsSystemMemorySize );
    }
    // グラフィックス開発者向けツールおよびデバッグレイヤのためのメモリアロケータを設定します。
    nv::SetGraphicsDevtoolsAllocator( Allocate, Free, Reallocate, NULL );
#endif

    nn::vi::Initialize();
    InitializeLayer();

    nn::gfx::Initialize();
    InitializeGfxObjects();

    g_BufferDescriptorPool.BeginUpdate();
    {
        nn::gfx::GpuAddress gpuAddress;
        g_ConstantBuffer.GetGpuAddress( &gpuAddress );
        g_BufferDescriptorPool.SetBufferView( g_BufferDescriptorBaseIndex, gpuAddress, sizeof( float ) * 4 );
    }
    g_BufferDescriptorPool.EndUpdate();

    g_TextureDescriptorPool.BeginUpdate();
    {
        g_TextureDescriptorPool.SetTextureView( g_TextureDescriptorBaseIndex, &g_TextureView);
    }
    g_TextureDescriptorPool.EndUpdate();

    g_SamplerDescriptorPool.BeginUpdate();
    {
        g_SamplerDescriptorPool.SetSampler( g_SamplerDescriptorBaseIndex, &g_Sampler );
    }
    g_SamplerDescriptorPool.EndUpdate();

    // 毎フレームのレンダリング
    for( int frame = 0; frame < 60 * 10; ++frame )
    {
        MakeCommand();

        // 定数バッファを更新
        float* pConstantBuffer = g_ConstantBuffer.Map< float >();
        {
            pConstantBuffer[ 0 ] = 0.0f;
            pConstantBuffer[ 1 ] = 0.0f;
            pConstantBuffer[ 2 ] = 0.0f;
            pConstantBuffer[ 3 ] = 0.0f;
        }
        g_ConstantBuffer.Unmap();

        // コマンドの実行
        g_Queue.ExecuteCommand( &g_CommandBuffer, NULL );

        // 結果の表示
        g_Queue.Present( &g_SwapChain, 1 );
        g_Queue.Sync();
    }

    FinalizeGfxObjects();
    nn::gfx::Finalize();

    nn::vi::DestroyLayer( g_pLayer );
    nn::vi::CloseDisplay( g_pDisplay );
    nn::vi::Finalize();

    nn::fs::Unmount("Contents");
    free(mountRomCacheBuffer);
}
