﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/gfx.h>
#include <nvn/nvn_FuncPtrInline.h>
#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN )
#include <nv/nv_MemoryManagement.h>
#endif

namespace nn { namespace g3d { namespace demo {

namespace {

//------------------------------------------------------------------------------
// グラフィックスシステム用メモリ割り当て・破棄関数
//------------------------------------------------------------------------------
#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN ) && defined( NN_BUILD_APISET_NX )
void* NvAllocate(size_t size, size_t alignment, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return aligned_alloc(alignment, nn::util::align_up(size, alignment));
}

void NvFree(void* addr, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    free(addr);
}

void* NvReallocate(void* addr, size_t newSize, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return realloc(addr, newSize);
}
#endif

NN_FORCEINLINE void InitializeGraphicMemory() NN_NOEXCEPT
{
#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN )
    const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;
    nv::SetGraphicsAllocator(NvAllocate, NvFree, NvReallocate, nullptr);
    nv::SetGraphicsDevtoolsAllocator(NvAllocate, NvFree, NvReallocate, nullptr);
    nv::InitializeGraphics(malloc(GraphicsSystemMemorySize), GraphicsSystemMemorySize);
#endif
    return;
}

NN_FORCEINLINE int GetShaderMemoryAlignment(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    int alignment;
    nvnDeviceGetInteger( pDevice->ToData()->pNvnDevice,
                         NVN_DEVICE_INFO_SHADER_SCRATCH_MEMORY_ALIGNMENT, &alignment );
    return alignment;
}

NN_FORCEINLINE size_t GetShaderMemorySize(nn::gfx::Device* pDevice, const nn::gfx::ResShaderFile* const* shaderFileList, int count) NN_NOEXCEPT
{
    return nn::gfx::NvnGetMaxRecommendedScratchMemorySize(pDevice, shaderFileList, count);
}

NN_FORCEINLINE void SetShaderMemory(nn::gfx::CommandBuffer* pCommandBuffer, nn::gfx::MemoryPool* pMemoryPool, ptrdiff_t offset, size_t size) NN_NOEXCEPT
{
    nvnCommandBufferSetShaderScratchMemory(pCommandBuffer->ToData()->pNvnCommandBuffer,
                                           pMemoryPool->ToData()->pNvnMemoryPool, offset, size);
}

}

}}} // namespace nn::g3d::demo

