﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "g3ddemo_ModelUtility.h"
#include <nn/g3d/g3d_ResFile.h>
#include <nn/g3d/g3d_ResShader.h>
#include <nn/g3d/g3d_MaterialObj.h>
#include <nn/g3d/g3d_ShaderUtility.h>
#include "g3ddemo_DemoUtility.h"

namespace nn { namespace g3d { namespace demo {

//--------------------------------------------------------------------------------------------------

void WriteSkinningOption(nns::g3d::RenderModelObj* pRenderModelObj, int passIndex) NN_NOEXCEPT
{
    int shapeCount = pRenderModelObj->GetModelObj()->GetShapeCount();
    for (int shapeIndex = 0; shapeIndex < shapeCount; ++shapeIndex)
    {
        nns::g3d::RenderUnitObj* pRenderUnitObj = pRenderModelObj->GetRenderUnitObj(shapeIndex);
        if (!pRenderUnitObj->IsInitialized())
        {
            continue;
        }

        nn::g3d::ShaderSelector* pShaderSelector = pRenderUnitObj->GetShaderSelector(passIndex);
        int optionIndex = pShaderSelector->FindDynamicOptionIndex("skinning");
        if (optionIndex >= 0)
        {
            const nn::g3d::ResShaderOption* pResShaderOption = pShaderSelector->GetDynamicOption(optionIndex);
            const nn::g3d::ResVertex* pResVertex = pRenderUnitObj->GetShapeObj()->GetResource()->GetVertex();
            NN_ASSERT(pResVertex->GetVertexInfluenceCount() <= 8);
            // skinning="-1" のバリエーションでは、Shape の UniformBlock 中の VtxSkinCount による
            // ユニフォーム変数分岐を使用してスキニングの種別を切り分ける実装を行っています。
            // skinning>="0" 以降のバリエーションではユニフォーム変数分岐を使用しない実装を行っています。
            // 開発中のシェーダーコンパイル時間を短縮するために、skinning="-1" のバリエーションを
            // 利用することが出来ます。
            char choice[2] = { static_cast<char>('0' + pResVertex->GetVertexInfluenceCount()), '\0' };
            int choiceIndex = pResShaderOption->FindChoiceIndex(choice);
            pShaderSelector->WriteDynamicKey(optionIndex, choiceIndex);
        }
    }
}

void WriteSkinningOption(nns::g3d::RenderModelObj* pRenderModelObj) NN_NOEXCEPT
{
    for (int passIndex = 0, passCount = pRenderModelObj->GetRenderModel()->GetPassCount(); passIndex < passCount; ++passIndex)
    {
        WriteSkinningOption(pRenderModelObj, passIndex);
    }
}

//--------------------------------------------------------------------------------------------------

void RegistAnim(ResourceHolder* pHolder) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pHolder);

    for (int fileIndex = 0; fileIndex < pHolder->files.GetCount(); ++fileIndex)
    {
        nn::g3d::ResFile* pResFile = pHolder->files[fileIndex];

        for (int index = 0; index < pResFile->GetSkeletalAnimCount(); ++index)
        {
            nn::g3d::ResSkeletalAnim* pAnim = pResFile->GetSkeletalAnim(index);
            void* ptr = pAnim;
            pHolder->modelAnims.PushBack(ptr);
        }
        for (int index = 0; index < pResFile->GetBoneVisibilityAnimCount(); ++index)
        {
            nn::g3d::ResBoneVisibilityAnim* pAnim = pResFile->GetBoneVisibilityAnim( index );
            void* ptr = pAnim;
            pHolder->modelAnims.PushBack(ptr);
        }
        for (int index = 0; index < pResFile->GetMaterialAnimCount(); ++index)
        {
            nn::g3d::ResMaterialAnim* pAnim = pResFile->GetMaterialAnim( index );
            void* ptr = pAnim;
            pHolder->modelAnims.PushBack(ptr);
        }
        for (int index = 0; index < pResFile->GetShapeAnimCount(); ++index)
        {
            nn::g3d::ResShapeAnim* pAnim = pResFile->GetShapeAnim( index );
            void* ptr = pAnim;
            pHolder->modelAnims.PushBack(ptr);
        }
    }
}

void DestroyAll(nn::gfx::Device* pDevice, ResourceHolder* pHolder) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pDevice);
    NN_ASSERT_NOT_NULL(pHolder);

    for (int index = 0; index < pHolder->modelObjs.GetCount(); ++index)
    {
        nns::g3d::DestroyModelObj(pDevice, pHolder->modelObjs[index]);
    }

    for (int index = 0; index < pHolder->modelAnimObjs.GetCount(); ++index)
    {
        nns::g3d::DestroyModelAnimObj(pDevice, pHolder->modelAnimObjs[index]);
    }

    for (int index = 0; index < pHolder->renderModels.GetCount(); ++index)
    {
        nns::g3d::DestroyRenderModel(pDevice, pHolder->renderModels[index]);
    }

    for (int index = 0; index < pHolder->renderModelObjs.GetCount(); ++index)
    {
        nns::g3d::DestroyRenderModelObj(pHolder->renderModelObjs[index]);
    }


    for (int index = 0; index < pHolder->files.GetCount(); ++index)
    {
        nn::g3d::ResFile* pResFile = pHolder->files[index];
        CleanupTexture(pDevice, pResFile);
        UnregisterSamplerFromDescriptorPool(pResFile);
        pResFile->Cleanup(pDevice);
        FreeMemory(pResFile);
    }

    for (int index = 0; index < pHolder->shaderFiles.GetCount(); ++index)
    {
        nn::g3d::ResShaderFile* pFile = pHolder->shaderFiles[index];
        nn::g3d::ResShaderArchive* pShaderArchive = pFile->GetResShaderArchive();
        pShaderArchive->Cleanup(pDevice);
        FreeMemory(pFile);
    }

    for (int index = 0; index < pHolder->animationIds.GetCount(); ++index)
    {
        FreeMemory(pHolder->animationIds[index]);
    }

    pHolder->Shutdown();
}

//--------------------------------------------------------------------------------------------------
void SetupTexture(nn::gfx::Device* pDevice, nn::gfx::ResTextureFile* pTextureFile) NN_NOEXCEPT
{
    pTextureFile->Initialize(pDevice);
    for (int textureIndex = 0, texCount = pTextureFile->GetTextureDic()->GetCount(); textureIndex < texCount; ++textureIndex)
    {
        nn::gfx::ResTexture* pResTexture = pTextureFile->GetResTexture(textureIndex);
        pResTexture->Initialize(pDevice);
    }
    RegisterTextureFileToDescriptorPool(pTextureFile);
}

void CleanupTexture(nn::gfx::Device* pDevice, nn::gfx::ResTextureFile* pTextureFile) NN_NOEXCEPT
{
    for (int textureIndex = 0; textureIndex < pTextureFile->GetTextureDic()->GetCount(); ++textureIndex)
    {
        nn::gfx::ResTexture* pResTexture = pTextureFile->GetResTexture(textureIndex);
        pResTexture->Finalize(pDevice);
    }
    pTextureFile->Finalize(pDevice);
    UnregisterTextureFileFromDescriptorPool(pTextureFile);
}

void SetupTexture(nn::gfx::Device* pDevice, nn::g3d::ResFile* pResFile, nn::g3d::TextureBindCallback pCallback) NN_NOEXCEPT
{
    char textureFileName[32];
    //テクスチャーのファイル名はモデルファイルと同名と想定。
    sprintf(textureFileName, "%s.bntx", pResFile->GetName());
    if (nn::g3d::ResExternalFile* pExternalFile = pResFile->FindExternalFile(textureFileName))
    {
        // 読み込んだファイルをResTextureFileに変換します。
        nn::gfx::ResTextureFile* pTextureFile = nn::gfx::ResTextureFile::ResCast(pExternalFile->GetData());
        SetupTexture(pDevice, pTextureFile);
        // テクスチャーをバインド
        pResFile->BindTexture(pCallback, pTextureFile);
    }
}

void CleanupTexture(nn::gfx::Device* pDevice, nn::g3d::ResFile* pResFile) NN_NOEXCEPT
{
    char textureFileName[32];
    //テクスチャーのファイル名はモデルファイルと同名と想定。
    sprintf(textureFileName, "%s.bntx", pResFile->GetName());
    if (nn::g3d::ResExternalFile* pExternalFile = pResFile->FindExternalFile(textureFileName))
    {
        nn::gfx::ResTextureFile* pTextureFile = reinterpret_cast<nn::gfx::ResTextureFile*>(pExternalFile->GetData());
        CleanupTexture(pDevice, pTextureFile);
    }
}

nn::g3d::TextureRef GetDummyTextureRef()
{
    // 3DEditor で編集中などで該当のテクスチャーが空だった場合は、ダミーテクスチャーを割り当てる
    const nn::gfx::ResTexture* pResTexture = GetDummyTexture();
    NN_ASSERT_NOT_NULL(pResTexture);
    nn::gfx::DescriptorSlot descriptorSlot;
    pResTexture->GetUserDescriptorSlot(&descriptorSlot);

    nn::g3d::TextureRef textureRef;
    textureRef.SetTextureView(pResTexture->GetTextureView());
    textureRef.SetDescriptorSlot(descriptorSlot);
    return textureRef;
}

nn::g3d::TextureRef TextureBindCallback(const char* name, void* pUserData) NN_NOEXCEPT
{
    if (pUserData == nullptr)
    {
        return GetDummyTextureRef();
    }

    const nn::gfx::ResTextureFile* pTextureFile = reinterpret_cast<const nn::gfx::ResTextureFile*>(pUserData);
    const nn::util::ResDic* pTextureDic = pTextureFile->GetTextureDic();
    int index = pTextureDic->FindIndex(name);
    if (index == nn::util::ResDic::Npos)
    {
        return GetDummyTextureRef();
    }

    const nn::gfx::ResTexture* pResTexture = pTextureFile->GetResTexture(index);
    nn::gfx::DescriptorSlot descriptorSlot;
    pResTexture->GetUserDescriptorSlot(&descriptorSlot);

    nn::g3d::TextureRef textureRef;
    textureRef.SetTextureView(pResTexture->GetTextureView());
    textureRef.SetDescriptorSlot(descriptorSlot);
    return textureRef;
}

//--------------------------------------------------------------------------------------------------

void MoveModelToFrontOfCamera(
    nns::g3d::ModelAnimObj* pModelAnimObj,
    const nn::util::Vector3f& cameraPosition,
    const nn::util::Vector3f& cameraAimTarget) NN_NOEXCEPT
{
    nn::util::Vector3f eyeVec;
    nn::util::VectorSubtract(&eyeVec, cameraAimTarget, cameraPosition);
    nn::util::VectorNormalize(&eyeVec, eyeVec);

    nn::g3d::ModelObj* pModelObj = pModelAnimObj->GetModelObj();
    pModelAnimObj->Calculate();
    pModelObj->CalculateBounding();
    nn::g3d::Sphere* pBoundingSphere = pModelObj->GetBounding();

    float modelOffsetFactor = pBoundingSphere->radius * 2;
    nn::util::Vector3f modelOffset;
    nn::util::VectorMultiply(&modelOffset, eyeVec, modelOffsetFactor);

    nn::util::Vector3f modelPosition;
    nn::util::VectorAdd(&modelPosition, cameraPosition, modelOffset);

    nn::util::Vector3f modelCenter;
    nn::util::VectorSubtract(&modelCenter, pBoundingSphere->center, pModelAnimObj->GetTranslate());

    nn::util::VectorSubtract(&modelPosition, modelPosition, modelCenter);

    pModelAnimObj->SetTranslate(modelPosition);
}

}}} // namespace nn::g3d::demo
