﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

layout( local_size_x = 64, local_size_y = 1, local_size_z = 1 ) in;

// 頂点数とキーシェイプ数を受け取るユニフォームブロック
layout(std140) uniform ShapeAnimParamBuffer  // @@ id="shape_anim_param_buf" type="none"
{
    int vertexCount;
    int keyShapeCount;
};

// 全キーシェイプの position と normal のリスト
layout(std430) buffer PositionBuffer	// @@ id="pos_buf" type="none"
{
    vec4 positions[]; // [keyshapeCount * vertexCount]
};
layout(std430) buffer NormalBuffer	// @@ id="norm_buf" type="none"
{
    vec4 normals[]; // [keyshapeCount * vertexCount]
};

// 全キーシェイプの weight のリスト
layout(std430) buffer WeightBuffer	// @@ id="weight_buf" type="none"
{
    float weights[]; // [keyshapeCount]
};

// position と normal のブレンド結果を格納するバッファー
layout(std430) buffer ExportPositionBuffer	// @@ id="export_pos_buf" type="none"
{
    vec4 positionResult[]; // [vertexCount]
};
layout(std430) buffer ExportNormalBufffer	// @@ id="export_norm_buf" type="none"
{
    vec4 normalResult[]; // [vertexCount]
};

void main()
{
    uint target = gl_WorkGroupID.x * gl_WorkGroupSize.x + gl_LocalInvocationID.x;

    // ワークグループサイズと頂点数の関係で、target が頂点数を超える場合があるのでチェックする
    if(target < vertexCount)
    {
        positionResult[target] = vec4(0.0, 0.0, 0.0, 0.0);
        normalResult[target] = vec4(0.0, 0.0, 0.0, 0.0);

        // 各キーシェイプのブレンド処理を行う
        for(int keyShapeIndex = 0; keyShapeIndex < keyShapeCount; ++keyShapeIndex)
        {
            positionResult[target] += positions[target + (vertexCount * keyShapeIndex)] * weights[keyShapeIndex];
            normalResult[target] += normals[target + (vertexCount * keyShapeIndex)] * weights[keyShapeIndex];
        }

        // 法線は正規化する
        normalResult[target] = normalize(normalResult[target]);
    }
}
