﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{FsSaveDataForDebug.cpp,PageSampleFsSaveDataForDebug}
 *
 * @brief
 *  デバッグ用途のセーブデータ機能のサンプルプログラム
 */

/**
 * @page PageSampleFsSaveDataForDebug デバッグ用途のセーブデータ機能
 * @tableofcontents
 *
 * @brief
 *  デバッグ用途のセーブデータ機能のサンプルプログラムの解説です。
 *
 * @section PageSampleFsSaveDataForDebug_SectionBrief 概要
 *  ここでは、デバッグ用途のセーブデータ機能のサンプルプログラムの説明をします。
 *
 *  ファイル・ディレクトリの操作方法については、
 *  @ref nn::fs "ファイルシステムライブラリの関数リファレンス" も併せて参照して下さい。
 *
 * @section PageSampleFsSaveDataForDebug_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/FsSaveDataForDebug Samples/Sources/Applications/FsSaveDataForDebug @endlink 以下にあります。
 *
 * @section PageSampleFsSaveDataForDebug_SectionNecessaryEnvironment 必要な環境
 *  特になし。
 *
 * @section PageSampleFsSaveDataForDebug_SectionHowToOperate 操作方法
 *  特になし。
 *
 * @section PageSampleFsSaveDataForDebug_SectionPrecaution 注意事項
 *  このデモは画面上に何も表示されません。実行結果はログに出力されます。
 *
 * @section PageSampleFsSaveDataForDebug_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleFsSaveDataForDebug_SectionDetail 解説
 *
 * @subsection PageSampleFsSaveDataForDebug_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  FsSaveDataForDebug.cpp
 *  @includelineno FsSaveDataForDebug.cpp
 *
 * @subsection PageSampleFsSaveDataForDebug_SectionSampleDetail サンプルプログラムの解説
 *  サンプルプログラムの全体像は以下の通りです。
 *
 *  - MountSaveDataForDebug 関数によるセーブデータのマウント
 *  - セーブデータに格納するファイルが無い場合は作成する
 *  - セーブデータに格納するファイルの読み込み・更新
 *  - Commit 関数によるセーブデータ更新内容のコミット
 *
 *  セーブデータ機能の詳細については、Nintendo SDK ドキュメントのファイルシステム機能のマニュアルにある「セーブデータ」の項目も併せて参照してください。
 *
 *  このサンプルプログラムの実行結果を以下に示します。
 *
 *  @verbinclude  FsSaveDataForDebug_OutputExample.txt
 *
 */

#include <cstdlib>
#include <cstring>

#include <nn/fs.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/fs/fs_Debug.h>

namespace {
/**
   @brief セーブデータ用の構造体
*/
    struct SaveDataStructure
    {
        int value;
    };
} // namespace

extern "C" void nnMain()
{
    nn::Result result;

    // マウント名 "save" としてセーブデータをマウントします。
    NN_LOG("Mount save data as \"save\"\n");
    result = nn::fs::MountSaveDataForDebug("save");
    // 失敗した際は必ずアボートしてください。
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    const size_t FileSize = sizeof(SaveDataStructure);

    // ファイルの存在を確認します。
    {
        nn::fs::DirectoryEntryType directoryEntryType;
        result = nn::fs::GetEntryType(&directoryEntryType, "save:/file");
        if( nn::fs::ResultPathNotFound::Includes(result) )
        {
            // ファイルが存在しない場合は作成します。
            result = nn::fs::CreateFile("save:/file", FileSize);
            if( nn::fs::ResultPathNotFound::Includes(result) )
            {
                // パスに含まれるディレクトリが存在しません。
                // 親ディレクトリが必ず存在する場合は、このエラーハンドリングは不要です。
            }
            else if( nn::fs::ResultPathAlreadyExists::Includes(result)
                  || nn::fs::ResultTargetLocked::Includes(result) )
            {
                // 対象ファイルが既に存在しています。
                // ファイルが既に存在していても構わない場合は、このエラーハンドリングは不要です。
                // エラーハンドリングしない場合、ファイルのサイズが FileSize である保証が無いことに注意してください。
                // 必要であれば対象ファイルを削除してから再度作成してください。
            }
            else if( nn::fs::ResultUsableSpaceNotEnough::Includes(result) )
            {
                // セーブデータのデータ保存領域が不足しています。
                // データ保存領域を増やす処理を行ってください。
                NN_ABORT("Usable space not enough.\n");
                return;
            }
            // 上記以外の原因で失敗した場合はライブラリ内でアボートするため、
            // これ以上のエラーハンドリングは不要です。

            // 初期データを書き込みます。
            NN_LOG("Create initial data as save:/file\n");
            nn::fs::FileHandle fileHandle;
            result = nn::fs::OpenFile(&fileHandle, "save:/file", nn::fs::OpenMode_Write);
            if( nn::fs::ResultPathNotFound::Includes(result) )
            {
                // 対象ファイルが存在しません。
                // 存在するファイルしか開かない場合は、このエラーハンドリングは不要です。
            }
            else if( nn::fs::ResultTargetLocked::Includes(result) )
            {
                // 対象ファイルが既にオープンされています。
                // ファイルが既にオープンされている可能性が無い場合は、このエラーハンドリングは不要です。
            }
            // 上記以外の原因で失敗した場合はライブラリ内でアボートするため、
            // これ以上のエラーハンドリングは不要です。

            SaveDataStructure data = {0};
            result = nn::fs::WriteFile(fileHandle, 0, &data, sizeof(data), nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush));
            if( nn::fs::ResultUsableSpaceNotEnough::Includes(result) )
            {
                // セーブデータのデータ保存領域が不足しています。
                // ファイルをオープンする際に OpenMode_AllowAppend を指定していない場合は、このエラーハンドリングは不要です。
            }
            // 上記以外の原因で失敗した場合はライブラリ内でアボートするため、
            // これ以上のエラーハンドリングは不要です。

            nn::fs::CloseFile(fileHandle);
        }
        // 上記以外の原因で失敗した場合はライブラリ内でアボートするため、
        // これ以上のエラーハンドリングは不要です。
    }

    // データを読み込み、更新します。
    {
        SaveDataStructure data;

        nn::fs::FileHandle fileHandle;
        result = nn::fs::OpenFile(&fileHandle, "save:/file", nn::fs::OpenMode_Read | nn::fs::OpenMode_Write);
        if( nn::fs::ResultPathNotFound::Includes(result) )
        {
            // 対象ファイルが存在しません。
            // 存在するファイルしか開かない場合は、このエラーハンドリングは不要です。
        }
        else if( nn::fs::ResultTargetLocked::Includes(result) )
        {
            // 対象ファイルが既にオープンされています。
            // ファイルが既にオープンされている可能性が無い場合は、このエラーハンドリングは不要です。
        }
        // 上記以外の原因で失敗した場合はライブラリ内でアボートするため、
        // これ以上のエラーハンドリングは不要です。

        // データを読み込みます。
        size_t readSize = 0;
        NN_LOG("Read save:/file\n");

        // 読み込みに失敗した場合はライブラリ内でアボートするため、エラーハンドリングは不要です。
        (void)nn::fs::ReadFile(&readSize, fileHandle, 0, &data, sizeof(data));

        NN_ASSERT_EQUAL(readSize, sizeof(data));

        NN_LOG("data.value: 0x%x\n", data.value);

        // データを更新します。
        NN_LOG("Increment data.value: 0x%x -> 0x%x\n", data.value, data.value + 1);
        data.value++;

        // 更新したデータを書き込みます。
        NN_LOG("Write save:/file\n");
        result = nn::fs::WriteFile(fileHandle, 0, &data, sizeof(data), nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush));
        if( nn::fs::ResultUsableSpaceNotEnough::Includes(result) )
        {
            // セーブデータのデータ保存領域が不足しています。
            // ファイルをオープンする際に OpenMode_AllowAppend を指定していない場合は、このエラーハンドリングは不要です。
        }
        // 上記以外の原因で失敗した場合はライブラリ内でアボートするため、
        // これ以上のエラーハンドリングは不要です。


        NN_LOG("Close save:/file\n");
        nn::fs::CloseFile(fileHandle);

        // セーブデータの更新内容をコミットします。
        // コミットに失敗した場合はライブラリ内でアボートするため、エラーハンドリングは不要です。
        NN_LOG("Commit \"save\"\n");
        (void)nn::fs::Commit("save");
    }

    // アンマウントします。
    NN_LOG("Unmount \"save\"\n\n");
    nn::fs::Unmount("save");

    return;
}
