﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{FsFileDataCache.cpp,PageSampleFsFileDataCache}
 *
 * @brief
 *  ファイルデータキャッシュのベンチマークデモ
 */

/**
 * @page PageSampleFsFileDataCache ファイルデータキャッシュのベンチマークデモ
 * @tableofcontents
 *
 * @brief
 *  ファイルデータキャッシュを使用したサンプルプログラムの解説です。
 *
 * @section PageSampleFsFileDataCache_SectionBrief 概要
 *  ファイルデータキャッシュを使用したサンプルプログラムです。ファイルデータキャッシュの有無によってファイルアクセスのパフォーマンスがどの程度変動するかを簡単なベンチマークで示します。
 *
 *  ファイルデータキャッシュの使用方法については、 @ref nn::fs "ファイルシステムライブラリの関数リファレンス" も併せて参照して下さい。
 *
 * @section PageSampleFsFileDataCache_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/FsFileDataCache Samples/Sources/Applications/FsFileDataCache @endlink 以下にあります。
 *
 * @section PageSampleFsFileDataCache_SectionNecessaryEnvironment 必要な環境
 *  本サンプルプログラムの実行には NX 開発機が必要です。
 *
 * @section PageSampleFsFileDataCache_SectionHowToOperate 操作方法
 *  特にありません。
 *
 * @section PageSampleFsFileDataCache_SectionPrecaution 注意事項
 *  このデモは画面上に何も表示しません。実行結果はログに出力されます。
 *
 *  本サンプルプログラムは NSP 形式のアプリケーションイメージから起動されることを想定しています。本サンプルプログラムの実行結果が期待される結果と大きく異なる場合は、アプリケーションが NSP 形式でビルドされていることを確認してください。
 *
 * @section PageSampleFsFileDataCache_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleFsFileDataCache_SectionDetail 解説
 *
 * @subsection PageSampleFsFileDataCache_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  FsFileDataCache.cpp
 *  @includelineno FsFileDataCache.cpp
 *
 * @subsection PageSampleFsFileDataCache_SectionSampleDetail サンプルプログラムの解説
 *  本サンプルプログラムはリソースデータ (ROM) 内にある約 4MB のファイルを使用して次の簡単なベンチマークを行います。
 *
 *  - 4KB のファイルバッファを使用したファイル全体のシーケンシャルリード
 *  - 4KB のファイルバッファを使用して合計 4MB 分のリードを行うランダムリード
 *
 *  それぞれのアクセスパターンについて、ファイルデータキャッシュを使用した場合とそうでない場合の実行時間を計測し、結果をログに出力します。
 *
 *  このサンプルプログラムの実行結果の例を以下に示します。実行結果はサンプルプログラムの実行ごとに若干の変動があります。
 *
 *  @verbinclude  FsFileDataCache_OutputExample.txt
 *
 */

#include <cstdlib>
#include <random>

#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/nn_TimeSpan.h>

#include <nn/fs.h>
#include <nn/os.h>


namespace
{

/*
 *   path に指定されたファイル全体のシーケンシャルリードを行います。
 *   バッファサイズとして bufferSize を使用します。
 */
void RunSequentialRead(const char* path, size_t bufferSize)
{
    nn::fs::FileHandle fileHandle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&fileHandle, path, nn::fs::OpenMode_Read));

    int64_t fileSize;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetFileSize(&fileSize, fileHandle));

    void* buffer = std::malloc(bufferSize);
    NN_ABORT_UNLESS_NOT_NULL(buffer);

    int64_t offset = 0;
    while (offset < fileSize)
    {
        size_t readSize;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(&readSize, fileHandle, offset, buffer, bufferSize));

        offset += static_cast<int64_t>(readSize);
    }

    std::free(buffer);

    nn::fs::CloseFile(fileHandle);
}

/*
 *   path に指定されたファイルを対象に、そのファイルサイズ分のランダムリードを行います。
 *   バッファサイズとして bufferSize を使用します。
 */
void RunRandomRead(const char* path, size_t bufferSize)
{
    nn::fs::FileHandle fileHandle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&fileHandle, path, nn::fs::OpenMode_Read));

    int64_t fileSize;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetFileSize(&fileSize, fileHandle));

    void* buffer = std::malloc(bufferSize);
    NN_ABORT_UNLESS_NOT_NULL(buffer);

    std::mt19937 mt;
    NN_ABORT_UNLESS(bufferSize <= fileSize, "bufferSize must be less than or equal to the size of the given path.");
    std::uniform_int_distribution<int64_t> offsetDistribution(0, fileSize - static_cast<int64_t>(bufferSize));

    int64_t totalReadSize = 0;
    while (totalReadSize < fileSize)
    {
        int64_t offset = offsetDistribution(mt);

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(fileHandle, offset, buffer, bufferSize));

        totalReadSize += static_cast<int64_t>(bufferSize);
    }

    std::free(buffer);

    nn::fs::CloseFile(fileHandle);
}

/*
 *   callable に指定された呼び出し可能オブジェクトの実行時間を計測します。
 */
template <typename T>
nn::TimeSpan MeasureExecutionTime(T callable)
{
    nn::os::Tick startTick = nn::os::GetSystemTick();
    callable();
    nn::os::Tick endTick = nn::os::GetSystemTick();
    return (endTick - startTick).ToTimeSpan();
}

}  // namespace unnamed


extern "C" void nnMain()
{
    // リソースデータのマウント名と、テスト対象のファイルパスです。
    const char* mountName = "rom";
    const char* testTargetPath = "rom:/dummy.bmp";

    // リード時のバッファサイズです。
    const size_t readBufferSize = 4 * 1024;

    // ファイルデータキャッシュ用のキャッシュメモリとして 32MB を確保します。
    // 最適なキャッシュメモリサイズはアプリケーションにより異なります。通常のアプリケーションでは
    // これ以上のキャッシュメモリを確保することも検討してください。
    const size_t fileDataCacheSize = 32 * 1024 * 1024;
    void* fileDataCache = std::malloc(fileDataCacheSize);
    NN_ABORT_UNLESS_NOT_NULL(fileDataCache);

    // ファイルシステム用のキャッシュメモリを確保してリソースデータのファイルシステムをマウントします。
    // (ここで確保するキャッシュメモリはファイルシステムの管理情報保持のために使用され、ファイルデータキャッシュとは無関係です)
    size_t fileSystemCacheSize;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&fileSystemCacheSize));

    void* fileSystemCache = std::malloc(fileSystemCacheSize);
    NN_ABORT_UNLESS_NOT_NULL(fileSystemCache);

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountRom(mountName, fileSystemCache, fileSystemCacheSize));

    // シーケンシャルリードのベンチマークを開始します。
    NN_LOG("sequential reading test\n");

    // ファイルデータキャッシュを有効化してシーケンシャルリードを行います。
    {
        nn::fs::EnableGlobalFileDataCache(fileDataCache, fileDataCacheSize);

        nn::TimeSpan elapsed = MeasureExecutionTime([=]() {
            RunSequentialRead(testTargetPath, readBufferSize);
        });
        NN_LOG("  cache enabled  = %lld ms.\n", elapsed.GetMilliSeconds());

        nn::fs::DisableGlobalFileDataCache();
    }
    // ファイルデータキャッシュを有効化せずにシーケンシャルリードを行います。
    {
        nn::TimeSpan elapsed = MeasureExecutionTime([=]() {
            RunSequentialRead(testTargetPath, readBufferSize);
        });
        NN_LOG("  cache disabled = %lld ms.\n", elapsed.GetMilliSeconds());
    }
    NN_LOG("\n");

    // ランダムリードのベンチマークを開始します。
    NN_LOG("random reading test\n");

    // ファイルデータキャッシュを有効化してランダムリードを行います。
    {
        nn::fs::EnableGlobalFileDataCache(fileDataCache, fileDataCacheSize);

        nn::TimeSpan elapsed = MeasureExecutionTime([=]() {
            RunRandomRead(testTargetPath, readBufferSize);
        });
        NN_LOG("  cache enabled  = %lld ms.\n", elapsed.GetMilliSeconds());

        nn::fs::DisableGlobalFileDataCache();
    }
    // ファイルデータキャッシュを有効化せずにランダムリードを行います。
    {
        nn::TimeSpan elapsed = MeasureExecutionTime([=]() {
            RunRandomRead(testTargetPath, readBufferSize);
        });
        NN_LOG("  cache disabled = %lld ms.\n", elapsed.GetMilliSeconds());
    }
    NN_LOG("\n");

    // ファイルシステムをアンマウントして、各種メモリを解放します。
    nn::fs::Unmount(mountName);

    std::free(fileSystemCache);

    std::free(fileDataCache);
}
