﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{FontDemo/Main.cpp,PageSampleFontDemo}
 *
 * @brief
 *  Font ライブラリの使い方を示したサンプルプログラム
 */

/**
 * @page PageSampleFontDemo FontDemo
 * @tableofcontents
 *
 * @image html Applications\FontDemo\fontdemo.png
 *
 * @brief
 *  Font ライブラリの使い方を示したサンプルプログラムです。
 *
 * @section PageSampleFontDemo_SectionBrief 概要
 *  ここでは、FontDemo の説明をします。
 *
 * @section PageSampleFontDemo_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/FontDemo Samples/Sources/Applications/FontDemo @endlink 以下にあります。
 *
 * @section PageSampleFontDemo_SectionNecessaryEnvironment 必要な環境
 *  特にありません。
 *
 * @section PageSampleFontDemo_SectionHowToOperate 操作方法
 * Joy-Con、DebugPad、マウス、タッチパネルを利用して以下の操作が可能です。
 * <table>
 * <tr><th> 入力 </th><th> 動作 </th></tr>
 * <tr><td> 十字ボタン上下、タッチパネルまたはマウスで項目を選択 </td><td> サンプルを選択します。 </td></tr>
 * <tr><td> A ボタン、マウス左ボタン押し </td><td> サンプルを実行します。 </td></tr>
 * <tr><td> B ボタン、マウス右ボタン押し </td><td> メニューに戻ります。 </td></tr>
 * </table>
 *
 * @section PageSampleFontDemo_SectionPrecaution 注意事項
 *  特にありません。
 *
 * @section PageSampleFontDemo_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、メニューから選び実行するサンプルを切り替えることができます。
 *
 * @section PageSampleFontDemo_SectionDetail 解説
 * 各サンプルはフォントをディスクから読み込んで、表示する最も基本的なデモです。
 *
 * メニューから選択または g_SampleMode の値の変更、または引数に 0 から始まる番号を入力することで、以下のサンプルを確認できます。
 *
 * - SampleMode_Simple          ビットマップフォントを使った表示を行います。
 * - SampleMode_PairFont        複数のフォントを組み合わせて利用するペアフォントを使って表示します。
 * - SampleMode_WordWrapping    文字列の折り返し処理の機能を使って表示します。
 * - SampleMode_ScalableFont    ttf(otf)、bfttf というスケーラブルフォント形式のフォントを使って表示します。実行時にメモリに文字画像を作成して表示を行っています。
 *
 * 他にも...
 * サイズスケール、行間隔指定、文字間隔指定、カラー変更、
 * カラーグラデーション、斜体描画、カゲ描画、
 * フチ付フォントのフチ表示ON/OFF
 * ...など多彩な機能があります。
 *
 * 詳しくは、API リファレンスをごらんください。
 * また、レイアウトライブラリのソースコードもフォントライブラリのコードサンプルとして
 * 参考にできます（font::で grep すると該当コードにたどり着けます）。
 *
 * なお、ビットマップフォントの生成には、専用のコンバータを利用します。
 * (Tools/Graphics/FontConverter 以下)
 *
 */

#include <cstdlib>

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/mem.h>
#include <nn/os.h>
#include <nn/fs.h>
#include <nn/util/util_Matrix.h>
#include <nns/gfx/gfx_GraphicsFramework.h>

#include <nn/font.h>

#include "FontCommon.h"
#include "FontSimple.h"
#include "FontPairFont.h"
#include "FontWordWrapping.h"
#include "FontScalableFont.h"
#include "FontTagProcessor.h"
#include "InputController.h"
#include "FontMenu.h"

#if defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)
#include <filesystem>
#define WIN32_LEAN_AND_MEAN
#define NOMINMAX
#include <nn/nn_Windows.h>
#endif

//------------------------------------------------------------------------------
//   定数定義
//------------------------------------------------------------------------------

static const uint32_t   ApplicationHeapSize    = 128 * 1024 * 1024;
static const uint32_t   CharCount   = 2048;
static uint32_t         ScreenWidth = 1280;
static uint32_t         ScreenHeight = 720;

// フォントバイナリファイル
static const char* const DemoFontFileName = "Contents:/sample.bffnt";

const uint16_t* commandMenuNames[] =
{
    reinterpret_cast<const uint16_t*>(NN_CHAR16LITERAL("Simple")),
    reinterpret_cast<const uint16_t*>(NN_CHAR16LITERAL("PairFont")),
    reinterpret_cast<const uint16_t*>(NN_CHAR16LITERAL("WordWrapping")),
    reinterpret_cast<const uint16_t*>(NN_CHAR16LITERAL("ScalableFont")),
    reinterpret_cast<const uint16_t*>(NN_CHAR16LITERAL("TagProcessor")),
};

//------------------------------------------------------------------------------
//   変数定義
//------------------------------------------------------------------------------

// サンプルの動作モード
static SampleMode g_SampleMode = SampleMode_CommandMenu;

SampleMode  g_NextSampleMode = g_SampleMode;

// グラフィックスフレームワーク
nns::gfx::GraphicsFramework g_GfxFramework;

static int g_AllocationCount = 0;

// メモリヒープ
static nn::util::BytePtr g_pMemoryHeap(NULL);
static nn::mem::StandardAllocator g_ApplicationHeapHandle;

// フレーム数のカウント(アニメーション用)
static int g_FrameCount = 0;

// デモで使用するフォント
static nn::font::ResFont* g_pResFont;

// フォントの描画に使用するコンスタントバッファ
static nn::font::GpuBuffer  g_ConstantBuffer;

// フォントの描画に使用するメモリプールのオフセット
static ptrdiff_t g_OffsetForMemoryPoolOfFontGpuBuffer = 0;

// RectDrawer が更新するコンスタントバッファの数
static int g_ConstantBufferCount = 0;

// フォント描画クラス
static nn::font::RectDrawer g_RectDrawer;

// RectDrawer に利用するメモリプールのオフセット
static ptrdiff_t g_OffsetForMemoryPoolOfRectDrawer = 0;

// RectDrawer に外部のメモリプールを利用するかどうか
static const bool g_IsUseExternalMemoryPoolForRectDrawer = true;

// フォント描画文字列バッファ
static nn::font::DispStringBuffer g_DispStringBuffer;

// 投影行列
static nn::util::MatrixT4x4fType g_Projection;

// ファイルシステムのメタデータキャッシュバッファ
static char* g_MountRomCacheBuffer = NULL;

// 文字列を動的に変更するかどうか（文字の頂点カラーの変更も含みます）
static bool g_DynamicTextModificationEnabled = false;

static bool g_OneTimeSample = false;

//------------------------------------------------------------------------------
//  グラフィックスフレームワーク用コールバック
//------------------------------------------------------------------------------
void CalculateCallback(nns::gfx::GraphicsFramework* pGraphicsFramework, void* pUserData);

//------------------------------------------------------------------------------
void MakeCommandCallback(nns::gfx::GraphicsFramework* pGraphicsFramework, int bufferIndex, void* pUserData);

//------------------------------------------------------------------------------
//  アプリケーションヒープからメモリを確保する
//------------------------------------------------------------------------------
void* AllocateFromApplicationHeap(size_t size, size_t alignment)
{
    void* ptr = g_ApplicationHeapHandle.Allocate(size, alignment);
    if (ptr != NULL)
    {
        g_AllocationCount++;
    }
    return ptr;
}

//------------------------------------------------------------------------------
void* AllocateFromApplicationHeap(size_t size)
{
    return AllocateFromApplicationHeap(size, 4);
}

//------------------------------------------------------------------------------
void* AllocateFromApplicationHeap(size_t size, size_t alignment, void* pUserData)
{
    NN_UNUSED(pUserData);
    return AllocateFromApplicationHeap(size, alignment);
}

//------------------------------------------------------------------------------
//  アプリケーションヒープから確保したメモリを開放する
//------------------------------------------------------------------------------
void DeallocateApplicationHeap(void* ptr)
{
    if (ptr == NULL)
    {
        return;
    }
    g_AllocationCount--;
    g_ApplicationHeapHandle.Free(ptr);
}

//------------------------------------------------------------------------------
void DeallocateApplicationHeap(void* ptr, void* pUserData)
{
    NN_UNUSED(pUserData);
    DeallocateApplicationHeap(ptr);
}

//------------------------------------------------------------------------------
bool RegisterSlotForTexture(nn::gfx::DescriptorSlot* pDstSlot, const nn::gfx::TextureView& textureView, void* pUserData)
{
    if (pUserData)
    {
        nns::gfx::GraphicsFramework* graphicsFramework = reinterpret_cast<nns::gfx::GraphicsFramework*>(pUserData);

        const int slotIndex = graphicsFramework->AllocateDescriptorSlot(nn::gfx::DescriptorPoolType::DescriptorPoolType_TextureView, 1);
        if (slotIndex == nn::gfx::util::DescriptorPoolAllocator::InvalidIndex)
        {
            return false;
        }

        nn::gfx::DescriptorPool* pPool = graphicsFramework->GetDescriptorPool(nn::gfx::DescriptorPoolType::DescriptorPoolType_TextureView);

        pPool->BeginUpdate();
        {
            pPool->SetTextureView(slotIndex, &textureView);
        }
        pPool->EndUpdate();
        pPool->GetDescriptorSlot(pDstSlot, slotIndex);
        return true;
    }

    return false;
}

//------------------------------------------------------------------------------
bool RegisterSlotForSampler(nn::gfx::DescriptorSlot* pDstSlot, const nn::gfx::Sampler& sampler, void* pUserData)
{
    if (pUserData)
    {
        nns::gfx::GraphicsFramework* graphicsFramework = reinterpret_cast<nns::gfx::GraphicsFramework*>(pUserData);

        const int slotIndex = graphicsFramework->AllocateDescriptorSlot(nn::gfx::DescriptorPoolType::DescriptorPoolType_Sampler, 1);
        if (slotIndex == nn::gfx::util::DescriptorPoolAllocator::InvalidIndex)
        {
            return false;
        }

        nn::gfx::DescriptorPool* pPool = graphicsFramework->GetDescriptorPool(nn::gfx::DescriptorPoolType::DescriptorPoolType_Sampler);

        pPool->BeginUpdate();
        {
            pPool->SetSampler(slotIndex, &sampler);
        }
        pPool->EndUpdate();
        pPool->GetDescriptorSlot(pDstSlot, slotIndex);
        return true;
    }

    return false;
}

//------------------------------------------------------------------------------
void UnregisterSlotForTexture(nn::gfx::DescriptorSlot* pDstSlot, const nn::gfx::TextureView& textureView, void* pUserData)
{
    NN_UNUSED(textureView);
    if (pUserData)
    {
        nns::gfx::GraphicsFramework* graphicsFramework = reinterpret_cast<nns::gfx::GraphicsFramework*>(pUserData);
        nn::gfx::DescriptorPool* pPool = graphicsFramework->GetDescriptorPool(nn::gfx::DescriptorPoolType::DescriptorPoolType_TextureView);
        int slotIndex = pPool->GetDescriptorSlotIndex(*pDstSlot);
        graphicsFramework->FreeDescriptorSlot(nn::gfx::DescriptorPoolType::DescriptorPoolType_TextureView, slotIndex);
    }
}

//------------------------------------------------------------------------------
void UnregisterSlotForSampler(nn::gfx::DescriptorSlot* pDstSlot, const nn::gfx::Sampler& sampler, void* pUserData)
{
    NN_UNUSED(sampler);
    if (pUserData)
    {
        nns::gfx::GraphicsFramework* graphicsFramework = reinterpret_cast<nns::gfx::GraphicsFramework*>(pUserData);
        nn::gfx::DescriptorPool* pPool = graphicsFramework->GetDescriptorPool(nn::gfx::DescriptorPoolType::DescriptorPoolType_Sampler);
        const int slotIndex = pPool->GetDescriptorSlotIndex(*pDstSlot);
        graphicsFramework->FreeDescriptorSlot(nn::gfx::DescriptorPoolType::DescriptorPoolType_Sampler, slotIndex);
    }

}

//------------------------------------------------------------------------------
// GpuBuffer 用メモリ割り当て・破棄関数
//------------------------------------------------------------------------------
static void* GpuBufferAllocateFunction(size_t size, size_t alignment, void* pUserData)
{
    NN_UNUSED(pUserData);
    return AllocateFromApplicationHeap(size, alignment);
}

//------------------------------------------------------------------------------
static void GpuBufferDeallocateFunction(void* ptr, void* pUserData)
{
    NN_UNUSED(pUserData);
    DeallocateApplicationHeap(ptr);
}

//------------------------------------------------------------------------------
// nn::fs用メモリ割り当て・破棄関数
//------------------------------------------------------------------------------
static void* FsAllocateFunction( size_t size )
{
    return malloc(size);
}

//------------------------------------------------------------------------------
static void FsDeallocateFunction( void* ptr, size_t size )
{
    NN_UNUSED( size );
    free(ptr);
}

//---------------------------------------------------------------------------
// 表示する文字列の最大文字数を指定して、表示文字列用バッファを確保します。
//---------------------------------------------------------------------------
static void InitDispStringBuffer()
{
    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();

    nn::font::DispStringBuffer::InitializeArg initializeArg;
    initializeArg.SetCharCountMax( CharCount );

    // 指定した文字数分バッファを確保
    const size_t drawBufSize = nn::font::DispStringBuffer::GetRequiredDrawBufferSize( initializeArg );
    if( drawBufSize > 0 )
    {
        initializeArg.SetDrawBuffer(AllocateFromApplicationHeap(drawBufSize));
    }

    nn::gfx::MemoryPoolInfo memPoolInfo;
    memPoolInfo.SetDefault();
    memPoolInfo.SetMemoryPoolProperty( nn::gfx::MemoryPoolProperty_CpuUncached | nn::gfx::MemoryPoolProperty_GpuCached );

    // コンスタントバッファは毎フレーム更新するためダブルバッファリング用に２つ確保します。
    g_ConstantBufferCount = 2;

    nn::font::GpuBuffer::InitializeArg  arg;
    arg.SetGpuAccessFlag(nn::gfx::GpuAccess_ConstantBuffer);

    const size_t constantBufferSize = nn::font::DispStringBuffer::GetRequiredConstantBufferSize(pDevice, initializeArg);
    const size_t memoryPoolGranularity = nn::gfx::MemoryPool::GetPoolMemorySizeGranularity(pDevice, memPoolInfo);
    const size_t memoryPoolSize = nn::util::align_up(constantBufferSize * g_ConstantBufferCount, memoryPoolGranularity);
    const size_t MemoryPoolAlignment = nn::gfx::MemoryPool::GetPoolMemoryAlignment(pDevice, memPoolInfo);

    // フレームワークからメモリプールをアロケート(オフセットを受け取る)
    nn::gfx::MemoryPool *pMemoryPool = g_GfxFramework.GetMemoryPool(nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer);
    g_OffsetForMemoryPoolOfFontGpuBuffer = g_GfxFramework.AllocatePoolMemory(
        nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer, memoryPoolSize, MemoryPoolAlignment);

    // font 用のコンスタントバッファを作成する。
    arg.SetMemoryPool(pMemoryPool);
    arg.SetMemoryPoolOffset(g_OffsetForMemoryPoolOfFontGpuBuffer);

    arg.SetBufferSize(constantBufferSize);
    arg.SetBufferCount(g_ConstantBufferCount);
    arg.SetAllocator(GpuBufferAllocateFunction, NULL);
    arg.SetAllocateSyncFlag(false);

    g_ConstantBuffer.Initialize(pDevice, arg);

    initializeArg.SetConstantBuffer(&g_ConstantBuffer);

    // 表示文字列バッファを初期化
    bool result = g_DispStringBuffer.Initialize( pDevice, initializeArg );

    NN_ASSERT( result );
}

//------------------------------------------------------------------------------
// 描画クラスを初期化します。
//------------------------------------------------------------------------------
static void InitRectDrawer()
{
    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();

    const size_t workBufferSize = nn::font::RectDrawer::GetWorkBufferSize(pDevice, CharCount);
    const size_t WorkBufferAlignment = nn::font::RectDrawer::GetWorkBufferAlignment();

    // RectDrawer::Initialize でメモリープールの指定を省略した場合内部で自動的に初期化されます。
    if (NN_STATIC_CONDITION(g_IsUseExternalMemoryPoolForRectDrawer))
    {
        // フレームワークからメモリプールをアロケート(オフセットを受け取る)
        const size_t ExternalMemoryPoolSize = nn::font::RectDrawer::CalculateMemoryPoolSize(pDevice, CharCount);
        g_OffsetForMemoryPoolOfRectDrawer = g_GfxFramework.AllocatePoolMemory(
            nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer,
            ExternalMemoryPoolSize,
            nn::font::RectDrawer::CalculateMemoryPoolAlignment(pDevice));

        // workBufferSize から、externalMemoryPoolSize を減らしていることに注意してください。
        void* pWorkBuffer = AllocateFromApplicationHeap(workBufferSize - ExternalMemoryPoolSize, WorkBufferAlignment);
        g_RectDrawer.Initialize(
            pDevice,
            pWorkBuffer,
            CharCount,
            g_GfxFramework.GetMemoryPool(nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer),
            g_OffsetForMemoryPoolOfRectDrawer,
            ExternalMemoryPoolSize);
    }
    else
    {
        g_OffsetForMemoryPoolOfRectDrawer = NULL;

        void* pWorkBuffer = AllocateFromApplicationHeap(workBufferSize, WorkBufferAlignment);
        g_RectDrawer.Initialize(pDevice, pWorkBuffer, CharCount);
    }

    g_RectDrawer.RegisterSamplerToDescriptorPool(RegisterSlotForSampler, &g_GfxFramework);
}

//------------------------------------------------------------------------------
// フォントの処理
//------------------------------------------------------------------------------
static void InitializeFont()
{
    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();

    {
        void* binary = NULL;

        {
            nn::Result result;
            nn::fs::FileHandle fileHandle;
            int64_t fileSize;

            result = nn::fs::OpenFile(&fileHandle, DemoFontFileName, nn::fs::OpenMode_Read);
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);

            result = nn::fs::GetFileSize(&fileSize, fileHandle);
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);

            binary = AllocateFromApplicationHeap(static_cast<size_t>(fileSize), nn::font::ResourceAlignment);
            size_t readSize;
            result = nn::fs::ReadFile(&readSize, fileHandle, 0, binary, static_cast<size_t>(fileSize));
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);

            nn::fs::CloseFile(fileHandle);
        }
        NN_ASSERT_NOT_NULL( binary );

        //-----------------------------------------
        // フォントの初期化
        g_pResFont = AllocAndConstruct<nn::font::ResFont>();
        NN_ASSERT_NOT_NULL(g_pResFont);
        bool result = g_pResFont->SetResource( pDevice, binary );
        NN_ASSERT( result );
        g_pResFont->RegisterTextureViewToDescriptorPool(RegisterSlotForTexture, &g_GfxFramework);
    }

    //-----------------------------------------
    // RectDrawer の初期化
    InitRectDrawer();

    //-----------------------------------------
    // DispStringBuffer の初期化
    InitDispStringBuffer();

    //-----------------------------------------
    // 各サンプルの初期化
    switch (g_SampleMode)
    {
    case SampleMode_Simple:
        InitializeSimple();
        break;
    case SampleMode_PairFont:
        InitializePairFont(g_pResFont);
        break;
    case SampleMode_WordWrapping:
        // ワードラップデモでは、動的に文字内容を変更します。
        g_DynamicTextModificationEnabled = true;
        InitializeWordWrappingText();
        InitializeWordWrapping(ScreenHeight, g_pResFont->GetHeight());
        break;
    case SampleMode_ScalableFont:
        InitializeScalableFont();
        break;
    case SampleMode_TagProcessor:
        InitializeTagProcessor();
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }

    //-----------------------------------------
    // プロジェクション行列の計算
    nn::util::MatrixOrthographicOffCenterRightHanded(
        &g_Projection,
        0 /*left*/,
        static_cast<float>( ScreenWidth ) /*right*/,
        static_cast<float>( ScreenHeight ) /*bottom*/,
        0 /*top*/,
        0.0f /*near*/,
        300.0f /*far*/ );

    g_GfxFramework.SetMakeCommandCallback(MakeCommandCallback, nullptr);
}

//------------------------------------------------------------------------------
// デモ共有の初期化
//------------------------------------------------------------------------------
static void InitDemoCommon()
{
    // nn::fs の初期化(gfx の初期化よりも前に実行します)
    {
        nn::fs::SetAllocator( FsAllocateFunction, FsDeallocateFunction );

        size_t cacheSize = 0;
        nn::Result result = nn::fs::QueryMountRomCacheSize(&cacheSize);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        g_MountRomCacheBuffer = new(std::nothrow) char[cacheSize];
        NN_ABORT_UNLESS_NOT_NULL(g_MountRomCacheBuffer);

        result = nn::fs::MountRom("Contents", g_MountRomCacheBuffer, cacheSize);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        NN_UNUSED(result);
    }

    // フレームワーク初期化
    nns::gfx::GraphicsFramework::InitializeGraphicsSystem(1024 * 1024 * 32);
    nns::gfx::GraphicsFramework::FrameworkInfo fwInfo;
    fwInfo.SetDefault();
    fwInfo.SetDisplayWidth(1280);
    fwInfo.SetDisplayHeight(720);
    fwInfo.SetBufferCount(2);
    fwInfo.SetSwapChainBufferCount(2);

    g_GfxFramework.Initialize(fwInfo);
    g_GfxFramework.SetCalculateCallback(CalculateCallback, nullptr);
}

//------------------------------------------------------------------------------
// 初期化処理の呼び出し
//------------------------------------------------------------------------------
static void Initialize()
{
    // メモリヒープの初期化
    g_pMemoryHeap.Reset(new uint8_t[ApplicationHeapSize]);
    g_ApplicationHeapHandle.Initialize(g_pMemoryHeap.Get(), ApplicationHeapSize);

    // デモシステム共有の初期化
    InitDemoCommon();
}

//------------------------------------------------------------------------------
// フォントの終了処理
//------------------------------------------------------------------------------
static void FinalizeFont()
{
    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();
    DeallocateApplicationHeap(g_DispStringBuffer.GetDrawBuffer());

    g_DispStringBuffer.Finalize( pDevice );

    DeallocateApplicationHeap(g_RectDrawer.GetBuffer());
    g_RectDrawer.UnregisterSamplerFromDescriptorPool(UnregisterSlotForSampler, &g_GfxFramework);
    g_RectDrawer.Finalize( pDevice );

    // フォントの破棄
    g_pResFont->UnregisterTextureViewFromDescriptorPool(UnregisterSlotForTexture, &g_GfxFramework);

    void* pResource = g_pResFont->RemoveResource( pDevice );
    DeallocateApplicationHeap(pResource);
    g_pResFont->Finalize( pDevice );

    DestructAndFree<nn::font::ResFont>(g_pResFont);
    g_pResFont = NULL;

    switch (g_SampleMode)
    {
    case SampleMode_Simple:
        FinalizeSimple();
        break;
    case SampleMode_PairFont:
        FinalizePairFont();
        break;
    case SampleMode_WordWrapping:
        FinalizeWordWrapping();
        break;
    case SampleMode_ScalableFont:
        FinalizeScalableFont();
        break;
    case SampleMode_TagProcessor:
        FinalizeTagProcessor();
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }

    g_ConstantBuffer.Finalize(pDevice, GpuBufferDeallocateFunction, NULL);

    g_DynamicTextModificationEnabled = false;
}

//------------------------------------------------------------------------------
// 終了処理の呼び出し
//------------------------------------------------------------------------------
static void Finalize()
{
    g_GfxFramework.Finalize();

    nn::fs::Unmount("Contents");

    delete[] g_MountRomCacheBuffer;
    g_MountRomCacheBuffer = NULL;

    // アロケータの終了処理
    g_ApplicationHeapHandle.Finalize();

    delete[] reinterpret_cast<uint8_t*>(g_pMemoryHeap.Get());
}

//---------------------------------------------------------------------------
// バッファ生成処理
//---------------------------------------------------------------------------
static void PrintStaticText()
{
    // 表示する文字列を設定します。
    //
    // 描画する文字列や追加のパラメータが変わった時は文字列描画情報の再生成が必要です。
    {
        // 描画設定
        nn::font::WideTextWriter writer;
        writer.SetDispStringBuffer( &g_DispStringBuffer );

        switch(g_SampleMode)
        {
        case SampleMode_Simple:
            writer.SetFont(g_pResFont);
            break;
        case SampleMode_PairFont:
            SetPairFont(&writer);
            break;
        case SampleMode_WordWrapping:
            writer.SetFont(g_pResFont);
            break;
        case SampleMode_ScalableFont:
            SetScalableFont(&writer);
            break;
        case SampleMode_TagProcessor:
            writer.SetFont(g_pResFont);
            break;
        default: NN_UNEXPECTED_DEFAULT;
        }

        writer.SetCursor( 50, 50 );
        writer.SetScale( 1.0f, 1.0f );

        // 文字上端、下端カラーの設定はここで行います。
        // 上端、下端カラー文字列描画情報に格納されるため、
        // カラーを変更した場合、Print 処理を再度行わなければならない点に注意してください。
        //nn::util::Unorm8x4 topColor = {0, 255, 0, 255};
        //nn::util::Unorm8x4 bottomColor = {0, 255, 0, 255};
        //writer.SetTextColor(topColor, bottomColor);

        writer.StartPrint();
        {
            writer.Print( FONT_SAMPLE_LITERAL("SAMPLE: FontDemo\n") );
            writer.Print( FONT_SAMPLE_LITERAL("\n"));
            writer.Print( FONT_SAMPLE_LITERAL("abcdefghijklmnopqrstuvwxyz\n") );
            writer.Print( FONT_SAMPLE_LITERAL("ABCDEFGHIJKLMNOPQRSTUVWXYZ\n") );
            writer.Print( FONT_SAMPLE_LITERAL("漢字「本日は晴天なり」\n") );

            if (NN_STATIC_CONDITION(g_SampleMode == SampleMode_ScalableFont))
            {
                PrintScalableFont(&writer);
            }
        }
        writer.EndPrint();
    }
}

//---------------------------------------------------------------------------
// 更新処理
//---------------------------------------------------------------------------
static void Calc()
{
    g_ConstantBuffer.Map(g_FrameCount % g_ConstantBufferCount);

    if (NN_STATIC_CONDITION(g_SampleMode == SampleMode_WordWrapping))
    {
        CalcWordWrapping(g_DispStringBuffer, *g_pResFont, ScreenWidth);
    }

    // 定数バッファを生成します。
    //
    // 定数バッファには射影行列および nn::font::ConstantBufferAdditionalContent で指定される
    // 白黒補間カラーなどの追加の描画パラメータが反映されます。
    //
    // 射影行列や追加の描画パラメータの内容が変わったときは定数バッファの再生成が必要です。
    {
        nn::font::ConstantBufferAdditionalContent additionalContent;

        // 白黒補間カラー:
        // TextWriter::Print 処理は高負荷の処理なので、
        // 必要がなければカラーのアニメーションは定数バッファの白黒カラー補間で済ませる方が
        // パフォーマンス的に有利です。
        const nn::util::Unorm8x4 black = {{0, 0, 0, 255}};
        const nn::util::Unorm8x4 white = {{255, 255, 255, 255}};
        additionalContent
            .SetInterpolateBlack( black )
            .SetInterpolateWhite( white )
            .SetInterpolateAlpha( g_FrameCount % 256 );

        g_DispStringBuffer.BuildConstantBuffer(
            g_Projection,
            &additionalContent);
    }
    g_ConstantBuffer.Unmap();
    g_ConstantBuffer.SetGpuAccessBufferIndex(g_FrameCount % g_ConstantBufferCount);
}

//------------------------------------------------------------------------------
// 描画ステートを設定します。
//
// フォントは基本的にデプステストなし、カリングなしで描画します。
// これらの設定はフォント内では行っていないため、Drawを呼び出す前に設定しておく必要があります。
// 逆にアルファテスト及びブレンドの設定はフォント内で書き換えるため注意が必要です。
//------------------------------------------------------------------------------
static void SetupDrawState(nn::gfx::CommandBuffer* pCommandBuffer)
{
    nn::gfx::ColorTargetView* pTarget = g_GfxFramework.GetColorTargetView();
    nn::gfx::DepthStencilView* pDepthStencilView = g_GfxFramework.GetDepthStencilView();

    pCommandBuffer->ClearColor(
        pTarget,
        0.2f /*red*/,
        0.2f /*green*/,
        0.2f /*blue*/,
        1.0f /*alpha*/,
        NULL /*pArrayRange*/ );

    pCommandBuffer->ClearDepthStencil(
        pDepthStencilView,
        1.0f /*depth*/,
        0 /*stencil*/,
        nn::gfx::DepthStencilClearMode_DepthStencil,
        NULL /*pArrayRange*/ );

    pCommandBuffer->SetRenderTargets( 1, &pTarget, pDepthStencilView );
    pCommandBuffer->SetViewportScissorState( g_GfxFramework.GetViewportScissorState() );

    pCommandBuffer->SetBlendState(
        g_GfxFramework.GetBlendState(nns::gfx::GraphicsFramework::BlendStateType_Alpha) );
    pCommandBuffer->SetDepthStencilState(
        g_GfxFramework.GetDepthStencilState(nns::gfx::GraphicsFramework::DepthStencilStateType_Disabled) );
    pCommandBuffer->SetRasterizerState(
        g_GfxFramework.GetRasterizerState(nns::gfx::GraphicsFramework::RasterizerStateType_FillSolid_CullNone) );
}

//------------------------------------------------------------------------------
//  計算処理コールバック
//------------------------------------------------------------------------------
void CalculateCallback(nns::gfx::GraphicsFramework* pGraphicsFramework, void* pUserData)
{
    NN_UNUSED(pGraphicsFramework);
    NN_UNUSED(pUserData);

    // 更新
    switch (g_SampleMode)
    {
    case SampleMode_Simple:
    case SampleMode_PairFont:
    case SampleMode_WordWrapping:
    case SampleMode_TagProcessor:
        Calc();
        break;
    case SampleMode_ScalableFont:
        if (g_FrameCount == 60 * 5)
        {
            // フォントの優先順位を変更して再描画する
            ChangeFontOrder();
            PrintStaticText();
        }
        Calc();
        break;
    case SampleMode_CommandMenu:
        CalculateMenu();
        break;
    default: NN_UNEXPECTED_DEFAULT;
        break;
    }
}

//------------------------------------------------------------------------------
//  コマンド生成コールバック
//------------------------------------------------------------------------------
void MakeCommandCallback(nns::gfx::GraphicsFramework* pGraphicsFramework, int bufferIndex, void* pUserData)
{
    NN_UNUSED(pGraphicsFramework);
    NN_UNUSED(pUserData);

    g_GfxFramework.BeginFrame(bufferIndex);
    {
        nn::gfx::CommandBuffer* pCommandBuffer = g_GfxFramework.GetRootCommandBuffer(bufferIndex);

        // 描画ステートの設定
        SetupDrawState(pCommandBuffer);

        // 描画
        g_RectDrawer.Draw(*pCommandBuffer, g_DispStringBuffer);
    }
    g_GfxFramework.EndFrame(bufferIndex);
}

//------------------------------------------------------------------------------
// アプリケーションの初期化処理
//------------------------------------------------------------------------------
static void InitializeApp()
{
    // ランタイムの初期化処理
    switch (g_SampleMode)
    {
    case SampleMode_Simple:
    case SampleMode_PairFont:
    case SampleMode_WordWrapping:
    case SampleMode_ScalableFont:

        InitializeFont();

        // 描画文字列から描画情報を生成します。
        // 動的な文字変更を行わない場合、一度だけ行えばよいで処理です。
        if (!g_DynamicTextModificationEnabled)
        {
            PrintStaticText();
        }
        break;
    case SampleMode_TagProcessor:
        InitializeFont();
        // タグを使った文字列から描画情報を生成します。
        PrintStaticTextOfTagProcessor(g_DispStringBuffer, *g_pResFont);
        break;
    case SampleMode_CommandMenu:
        InitializeMenu(commandMenuNames, NN_ARRAY_SIZE(commandMenuNames));
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }
}

//------------------------------------------------------------------------------
// アプリケーションの終了処理
//------------------------------------------------------------------------------
static void FinalizeApp()
{
    switch (g_SampleMode)
    {
    case SampleMode_Simple:
    case SampleMode_PairFont:
    case SampleMode_WordWrapping:
    case SampleMode_ScalableFont:
    case SampleMode_TagProcessor:
        // フォントの終了処理
        FinalizeFont();
        break;
    case SampleMode_CommandMenu:
        FinalizeMenu();
        break;
    default: NN_UNEXPECTED_DEFAULT;
        break;
    }
}

//------------------------------------------------------------------------------
//  メイン 関数
//------------------------------------------------------------------------------
extern "C" void nnMain()
{
    // 引数から起動
    if (nn::os::GetHostArgc() > 1)
    {
        int idxDemo = atoi(nn::os::GetHostArgv()[1]);
        g_NextSampleMode = g_SampleMode = (SampleMode)std::max(std::min(idxDemo, (int)SampleMode_ScalableFont), (int)SampleMode_Simple);
    }

    // メニュー以外の直接起動は１度のみの再生モードで実行
    if (g_SampleMode != SampleMode_CommandMenu)
    {
        g_OneTimeSample = true;
    }

    // 初期化
    Initialize();

    // ウィンドウハンドルを取得する
    nn::vi::NativeWindowHandle hwnd = NULL;
    nn::Result result = nn::vi::GetNativeWindow(&hwnd, g_GfxFramework.GetLayer());
    NN_ASSERT(result.IsSuccess());

#if defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)
    SetForegroundWindow(static_cast<HWND>(hwnd));
#endif

    const int displayWidth = g_GfxFramework.GetDisplayWidth();
    const int displayHeight = g_GfxFramework.GetDisplayHeight();
    InitializeInputController(displayWidth, displayHeight, reinterpret_cast<void*>(hwnd));

    // メインループ
    NN_LOG("Start demo.\n");
    for(;;)
    {
        //------------------------------------------
        // 初期化
        InitializeApp();

        //------------------------------------------
        g_GfxFramework.ResetFrame();

        g_FrameCount = 0;

        for(;;)
        {
            UpdateInputController();

            g_GfxFramework.ProcessFrame();

            if (g_SampleMode != SampleMode_CommandMenu)
            {
                g_FrameCount++;

                if (g_FrameCount >= 60 * 10 || GetPad().IsTriggered(Pad::BUTTON_B) || GetMouseState().buttons.Test(nn::hid::MouseButton::Right::Index))
                {
                    if (g_OneTimeSample == true)
                    {
                        g_NextSampleMode = SampleMode_Exit;
                    }
                    else
                    {
                        g_NextSampleMode = SampleMode_CommandMenu;
                    }
                }
            }

            // モードが変わった場合
            if (g_NextSampleMode != g_SampleMode)
            {
                g_GfxFramework.QueueFinish();

                //------------------------------------------
                // 終了処理
                FinalizeApp();

                // グラフィックスフレームワーク 終了処理
                if (g_OffsetForMemoryPoolOfFontGpuBuffer != 0)
                {
                    g_GfxFramework.FreePoolMemory(
                        nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer,
                        g_OffsetForMemoryPoolOfFontGpuBuffer);
                    g_OffsetForMemoryPoolOfFontGpuBuffer = 0;
                }

                if (g_OffsetForMemoryPoolOfRectDrawer != 0)
                {
                    g_GfxFramework.FreePoolMemory(
                        nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer,
                        g_OffsetForMemoryPoolOfRectDrawer);
                    g_OffsetForMemoryPoolOfRectDrawer = 0;
                }

                g_SampleMode = g_NextSampleMode;
                break;
            }
        }

        if (g_SampleMode == SampleMode_Exit)
        {
            break;
        }
    }

    // 終了処理
    Finalize();

    NN_ASSERT(g_AllocationCount == 0);
    NN_LOG( "End demo.\n" );
}
