﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/ovln/ovln_CommonTypes.h>
#include <nn/ovln/ovln_FormatForOverlay.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <cstring>

namespace
{

const char* GetBatteryStateChangeCauseName(uint32_t tag)
{
    switch (tag)
    {
    case nn::ovln::format::PowerManagementChargerStateChangeDataTag:
        return "ChargerStateChanged";
    case nn::ovln::format::PowerManagementLowBatteryDataTag:
        return "LowBattery";
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

const char* GetChargerTypeName(nn::ovln::format::ChargerType chargerType)
{
    switch (chargerType)
    {
    case nn::ovln::format::ChargerType_Unconnected:
        return "Unconnected";
    case nn::ovln::format::ChargerType_EnoughPower:
        return "EnoughPower";
    case nn::ovln::format::ChargerType_LowPower:
        return "LowPower";
    case nn::ovln::format::ChargerType_NotSupported:
        return "NotSupported";
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

}

void DispatchPowerManagementMessage(
    const nn::ovln::Message& message) NN_NOEXCEPT
{
    switch (message.tag)
    {
        case nn::ovln::format::PowerManagementChargerStateChangeDataTag:
        case nn::ovln::format::PowerManagementLowBatteryDataTag:
        {
            nn::ovln::format::BatteryStateChangeData data;
            if (message.dataSize == sizeof(data))
            {
                std::memcpy(&data, &message.data, sizeof(data));
                NN_LOG("[DispatchPowerManagementMessage] battery state change message:\n");

                NN_LOG("    cause: %s\n", GetBatteryStateChangeCauseName(message.tag));

                int batteryCharge = nn::ovln::format::GetBatteryChargePercentage(data);
                NN_LOG("    battery charge: %d %%\n", batteryCharge);

                nn::ovln::format::ChargerType chargerType
                    = nn::ovln::format::GetChargerType(data);
                NN_LOG("    usb: %s\n", GetChargerTypeName(chargerType));
            }
            else
            {
                NN_LOG("[DispatchPowerManagementMessage] invalid message: tag = %u\n", message.tag);
            }
            return;
        }
        default:
        {
            NN_LOG("[DispatchPowerManagementMessage] unknown message: tag = %u\n", message.tag);
            return;
        }
    }
}
