﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/ovln/ovln_CommonTypes.h>
#include <nn/ovln/ovln_FormatForOverlay.h>
#include <nn/capsrv/capsrv_AlbumEntry.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <cstring>

namespace {

    void PrintReason(nn::ovln::format::ScreenShotReason reason)
    {
        if(reason == nn::ovln::format::ScreenShotReason::CaptureButtonPressed)
        {
            NN_LOG("    reason   : capture button is pressed\n");
        }
        else
        {
            NN_LOG("    reason   : unknown\n");
        }
    }

    void PrintResult(nn::ovln::format::ScreenShotResult result)
    {
        if(result == nn::ovln::format::ScreenShotResult::Success)
        {
            NN_LOG("    result   : success\n");
        }
        else if(result == nn::ovln::format::ScreenShotResult::Failure)
        {
            NN_LOG("    result   : failure\n");
        }
        else if(result == nn::ovln::format::ScreenShotResult::NotPermitted)
        {
            NN_LOG("    result   : not permitted\n");
        }
        else if(result == nn::ovln::format::ScreenShotResult::AlbumIsFull)
        {
            NN_LOG("    result   : album is full\n");
        }
        else if(result == nn::ovln::format::ScreenShotResult::AlbumFileCountLimit)
        {
            NN_LOG("    result   : album file count limit\n");
        }
        else if(result == nn::ovln::format::ScreenShotResult::ThumbnailIsReady)
        {
            NN_LOG("    result   : thumbnail is ready\n");
        }
        else if(result == nn::ovln::format::ScreenShotResult::AlbumFileSystemError)
        {
            NN_LOG("    result   : album file system error\n");
        }
        else if(result == nn::ovln::format::ScreenShotResult::ScreenCaptureError)
        {
            NN_LOG("    result   : screen capture error\n");
        }
        else
        {
            NN_LOG("    result   : unknown\n");
        }
    }

    void PrintIsThumbnailAvailable(bool value)
    {
        if(value)
        {
            NN_LOG("    thumbnail: available\n");
        }
        else
        {
            NN_LOG("    thumbnail: none\n");
        }
    }

    void PrintAlbumEntry(const nn::capsrv::AlbumEntry& entry)
    {
        NN_LOG("    fileEntry:\n");
        // ファイルサイズ
        int64_t size = entry.size;
        NN_LOG("        size       : %lld\n", size);
        // アプリケーション ID
        auto appId = entry.fileId.applicationId;
        NN_LOG("        application: %016llX\n", appId.value);
        // 撮影時刻
        const auto& time = entry.fileId.time;
        NN_LOG("        date       : %04d/%02d/%02d %02d:%02d:%02d-%02d\n",
            static_cast<int>(time.year),
            static_cast<int>(time.month),
            static_cast<int>(time.day),
            static_cast<int>(time.hour),
            static_cast<int>(time.minute),
            static_cast<int>(time.second),
            static_cast<int>(time.id)
        );
        // 保存ストレージ
        auto storage = entry.fileId.storage;
        if(storage == nn::capsrv::AlbumStorage_Nand)
        {
            NN_LOG("        storage    : NAND\n");
        }
        else if(storage == nn::capsrv::AlbumStorage_Sd)
        {
            NN_LOG("        storage    : SD\n");
        }
        else
        {
            NN_LOG("        storage    : unknown\n");
        }
        // ファイルの種類
        auto contents = entry.fileId.contents;
        if(contents == nn::capsrv::AlbumFileContents_ScreenShot)
        {
            NN_LOG("        contents   : ScreenShot\n");
        }
        else
        {
            NN_LOG("        contents   : unknown\n");
        }
    }


}

void DispatchCapsrvMessage(const nn::ovln::Message& message) NN_NOEXCEPT
{
    switch (message.tag)
    {
    case nn::ovln::format::ScreenShotSuccessDataTag:
        {
            nn::ovln::format::ScreenShotCapturedData data;
            if (message.dataSize == sizeof(data))
            {
                std::memcpy(&data, &message.data, sizeof(data));
                NN_LOG("[DispatchCapsrvMessage] ScreenShotSuccess message: tag = %u\n", message.tag);
                PrintReason(data.reason);
                PrintResult(data.result);
                PrintIsThumbnailAvailable(data.IsThumbnailAvailable());
                PrintAlbumEntry(data.GetAlbumEntry());
            }
            else
            {
                NN_LOG("[DispatchCapsrvMessage] invalid message: tag = %u\n", message.tag);
            }
            return;
        }
    case nn::ovln::format::ScreenShotFailureDataTag:
        {
            nn::ovln::format::ScreenShotCapturedData data;
            if (message.dataSize == sizeof(data))
            {
                std::memcpy(&data, &message.data, sizeof(data));
                NN_LOG("[DispatchCapsrvMessage] ScreenShotFailure message: tag = %u\n", message.tag);
                PrintReason(data.reason);
                PrintResult(data.result);
                if (
                    data.result == nn::ovln::format::ScreenShotResult::AlbumIsFull ||
                    data.result == nn::ovln::format::ScreenShotResult::AlbumFileCountLimit
                    )
                {
                    PrintAlbumEntry(data.GetAlbumEntry());
                }
            }
            else
            {
                NN_LOG("[DispatchCapsrvMessage] invalid message: tag = %u\n", message.tag);
            }
            return;
        }
    case nn::ovln::format::ScreenShotIsNotPermittedDataTag:
        {
            nn::ovln::format::ScreenShotCapturedData data;
            if (message.dataSize == sizeof(data))
            {
                std::memcpy(&data, &message.data, sizeof(data));
                NN_LOG("[DispatchCapsrvMessage] ScreenShotIsNotPermitted message: tag = %u\n", message.tag);
                PrintReason(data.reason);
                PrintResult(data.result);
            }
            else
            {
                NN_LOG("[DispatchCapsrvMessage] invalid message: tag = %u\n", message.tag);
            }
            return;
        }
    case nn::ovln::format::ScreenShotThumbnailIsReadyDataTag:
        {
            nn::ovln::format::ScreenShotCapturedData data;
            if (message.dataSize == sizeof(data))
            {
                std::memcpy(&data, &message.data, sizeof(data));
                NN_LOG("[DispatchCapsrvMessage] ScreenShotThumbnailIsReady message: tag = %u\n", message.tag);
                PrintReason(data.reason);
                PrintResult(data.result);
                PrintIsThumbnailAvailable(data.IsThumbnailAvailable());
                PrintAlbumEntry(data.GetAlbumEntry());
            }
            else
            {
                NN_LOG("[DispatchCapsrvMessage] invalid message: tag = %u\n", message.tag);
            }
            return;
        }
    default:
        {
            NN_LOG("[DispatchCapsrvMessage] unknown message: tag = %u\n", message.tag);
            return;
        }
    }
}

