﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#define NN_PERF_PROFILE_ENABLED

#include <nn/perf.h>

#include <nn/util/util_Color.h>
#include <nn/util/util_FormatString.h>

#include "Demo1Npad.h"
#include "Demo1PluginManager.h"
#include "Demo1Time.h"


//!< 共通操作説明を描画します。
void ScenePluginBase::WriteCommonGuide(
    nn::gfx::util::DebugFontTextWriter* pTextWriter,
    const char* pName) const NN_NOEXCEPT
{
    const float CommonGuideOffsetX = 20;
    const float CommonGuideOffsetY = 150.0f;
    const float ClockX = 20.0f;
    const float ClockY = 80.0f;
    const float NoUserAccountX = 20.0f;
    const float NoUserAccountY = 100.0f;
    const float NoSpaceX = NoUserAccountX;
    const float NoSpaceY = NoUserAccountY;

    pTextWriter->SetTextColor(nn::util::Color4u8::White());
    pTextWriter->SetScale(1.0f, 1.0f);
    float fNameStart = static_cast<float>(
        m_pGraphicsSystem->GetDisplayWidth() - pTextWriter->CalculateStringWidth(pName) - 10.0);
    pTextWriter->SetCursor(fNameStart, 0.0f );

    pTextWriter->Print(pName);

    pTextWriter->SetTextColor(nn::util::Color4u8::White());
    pTextWriter->SetScale(NormalFontScaleX, NormalFontScaleY);
    pTextWriter->SetCursor(CommonGuideOffsetX, CommonGuideOffsetY + 40.0f);
    pTextWriter->Print("Next Demo");

    pTextWriter->SetCursor(CommonGuideOffsetX, CommonGuideOffsetY + 60.0f);
    pTextWriter->Print("Prev Demo");

    pTextWriter->SetCursor(CommonGuideOffsetX, CommonGuideOffsetY + 80.0f);
    pTextWriter->Print("Finish Demo");

    pTextWriter->SetCursor(CommonGuideOffsetX + 105.0f, CommonGuideOffsetY + 40.0f);
    pTextWriter->Print(":");

    pTextWriter->SetCursor(CommonGuideOffsetX + 105.0f, CommonGuideOffsetY + 60.0f);
    pTextWriter->Print(":");

    pTextWriter->SetCursor(CommonGuideOffsetX + 105.0f, CommonGuideOffsetY + 80.0f);
    pTextWriter->Print(":");

    pTextWriter->SetCursor(CommonGuideOffsetX + 115.0f, CommonGuideOffsetY + 40.0f);
    pTextWriter->Print("[+]");

    pTextWriter->SetCursor(CommonGuideOffsetX + 115.0f, CommonGuideOffsetY + 60.0f);
    pTextWriter->Print("[-]");

    pTextWriter->SetCursor(CommonGuideOffsetX + 115.0f, CommonGuideOffsetY + 80.0f);
    pTextWriter->Print("[+] + [-]");

    pTextWriter->SetTextColor(nn::util::Color4u8::Red());
    pTextWriter->SetScale(NormalFontScaleX, NormalFontScaleY);

    pTextWriter->SetTextColor(nn::util::Color4u8::White());
    pTextWriter->SetScale(NormalFontScaleX, NormalFontScaleY);

    // 2つの SystemClockContext を比較して時計操作が行われたかを確認できます。
    nn::time::CalendarTime calendarTime;
    nn::time::CalendarAdditionalInfo calendarAdditionalInfo;
    GetCalendarTime(&calendarTime, &calendarAdditionalInfo);

    pTextWriter->SetCursor(ClockX, ClockY);
    pTextWriter->Print("Current calendar(%s): %04d/%02d/%02d %02d:%02d:%02d",
        calendarAdditionalInfo.timeZone.standardTimeName,
        calendarTime.year, calendarTime.month, calendarTime.day,
        calendarTime.hour, calendarTime.minute, calendarTime.second);

    pTextWriter->SetTextColor(nn::util::Color4u8::Red());
    if (!m_pSaveDataAccessor->HasAccount())
    {
        // アカウントがない場合表示
        pTextWriter->SetCursor(NoUserAccountX, NoUserAccountY);
        pTextWriter->Print("userCount = 0, Could not mount save data as \"save\"");
    }
    else if (!m_pSaveDataAccessor->HasSpace())
    {
        // Spaceがない場合表示
        pTextWriter->SetCursor(NoSpaceX, NoSpaceY);
        pTextWriter->Print("Usable space not enough to save data.");
    }
}

// GraphicsSystem での関数呼び出しを簡略化
nns::gfx::PrimitiveRenderer::Renderer& ScenePluginBase::GetPrimitiveRenderer() const NN_NOEXCEPT
{
    return m_pGraphicsSystem->GetPrimitiveRenderer();
}

nn::gfx::CommandBuffer& ScenePluginBase::GetCommandBuffer() const NN_NOEXCEPT
{
    return m_pGraphicsSystem->GetCommandBuffer();
}

nn::gfx::util::DebugFontTextWriter& ScenePluginBase::GetDebugFont() const NN_NOEXCEPT
{
    return m_pGraphicsSystem->GetDebugFont();
}

typedef nn::util::IntrusiveList<
    nn::perf::LoadMeterBase,
    nn::util::IntrusiveListBaseNodeTraits<nn::perf::LoadMeterBase>> LoadMeterList;

// 負荷メーターを描画する
void ScenePluginBase::DrawLoadMeter() const NN_NOEXCEPT
{
    // NN_PERF_IS_ENABLED() が false の場合はメーターの描画をしない
    if (NN_STATIC_CONDITION(NN_PERF_IS_ENABLED()))
    {
        const int frameRate = 60;
        const nn::TimeSpan ReferenceFrame = nn::TimeSpan::FromMicroSeconds(
            static_cast<int64_t>(1000000.f / frameRate)
        );
        nn::perf::CpuMeter* pFrameMeter = NN_PERF_GET_FRAME_METER();
        int i = 0;
        for (LoadMeterList::iterator it = pFrameMeter->m_LoadMeterList.begin();
             it != pFrameMeter->m_LoadMeterList.end();
             ++it)
        {
            float rate = 100.f *
                         it->GetLastTotalSpan().GetMicroSeconds() /
                         ReferenceFrame.GetMicroSeconds();
            const char* name = it->GetName();

            GetDebugFont().SetTextColor(nn::util::Color4u8::White());
            GetDebugFont().SetScale(NormalFontScaleX, NormalFontScaleY);
            GetDebugFont().SetCursor(20.0f, 20.0f * i);
            GetDebugFont().Print(name);

            GetDebugFont().SetCursor(70.0f, 20.0f * i);
            GetDebugFont().Print("%3.1f", rate);

            GetDebugFont().SetCursor(120.0f, 20.0f * i);
            GetDebugFont().Print("%%", rate);
            i++;
        }
    }
}
