﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/gfx.h>

#include <nn/util/util_Color.h>

#include <nns/gfx/gfx_PrimitiveRenderer.h>
#include <nns/gfx/gfx_PrimitiveRendererMeshRes.h>
#include <nns/gfx/gfx_PrimitiveRendererMeterDrawer.h>

#include "Demo1.h"
#include "Demo1SaveData.h"
#include "Demo1PluginManager.h"

namespace
{

SET_PLUGIN( "SaveData", SaveDataDemo, PluginProperty_None );

/**
*   @brief セーブデータ用の構造体
*/
struct SaveDataStructure
{
    int value;
};
} // namespace

SaveDataDemo::~SaveDataDemo() NN_NOEXCEPT
{

}

void SaveDataDemo::Initialize() NN_NOEXCEPT
{

    m_UserCount = 0;
    // アカウントライブラリを初期化します。
    nn::account::Initialize();

    // セーブデータをマウントします。
    {
        nn::account::Uid user = nn::account::InvalidUid;
        nn::Result result = nn::account::ListAllUsers(&m_UserCount, &user, 1);

        if (result.IsSuccess() && m_UserCount > 0)
        {
            m_pSaveDataAccessor->EnableAccount(true);
            result = nn::fs::EnsureSaveData(user);

            if (result.IsSuccess())
            {
                m_pSaveDataAccessor->EnableSpace(true);
                // マウント名 "save" としてセーブデータをマウントします。
                NN_ABORT_UNLESS_RESULT_SUCCESS (nn::fs::MountSaveData("save", user));

                m_pThreadStack = reinterpret_cast<char*>(
                    GetStandardAllocator().Allocate(
                        ThreadStackSize,
                        nn::os::ThreadStackAlignment
                    )
                );

                // スレッドを作成
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
                    &m_SaveDataThread, SaveDataThreadFunc, this,
                    m_pThreadStack, ThreadStackSize, nn::os::DefaultThreadPriority));
            }
            else if (nn::fs::ResultUsableSpaceNotEnough::Includes(result))
            {
                //「十分な空き容量が無い場合」
                m_pSaveDataAccessor->EnableSpace(false);
            }
            else
            {
                //「十分な空き容量が無い場合」を除くエラー結果のハンドリング
                NN_ABORT_UNLESS_RESULT_SUCCESS( result );
            }
        }
        else
        {
            m_pSaveDataAccessor->EnableAccount(false);
        }
    }
}

void SaveDataDemo::Start() NN_NOEXCEPT
{
    if (m_pSaveDataAccessor->HasAccount() && m_pSaveDataAccessor->HasSpace())
    {
        // スレッドをStart
        nn::os::StartThread(&m_SaveDataThread);
    }
}

void SaveDataDemo::Finalize() NN_NOEXCEPT
{
    if (m_pSaveDataAccessor->HasAccount() && m_pSaveDataAccessor->HasSpace())
    {
        // スレッドの破棄
        nn::os::DestroyThread(&m_SaveDataThread);
        GetStandardAllocator().Free(m_pThreadStack);

        // アンマウントします。
        nn::fs::Unmount("save");
    }
}

void SaveDataDemo::Wait() NN_NOEXCEPT
{
    if (m_pSaveDataAccessor->HasAccount() && m_pSaveDataAccessor->HasSpace())
    {
        nn::os::WaitThread(&m_SaveDataThread);
    }
}


void SaveDataDemo::SaveDataThreadFunc(void* pArg) NN_NOEXCEPT
{
    SaveDataDemo * pSaveDataDemo = reinterpret_cast<SaveDataDemo*>(pArg);
    pSaveDataDemo->SaveDataThreadFuncImpl();
}

void SaveDataDemo::SaveDataThreadFuncImpl() const NN_NOEXCEPT
{
    const size_t FileSize = sizeof(SaveDataStructure);
    const char* FilePath = "save:/file";

    while (!nn::os::TryWaitEvent(&GetGlobalEvent()))
    {
        // ファイルの存在を確認します。
        nn::fs::DirectoryEntryType directoryEntryType;
        nn::Result result = nn::fs::GetEntryType(&directoryEntryType, FilePath);
        if (nn::fs::ResultPathNotFound::Includes(result))
        {
            // 対象ファイルが存在しません。
            // ファイルを作成します。
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::CreateFile(FilePath, FileSize));

            // 作成したファイルに初期データを書き込みます。
            nn::fs::FileHandle fileHandle;

            // ファイルをオープンします。
            NN_ABORT_UNLESS_RESULT_SUCCESS(
                nn::fs::OpenFile(&fileHandle, FilePath, nn::fs::OpenMode_Write)
            );

            // ファイルに書き込みます。
            SaveDataStructure data = {};
            NN_ABORT_UNLESS_RESULT_SUCCESS(
                nn::fs::WriteFile(fileHandle, 0, &data, sizeof(data),
                    nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));

            nn::fs::CloseFile(fileHandle);
        }

        // ファイルのデータを読み込み、更新します。
        SaveDataStructure data;

        nn::fs::FileHandle fileHandle;

        // ファイルをオープンします
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::fs::OpenFile(
                &fileHandle,
                FilePath,
                nn::fs::OpenMode_Read | nn::fs::OpenMode_Write)
        );

        // ファイルを読み込みます。
        size_t readSize;

        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::fs::ReadFile(
                &readSize,
                fileHandle,
                0,
                &data,
                sizeof(data))
        );
        NN_ASSERT_EQUAL(readSize, sizeof(data));

        // 読み込んだデータを更新します。
        data.value++;

        // 更新したデータを書き込みます。
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::fs::WriteFile(fileHandle, 0, &data, sizeof(data),
                nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));

        nn::fs::CloseFile(fileHandle);

        // セーブデータの更新内容をコミットします。
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::Commit("save"));
    }
}
