﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>

#include <nns/hid.h>

#include "Demo1NpadPluginBase.h"
#include "Demo1Plugin.h"
#include "Demo1PlatformDependentPluginBase.h"
#include "Demo1ScenePluginBase.h"

class PluginManager
{
    NN_DISALLOW_COPY(PluginManager);
    NN_DISALLOW_MOVE(PluginManager);
public:
    PluginManager() NN_NOEXCEPT
    : m_Mutex(false),
      m_DrawIndex(0)
    {}

    void AddPlugin(ScenePluginBase* pSceneBase) NN_NOEXCEPT
    {
        m_Mutex.Lock();
        m_SceneList.push_back(pSceneBase);
        m_Mutex.Unlock();
    }

    void AddBeginPlugin(ScenePluginBase* pSceneBase) NN_NOEXCEPT
    {
        m_Mutex.Lock();
        m_SceneList.insert(m_SceneList.begin(), pSceneBase);
        m_Mutex.Unlock();
    }

    void AddPlugin(PlatformDependentPluginBase* pPlatformDependent) NN_NOEXCEPT
    {
        m_Mutex.Lock();
        m_PlatformDependentList.push_back(pPlatformDependent);
        m_Mutex.Unlock();
    }

    void AddBeginPlugin(PlatformDependentPluginBase* pPlatformDependent) NN_NOEXCEPT
    {
        m_Mutex.Lock();
        m_PlatformDependentList.insert(m_PlatformDependentList.begin(), pPlatformDependent);
        m_Mutex.Unlock();
    }

    void AddPlugin(NpadPluginBase* pNpadBase) NN_NOEXCEPT
    {
        m_Mutex.Lock();
        m_NpadList.push_back(pNpadBase);
        m_Mutex.Unlock();
    }

    void AddBeginPlugin(NpadPluginBase* pNpadBase) NN_NOEXCEPT
    {
        m_Mutex.Lock();
        m_NpadList.insert(m_NpadList.begin(), pNpadBase);
        m_Mutex.Unlock();
    }

    void Procedure() NN_NOEXCEPT
    {
        for (std::vector<PlatformDependentPluginBase*>::iterator it =
                m_PlatformDependentList.begin();
             it != m_PlatformDependentList.end();
             ++it)
        {
            (*it)->Procedure();
        }
        for (std::vector<ScenePluginBase*>::iterator it = m_SceneList.begin();
             it != m_SceneList.end();
             ++it)
        {
            (*it)->Procedure();
        }
        for (std::vector<NpadPluginBase*>::iterator it = m_NpadList.begin();
             it != m_NpadList.end();
             ++it)
        {
            (*it)->Procedure();
        }
    }

    void ChangeScene(int32_t num) NN_NOEXCEPT
    {
        int32_t moveCount = (num < 0)?(-num):(num);
        int32_t addValue = (num < 0)?(-1):(1);

        if(moveCount == 0 && m_SceneList[m_DrawIndex]->IsDraw() == false)
        {
            moveCount = 1;
        }
        while(moveCount > 0)
        {
            m_DrawIndex += addValue;
            if(m_DrawIndex < 0)
            {
                m_DrawIndex = static_cast<int>(m_SceneList.size()) - 1;
            }
            else if(m_DrawIndex >= static_cast<int>(m_SceneList.size()))
            {
                m_DrawIndex = 0;
            }

            if(m_SceneList[m_DrawIndex]->IsDraw())
            {
                moveCount--;
            }
        }

        for (int i = 0; i < static_cast<int>(m_SceneList.size()); i++)
        {
            if (i == m_DrawIndex)
            {
                m_SceneList[i]->SetActive(true);
            }
            else
            {
                m_SceneList[i]->SetActive(false);
            }
        }

        return;
    }

    void DrawScene() NN_NOEXCEPT
    {
        m_SceneList[m_DrawIndex]->Draw();
    }

    void InitializeScenePlugin(GraphicsSystem* pGraphicsSystem,
                         SaveDataAccessor* pSaveDataAccessor) NN_NOEXCEPT
    {
        // 各シーンの初期化
        for (std::vector<ScenePluginBase*>::iterator it = m_SceneList.begin();
             it != m_SceneList.end();
             ++it)
        {
            (*it)->SetGraphicSystem(pGraphicsSystem);
            (*it)->SetSaveDataAccessor(pSaveDataAccessor);
            (*it)->Initialize();
        }

        // 各シーンの開始
        for (std::vector<ScenePluginBase*>::iterator it = m_SceneList.begin();
             it != m_SceneList.end();
             ++it)
        {
            (*it)->Start();
        }
        return;
    }

    void FinalizeScenePlugin() NN_NOEXCEPT
    {
        // 各シーンの停止
        for (std::vector<ScenePluginBase*>::iterator it = m_SceneList.begin();
             it != m_SceneList.end();
             ++it)
        {
            (*it)->Wait();
        }

        for (std::vector<ScenePluginBase*>::iterator it = m_SceneList.begin();
             it != m_SceneList.end();
             ++it)
        {
            (*it)->End();
        }

        // 各シーンの終了処理
        // 初期化順と逆順で終了処理を行う
        for (std::vector<ScenePluginBase*>::reverse_iterator it = m_SceneList.rbegin();
             it != m_SceneList.rend();
             ++it)
        {
            (*it)->Finalize();
            delete (*it);
        }
        m_SceneList.clear();
    }

    void InitializePlatformPlugin() NN_NOEXCEPT
    {
        for (std::vector<PlatformDependentPluginBase*>::iterator it =
                m_PlatformDependentList.begin();
            it != m_PlatformDependentList.end();
            ++it)
        {
            (*it)->Initialize();
        }
    }

    void FinalizePlatformPlugin() NN_NOEXCEPT
    {
        for (std::vector<PlatformDependentPluginBase*>::reverse_iterator it =
                m_PlatformDependentList.rbegin();
            it != m_PlatformDependentList.rend();
            ++it)
        {
            (*it)->Finalize();
            delete (*it);
        }
        m_PlatformDependentList.clear();
    }

    void InitializeNpadPlugin() NN_NOEXCEPT
    {
        for (std::vector<NpadPluginBase*>::iterator it =
                m_NpadList.begin();
            it != m_NpadList.end();
            ++it)
        {
            (*it)->Initialize();
        }
    }

    void FinalizeNpadPlugin() NN_NOEXCEPT
    {
        for (std::vector<NpadPluginBase*>::reverse_iterator it =
                m_NpadList.rbegin();
            it != m_NpadList.rend();
            ++it)
        {
            (*it)->Finalize();
            delete (*it);
        }
        m_NpadList.clear();
    }

    nn::hid::NpadStyleSet GetSupportedNpadStyleSet() NN_NOEXCEPT
    {
        nn::hid::NpadStyleSet mask = {};

        for (std::vector<NpadPluginBase*>::iterator it =
                m_NpadList.begin();
            it != m_NpadList.end();
            ++it)
        {
            mask |= (*it)->GetNpadStyleSet();
        }

        return mask;
    }

    NpadPluginBase* GetEnableNpad(nn::hid::NpadStyleSet style) NN_NOEXCEPT
    {
        for (std::vector<NpadPluginBase*>::iterator it =
                m_NpadList.begin();
            it != m_NpadList.end();
            ++it)
        {
            if ((*it)->IsEnableNpadStyle(style))
            {
                return (*it);
            }
        }
        return NULL;
    }

    nns::hid::ControllerManager& GetControllerManager()
    {
        return m_ControllerManager;
    }

private:
    std::vector<ScenePluginBase*> m_SceneList;
    std::vector<PlatformDependentPluginBase*> m_PlatformDependentList;
    std::vector<NpadPluginBase*> m_NpadList;
    nns::hid::ControllerManager m_ControllerManager;

    nn::os::Mutex m_Mutex;
    int32_t m_DrawIndex;
};

PluginManager& GetPluginManager() NN_NOEXCEPT;

#define SET_PLUGIN( progname, device, property ) \
class device##Plugin \
{ \
public: \
    device##Plugin () NN_NOEXCEPT \
    { \
        const char ProgramName[] = progname; \
        device* pDevice = new device(); \
        pDevice->SetName(ProgramName); \
        pDevice->SetProperty(property); \
        GetPluginManager().AddPlugin(pDevice); \
    }; \
}; \
device##Plugin g_##device##Plugin \

