﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <vector>

#include <nn/gfx.h>
#include <nn/nn_Log.h>

#include <nn/swkbd/swkbd_Api.h>
#include <nn/swkbd/swkbd_Result.h>
#include <nn/util/util_Color.h>

#include <nns/gfx/gfx_PrimitiveRenderer.h>
#include <nns/gfx/gfx_PrimitiveRendererMeterDrawer.h>
#include <nns/gfx/gfx_PrimitiveRendererMeshRes.h>

#include "Demo1.h"
#include "Demo1Keyboard.h"
#include "Demo1PluginManager.h"

namespace {

SET_PLUGIN( "Keyboard", KeyboardDemo, PluginProperty_Drawable );

}

// キーの状態を取得する
void KeyboardDemo::GetKeyState() NN_NOEXCEPT
{
    for (int i = 0; i < NN_ARRAY_SIZE(NpadIds); i++)
    {
        // 操作形態(NpadStyleSet)を取得
        nn::hid::NpadStyleSet style = nn::hid::GetNpadStyleSet(NpadIds[i]);

        NpadPluginBase* pNpad = GetPluginManager().GetEnableNpad(style);
        if(pNpad == NULL)
        {
            continue;
        }

        m_OldNpadButtonState[i] = m_CurrentNpadButtonState[i];
        m_CurrentNpadButtonState[i] = pNpad->GetNpadButtonState(NpadIds[i]);
    }
}


// ボタンが押されたか判断する
template<typename BUTTON>
bool KeyboardDemo::IsButtonPressed() const NN_NOEXCEPT
{
    bool isButtonPressed = false;
    for (int i = 0; i < NN_ARRAY_SIZE(NpadIds); i++)
    {
        // 操作形態(NpadStyleSet)を取得
        nn::hid::NpadStyleSet style = nn::hid::GetNpadStyleSet(NpadIds[i]);

        NpadPluginBase* pNpad = GetPluginManager().GetEnableNpad(style);
        if(pNpad == NULL)
        {
            continue;
        }

        if (m_CurrentNpadButtonState[i].buttons.Test<BUTTON>() &&
            !m_OldNpadButtonState[i].buttons.Test<BUTTON>())
        {
            isButtonPressed = true;
        }
    }
    return isButtonPressed;
}

// 15 個までの文字列に制限
void KeyboardDemo::AddStringButton(
    std::vector <std::string>* pAddDispString,
    const char* pString) NN_NOEXCEPT
{
    if (pAddDispString == NULL)
    {
        return;
    }

    if (pAddDispString->size() > 15)
    {
        pAddDispString->erase(pAddDispString->begin() + 15);
    }
    pAddDispString->insert(pAddDispString->begin(), pString);
}

// ソフトウェアキーボードアプレットを起動
bool KeyboardDemo::LaunchKeyboard() NN_NOEXCEPT
{
    nn::swkbd::ShowKeyboardArg showKeyboardArg;
    nn::swkbd::MakePreset(&(showKeyboardArg.keyboardConfig), nn::swkbd::Preset_Default);

    // ガイド文字列の設定
    const char* pGuideString = "please input word.";
    nn::swkbd::SetGuideTextUtf8(&showKeyboardArg.keyboardConfig, pGuideString);

    // 共有メモリ用バッファの割り当て
    size_t inHeapSize = nn::swkbd::GetRequiredWorkBufferSize(false);
    void* pSwkbdWorkBuffer = GetStandardAllocator().Allocate(inHeapSize, nn::os::MemoryPageSize);

    showKeyboardArg.workBuf = pSwkbdWorkBuffer;
    showKeyboardArg.workBufSize = inHeapSize;

    // 終了パラメータの設定
    size_t outHeapSize = nn::swkbd::GetRequiredStringBufferSize();
    nn::swkbd::String outputString;
    outputString.ptr = reinterpret_cast<uint16_t*>(
        GetStandardAllocator().Allocate(outHeapSize, nn::os::MemoryPageSize));
    outputString.bufSize = outHeapSize;

    nn::Result result = nn::swkbd::ShowKeyboard(&outputString, showKeyboardArg);
    if (nn::swkbd::ResultCanceled::Includes(result))
    {
        GetStandardAllocator().Free(pSwkbdWorkBuffer);
        GetStandardAllocator().Free(outputString.ptr);
        return false;
    }

    // 結果文字列を受け取る
    uint16_t* pString = reinterpret_cast< uint16_t* >(outputString.ptr);
    int strIndex = 0;
    std::string tempString;
    while (pString[strIndex] != 0)
    {
        tempString = tempString + static_cast<char>(pString[strIndex]);
        strIndex++;
    }

    // 一度に入力できる文字数を 30 文字に制限
    if (strIndex > 30)
    {
        tempString.erase(tempString.begin() + 30, tempString.begin() + strIndex);
    }
    AddStringButton(&m_OutString, tempString.c_str());

    GetStandardAllocator().Free(pSwkbdWorkBuffer);
    GetStandardAllocator().Free(outputString.ptr);
    return true;
}

// ソフトウェアキーボードの入力状態を描画
void KeyboardDemo::WriteKeyboardState(
    nn::gfx::util::DebugFontTextWriter* pTextWriter,
    const float offsetX,
    const float offsetY) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pTextWriter);

    const nn::util::Unorm8x4& TextColor = Color::White;

    pTextWriter->SetTextColor(TextColor);
    pTextWriter->SetScale(NormalFontScaleX, NormalFontScaleY);
    pTextWriter->SetCursor(offsetX, offsetY);

    float displayHeight = static_cast<float>(m_pGraphicsSystem->GetDisplayHeight());
    float lineStep = displayHeight / 40.0f;
    float heightOffset = lineStep;

    // A ボタンを押したら、ソフトウェアキーボードアプレットを起動します。
    if (IsButtonPressed<nn::hid::NpadButton::A>())
    {
        if (LaunchKeyboard() == false)
        {
            m_IsLaunchKeyboard = false;
            return;
        }
        else
        {
            m_IsLaunchKeyboard = true;
        }
    }

    if (m_OutString.size() > 0)
    {
        nn::gfx::util::DebugFontTextWriter& debugFontTextWriter = m_pGraphicsSystem->GetDebugFont();
        int stepCounter = 2;
        for (size_t i = 0; i < m_OutString.size(); i++)
        {
            pTextWriter->SetCursor(offsetX, offsetY);
            debugFontTextWriter.SetCursor(offsetX, heightOffset + lineStep * stepCounter++);
            pTextWriter->Print(m_OutString[i].c_str());
            if (stepCounter > 16)
            {
                break;
            }
        }
    }
}

void KeyboardDemo::WriteKeyboardTemplate(
    nn::gfx::util::DebugFontTextWriter* pTextWriter
) const NN_NOEXCEPT
{
    pTextWriter->SetScale(NormalFontScaleX, NormalFontScaleY);

    // テスト用にテキスト出力
    float displayHeight = static_cast<float>(m_pGraphicsSystem->GetDisplayHeight());
    float lineStep = displayHeight / 40.0f;
    float widthOffset = 700.0f;
    float heightOffset = lineStep;

    pTextWriter->SetTextColor(nn::util::Color4u8::White());

    pTextWriter->SetCursor(widthOffset, heightOffset + lineStep * 1);
    pTextWriter->Print("---------------------------------------------------------------------");
    pTextWriter->SetCursor(widthOffset, heightOffset + lineStep * 2);
    pTextWriter->Print("  [01] input = ");
    pTextWriter->SetCursor(widthOffset, heightOffset + lineStep * 3);
    pTextWriter->Print("  [02] input = ");
    pTextWriter->SetCursor(widthOffset, heightOffset + lineStep * 4);
    pTextWriter->Print("  [03] input = ");
    pTextWriter->SetCursor(widthOffset, heightOffset + lineStep * 5);
    pTextWriter->Print("  [04] input = ");
    pTextWriter->SetCursor(widthOffset, heightOffset + lineStep * 6);
    pTextWriter->Print("  [05] input = ");
    pTextWriter->SetCursor(widthOffset, heightOffset + lineStep * 7);
    pTextWriter->Print("  [06] input = ");
    pTextWriter->SetCursor(widthOffset, heightOffset + lineStep * 8);
    pTextWriter->Print("  [07] input = ");
    pTextWriter->SetCursor(widthOffset, heightOffset + lineStep * 9);
    pTextWriter->Print("  [08] input = ");
    pTextWriter->SetCursor(widthOffset, heightOffset + lineStep * 10);
    pTextWriter->Print("  [09] input = ");
    pTextWriter->SetCursor(widthOffset, heightOffset + lineStep * 11);
    pTextWriter->Print("  [10] input = ");
    pTextWriter->SetCursor(widthOffset, heightOffset + lineStep * 12);
    pTextWriter->Print("  [11] input = ");
    pTextWriter->SetCursor(widthOffset, heightOffset + lineStep * 13);
    pTextWriter->Print("  [12] input = ");
    pTextWriter->SetCursor(widthOffset, heightOffset + lineStep * 14);
    pTextWriter->Print("  [13] input = ");
    pTextWriter->SetCursor(widthOffset, heightOffset + lineStep * 15);
    pTextWriter->Print("  [14] input = ");
    pTextWriter->SetCursor(widthOffset, heightOffset + lineStep * 16);
    pTextWriter->Print("  [15] input = ");
    pTextWriter->SetCursor(widthOffset, heightOffset + lineStep * 17);
    pTextWriter->Print("---------------------------------------------------------------------");

    pTextWriter->SetCursor(widthOffset, heightOffset + lineStep * 19);
    pTextWriter->Print("Press A Button  :  Start SoftWareKeyBoard");
    pTextWriter->SetCursor(widthOffset, heightOffset + lineStep * 20);
    pTextWriter->Print("Limit 30 Characters");

    if (m_IsLaunchKeyboard == false)
    {
        pTextWriter->SetTextColor(nn::util::Color4u8::Red());
        pTextWriter->SetCursor(widthOffset, heightOffset);
        pTextWriter->Print("SoftWareKeyBoard Cancel or Failed.");
        pTextWriter->SetTextColor(nn::util::Color4u8::White());
    }
}

void KeyboardDemo::MakeCommand(int64_t frame, const char* pName) NN_NOEXCEPT
{
    NN_UNUSED(frame);

    nn::gfx::CommandBuffer& commandBuffer = m_pGraphicsSystem->GetCommandBuffer();
    nn::gfx::util::DebugFontTextWriter& debugFontTextWriter = m_pGraphicsSystem->GetDebugFont();

    // キーの状態を取得する
    GetKeyState();

    // ソフトウェアキーボードの入力状態を描画
    WriteKeyboardState(&debugFontTextWriter, 800, 16);

    // ソフトウェアキーボードの入力フレームを描画
    WriteKeyboardTemplate(&debugFontTextWriter);

    //!< 共通操作説明を描画します。
    WriteCommonGuide(&debugFontTextWriter, pName);

    // 負荷メータを表示する
    DrawLoadMeter();

    // テキストを描画
    debugFontTextWriter.Draw(&commandBuffer);
}

void KeyboardDemo::Draw() NN_NOEXCEPT
{
    static int64_t s_Frame = 0;
    MakeCommand(s_Frame, m_Name.c_str());
    s_Frame++;
}
