﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs.h>
#include <nn/os.h>

#include "Demo1.h"
#include "Demo1Fs.h"
#include "Demo1PluginManager.h"

namespace {

SET_PLUGIN( "Fs", FsDemo, PluginProperty_None );

}

FsDemo::FsDemo() NN_NOEXCEPT
{

}

FsDemo::~FsDemo() NN_NOEXCEPT
{

}

void FsDemo::Initialize() NN_NOEXCEPT
{
    size_t cacheSize = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&cacheSize));
    m_pMountRomCacheBuffer = new char[cacheSize];
    NN_ABORT_UNLESS_NOT_NULL(m_pMountRomCacheBuffer);

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::fs::MountRom("rom", m_pMountRomCacheBuffer, cacheSize)
    );

    m_pThreadStack = reinterpret_cast<char*>(
        GetStandardAllocator().Allocate(
            ThreadStackSize,
            nn::os::ThreadStackAlignment
        )
        );

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
        &m_FsThread, FsThreadFunc, this, m_pThreadStack,
        ThreadStackSize, nn::os::DefaultThreadPriority));
}

void FsDemo::Finalize() NN_NOEXCEPT
{
    nn::os::DestroyThread(&m_FsThread);

    GetStandardAllocator().Free(m_pThreadStack);
}

void FsDemo::Start() NN_NOEXCEPT
{
    // ファイルを開く
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&m_RomFsFile, "rom:/res/test.txt",
        nn::fs::OpenMode_Read));

    // ファイルの内容を読み出すバッファを作成
    int64_t fileSize = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetFileSize(&fileSize, m_RomFsFile));
    m_pFileBuffer = new char[static_cast<unsigned int>(fileSize)];

    nn::os::StartThread(&m_FsThread);
}

void FsDemo::Wait() NN_NOEXCEPT
{
    nn::os::WaitThread(&m_FsThread);
}

void FsDemo::End() NN_NOEXCEPT
{
    // バッファを破棄
    delete[] m_pFileBuffer;

    // ファイルを閉じる
    nn::fs::CloseFile(m_RomFsFile);

    nn::fs::Unmount("rom");

    delete[] reinterpret_cast<unsigned char*>(m_pMountRomCacheBuffer);
    m_pMountRomCacheBuffer = NULL;
}

void FsDemo::FsThreadFuncImpl() const NN_NOEXCEPT
{
    // ファイル読み込みはスリープ時には自動で停止するので、
    // 排他ロック等は必要ありません
    while (!nn::os::TryWaitEvent(&GetGlobalEvent()))
    {
        // ファイルの内容をバッファにコピー
        int64_t fileSize = 0;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetFileSize(&fileSize, m_RomFsFile));
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(m_RomFsFile, 0, m_pFileBuffer,
            static_cast<size_t>(fileSize)));

        ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
    }
}

void FsDemo::FsThreadFunc(void* pArg) NN_NOEXCEPT
{
    FsDemo * pFsDemo = reinterpret_cast<FsDemo*>(pArg);
    pFsDemo->FsThreadFuncImpl();
}
