﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{Demo1/Demo1.cpp,PageSampleDemo1Sample}
 *
 * @brief
 *  NintendoSDK の複数の機能を同時に使用するサンプルプログラムです。
 */

/**
 * @page PageSampleDemo1Sample Demo1
 * @tableofcontents
 *
 * @brief
 *  NintendoSDK の複数の機能を同時に使用するサンプルプログラムの解説です。
 *
 * @image html Applications\Demo1\Demo1Sample.png
 *
 * @section PageSampleDemo1Sample_SectionBrief 概要
 *  以下の機能から構成される統合サンプルです。
 *  - nns::gfx::PrimitiveRenderer ライブラリを用いたプリミティブ描画
 *  - nn::hid ライブラリを用いたヒューマンインターフェイスデバイスの状態取得
 *  - オーディオレンダラを用いたオーディオ再生
 *  - nn::fs ライブラリを用いたファイルの読み出し
 *  - nn::fs ライブラリを用いたユーザーセーブデータの書き込み
 *  - nn::time::StandardUserSystemClock ライブラリを用いた現在時刻の表示と時刻操作の検知
 *  - Npad を用いた SixAxisSensor の状態取得
 *
 * @section PageSampleDemo1Sample_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/Demo1
 *  Samples/Sources/Applications/Demo1 @endlink 以下にあります。
 *
 * @section PageSampleDemo1Sample_SectionNecessaryEnvironment 必要な環境
 *  画面表示が利用可能である必要があります。
 *
 * @section PageSampleDemo1Sample_SectionHowToOperate 操作方法
 * 『 + 』ボタンまたは『 - 』ボタンを押下すると画面が遷移します。
 * 『 + 』ボタンと『 - 』ボタンを同時に押下すると Demo1 サンプルプログラムを終了します。
 *  その他、各機能の操作方法においては、移植元の各機能サンプルの操作方法に準じます。
 *
 * @section PageSampleDemo1Sample_SectionPrecaution 注意事項
 *  サンプルを実行前にアカウントを作成してください。
 *  サンプル実行後にユーザーセーブデータ処理によりファイルが作成されます。
 *
 * @section PageSampleDemo1Sample_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleDemo1Sample_SectionDetail 解説
 *
 *
 * このサンプルプログラムの処理の流れは以下の通りです。
 *
 * - アプリケーション動作環境操作ライブラリを初期化
 * - ノーマルモード時の性能を選択
 * - ブーストモード時の性能を選択
 * - グラフィックスシステムのためのメモリ周りの初期化
 * - グラフィックス開発者向けツールおよびデバッグレイヤのためのメモリアロケータを設定
 * - 各種デモプログラムを開始
 * - 各種オブジェクトを破棄
 */

#include <cstdlib>
#include <cstring>

#include <nn/fs.h>
#include <nn/gfx.h>
#include <nn/init.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/vi.h>

#include <nn/lmem/lmem_ExpHeap.h>

#include "Demo1.h"
#include "Demo1Gfx.h"
#include "Demo1GraphicsSystem.h"
#include "Demo1Npad.h"
#include "Demo1PluginManager.h"
#include "Demo1SaveDataAccessor.h"
#include "Demo1Time.h"

nn::os::EventType& GetGlobalEvent() NN_NOEXCEPT
{
    NN_FUNCTION_LOCAL_STATIC(nn::os::EventType, s_EventType);
    return s_EventType;
}

nn::mem::StandardAllocator& GetStandardAllocator() NN_NOEXCEPT
{
    NN_FUNCTION_LOCAL_STATIC(nn::mem::StandardAllocator, s_StandardAllocator);
    return s_StandardAllocator;
}

namespace {
void StartDemo() NN_NOEXCEPT
{
    // ##################################################################################
    // アプリケーションの初期化シーケンス
    // ##################################################################################
    GraphicsSystem* pGraphicsSystem = new GraphicsSystem();
    pGraphicsSystem->SetApplicationHeap(&GetStandardAllocator());
    pGraphicsSystem->Initialize();

    InitializeTimeDemo();

    // SaveDataAccessorの初期化
    SaveDataAccessor* pSaveDataAccessor = new SaveDataAccessor();

    // パッドデモの初期化
    const char NpadProgramName[] = "               Npad";
    NpadDemo* pNpadDemo = new NpadDemo();
    pNpadDemo->SetName(NpadProgramName);
    pNpadDemo->SetProperty(PluginProperty_Drawable);
    GetPluginManager().AddBeginPlugin(pNpadDemo);

    // グラフィックスデモの初期化
    const char GfxProgramName[] = "PrimitiveRenderer";
    GfxDemo* pGfxDemo = new GfxDemo();
    pGfxDemo->SetName(GfxProgramName);
    GetPluginManager().AddBeginPlugin(pGfxDemo);

    pGraphicsSystem->InitializeDebugFont();

    // シーンプラグインの初期化
    GetPluginManager().InitializeScenePlugin(pGraphicsSystem,pSaveDataAccessor);

    // NPadプラグインの初期化
    GetPluginManager().InitializeNpadPlugin();

    // ##################################################################################
    // メインループ
    // ##################################################################################
    bool isFirstDrawing  = true;
    while( NN_STATIC_CONDITION(true) )
    {
        int moveCount = pNpadDemo->GetSceneChangeCount();
        GetPluginManager().ChangeScene(moveCount);

        GetPluginManager().Procedure();

        pGraphicsSystem->BeginDraw();
        pGfxDemo->Draw();
        GetPluginManager().DrawScene();
        pGraphicsSystem->EndDraw();

        if ( isFirstDrawing )
        {
            // Demo1の初期化が正常に行われた場合ログを表示する
            NN_LOG("Demo1 Initialize Success\n");
            isFirstDrawing = false;
        }
        const int FrameRate = 60;
        pGraphicsSystem->Synchronize(nn::TimeSpan::FromNanoSeconds(1000 * 1000 * 1000 / FrameRate));

        // 終了通知が来ていないかを確認します。
        if (pNpadDemo->EndTrigger())
        {
            break;
        }
    }

    // ##################################################################################
    // アプリケーションの終了処理
    // ##################################################################################
    // nn::applet::IsExpectedToCloseApplication() == true となると、
    // スリープ系コールバックは自動的に呼ばれなくなります。

    // NPadプラグインの終了処理
    GetPluginManager().FinalizeNpadPlugin();

    // シーンプラグインの終了処理
    GetPluginManager().FinalizeScenePlugin();

    pGraphicsSystem->FinalizeDebugFont();

    delete pSaveDataAccessor;

    FinalizeTimeDemo();

    pGraphicsSystem->Finalize();
    delete pGraphicsSystem;
}
}

//
//  Main Function
//  メイン関数です。
//

extern "C" void nnMain() NN_NOEXCEPT
{
    size_t cacheSize = 0;
    nn::Result result;

    const size_t MemorySize = (32 + 64 * 2 + 64 + 128 + 128) * 1024 * 1024;

    NN_ALIGNAS(4096) static nn::Bit8 s_Memory[MemorySize];

    GetStandardAllocator().Initialize(
        s_Memory, MemorySize
    );

    // ##################################################################################
    // メモリ周りの初期化
    // ##################################################################################

    result = nn::fs::QueryMountRomCacheSize(&cacheSize);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    void* mountRomCacheBuffer = new unsigned char[cacheSize];
    NN_ASSERT_NOT_NULL(mountRomCacheBuffer);

    result = nn::fs::MountRom("Contents", mountRomCacheBuffer, cacheSize);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    GetPluginManager().InitializePlatformPlugin();

    nn::os::InitializeEvent(&GetGlobalEvent(), false, nn::os::EventClearMode_ManualClear);

    // デモを開始する
    StartDemo();

    nn::fs::Unmount("Contents");

    GetPluginManager().FinalizePlatformPlugin();

    GetStandardAllocator().Finalize();
    delete[] reinterpret_cast<unsigned char*>(mountRomCacheBuffer);
}
