﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{DdDeviceAddressSpace.cpp,PageSampleDdDeviceAddressSpace}
 *
 * @brief
 *  デバイスアドレス空間機能のサンプルプログラム
 */

/**
 * @page PageSampleDdDeviceAddressSpace DeviceAddressSpace サンプルプログラム
 * @tableofcontents
 *
 * @brief
 *  デバイスアドレス空間機能のサンプルプログラムの解説です。
 *
 * @section PageSampleDdDeviceAddressSpace_SectionBrief 概要
 *  ここでは、デバイスアドレス空間機能を使ったサンプルプログラムの説明をします。
 *
 *  デバイスアドレス空間機能の API 詳細については、
 *  @ref nn::dd "DD の関数リファレンス" も併せて参照して下さい。
 *
 * @section PageSampleDdDeviceAddressSpace_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/DdDeviceAddressSpace Samples/Sources/Applications/DdDeviceAddressSpace @endlink 以下にあります。
 *
 * @section PageSampleDdDeviceAddressSpace_SectionNecessaryEnvironment 必要な環境
 *  実機ターゲット環境でのみデバイスアドレス空間機能が使用できます。@n
 *  ただし、デバイスアドレス空間機能は LSI によってサポートされている場合のみ
 *  利用可能ですので、ご注意ください。
 *
 * @section PageSampleDdDeviceAddressSpace_SectionHowToOperate 操作方法
 *  特になし
 *
 * @section PageSampleDdDeviceAddressSpace_SectionPrecaution 注意事項
 *  このデモは画面上に何も表示されません。実行結果はログに出力されます。
 *
 * @section PageSampleDdDeviceAddressSpace_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleDdDeviceAddressSpace_SectionDetail 解説
 *
 * @subsection PageSampleDdDeviceAddressSpace_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  DdDeviceAddressSpace.cpp
 *  @includelineno DdDeviceAddressSpace.cpp
 *
 * @subsection PageSampleDdDeviceAddressSpace_SectionSampleDetail サンプルプログラムの解説
 *  先のサンプルプログラムの全体像は以下の通りです。
 *
 *  - nnMain() にて nn::dd::DeviceAddressSpaceType オブジェクトを用意
 *  - SetupDeviceAddressSpace() 関数を呼び出し、デバイスアドレス空間を設定
 *  - DeviceDriverMain() 関数を呼び出し、デバイスドライバの本体処理を実行
 *  - CleanDeviceAddressSpace() 関数を呼び出し、デバイスアドレス空間を破棄
 *
 *  SetupDeviceAddressSpace() 関数では以下の処理を行なっています。
 *
 *  1. nn::dd::CreateDeviceAddressSpace() でデバイスアドレス空間を生成します。
 *     この時、デバイスアドレス空間サイズを 32bit に設定しています。
 *  2. 上記 1 で生成したデバイスアドレス空間にマップするメモリ領域を 2 つ用意
 *     しています。g_Address1 にはメモリヒープから 4MB のメモリを確保し、
 *     g_Address2 にはグローバルに定義した 8KB のメモリ領域を設定しています。
 *  3. 上記 2 で用意した 2 つのメモリ領域を
 *     nn::dd::MapDeviceAddressSpaceAligned() を
 *     使ってデバイスアドレス空間オブジェクトに関連付けしています。
 *     この時、マップ先の nn::dd::DeviceVirtualAddress の下位 22bit は、
 *     g_Address1 や g_Address2 の下位 22bit と同じになるようにします。
 *  4. 上記 3 で用意したデバイスアドレス空間オブジェクトを、
 *     nn::dd::DeviceName で指定されたデバイスに
 *     nn::dd::AttachDeviceAddressSpace() で関連付けします。
 *     これにより、そのデバイスのアドレス空間に対して上記 2 で関連付けした
 *     2 つのメモリ領域がミラーリングされ、アクセスできるようになります。
 *
 *  DeviceDriverMain() 関数ではデバイスドライバ本体の処理を行ないます。
 *  本プログラムはサンプルであるため、実質的には何も行なっていませんが、
 *  実際のデバイスドライバである場合には、このタイミングで各デバイスの
 *  I/O レジスタなどにアクセスし、必要に応じて上記 2 で用意したメモリに対して
 *  対象デバイスから Read/Write させることが出来ます。
 *
 *  最後にデバイスドライバの後処理として、CleanDeviceAddressSpace() で
 *  デバイスアドレス空間の破棄を行なっています。手順は以下の通りです。
 *
 *  1. nn::dd::DetachDeviceAddressSpace() を呼び出して、アタッチしていた
 *     デバイスへの関連付けを解除します。これにより、対象デバイスへ
 *     ミラーリングされていたメモリは見えなくなります。
 *  2. nn::dd::UnmapDeviceAddressSpace() を呼び出して、デバイスアドレス空間
 *     オブジェクトから、マップしていたメモリを切り離します。
 *  3. ヒープから確保していたメモリを、ヒープに返却します。
 *  4. 最後に nn::dd::DestroyDeviceAddressSpace() を呼び出して、
 *     デバイスアドレス空間オブジェクトを破棄します。
 *
 *  このサンプルプログラムの実行結果を以下に示します。@n
 *  ここでは、 nn::dd::MapDeviceAddressSpaceAligned() に成功した
 *  ミラーリング対象である自プロセスのメモリアドレスとサイズ、および、
 *  マップ先のデバイスアドレス空間のアドレスをログに出力しています。
 *
 *  @verbinclude  DdDeviceAddressSpace_OutputExample.txt
 *
 *  ログに出力されるアドレスやサイズの値やビット幅は、
 *  32bit もしくは 64bit の動作環境などによって異なります。
 *
 */

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/dd.h>

//-----------------------------------------------------------------------------
//  獲得する２種類のメモリ領域のサイズなどを定義

namespace
{
    const size_t                    g_Size1 = 4 * 1024 * 1024;
    uintptr_t                       g_Address1;
    nn::dd::DeviceVirtualAddress    g_MapAddress1;

    const size_t                    g_Size2 = 8 * 1024;
    uintptr_t                       g_Address2;
    nn::dd::DeviceVirtualAddress    g_MapAddress2;

    NN_ALIGNAS(4096) uint8_t    g_Buffer[ g_Size2 ];
}

//-----------------------------------------------------------------------------
//  メモリヒープの確保
//  この関数はプログラム起動時に nnMain() より前に自動的に呼ばれる。
//
extern "C" void nninitStartup()
{
    nn::os::SetMemoryHeapSize( 16 * 1024 * 1024 );
}

//----------------------------------------------------------------------------
//  デバイスアドレス空間の設定
//
void SetupDeviceAddressSpace(nn::dd::DeviceAddressSpaceType* pDas, nn::dd::DeviceName deviceName)
{
    // デバイスアドレス空間を 32bit アドレス空間で構築する
    auto result = nn::dd::CreateDeviceAddressSpace(pDas, 0x100000000ull);
    NN_ASSERT(result.IsSuccess(), "Cannot create DeviceAddressSpace.");

    // ヒープからメモリを獲得する
    result = nn::os::AllocateMemoryBlock(&g_Address1, g_Size1);
    NN_ASSERT(result.IsSuccess(), "Cannot allocate memory.");
    g_MapAddress1 = 0x10000000 + (g_Address1 & 0x3fffff);

    // もう１つの領域は静的に確保した領域を利用する
    g_Address2    = reinterpret_cast<uintptr_t>( g_Buffer );
    g_MapAddress2 = 0x20000000 + (g_Address2 & 0x3fffff);

    // デバイスアドレス空間にメモリをマップする
    auto currentProcess = nn::dd::GetCurrentProcessHandle();
    result = nn::dd::MapDeviceAddressSpaceAligned(
                                        pDas,
                                        currentProcess,
                                        g_Address1,
                                        g_Size1,
                                        g_MapAddress1,
                                        nn::dd::MemoryPermission_ReadWrite);
    NN_ASSERT(result.IsSuccess(), "Cannot map memory to DeviceAddressSpace.");

    NN_LOG("Map allocated memory to DeviceAddressSpace:\n");
    NN_LOG("  process address=0x%p\n",      g_Address1);
    NN_LOG("             size=0x%p\n",      g_Size1);
    NN_LOG("   device address=0x%016llx\n", g_MapAddress1);

    result = nn::dd::MapDeviceAddressSpaceAligned(
                                        pDas,
                                        currentProcess,
                                        g_Address2,
                                        g_Size2,
                                        g_MapAddress2,
                                        nn::dd::MemoryPermission_ReadWrite);
    NN_ASSERT(result.IsSuccess(), "Cannot map memory to DeviceAddressSpace.");

    NN_LOG("Map static memory to DeviceAddressSpace:\n");
    NN_LOG("  process address=0x%p\n",      g_Address2);
    NN_LOG("             size=0x%p\n",      g_Size2);
    NN_LOG("   device address=0x%016llx\n", g_MapAddress2);

    // 指定したデバイスに、構築したデバイスアドレス空間を関連付ける
    result = nn::dd::AttachDeviceAddressSpace(pDas, deviceName);
    NN_ASSERT(result.IsSuccess(), "Cannot attach DeviceAddressSpace to device.");

    NN_LOG("Attach to DeviceAddressSpace: deviceName=%d\n", deviceName);
}

//-----------------------------------------------------------------------------
//  デバイスアドレス空間の後始末
//
void CleanDeviceAddressSpace(nn::dd::DeviceAddressSpaceType* pDas, nn::dd::DeviceName deviceName)
{
    // 指定されたデバイスへの関連付けを解除する
    nn::dd::DetachDeviceAddressSpace(pDas, deviceName);

    // デバイスアドレス空間からメモリを切り離す
    auto currentProcess = nn::dd::GetCurrentProcessHandle();
    nn::dd::UnmapDeviceAddressSpace(pDas,
                                    currentProcess,
                                    g_Address1,
                                    g_Size1,
                                    g_MapAddress1);
    nn::dd::UnmapDeviceAddressSpace(pDas,
                                    currentProcess,
                                    g_Address2,
                                    g_Size2,
                                    g_MapAddress2);

    // ヒープにメモリを返却する
    nn::os::FreeMemoryBlock(g_Address1, g_Size1);

    // デバイスアドレス空間オブジェクトを破棄する
    nn::dd::DestroyDeviceAddressSpace(pDas);
}

//-----------------------------------------------------------------------------
//  デバイスドライバのメイン処理
//
void DeviceDriverMain()
{
    // Do nothing because this is just example of DeviceAddressSpace API usage.
}

//-----------------------------------------------------------------------------
//  メイン関数です。
//
extern "C" void nnMain()
{
    nn::dd::DeviceAddressSpaceType  das;

    // デバイスアドレス空間の設定
    SetupDeviceAddressSpace(&das, nn::dd::DeviceName_Sata);

    // デバイスドライバのメイン処理
    DeviceDriverMain();

    // デバイスアドレス空間の後始末
    CleanDeviceAddressSpace(&das, nn::dd::DeviceName_Sata);
}

