﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{CodecOpusDecoder.cpp,PageSampleCodecOpusDecoder}
 *
 * @brief
 * Opus デコーダのサンプルプログラム
 */

/**
 * @page PageSampleCodecOpusDecoder Opus デコーダ
 * @tableofcontents
 *
 * @brief
 * Opus デコーダのサンプルプログラムの解説です。
 *
 * @section PageSampleCodecOpusDecoder_SectionBrief 概要
 * Opus デコーダを用いて、Opus データをデコードしながらオーディオ再生を行うサンプルです。
 * オーディオ再生にはオーディオレンダラを利用します。
 *
 * @section PageSampleCodecOpusDecoder_SectionFileStructure ファイル構成
 * 本サンプルプログラムは @link ../../../Samples/Sources/Applications/CodecOpusDecoder
 * Samples/Sources/Applications/CodecOpusDecoder @endlink 以下にあります。
 *
 * @section PageSampleCodecOpusDecoder_SectionNecessaryEnvironment 必要な環境
 * オーディオ出力が利用可能である必要があります。
 *
 * @section PageSampleCodecOpusDecoder_SectionHowToOperate 操作方法
 * サンプルプログラムを実行すると、自動的に BGM が再生されます。
 * 再生中はキーボードや DebugPad による入力を用いて、再生される BGM の再生位置をコントロールできます。
 *
 * <p>
 * <table>
 * <tr><th> 入力 </th><th> 動作 </th></tr>
 * <tr><td> [A]             </td><td> 再生位置のシーク操作を実行                     </td></tr>
 * <tr><td> [Left][Right]   </td><td> シーク位置のコントロール (サンプル単位)        </td></tr>
 * <tr><td> [Up][Down]      </td><td> シーク位置のコントロール (OpusFrameSize 単位)  </td></tr>
 * <tr><td> [L][R]          </td><td> シーク位置のコントロール (波形の頭と末尾)      </td></tr>
 * <tr><td> [Select/-]      </td><td> 使い方の表示                                   </td></tr>
 * <tr><td> [Start/Space]   </td><td> プログラムの終了                               </td></tr>
 * </table>
 * </p>
 *
 *
 * @section PageSampleCodecOpusDecoder_SectionPrecaution 注意事項
 * 特にありません。
 *
 * @section PageSampleCodecOpusDecoder_SectionHowToExecute 実行手順
 * サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleCodecOpusDecoder_SectionDetail 解説
 *
 * @subsection PageSampleCodecOpusDecoder_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  CodecOpusDecoder.cpp
 *  @includelineno CodecOpusDecoder.cpp
 *
 * @subsection PageSampleCodecOpusDecoder_SectionSampleDetail サンプルプログラムの解説
 *
 * このサンプルプログラムは、あらかじめエンコードされた Opus データをデコードしながら、オーディオ再生を行うものです。
 *
 * サンプルプログラムの処理の流れは以下の通りです。
 *
 * - ファイルシステムの初期化
 * - Opus データの読み込み
 * - Opus デコーダの初期化
 * - オーディオレンダラの初期化、および開始
 * - Opus デコーダの実行と、デコード結果の再生
 * - オーディオレンダラの終了
 * - Opus デコーダの終了
 *
 * サンプルプログラムでは、デフォルトで Opus デコーダに nn::codec::OpusDecoder が使用されます。
 * USE_HARDWARE_DECODER マクロを定義してビルドすると nn::codec::HardwareOpusDecoder が使用されます。
 * 以下の説明は nn::codec::OpusDecoder を代表例としています。
 * nn::codec::HardwareOpusDecoder を使用する場合は、OpusDecoder を HardwareOpusDecoder に置き換えてお読み下さい。
 *
 * 最初にファイルシステムの初期化を行います。
 * リソースファイルはビルド時にコンバートされるものを利用します。
 * リソースディレクトリの場所は適宜変更して利用してください。
 *
 * Opus データをファイルから読み込みます。
 *
 * ヘッダ情報をもとに Opus デコーダを初期化します。
 *
 * 初期化関数 nn::codec::OpusDecoder::Initialize() にはサンプルレートとチャンネル数、さらにそれらによってサイズが決まるワークバッファを与える必要があります。
 *
 * ワークバッファのサイズは nn::codec::OpusDecoder::GetWorkBufferSize() により取得可能です。
 *
 * オーディオレンダラを準備します。
 * オーディオレンダラの詳細に関しては @link ../../../Samples/Sources/Applications/AudioRenderer
 * Samples/Sources/Applications/AudioRenderer @endlink サンプルおよび関連する API リファレンスを参照してください。
 *
 * メインループでは、ストリーム再生のような処理を行います。
 * 複数の PCM バッファを用意し、 nn::audio::VoiceType 経由で再生しながら、
 * 再生が終わったバッファに対して順次 Opus データのデコードを行います。
 *
 * Opus データをすべてデコードし、再生を終えると、自動的に波形の冒頭にシークしループ再生を行います。
 *
 * プログラムの終了が要求されると、上記の繰り返し処理を抜けて、オーディオレンダラを停止し、
 * オーディオレンダラおよび Opus デコーダを終了し、メモリの破棄を行います。
 */

#include <algorithm>  // std::min, std::max
#include <cstdlib>  // std::malloc, std::free
#include <new>
#include <nn/nn_Abort.h>
#include <nn/audio.h>
#include <nn/codec.h>
#include <nn/fs.h>
#include <nn/mem.h>
#include <nn/os.h>
#include <nn/hid.h>
#include <nn/hid/hid_KeyboardKey.h>
#include <nn/hid/hid_Npad.h>
#include <nn/settings/settings_DebugPad.h>

#include <nns/audio/audio_HidUtilities.h>
#include <nns/nns_Log.h>


//==============================================================================
// USE_HARDWARE_DECODER を有効にすると、HardwareOpusDecoder に切り替わります。
//#define USE_HARDWARE_DECODER
//==============================================================================

namespace {

const char Title[] = "CodecOpusDecoder";

char g_HeapBuffer[8 * 1024 * 1024];

uint8_t g_OpusData[4 * 1024 * 1024];

char* g_MountRomCacheBuffer = NULL;

const int BufferCount = 4;
const int OpusFrameSampleCount = 960;
const int OpusChannelCount = 2;

// シーク再生時にデコーダのコンテキストを調整するためにアイドルデコードを行うフレーム数
const int OpusPreRollFrameCount = 4;

NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN int16_t g_PcmBuffer[1024 * 1024];
NN_STATIC_ASSERT(BufferCount * OpusFrameSampleCount * OpusChannelCount * sizeof(int16_t) <= sizeof(g_PcmBuffer));

void InitializeFileSystem()
{
    size_t cacheSize = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&cacheSize));
    g_MountRomCacheBuffer = new(std::nothrow) char[cacheSize];
    NN_ABORT_UNLESS_NOT_NULL(g_MountRomCacheBuffer);

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::fs::MountRom("asset", g_MountRomCacheBuffer, cacheSize)
    );
}

void FinalizeFileSystem()
{
    nn::fs::Unmount("asset");

    delete[] g_MountRomCacheBuffer;
    g_MountRomCacheBuffer = NULL;
}

void InitializeHidDevices()
{
    nn::hid::InitializeDebugPad();
    nn::hid::InitializeNpad();
    const nn::hid::NpadIdType npadIds[2] = { nn::hid::NpadId::No1, nn::hid::NpadId::Handheld };
    nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleFullKey::Mask | nn::hid::NpadStyleHandheld::Mask);
    nn::hid::SetSupportedNpadIdType(npadIds, sizeof(npadIds) / sizeof(npadIds[0]));

    //キーボードのキーを DebugPad のボタンに割り当てます。
    nn::settings::DebugPadKeyboardMap map;
    nn::settings::GetDebugPadKeyboardMap(&map);
    map.buttonA     = nn::hid::KeyboardKey::A::Index;
    map.buttonB     = nn::hid::KeyboardKey::B::Index;
    map.buttonX     = nn::hid::KeyboardKey::X::Index;
    map.buttonY     = nn::hid::KeyboardKey::Y::Index;
    map.buttonL     = nn::hid::KeyboardKey::L::Index;
    map.buttonR     = nn::hid::KeyboardKey::R::Index;
    map.buttonZL    = nn::hid::KeyboardKey::U::Index;
    map.buttonZR    = nn::hid::KeyboardKey::V::Index;
    map.buttonLeft  = nn::hid::KeyboardKey::LeftArrow::Index;
    map.buttonRight = nn::hid::KeyboardKey::RightArrow::Index;
    map.buttonUp    = nn::hid::KeyboardKey::UpArrow::Index;
    map.buttonDown  = nn::hid::KeyboardKey::DownArrow::Index;
    map.buttonStart = nn::hid::KeyboardKey::Space::Index;
    nn::settings::SetDebugPadKeyboardMap(map);
}

void PrintUsage()
{
    NNS_LOG("--------------------------------------------------------\n");
    NNS_LOG("%s Sample\n", Title);
    NNS_LOG("--------------------------------------------------------\n");
    NNS_LOG("[A]             Seek Playback Position                  \n");
    NNS_LOG("[L][R]          Control Seek Position (Sample)          \n");
    NNS_LOG("[Left][Right]   Control Seek Position (OpusFrame)       \n");
    NNS_LOG("[Up][Down]      Control Seek Position (Head/Tail)       \n");
    NNS_LOG("[Select/-]      Print Usage               　　　　　　　\n");
    NNS_LOG("[Start/Space]   Shut Down Sample Program  　　　　　　　\n");
    NNS_LOG("------------------------------------------------------- \n");
}

void GetHidState(nn::hid::NpadButtonSet& outNpadButtonCurrent,
                 nn::hid::NpadButtonSet& outNpadButtonDown,
                 nn::hid::AnalogStickState& outAnalogStickStateL,
                 nn::hid::AnalogStickState& outAnalogStickStateR)
{
    // Npad の入力を取得します。
    if (nn::hid::GetNpadStyleSet(nn::hid::NpadId::No1).Test<nn::hid::NpadStyleFullKey>())
    {
        static nn::hid::NpadFullKeyState npadFullKeyState = {};
        nn::hid::NpadFullKeyState state;
        nn::hid::GetNpadState(&state, nn::hid::NpadId::No1);
        outNpadButtonCurrent |= state.buttons;
        outNpadButtonDown |= state.buttons & ~npadFullKeyState.buttons;
        outAnalogStickStateL.x += state.analogStickL.x;
        outAnalogStickStateL.y += state.analogStickL.y;
        outAnalogStickStateR.x += state.analogStickR.x;
        outAnalogStickStateR.y += state.analogStickR.y;
        npadFullKeyState = state;
    }
    if (nn::hid::GetNpadStyleSet(nn::hid::NpadId::Handheld).Test<nn::hid::NpadStyleHandheld>())
    {
        static nn::hid::NpadHandheldState npadHandheldState = {};
        nn::hid::NpadHandheldState state;
        nn::hid::GetNpadState(&state, nn::hid::NpadId::Handheld);
        outNpadButtonCurrent |= state.buttons;
        outNpadButtonDown |= state.buttons & ~npadHandheldState.buttons;
        outAnalogStickStateL.x += state.analogStickL.x;
        outAnalogStickStateL.y += state.analogStickL.y;
        outAnalogStickStateR.x += state.analogStickR.x;
        outAnalogStickStateR.y += state.analogStickR.y;
        npadHandheldState = state;
    }

    // DebugPad の入力を取得します。
    {
        static nn::hid::DebugPadState debugPadState = {};
        nn::hid::DebugPadButtonSet debugPadButtonCurrent = {};
        nn::hid::DebugPadButtonSet debugPadButtonDown = {};
        nn::hid::DebugPadState state;
        nn::hid::GetDebugPadState(&state);
        debugPadButtonCurrent |= state.buttons;
        debugPadButtonDown |= state.buttons & ~debugPadState.buttons;
        outAnalogStickStateL.x += state.analogStickL.x;
        outAnalogStickStateL.y += state.analogStickL.y;
        outAnalogStickStateR.x += state.analogStickR.x;
        outAnalogStickStateR.y += state.analogStickR.y;
        debugPadState = state;
        nns::audio::ConvertDebugPadButtonsToNpadButtons(&outNpadButtonCurrent, debugPadButtonCurrent);
        nns::audio::ConvertDebugPadButtonsToNpadButtons(&outNpadButtonDown, debugPadButtonDown);
    }
}

}

extern "C" void nnMain()
{
#ifdef USE_HARDWARE_DECODER
    NNS_LOG("CodecOpusDecoder sample (using nn::codec::HardwareOpusDecoder)\n");
#else  // USE_HARDWARE_DECODER
    NNS_LOG("CodecOpusDecoder sample (using nn::codec::OpusDecoder)\n");
#endif  // USE_HARDWARE_DECODER

    nn::mem::StandardAllocator allocator(g_HeapBuffer, sizeof(g_HeapBuffer));
    nn::mem::StandardAllocator pcmBufferAllocator(g_PcmBuffer, sizeof(g_PcmBuffer));

    // ファイルシステムを初期化します
    InitializeFileSystem();

    // HID を初期化します。
    InitializeHidDevices();

    // 操作方法を表示します。
    PrintUsage();

    // Opus データを読み込みます
    const char Filename[] = "asset:/SampleBgm0-1ch.opus";
    nn::fs::FileHandle handle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&handle, Filename, nn::fs::OpenMode_Read));
    int64_t size;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetFileSize(&size, handle));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(handle, 0, g_OpusData, static_cast<std::size_t>(size)));
    nn::fs::CloseFile(handle);

    // ヘッダより、サンプルレートとチャンネル数を取得します
    nn::codec::OpusBasicInfo opusBasicInfo;
    std::memcpy(&opusBasicInfo, g_OpusData, sizeof(opusBasicInfo));
    NN_ABORT_UNLESS(opusBasicInfo.header.type == nn::codec::OpusInfoType_BasicInfo);
    int sampleRate = opusBasicInfo.sampleRate;
    int channelCount = opusBasicInfo.channelCount;

    // Opus デコーダを初期化します
#ifndef USE_HARDWARE_DECODER
    nn::codec::OpusDecoder decoder;
#else  // #ifndef USE_HARDWARE_DECODER
    const int32_t noOption = 0; // 現在、指定可能なオプションはありません
    nn::codec::HardwareOpusDecoder decoder(noOption);
#endif // #ifndef USE_HARDWARE_DECODER
    void* workBuffer = std::malloc(decoder.GetWorkBufferSize(sampleRate, channelCount));
    NN_ABORT_UNLESS_NOT_NULL(workBuffer);
    nn::codec::OpusResult opusResult = decoder.Initialize(sampleRate, channelCount, workBuffer, decoder.GetWorkBufferSize(sampleRate, channelCount));
    NN_ABORT_UNLESS(opusResult == nn::codec::OpusResult_Success);

    //オーディオレンダラを構築します
    nn::audio::AudioRendererParameter audioRendererParameter;
    nn::audio::InitializeAudioRendererParameter(&audioRendererParameter);
    audioRendererParameter.sampleRate = 48000;
    audioRendererParameter.sampleCount = 240;
    audioRendererParameter.mixBufferCount = 2;
    audioRendererParameter.voiceCount = channelCount;
    audioRendererParameter.sinkCount = 1;
    NN_ABORT_UNLESS(nn::audio::IsValidAudioRendererParameter(audioRendererParameter));

    std::size_t audioRendererWorkBufferSize = nn::audio::GetAudioRendererWorkBufferSize(audioRendererParameter);
    void* audioRendererWorkBuffer = allocator.Allocate(audioRendererWorkBufferSize, nn::os::MemoryPageSize);
    NN_ABORT_UNLESS_NOT_NULL(audioRendererWorkBuffer);

    nn::audio::AudioRendererHandle audioRendererHandle;
    nn::os::SystemEvent systemEvent;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::OpenAudioRenderer(&audioRendererHandle, &systemEvent, audioRendererParameter, audioRendererWorkBuffer, audioRendererWorkBufferSize));

    std::size_t configBufferSize = nn::audio::GetAudioRendererConfigWorkBufferSize(audioRendererParameter);
    void* configBuffer = allocator.Allocate(configBufferSize, nn::os::MemoryPageSize);
    NN_ABORT_UNLESS_NOT_NULL(configBuffer);
    nn::audio::AudioRendererConfig config;
    nn::audio::InitializeAudioRendererConfig(&config, audioRendererParameter, configBuffer, configBufferSize);

    nn::audio::FinalMixType finalMix;
    nn::audio::AcquireFinalMix(&config, &finalMix, 2);

    nn::audio::DeviceSinkType sink;
    int8_t bus[2] = { 0, 1 };
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::AddDeviceSink(&config, &sink, &finalMix, bus, sizeof(bus) / sizeof(bus[0]), "MainAudioOut"));

    // WaveBuffer に追加するオーディオデータを保持するためのメモリプールを準備します。
    nn::audio::MemoryPoolType pcmBufferMemoryPool;
    NN_ABORT_UNLESS(nn::audio::AcquireMemoryPool(&config, &pcmBufferMemoryPool, g_PcmBuffer, sizeof(g_PcmBuffer)));
    NN_ABORT_UNLESS(nn::audio::RequestAttachMemoryPool(&pcmBufferMemoryPool));

    nn::audio::VoiceType voice;
    nn::audio::AcquireVoiceSlot(&config, &voice, sampleRate, channelCount, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0);
    nn::audio::SetVoiceDestination(&config, &voice, &finalMix);
    nn::audio::SetVoicePlayState(&voice, nn::audio::VoiceType::PlayState_Play);
    if(channelCount == 1)
    {
        nn::audio::SetVoiceMixVolume(&voice, &finalMix, 1.0f, 0, 0);
        nn::audio::SetVoiceMixVolume(&voice, &finalMix, 1.0f, 0, 1);
    }
    else
    {
        nn::audio::SetVoiceMixVolume(&voice, &finalMix, 1.0f, 0, 0);
        nn::audio::SetVoiceMixVolume(&voice, &finalMix, 1.0f, 1, 1);
    }

    // ストリーム再生用のバッファを用意します
    // 最初は 0 で初期化したものを渡し、この後のメインループで Opus データのデコード結果を与えています
    nn::audio::WaveBuffer waveBuffer[BufferCount];
    const std::size_t pcmBufferSize = OpusFrameSampleCount * OpusChannelCount * sizeof(int16_t);
    void* pcmBuffer[BufferCount];
    for (int i = 0; i < BufferCount; ++i)
    {
        pcmBuffer[i] = pcmBufferAllocator.Allocate(pcmBufferSize, nn::audio::BufferAlignSize);

        std::memset(pcmBuffer[i], 0, pcmBufferSize);

        waveBuffer[i].buffer = pcmBuffer[i];
        waveBuffer[i].size = pcmBufferSize;
        waveBuffer[i].startSampleOffset = 0;
        waveBuffer[i].endSampleOffset = static_cast<int32_t>(pcmBufferSize / sizeof(int16_t));
        waveBuffer[i].loop = false;
        waveBuffer[i].isEndOfStream = false;
        waveBuffer[i].pContext = nullptr;
        waveBuffer[i].contextSize = 0;

        nn::audio::AppendWaveBuffer(&voice, &waveBuffer[i]);
    }

    auto pcmBufferForPreRoll = static_cast<int16_t*>(pcmBufferAllocator.Allocate(pcmBufferSize, nn::audio::BufferAlignSize));

    // オーディオレンダラを開始します
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::StartAudioRenderer(audioRendererHandle));

    // Opus データ本体と、そのサイズを取得します
    unsigned char* p = reinterpret_cast<unsigned char*>(g_OpusData + opusBasicInfo.dataInfoOffset);
    nn::codec::OpusInfoHeader dataInfoHeader;
    std::memcpy(&dataInfoHeader, p, sizeof(dataInfoHeader));
    NN_ABORT_UNLESS(dataInfoHeader.type == nn::codec::OpusInfoType_DataInfo);
    std::size_t remainedCount = dataInfoHeader.size;
    p += sizeof(dataInfoHeader);
    NN_ABORT_UNLESS(opusBasicInfo.frameDataSize != 0, "This sample only supports CBR format opus file.");

    const auto opusDataSampleCount = static_cast<int>(remainedCount / opusBasicInfo.frameDataSize * OpusFrameSampleCount);

    int seekDestSamplePosition = opusDataSampleCount - 1;
    int playbackSamplePosition = 0;
    bool isSeekRequested = false;
    for (;;)
    {
        systemEvent.Wait();
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(audioRendererHandle, &config));

        nn::hid::NpadButtonSet npadButtonCurrent = {};
        nn::hid::NpadButtonSet npadButtonDown = {};
        nn::hid::AnalogStickState analogStickStateL = {};
        nn::hid::AnalogStickState analogStickStateR = {};

        GetHidState(npadButtonCurrent, npadButtonDown, analogStickStateL, analogStickStateR);

        if(npadButtonCurrent.Test< ::nn::hid::NpadButton::Plus >())
        {
            break;
        }

        const auto oldSeekDestSamlePosition = seekDestSamplePosition;

        if(npadButtonCurrent.Test< ::nn::hid::NpadButton::Left >())
        {
            --seekDestSamplePosition;
        }

        if(npadButtonCurrent.Test< ::nn::hid::NpadButton::Right >())
        {
            ++seekDestSamplePosition;
        }

        if(npadButtonCurrent.Test< ::nn::hid::NpadButton::Up >())
        {
            seekDestSamplePosition += OpusFrameSampleCount;
        }

        if(npadButtonCurrent.Test< ::nn::hid::NpadButton::Down >())
        {
            seekDestSamplePosition -= OpusFrameSampleCount;
        }

        if(npadButtonCurrent.Test< ::nn::hid::NpadButton::L >())
        {
            seekDestSamplePosition = 0;
        }

        if(npadButtonCurrent.Test< ::nn::hid::NpadButton::R >())
        {
            seekDestSamplePosition = opusDataSampleCount - 1;
        }

        seekDestSamplePosition = std::max(std::min(seekDestSamplePosition, static_cast<int>(opusDataSampleCount - 1)), 0);

        if(seekDestSamplePosition != oldSeekDestSamlePosition)
        {
            NNS_LOG("Seek Position: %d\n", seekDestSamplePosition);
        }

        if(npadButtonDown.Test< ::nn::hid::NpadButton::A >())
        {
            NNS_LOG("Seek To: %d\n", seekDestSamplePosition);
            isSeekRequested = true;
        }

        // 再生が完了したバッファがあれば、Opus デコードを実行します
        if (auto pReleasedWaveBuffer = nn::audio::GetReleasedWaveBuffer(&voice))
        {
            nn::audio::WaveBuffer* pWaveBuffer = nullptr;
            int16_t* pPcmBuffer = nullptr;
            for (int i = 0; i < BufferCount; ++i)
            {
                if (pReleasedWaveBuffer == &waveBuffer[i])
                {
                    pWaveBuffer = &waveBuffer[i];
                    pPcmBuffer = static_cast<int16_t*>(pcmBuffer[i]);
                    break;
                }
            }

            // Opus ファイルの末尾に到達したときは先頭にシークします
            if(remainedCount == 0)
            {
                seekDestSamplePosition = 0;
                isSeekRequested = true;
            }

            if(isSeekRequested)
            {
                isSeekRequested = false;

                const auto seekDestFrameIndex = seekDestSamplePosition / OpusFrameSampleCount;

                int idleDecodeFrameCount;
                if(seekDestFrameIndex < OpusPreRollFrameCount)
                {
                    decoder.Reset();
                    idleDecodeFrameCount = seekDestFrameIndex;
                }
                else
                {
                    idleDecodeFrameCount = OpusPreRollFrameCount;
                }

                const auto idleDecodeStartDataOffset = (seekDestFrameIndex - idleDecodeFrameCount) * opusBasicInfo.frameDataSize;

                // 再生位置などの情報をジャンプ先に更新します
                p = reinterpret_cast<unsigned char*>(g_OpusData + opusBasicInfo.dataInfoOffset + sizeof(nn::codec::OpusInfoHeader) + idleDecodeStartDataOffset);
                remainedCount = dataInfoHeader.size - idleDecodeStartDataOffset;
                playbackSamplePosition = idleDecodeStartDataOffset / opusBasicInfo.frameDataSize;

                // デコーダのコンテキスト調整のためにアイドルデコードします
                for(int i = 0; i < idleDecodeFrameCount; ++i)
                {
                    std::size_t consumed;
                    int sampleCount;
                    opusResult = decoder.DecodeInterleaved(&consumed, &sampleCount, pcmBufferForPreRoll, OpusFrameSampleCount * OpusChannelCount * sizeof(int16_t), p, remainedCount);
                    NN_ABORT_UNLESS(opusResult == nn::codec::OpusResult_Success);
                    NN_ABORT_UNLESS(sampleCount == OpusFrameSampleCount);

                    p += consumed;
                    remainedCount -= consumed;
                    playbackSamplePosition += sampleCount;
                }

                // シーク先フレームをデコードします
                {
                    std::size_t consumed;
                    int sampleCount;
                    opusResult = decoder.DecodeInterleaved(&consumed, &sampleCount, pPcmBuffer, OpusFrameSampleCount * OpusChannelCount * sizeof(int16_t), p, remainedCount);
                    NN_ABORT_UNLESS(opusResult == nn::codec::OpusResult_Success);
                    NN_ABORT_UNLESS(sampleCount == OpusFrameSampleCount);

                    const auto seekDestSamplePositionInFrame = seekDestSamplePosition % OpusFrameSampleCount;

                    p += consumed;
                    remainedCount -= consumed;

                    pWaveBuffer->size = sampleCount * sizeof(int16_t) * channelCount;

                    // 必要ならばエンコーダの内部遅延分をスキップします
                    if(seekDestSamplePositionInFrame < opusBasicInfo.preSkipSampleCount)
                    {
                        pWaveBuffer->startSampleOffset = opusBasicInfo.preSkipSampleCount;
                    }
                    else
                    {
                        pWaveBuffer->startSampleOffset = seekDestSamplePositionInFrame;
                    }
                    pWaveBuffer->endSampleOffset = sampleCount;
                    pWaveBuffer->loop = false;
                    pWaveBuffer->isEndOfStream = false;

                    nn::audio::AppendWaveBuffer(&voice, pWaveBuffer);

                    playbackSamplePosition += sampleCount;
                }
            }
            else
            {
                // 通常のデコード処理を実行します
                std::size_t consumed;
                int sampleCount;
                opusResult = decoder.DecodeInterleaved(&consumed, &sampleCount, pPcmBuffer, OpusFrameSampleCount * OpusChannelCount * sizeof(int16_t), p, remainedCount);
                NN_ABORT_UNLESS(opusResult == nn::codec::OpusResult_Success);
                NN_ABORT_UNLESS(sampleCount == OpusFrameSampleCount);
                p += consumed;
                remainedCount -= consumed;

                pWaveBuffer->size = sampleCount * sizeof(int16_t) * channelCount;

                // 必要ならばエンコーダの内部遅延分をスキップします
                if(playbackSamplePosition < opusBasicInfo.preSkipSampleCount)
                {
                    pWaveBuffer->startSampleOffset = opusBasicInfo.preSkipSampleCount;
                }
                else
                {
                    pWaveBuffer->startSampleOffset = 0;
                }

                pWaveBuffer->endSampleOffset = sampleCount;
                pWaveBuffer->loop = false;
                pWaveBuffer->isEndOfStream = false;

                nn::audio::AppendWaveBuffer(&voice, pWaveBuffer);

                playbackSamplePosition += sampleCount;
            }
        }
    }

    // 利用していたメモリプールにデタッチ要求をします。
    nn::audio::RequestDetachMemoryPool(&pcmBufferMemoryPool);
    // メモリプールがデタッチされるのを待ちます。
    while (nn::audio::IsMemoryPoolAttached(&pcmBufferMemoryPool))
    {
        nn::audio::GetReleasedWaveBuffer(&voice);

        systemEvent.Wait();
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(audioRendererHandle, &config));
    }
    // MemoryPool を解放する。
    nn::audio::ReleaseMemoryPool(&config, &pcmBufferMemoryPool);
    // バッファを解放する。
    for (int i = 0; i < BufferCount; ++i)
    {
        pcmBufferAllocator.Free(pcmBuffer[i]);
    }
    pcmBufferAllocator.Free(pcmBufferForPreRoll);

    // オーディオレンダラを終了し、メモリを破棄します
    nn::audio::StopAudioRenderer(audioRendererHandle);
    nn::audio::CloseAudioRenderer(audioRendererHandle);
    nn::os::DestroySystemEvent(systemEvent.GetBase());
    allocator.Free(audioRendererWorkBuffer);
    allocator.Free(configBuffer);

    // Opus デコーダを終了し、メモリを破棄します
    decoder.Finalize();
    std::free(workBuffer);

    // ファイルシステムの終了処理を行います
    FinalizeFileSystem();

}  // NOLINT(readability/fn_size)
