﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{CodecOpusCodecProcessingTime.cpp,PageSampleCodecOpusCodecProcessingTime}
 *
 * @brief
 * Opus エンコーダ/デコーダの処理時間計測サンプルプログラム
 */

/**
 * @page PageSampleCodecOpusCodecProcessingTime Opus エンコーダ/デコーダの処理時間計測
 * @tableofcontents
 *
 * @brief
 * Opus エンコーダ/デコーダの処理時間計測サンプルプログラムの解説です。
 *
 * @section PageSampleCodecOpusCodecProcessingTime_SectionBrief 概要
 * Opus エンコーダ/デコーダの処理時間を計測するサンプルです。
 *
 * @section PageSampleCodecOpusCodecProcessingTime_SectionFileStructure ファイル構成
 * 本サンプルプログラムは @link ../../../Samples/Sources/Applications/CodecOpusCodecProcessingTime
 * Samples/Sources/Applications/CodecOpusCodecProcessingTime @endlink 以下にあります。
 *
 * @section PageSampleCodecOpusCodecProcessingTime_SectionNecessaryEnvironment 必要な環境
 * デコード結果を再生する場合は、オーディオ出力が利用可能である必要があります。
 *
 * @section PageSampleCodecOpusCodecProcessingTime_SectionHowToOperate 操作方法
 * サンプルプログラムを実行すると、Opus エンコーダ/デコーダの処理時間計測がおこなわれます。
 * 各ケースの計測結果のログ出力をおこなった後、自動的に終了します。
 *
 * @section PageSampleCodecOpusCodecProcessingTime_SectionPrecaution 注意事項
 * デコード結果の再生は、プログラムの動作確認等、処理時間計測以外で使うことを想定しています。
 * デコード結果の再生を有効にすると、処理時間を正しく計測できない可能性があります。
 *
 * @section PageSampleCodecOpusCodecProcessingTime_SectionHowToExecute 実行手順
 * サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleCodecOpusCodecProcessingTime_SectionDetail 解説
 *
 * @subsection PageSampleCodecOpusCodecProcessingTime_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  CodecOpusCodecProcessingTime.cpp
 *  @includelineno CodecOpusCodecProcessingTime.cpp
 *
 * @subsection PageSampleCodecOpusCodecProcessingTime_SectionSampleDetail サンプルプログラムの解説
 *
 * このサンプルプログラムは、音源データをエンコードする処理時間と、エンコードされた Opus データをデコードする処理時間を計測するものです。
 * エンコードパラメータは次の通りです。
 *
 * - ビットレート (10, 29, 40, 64, 96, 128, 192 [kbps])
 * - フレーム (2.5, 5.0, 10.0, 20.0 [ms])
 * - 符号化モード (Silk, Celt)
 *
 * サンプルプログラムの処理の流れは以下の通りです。
 *
 * - ファイルシステムの初期化
 * - オーディオデータの読み込み
 * - Opus エンコーダ/デコーダの初期化
 * - オーディオレンダラの初期化、および開始 (再生時のみ)
 * - Opus エンコーダの実行
 * - Opus デコーダの実行
 * - デコード結果の再生  (再生時のみ)
 * - オーディオレンダラの終了  (再生時のみ)
 * - Opus エンコーダ/デコーダの終了
 * - Opus エンコード/デコード処理時間の表示
 *
 * ここでは、Opus エンコード/デコードの処理時間計測について説明します。
 * Opus デコーダの使い方については、@link ../../../Samples/Sources/Applications/CodecOpusDecoder Samples/Sources/Applications/CodecOpusDecoder @endlink をご参照ください。
 *
 * USE_HARDWARE_DECODER マクロを定義してビルドすると nn::codec::HardwareOpusDecoder が使用されます。
 * 以下の説明は nn::codec::OpusDecoder を代表例としています。
 * nn::codec::HardwareOpusDecoder を使用する場合は、OpusDecoder を HardwareOpusDecoder に置き換えてお読み下さい。
 *
 * ENABLE_PLAY マクロを定義してビルドすると、Opus デコードの結果を再生します (オーディオ出力が利用可能である必要があります)。
 * エンコード/デコード処理の動作確認等、処理時間計測以外の目的で使用して下さい。
 * 処理時間を計測する際は、計測をできる限り正確にするため、Opus デコード結果の再生は無効にして下さい。
 *
 * ファイルから読み込みんだオーディオデータを、フレーム単位でエンコードします。
 * エンコード処理の直後に、エンコーダが出力した Opus パケットをデコードします。
 * 本サンプルでは、エンコード/デコード一回毎の処理時間を計測しています。
 * 処理時間は、それぞれ
 * nn::codec::OpusEncoder::EncodeInterleaved()、
 * nn::codec::OpusDecoder::DecodeInterleaved() の呼び出し前後で取得した nn::os::Tick の差から計算しています。
 *
 * 全てのオーディオデータのエンコード/デコードが終わると、エンコード/デコード処理時間の平均値、最小値、最大値、標準偏差を計算して表示します。
 * この処理を、全てのエンコードパラメータに対して実施します。
 *
 * 全てのオーディオファイルに対して計測を終えると、プログラムは自動的に終了します。
 *
 */

#include <cstdlib>  // std::malloc, std::free
#include <cmath> // std::pow
#include <string> // std::string
#include <algorithm> // std::max, std::min

#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/audio.h>
#include <nn/codec.h>
#include <nn/codec/codec_OpusEncoder.h>
#include <nn/fs.h>
#include <nn/mem.h>
#include <nn/os.h>
#include <nn/util/util_FormatString.h>
#include <nns/audio/audio_WavFormat.h>

#include <nns/nns_Log.h>

//==============================================================================
// USE_HARDWARE_DECODER を有効にすると、HardwareOpusDecoder に切り替わります。
// #define USE_HARDWARE_DECODER
//==============================================================================

//==============================================================================
// ENABLE_PLAY を有効にすると、デコード結果を再生します。
// #define ENABLE_PLAY
//==============================================================================

namespace {

#if defined(ENABLE_PLAY)
const int AudioFrameSampleCount = 240;
#endif  // defined(ENABLE_PLAY)

enum ProcessingId
{
    ProcessingId_Encoding = 0,
    ProcessingId_Decoding = 1,
    ProcessingId_Count
};

// 計測プログラムに与えるエンコードパラメータを格納する構造体です。
struct CodecParameterSet
{
    int frameDuration;
    int bitRate;
    nn::codec::OpusBitRateControl bitRateControl;
    nn::codec::OpusCodingMode codingMode;
};

// Wav 形式のファイルを取り扱うクラスです。
class Wav
{
public:
    Wav()
        : m_SampleRate(0)
        , m_ChannelCount(0)
        , m_IsInitialized(false)
        , m_BufferAddress(nullptr)
    {};
    ~Wav()
    {
        m_PcmDataAddress = nullptr;
        if (nullptr != m_BufferAddress)
        {
            delete[] m_BufferAddress;
            m_BufferAddress = nullptr;
        }
        m_IsInitialized = false;
    };

    void Load(const char* filePath)
    {
        // Wav データをロードします。
        nn::fs::FileHandle handle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&handle, filePath, nn::fs::OpenMode_Read));
        int64_t fileSize;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetFileSize(&fileSize, handle));
        m_BufferAddress = new uint8_t[static_cast<size_t>(fileSize)];
        NN_ABORT_UNLESS_NOT_NULL(m_BufferAddress);
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(handle, 0, m_BufferAddress, static_cast<std::size_t>(fileSize)));
        nn::fs::CloseFile(handle);

        // ヘッダより、サンプルレートとチャンネル数を取得します
        nns::audio::WavFormat wavFormat;
        NN_ABORT_UNLESS_EQUAL(nns::audio::WavResult_Success, ParseWavFormat(&wavFormat, m_BufferAddress, static_cast<size_t>(fileSize)));
        m_ChannelCount = wavFormat.channelCount;
        m_SampleRate = wavFormat.sampleRate;
        m_SampleCountPerChannel = wavFormat.dataSize / (wavFormat.bitsPerSample >> 3) / wavFormat.channelCount;
        m_PcmDataAddress = reinterpret_cast<const int16_t*>(m_BufferAddress + wavFormat.dataOffset);

        m_IsInitialized = true;
    }

    inline int GetSampleRate() const NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        return m_SampleRate;
    }

    inline int GetChannelCount() const NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        return m_ChannelCount;
    }

    inline const int16_t* GetPcmDataAddress() const NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        return m_PcmDataAddress;
    }

    inline int64_t GetSampleCountPerChannel() const NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        return m_SampleCountPerChannel;
    }

    inline bool IsInitialized() const NN_NOEXCEPT
    {
        return m_IsInitialized;
    }

private:
    int m_SampleRate;
    int m_ChannelCount;
    int64_t m_SampleCountPerChannel;
    bool m_IsInitialized;
    uint8_t* m_BufferAddress;
    const int16_t* m_PcmDataAddress;
};

const int BufferCount = 4;
NN_STATIC_ASSERT(BufferCount <= nn::audio::VoiceType::WaveBufferCountMax);

//------------------------------
// エンコードパラメータを定義します。
//------------------------------
// ビットレート
const int BitRates[] =
{
     10000,
     20000,
     40000,
     64000,
     96000,
    128000,
    192000
};

// フレーム
const int FrameDurations[] =
{
      2500,
      5000,
     10000,
     20000
};

// 符号化モード
const nn::codec::OpusCodingMode CodingModes[] =
{
      nn::codec::OpusCodingMode_Celt,
      nn::codec::OpusCodingMode_Silk
};

// エンコード/デコード処理時間の各種結果を格納する構造体です。
struct ProcessingTime
{
    double total;
    double average;
    double minimum;
    double maximum;
    double standardDeviation;
    int iterationCount;
};

NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN int16_t g_PcmBuffer[1024 * 1024];

#if defined(ENABLE_PLAY)
char g_HeapBuffer[8 * 1024 * 1024];
#endif // defined(ENABLE_PLAY)

char* g_MountRomCacheBuffer = NULL;

const std::string GetMountPointRom()
{
    return std::string("asset");
}

void InitializeFileSystem()
{
    size_t cacheSize = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&cacheSize));
    g_MountRomCacheBuffer = new char[cacheSize];
    NN_ABORT_UNLESS_NOT_NULL(g_MountRomCacheBuffer);

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::fs::MountRom(GetMountPointRom().c_str(), g_MountRomCacheBuffer, cacheSize)
    );
}

void FinalizeFileSystem()
{
    nn::fs::Unmount(GetMountPointRom().c_str());

    delete[] g_MountRomCacheBuffer;
    g_MountRomCacheBuffer = NULL;
}

void CollectUpProcessingTime(struct ProcessingTime* pProcessingTime, const int64_t* pScores, int count)
{
    const double ScalingFactor = 1.0e+6;

    double sum = 0;
    int64_t maximum = pScores[0];
    int64_t minimum = pScores[0];

    for (auto k = 0; k < count; ++k)
    {
        sum += static_cast<double>(pScores[k]);
        maximum = std::max(maximum, pScores[k]);
        minimum = std::min(minimum, pScores[k]);
    }
    const double average = sum / count;
    double sd = 0;
    for (auto k = 0; k < count; ++k)
    {
        sd += std::pow(static_cast<double>(pScores[k]) - average, 2);
    }
    sd /= count;

    pProcessingTime->average = average / ScalingFactor;
    pProcessingTime->minimum = static_cast<double>(minimum / ScalingFactor);
    pProcessingTime->maximum = static_cast<double>(maximum / ScalingFactor);
    pProcessingTime->standardDeviation = std::sqrt(sd / ScalingFactor);
    pProcessingTime->iterationCount = count;
    pProcessingTime->total = static_cast<double>(sum / ScalingFactor);
}

}

void EvaluateOpusProcessingPerformance(struct ProcessingTime processingTime[], const Wav& wav, const struct CodecParameterSet& parameterSet)
{
#if defined(ENABLE_PLAY)
    nn::mem::StandardAllocator allocator(g_HeapBuffer, sizeof(g_HeapBuffer));
    nn::mem::StandardAllocator pcmBufferAllocator(g_PcmBuffer, sizeof(g_PcmBuffer));
#endif // defined(ENABLE_PLAY)

    const int sampleRate = wav.GetSampleRate();
    const int channelCount = wav.GetChannelCount();

    // Opus エンコーダを初期化します
    nn::codec::OpusEncoder encoder;
    void* workBufferForEncoder = std::malloc(encoder.GetWorkBufferSize(sampleRate, channelCount));
    NN_ABORT_UNLESS_NOT_NULL(workBufferForEncoder);
    NN_ABORT_UNLESS_EQUAL(
        nn::codec::OpusResult_Success,
        encoder.Initialize(sampleRate, channelCount, workBufferForEncoder, encoder.GetWorkBufferSize(sampleRate, channelCount))
    );
    encoder.SetBitRate(parameterSet.bitRate);
    encoder.SetBitRateControl(parameterSet.bitRateControl);
    encoder.BindCodingMode(parameterSet.codingMode);

    // Opus デコーダを初期化します
#ifndef USE_HARDWARE_DECODER
    nn::codec::OpusDecoder decoder;
#else  // #ifndef USE_HARDWARE_DECODER
    const int32_t noOption = 0; // 現在、指定可能なオプションはありません
    nn::codec::HardwareOpusDecoder decoder(noOption);
#endif // #ifndef USE_HARDWARE_DECODER
    void* workBufferForDecoder = std::malloc(decoder.GetWorkBufferSize(sampleRate, channelCount));
    NN_ABORT_UNLESS_NOT_NULL(workBufferForDecoder);
    NN_ABORT_UNLESS_EQUAL(
        nn::codec::OpusResult_Success,
        decoder.Initialize(sampleRate, channelCount, workBufferForDecoder, decoder.GetWorkBufferSize(sampleRate, channelCount))
    );

    const int sampleCountPerCodecProcessing = encoder.CalculateFrameSampleCount(parameterSet.frameDuration);
    const int sampleCountPerAudioProcessing = std::max(sampleRate / 1000 * 5, encoder.CalculateFrameSampleCount(parameterSet.frameDuration));
    const auto pcmBufferSize = sampleCountPerAudioProcessing * channelCount * sizeof(int16_t);

#if defined(ENABLE_PLAY)
    //オーディオレンダラを構築します
    nn::audio::AudioRendererParameter audioRendererParameter;
    nn::audio::InitializeAudioRendererParameter(&audioRendererParameter);
    audioRendererParameter.sampleRate = 48000;
    audioRendererParameter.sampleCount = AudioFrameSampleCount;
    audioRendererParameter.mixBufferCount = 2;
    audioRendererParameter.voiceCount = channelCount;
    audioRendererParameter.sinkCount = 1;
    NN_ABORT_UNLESS(nn::audio::IsValidAudioRendererParameter(audioRendererParameter));

    std::size_t audioRendererWorkBufferSize = nn::audio::GetAudioRendererWorkBufferSize(audioRendererParameter);
    void* audioRendererWorkBuffer = allocator.Allocate(audioRendererWorkBufferSize, nn::os::MemoryPageSize);
    NN_ABORT_UNLESS_NOT_NULL(audioRendererWorkBuffer);

    nn::audio::AudioRendererHandle audioRendererHandle;
    nn::os::SystemEvent systemEvent;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::OpenAudioRenderer(&audioRendererHandle, &systemEvent, audioRendererParameter, audioRendererWorkBuffer, audioRendererWorkBufferSize));

    std::size_t configBufferSize = nn::audio::GetAudioRendererConfigWorkBufferSize(audioRendererParameter);
    void* configBuffer = allocator.Allocate(configBufferSize, nn::os::MemoryPageSize);
    NN_ABORT_UNLESS_NOT_NULL(configBuffer);
    nn::audio::AudioRendererConfig config;
    nn::audio::InitializeAudioRendererConfig(&config, audioRendererParameter, configBuffer, configBufferSize);

    nn::audio::FinalMixType finalMix;
    nn::audio::AcquireFinalMix(&config, &finalMix, 2);

    nn::audio::DeviceSinkType sink;
    int8_t bus[2] = { 0, 1 };
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::AddDeviceSink(&config, &sink, &finalMix, bus, sizeof(bus) / sizeof(bus[0]), "MainAudioOut"));

    // WaveBuffer に追加するオーディオデータを保持するためのメモリプールを準備します。
    nn::audio::MemoryPoolType pcmBufferMemoryPool;
    NN_ABORT_UNLESS(nn::audio::AcquireMemoryPool(&config, &pcmBufferMemoryPool, g_PcmBuffer, sizeof(g_PcmBuffer)));
    NN_ABORT_UNLESS(nn::audio::RequestAttachMemoryPool(&pcmBufferMemoryPool));

    nn::audio::VoiceType voice;
    nn::audio::AcquireVoiceSlot(&config, &voice, sampleRate, channelCount, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0);
    nn::audio::SetVoiceDestination(&config, &voice, &finalMix);
    nn::audio::SetVoicePlayState(&voice, nn::audio::VoiceType::PlayState_Play);
    nn::audio::SetVoiceMixVolume(&voice, &finalMix, 1.0f, 0, 0);
    nn::audio::SetVoiceMixVolume(&voice, &finalMix, 1.0f, channelCount - 1, 1);

    // ストリーム再生用のバッファを用意します
    // 最初は 0 で初期化したものを渡し、この後のメインループで Opus データのデコード結果を与えています
    nn::audio::WaveBuffer waveBuffer[BufferCount];
    void* pcmBuffer[BufferCount];
    for (int i = 0; i < BufferCount; ++i)
    {
        pcmBuffer[i] = pcmBufferAllocator.Allocate(pcmBufferSize, nn::audio::BufferAlignSize);
        std::memset(pcmBuffer[i], 0, pcmBufferSize);

        waveBuffer[i].buffer = pcmBuffer[i];
        waveBuffer[i].size = pcmBufferSize;
        waveBuffer[i].startSampleOffset = 0;
        waveBuffer[i].endSampleOffset = sampleCountPerAudioProcessing;
        waveBuffer[i].loop = false;
        waveBuffer[i].isEndOfStream = false;
        waveBuffer[i].pContext = nullptr;
        waveBuffer[i].contextSize = 0;

        nn::audio::AppendWaveBuffer(&voice, &waveBuffer[i]);
    }

    // オーディオレンダラを開始します
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::StartAudioRenderer(audioRendererHandle));
#endif // #if defined(ENABLE_PLAY)

    // Wav データの先頭アドレスと、そのサイズを取得します
    auto* p = wav.GetPcmDataAddress();
    auto sampleCountRemained = wav.GetSampleCountPerChannel();

    int bufferCount = BufferCount;
    int samplingCount = 0;
    const auto packetCount = static_cast<std::size_t>(
        (sampleCountRemained + std::max(sampleCountPerCodecProcessing, sampleCountPerAudioProcessing) - 1) / sampleCountPerCodecProcessing );
    auto pEncodingScores = new int64_t[packetCount];
    NN_ABORT_UNLESS_NOT_NULL(pEncodingScores);
    auto pDecodingScores = new int64_t[packetCount];
    NN_ABORT_UNLESS_NOT_NULL(pDecodingScores);

    const std::size_t encodedDataSizeMaximum = nn::codec::OpusPacketSizeMaximum;
    const int encodeSampleCountMaximum = encoder.CalculateFrameSampleCount(20000);
    auto encodedDataBuffer = new uint8_t[encodedDataSizeMaximum];
    auto inputDataBuffer = new int16_t[encodeSampleCountMaximum * channelCount];

    for (;;)
    {
#if defined(ENABLE_PLAY)
        systemEvent.Wait();
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(audioRendererHandle, &config));
        // 再生が完了したバッファがあれば、Opus デコードを実行します
        if (auto pReleasedWaveBuffer = nn::audio::GetReleasedWaveBuffer(&voice))
        {
            nn::audio::WaveBuffer* pWaveBuffer = nullptr;
            int16_t* pPcmBuffer = nullptr;
            for (int i = 0; i < BufferCount; ++i)
            {
                if (pReleasedWaveBuffer == &waveBuffer[i])
                {
                    pWaveBuffer = &waveBuffer[i];
                    pPcmBuffer = static_cast<int16_t*>(pcmBuffer[i]);
                    break;
                }
            }
#else
            int16_t* pPcmBuffer = g_PcmBuffer;
#endif // defined(ENABLE_PLAY)

            // データが残っていればデコードを実行します
            if (sampleCountRemained > 0)
            {
#if defined(ENABLE_PLAY)
                for ( auto sampleCount = 0; sampleCount < sampleCountPerAudioProcessing; sampleCount += sampleCountPerCodecProcessing)
                {
#endif // defined(ENABLE_PLAY)
                    // エンコード
                    const int encodeSampleCount = sampleCountPerCodecProcessing;
                    std::size_t copySampleCount = encodeSampleCount;
                    // 端数の処理
                    if (sampleCountRemained < encodeSampleCount)
                    {
                        copySampleCount = static_cast<int>(sampleCountRemained);
                        std::memset(inputDataBuffer, 0x0, sizeof(int16_t) * channelCount * encodeSampleCount);
                    }
                    std::memcpy(inputDataBuffer, p, sizeof(int16_t) * channelCount * copySampleCount);
                    std::size_t encodedDataSize = 0;
                    const auto perEncodeBase = nn::os::GetSystemTick();
                    NN_ABORT_UNLESS_EQUAL(
                        nn::codec::OpusResult_Success,
                        encoder.EncodeInterleaved(
                            &encodedDataSize,
                            encodedDataBuffer,
                            encodedDataSizeMaximum,
                            inputDataBuffer,
                            encodeSampleCount)
                    );
                    pEncodingScores[samplingCount] = (nn::os::GetSystemTick() - perEncodeBase).ToTimeSpan().GetNanoSeconds();

                    p += encodeSampleCount * channelCount;

                    // デコード
                    std::size_t consumed;
                    int outputSampleCount;
                    const auto perDecodeBase = nn::os::GetSystemTick();
                    NN_ABORT_UNLESS_EQUAL(
                        nn::codec::OpusResult_Success,
                        decoder.DecodeInterleaved(&consumed, &outputSampleCount, pPcmBuffer, pcmBufferSize, encodedDataBuffer, encodedDataSize)
                    );
                    pDecodingScores[samplingCount] = (nn::os::GetSystemTick() - perDecodeBase).ToTimeSpan().GetNanoSeconds();
                    samplingCount++;
                    NN_ASSERT(consumed == encodedDataSize);
                    NN_ASSERT(outputSampleCount == encodeSampleCount);
                    sampleCountRemained -= copySampleCount;
#if defined(ENABLE_PLAY)
                    pPcmBuffer += outputSampleCount * channelCount;
                }
#endif // defined(ENABLE_PLAY)

#if defined(ENABLE_PLAY)
                pWaveBuffer->size = sampleCountPerAudioProcessing * channelCount * sizeof(int16_t);
                pWaveBuffer->endSampleOffset = sampleCountPerAudioProcessing;
                pWaveBuffer->loop = false;
                pWaveBuffer->isEndOfStream = false;
                nn::audio::AppendWaveBuffer(&voice, pWaveBuffer);
#endif // defined(ENABLE_PLAY)
            }
            // データが残っていなければ、全てのバッファの再生が完了するまで待ちます
            else if (--bufferCount == 0)
            {
                break;
            }
#if defined(ENABLE_PLAY)
        }
#endif // defined(ENABLE_PLAY)
    }
    // 計測を終了し、集計する。
    CollectUpProcessingTime(processingTime + ProcessingId_Encoding, pEncodingScores, samplingCount);
    CollectUpProcessingTime(processingTime + ProcessingId_Decoding, pDecodingScores, samplingCount);
    delete[] pEncodingScores;
    delete[] pDecodingScores;

#if defined(ENABLE_PLAY)
    // 利用していたメモリプールにデタッチ要求をします。
    nn::audio::RequestDetachMemoryPool(&pcmBufferMemoryPool);
    // メモリプールがデタッチされるのを待ちます。
    while (nn::audio::IsMemoryPoolAttached(&pcmBufferMemoryPool))
    {
        nn::audio::GetReleasedWaveBuffer(&voice);

        systemEvent.Wait();
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(audioRendererHandle, &config));
    }
    // MemoryPool を解放する。
    nn::audio::ReleaseMemoryPool(&config, &pcmBufferMemoryPool);
    // バッファを解放する。
    for (int i = 0; i < BufferCount; ++i)
    {
        pcmBufferAllocator.Free(pcmBuffer[i]);
    }

    // オーディオレンダラを終了し、メモリを破棄します
    nn::audio::StopAudioRenderer(audioRendererHandle);
    nn::audio::CloseAudioRenderer(audioRendererHandle);
    nn::os::DestroySystemEvent(systemEvent.GetBase());
    allocator.Free(audioRendererWorkBuffer);
    allocator.Free(configBuffer);
#endif // #if defined(ENABLE_PLAY)

    // Opus エンコーダ/デコーダを終了し、メモリを破棄します
    decoder.Finalize();
    std::free(workBufferForDecoder);
    encoder.Finalize();
    std::free(workBufferForEncoder);

    delete[] encodedDataBuffer;
    delete[] inputDataBuffer;

}  // NOLINT(readability/fn_size)


extern "C" void nnMain()
{
#ifdef USE_HARDWARE_DECODER
    NNS_LOG("CodecOpusCodecProcessingTime sample (using nn::codec::HardwareOpusDecoder)\n");
#else  // USE_HARDWARE_DECODER
    NNS_LOG("CodecOpusCodecProcessingTime sample (using nn::codec::OpusDecoder)\n");
#endif  // USE_HARDWARE_DECODER

    // ファイルシステムを初期化します
    InitializeFileSystem();

    nn::fs::DirectoryHandle directoryHandle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenDirectory(&directoryHandle, (GetMountPointRom() + ":/").c_str(), nn::fs::OpenDirectoryMode_File));
    int64_t entryCount;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetDirectoryEntryCount(&entryCount, directoryHandle));
    auto entries = new nn::fs::DirectoryEntry[static_cast<int32_t>(entryCount)];
    NN_ABORT_UNLESS_NOT_NULL(entries);
    int64_t tmp;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadDirectory(&tmp, entries, directoryHandle, entryCount));
    NN_ABORT_UNLESS(tmp == entryCount);

    // 結果のラベルを出力
    NNS_LOG("%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s\n",
        "SampleRate",
        "ChannelCount",
        "FrameDuration",
        "BitRate",
        "CodingMode",
        "Count",
        "Total (Encoder)",
        "Average (Encoder)",
        "Minimum (Encoder)",
        "Maximum (Encoder)",
        "StandardDeviation (Encoder)",
        "Total (Decoder)",
        "Average (Decoder)",
        "Minimum (Decoder)",
        "Maximum (Decoder)",
        "StandardDeviation (Decoder)"
    );
    for (int i = 0; i < entryCount; ++i)
    {
        // open input/output files
        struct ProcessingTime processingTime[ProcessingId_Count];
        NN_ABORT_UNLESS_EQUAL(entries[i].directoryEntryType, nn::fs::DirectoryEntryType_File);
        // Wav データを読み込みます
        Wav wav;
        const std::string filePath = GetMountPointRom() + ":/" + entries[i].name;
        wav.Load(filePath.c_str());

        for (auto codingMode : CodingModes)
        {
            for (auto bitRate : BitRates)
            {
                for (auto frameDuration : FrameDurations)
                {
                    const struct CodecParameterSet parameterSet =
                    {
                        frameDuration,
                        bitRate,
                        nn::codec::OpusBitRateControl_Vbr,
                        codingMode,
                    };
                    // 非サポートフレームをスキップ
                    if (codingMode == nn::codec::OpusCodingMode_Silk && frameDuration < 10000)
                    {
                        continue;
                    }
                    EvaluateOpusProcessingPerformance(processingTime, wav, parameterSet);

                    // 試行毎の結果を出力
                    const int id = static_cast<int>(ProcessingId_Encoding);
                    NNS_LOG("%d,%d,%.1f,%d,%s,%d,%.3f,%.3f,%.3f,%.3f,%.3f,%.3f,%.3f,%.3f,%.3f,%.3f\n",
                        wav.GetSampleRate(),
                        wav.GetChannelCount(),
                        static_cast<float>(frameDuration) / 1000.0,
                        bitRate,
                        codingMode == nn::codec::OpusCodingMode_Celt ? "Celt" : "Silk",
                        processingTime[id].iterationCount,
                        processingTime[id].total,
                        processingTime[id].average,
                        processingTime[id].minimum,
                        processingTime[id].maximum,
                        processingTime[id].standardDeviation,
                        processingTime[id + 1].total,
                        processingTime[id + 1].average,
                        processingTime[id + 1].minimum,
                        processingTime[id + 1].maximum,
                        processingTime[id + 1].standardDeviation
                    );
                }
            }
        }
    }
    delete[] entries;
    nn::fs::CloseDirectory(directoryHandle);
    // ファイルシステムの終了処理を行います
    FinalizeFileSystem();
}
