﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
    @page BtmDeviceInfo ペアリング情報の参照、追加、削除
    @tableofcontents

    @brief
    ペアリング情報の参照、追加、削除をおこなう方法を示すサンプルプログラムの解説です。

    @section BtmDeviceInfo_SectionDetail 解説
    サンプルプログラムの全体像は以下の通りです。
    - ペアリング済みデバイス情報の取得
    もし、ペアリング済みデバイスが存在していたら、以下も実行します。
    - ペアリング情報の削除
    - Btm へのペアリング情報の追加
 */

#include <cstdlib>
#include <cstdio>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_TickTypes.h>
#include <nn/settings/system/settings_BluetoothDevices.h>

#include <nn/btm/btm.h>
#include <nn/btm/system/btm_SystemApi.h>

namespace
{
    const char ProgramName[] = "BtmDeviceInfo Sample Application";

    const char ProfileStr[2][16] = {
        "NONE",
        "HID"
    };

    nn::os::SystemEventType g_DeviceInfoEvent;
} // anonymous-namespace

namespace
{

char* BdaddrToString(const nn::btm::BdAddress &bdaddr)
{
    static char strBdaddr[18];
    snprintf(strBdaddr, sizeof(strBdaddr), "%02X:%02X:%02X:%02X:%02X:%02X",
        bdaddr.address[0], bdaddr.address[1], bdaddr.address[2], bdaddr.address[3], bdaddr.address[4], bdaddr.address[5]);
    return strBdaddr;
}

}

extern "C" void nnMain()
{
    NN_LOG("%s Start.\n", ProgramName);

    nn::btm::InitializeBtmInterface();

    // DeviceInfo 変更完了を通知するイベントを登録
    nn::btm::RegisterSystemEventForRegisteredDeviceInfo(&g_DeviceInfoEvent);

    // ペアリング済みのデバイス情報を取得します
    nn::btm::DeviceInfoList list;
    nn::btm::GetDeviceInfo(&list);

    NN_LOG("-- Paired Device Info --\n");
    for (int i = 0; i < list.deviceCount; ++i)
    {
        NN_LOG("Dev:%d\n", i);
        NN_LOG("    BD_ADDR: %s\n", BdaddrToString(list.device[i].bdAddress));
        NN_LOG("    COD: %02X %02X %02X\n", list.device[i].classOfDevice.cod[0], list.device[i].classOfDevice.cod[1], list.device[i].classOfDevice.cod[2]);
        NN_LOG("    Name: %s\n", list.device[i].bdName.name);
        NN_LOG("    Profile:%s\n", ProfileStr[list.device[i].profile]);
    }
    NN_LOG("-- ------------------- --\n");

    if (list.deviceCount > 0)
    {
        nn::btm::BdAddress testAddress;
        nn::btm::DeviceInfo testInfo;
        uint8_t count;
        memcpy(testAddress.address, list.device[0].bdAddress.address, nn::btm::SIZE_OF_BDADDRESS);
        memcpy(&testInfo, &list.device[0], sizeof(nn::btm::DeviceInfo));
        count = list.deviceCount;

        // ペアリング情報をbtmから削除します。該当デバイスが接続されている場合、切断します
        nn::btm::RemoveDeviceInfo(&testAddress);
        NN_SDK_ASSERT(nn::os::TimedWaitSystemEvent(&g_DeviceInfoEvent, nn::TimeSpan::FromMilliSeconds(500)));

        nn::btm::GetDeviceInfo(&list);
        NN_SDK_ASSERT_EQUAL(list.deviceCount, count - 1);

        NN_LOG("-- Paired Device Info --\n");
        for (int i = 0; i < list.deviceCount; ++i)
        {
            NN_LOG("Dev:%d\n", i);
            NN_LOG("    BD_ADDR: %s\n", BdaddrToString(list.device[i].bdAddress));
            NN_LOG("    COD: %02X %02X %02X\n", list.device[i].classOfDevice.cod[0], list.device[i].classOfDevice.cod[1], list.device[i].classOfDevice.cod[2]);
            NN_LOG("    Name: %s\n", list.device[i].bdName.name);
            NN_LOG("    Profile:%s\n", ProfileStr[list.device[i].profile]);
        }
        NN_LOG("-- ------------------- --\n");
        count = list.deviceCount;

        nn::btm::AddDeviceInfo(&testInfo);
        NN_SDK_ASSERT(nn::os::TimedWaitSystemEvent(&g_DeviceInfoEvent, nn::TimeSpan::FromMilliSeconds(500)));

        nn::btm::GetDeviceInfo(&list);
        NN_SDK_ASSERT_EQUAL(list.deviceCount, count + 1);

        NN_LOG("-- Paired Device Info --\n");
        for (int i = 0; i < list.deviceCount; ++i)
        {
            NN_LOG("Dev:%d\n", i);
            NN_LOG("    BD_ADDR: %s\n", BdaddrToString(list.device[i].bdAddress));
            NN_LOG("    COD: %02X %02X %02X\n", list.device[i].classOfDevice.cod[0], list.device[i].classOfDevice.cod[1], list.device[i].classOfDevice.cod[2]);
            NN_LOG("    Name: %s\n", list.device[i].bdName.name);
            NN_LOG("    Profile:%s\n", ProfileStr[list.device[i].profile]);
        }
        NN_LOG("-- ------------------- --\n");

        nn::os::DestroySystemEvent(&g_DeviceInfoEvent);
    }

    nn::btm::FinalizeBtmInterface();

    NN_LOG("%s Done\n", ProgramName);
}

