﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/applet/applet.h>

#include "bluetooth_Client.h"
#include <cstdlib>

//static const nn::bluetooth::GattId gapServiceUuid = { 0, {2, { 0x1800 }} };
//static const nn::bluetooth::GattId deviceNameCharUuid = { 0, {2, { 0x2A00 }} };
//static const nn::bluetooth::GattId appearanceCharUuid = { 0, {2, { 0x2A01 }} };
static const nn::bluetooth::GattId cccDescriptorUuid = { 0, { nn::bluetooth::ClientCharacteristicConfigurationDescriptorUuid } };
static const uint16_t mtu = 300;

static const uint8_t smallServiceUuidRawByte[]         = { 0x00, 0x00, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 0x00 };
static const uint8_t smallReadCharUuidRawByte[]        = { 0x01, 0x01, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 0x00 };
static const uint8_t smallWriteReqCharUuidRawByte[]    = { 0x02, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 0x00 };
static const uint8_t smallWriteCmdCharUuidRawByte[]    = { 0x03, 0x03, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 0x00 };
static const uint8_t smallNotifCharUuidRawByte[]       = { 0x04, 0x04, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 0x00 };

static const uint8_t largeServiceUuidRawByte[]         = { 0x10, 0x10, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F, 0x10 };
static const uint8_t largeReadCharUuidRawByte[]        = { 0x11, 0x11, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F, 0x10 };
static const uint8_t largeWriteReqCharUuidRawByte[]    = { 0x12, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F, 0x10 };
static const uint8_t largeWriteCmdCharUuidRawByte[]    = { 0x13, 0x13, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F, 0x10 };
static const uint8_t largeNotifCharUuidRawByte[]       = { 0x14, 0x14, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F, 0x10 };

static const uint8_t manufacturerId[]   = {0x53, 0x05};
static const uint8_t manufacturerData[] = {0x00, 0x00, 0x01, 0x00, 0x00, 0x01, 0x00};
static const uint8_t filterMask[]       = {0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF};

struct BleGattAttribute
{
    nn::bluetooth::GattId               id;
    uint16_t                            handle;
    uint16_t                            endGroupHandle;     // used only for Gatt Service
    nn::bluetooth::GattAttributeType    type;
    uint8_t                             property;
    bool                                isPrimaryService;   // used only for Gatt Service
};

void getGattIdFromBytes(nn::bluetooth::GattAttributeUuid* pOutUuid, const uint8_t* bytes, size_t size)
{
    if ((size != 2 && size != 4 && size != 16) || !pOutUuid)    return;

    nn::bluetooth::GattAttributeUuid uuid;

    uuid.length = (nn::bluetooth::GattAttributeUuidLength)size;
    if (uuid.length == 2)          memcpy(reinterpret_cast<uint8_t*>(uuid.uu.uuid16), bytes, uuid.length);
    else if (uuid.length == 4)     memcpy(reinterpret_cast<uint8_t*>(uuid.uu.uuid32), bytes, uuid.length);
    else if (uuid.length == 16)    memcpy(uuid.uu.uuid128, bytes, uuid.length);

    memcpy(pOutUuid, &uuid, sizeof(nn::bluetooth::GattAttributeUuid));

    return;
}

const char* toHexString(const nn::bluetooth::GattAttributeUuid &uuid)
{
    if (uuid.length == 2)
    {
        uint8_t uuidArray[2];
        uuidArray[0] = uuid.uu.uuid16;
        uuidArray[1] = (uuid.uu.uuid16 >> 8);
        return bluetoothClient::toHexString(2, uuidArray);
    }
    else if (uuid.length == 4)
    {
        uint8_t uuidArray[4];
        uuidArray[0] = uuid.uu.uuid32;
        uuidArray[1] = (uuid.uu.uuid32 >> 8);
        uuidArray[2] = (uuid.uu.uuid32 >> 16);
        uuidArray[3] = (uuid.uu.uuid32 >> 24);
        return bluetoothClient::toHexString(4, uuidArray);
    }
    else
    {
        return bluetoothClient::toHexString(16, uuid.uu.uuid128);
    }
}

class myBtClient : public bluetoothClient
{
    public:

    bool discoverServiceComplete = false;
    uint8_t clientIf = 0xFF;
    uint16_t connectionId = 0xFF;
    uint16_t mtu = 0x00;
    bool isRegistered           = false;
    bool isScanFilterEnabled    = false;
    bool isScanning             = false;
    bool serverConnected        = false;
    bool mtuConfigured          = false;
    nn::bluetooth::BluetoothAddress     connectedDevice;
    nn::bluetooth::BleAdvertiseFilter   scanFilter;
    nn::bluetooth::LeConnectionParams   connParams;

    //-----------------------------------------------------------------------------
    // startBluetooth
    //
    void startBluetoothLowEnergyClient(nn::bluetooth::GattAttributeUuid uuid)
    {
        initTargetAttribute();

        nn::bluetooth::RegisterLeClient(uuid);

        scanFilter.filterIndex = 0;
        scanFilter.structure.length = sizeof(nn::bluetooth::BleAdType) + sizeof(manufacturerId) + sizeof(manufacturerData);
        scanFilter.structure.adType = nn::bluetooth::BleAdType_ManufactureSpecificData;
        memcpy(scanFilter.structure.data, manufacturerId, sizeof(manufacturerId));
        memcpy(scanFilter.structure.data + sizeof(manufacturerId), manufacturerData, sizeof(manufacturerData));
        scanFilter.maskeLength = sizeof(filterMask);
        memcpy(scanFilter.mask, filterMask, scanFilter.maskeLength);

        nn::bluetooth::ClearLeScanFilters();

        for (int i = 0; i < SMALL_WRITE_DATA_SIZE; ++i)
        {
            smallWriteData[i] = i;
        }

        for (int i = 0; i < LARGE_WRITE_DATA_SIZE; ++i)
        {
            largeWriteData[i] = i;
        }
    }

    int registerTestAttribute(BleGattAttribute* attribute)
    {
        if (!attribute)
        {
            return -1;
        }

        for (int i = 0; i < MAX_TARGET_ATTRIBUTE_NUM; ++i)
        {
            if (!attributeList[i].attribute)
            {
                attributeList[i].attribute = attribute;
                return 0;
            }
        }

        return -1;
    }

    bool isAttributeSupported(BleGattAttribute* attribute)
    {
        if (!attribute)
        {
            return false;
        }

        for (int i = 0; i < MAX_TARGET_ATTRIBUTE_NUM; ++i)
        {
            if (!attributeList[i].attribute)
            {
                continue;
            }

            if (!memcmp(&attributeList[i].attribute->id.uuid, &attribute->id.uuid, sizeof(nn::bluetooth::GattAttributeUuid)))
            {
                return attributeList[i].isSupported;
            }
        }

        return false;
    }

    enum
    {
        SMALL_WRITE_DATA_SIZE = 20,
        LARGE_WRITE_DATA_SIZE = 256,
    };

    uint8_t smallWriteData[SMALL_WRITE_DATA_SIZE];
    uint8_t largeWriteData[LARGE_WRITE_DATA_SIZE];

    private:

    enum {
        MAX_TARGET_ATTRIBUTE_NUM = 50,
    };

    struct TargetAttribute
    {
        BleGattAttribute* attribute;
        bool isSupported;
    };

    TargetAttribute attributeList[MAX_TARGET_ATTRIBUTE_NUM];

    void initTargetAttribute()
    {
        for (int i = 0; i < MAX_TARGET_ATTRIBUTE_NUM; ++i)
        {
            attributeList[i].attribute = NULL;
            attributeList[i].isSupported = false;
        }
    }

    enum
    {
        NONE,
        CANCELING_BLE_DISC,
        BLE_CONNECT
    } bleConnectState;

    void EventFromLeConnParamUpdateCallback(const nn::bluetooth::InfoFromLeConnParamUpdateCallback* pInfo)
    {
        NN_LOG("*****[%s]*****\n", __func__);

        if (pInfo->status != nn::bluetooth::BTHH_OK)
        {
            NN_LOG("Error. Failed to update connection parameters.\n");
            return;
        }

        NN_LOG("  Connection ID: %d\n", pInfo->connId);
        NN_LOG("  Connection Interval: %d (* 1.25msec)\n", pInfo->interval);
        NN_LOG("  Slave Latency: %d\n", pInfo->slaveLatency);
        NN_LOG("  Supervision Timeout: %d\n", pInfo->supervisionTimeout);

        return;
    }

    void EventFromLeScanStateChangedCallback(const nn::bluetooth::InfoFromLeScanStateChangedCallback* pInfo)
    {
        NN_LOG("*****[%s]*****\n", __func__);

        switch (pInfo->state)
        {
        case nn::bluetooth::BLE_SCAN_STATE_SCANNING:
            NN_LOG("  BLE_SCAN_STATE_STARTED\n");
            isScanning = true;
            break;
        case nn::bluetooth::BLE_SCAN_STATE_COMPLETED:
            NN_LOG("  BLE_SCAN_STATE_STOPPED\n");

            isScanning = false;
            if(bleConnectState == CANCELING_BLE_DISC)
            {
                nn::bluetooth::LeClientConnect(nn::applet::GetAppletResourceUserId(), clientIf, &connectedDevice, true);
                bleConnectState = BLE_CONNECT;
            }
            break;
        case nn::bluetooth::BLE_SCAN_STATE_FOUND_DEVICE:
            NN_LOG("  BLE_SCAN_STATE_FOUND_DEVICE\n");

            NN_LOG("    PDU Type: %d (0: ADV_IND, 1: DIR_IND, 2: SCAN_IND, 3: NONCONN_IND, 4: SCAN_RSP)\n", pInfo->pduType);
            NN_LOG("    Address Type: %d\n", pInfo->addressType);
            NN_LOG("    Address: %s\n", toHexString(pInfo->address));

            for (int i = 0; i < pInfo->adStructureNum; ++i)
            {
                NN_LOG("    AD Structure #%d\n", i);
                NN_LOG("      Length: %d\n", pInfo->adStructures[i].length);
                NN_LOG("      AD Type: %d\n", pInfo->adStructures[i].adType);
                NN_LOG("      Payload: ");

                for (int j = 0; j < pInfo->adStructures[i].length - 1; ++j)
                {
                    NN_LOG("%02X ", pInfo->adStructures[i].data[j]);
                }
                NN_LOG("\n");
            }

            NN_LOG("  Found filter matching Device\n");
            NN_LOG("  Cancel Low Energy Discovery\n");
            if (isScanning)
            {
                nn::bluetooth::StopLeScan();
                isScanning = false;

                memcpy(connectedDevice.address, pInfo->address.address, 6);
            }
            bleConnectState = CANCELING_BLE_DISC;
            break;
        default:
            NN_LOG("ERROR: Unexpected default.\n");
            break;
        }
    }

    void EventFromLeScanFilterStateChangedCallback(const nn::bluetooth::InfoFromLeScanFilterStateChangedCallback* pInfo)
    {
        NN_LOG("*****[%s]*****\n", __func__);

        switch (pInfo->operation)
        {
        case nn::bluetooth::BLE_SCAN_FILTER_OP_ENABLE:
            if (pInfo->status != nn::bluetooth::BTHH_OK)
            {
                NN_LOG("ERROR: Failed to enable BLE scan filter. status %d\n", pInfo->status);
                break;
            }
            NN_LOG("  Scan filter enabled\n");
            isScanFilterEnabled = true;
            break;
        case nn::bluetooth::BLE_SCAN_FILTER_OP_DISABLE:
            if (pInfo->status != nn::bluetooth::BTHH_OK)
            {
                NN_LOG("ERROR: Failed to disable BLE scan filter. status %d\n", pInfo->status);
                break;
            }

            NN_LOG("  Scan filter disabled\n");
            isScanFilterEnabled = false;
            break;
        case nn::bluetooth::BLE_SCAN_FILTER_OP_COND_ADD:
            if (pInfo->status != nn::bluetooth::BTHH_OK)
            {
                NN_LOG("ERROR: Failed to add BLE scan filter condition. status %d\n", pInfo->status);
                break;
            }

            NN_LOG("  Scan filter condition added\n");
            if (!isScanFilterEnabled)
            {
                NN_LOG("  Enable scan filter.\n");
                nn::bluetooth::EnableLeScanFilter(true);
            }
            break;
        case nn::bluetooth::BLE_SCAN_FILTER_OP_COND_DELETE:
            if (pInfo->status != nn::bluetooth::BTHH_OK)
            {
                NN_LOG("ERROR: Failed to delete BLE scan filter condition. status %d\n", pInfo->status);
                break;
            }
            NN_LOG("  Scan filter condition deleted.\n");
            break;
        case nn::bluetooth::BLE_SCAN_FILTER_OP_COND_CLEAR:
            if (pInfo->status != nn::bluetooth::BTHH_OK)
            {
                NN_LOG("ERROR: Failed to clear BLE scan filter condition. status %d\n", pInfo->status);
                break;
            }
            NN_LOG("  Scan filter condition cleared\n");

            if (isScanFilterEnabled)
            {
                NN_LOG("  Disable scan filter.\n");
                nn::bluetooth::EnableLeScanFilter(false);
            }
            break;
        default:
            break;
        }
    }

    void EventFromLeClientStateChangedCallback(const nn::bluetooth::InfoFromLeAppStateChangedCallback* pInfo)
    {
        NN_LOG("*****[%s]*****\n", __func__);
        if (pInfo->status != nn::bluetooth::BTHH_OK)
        {
            NN_LOG("ERROR. status: %d\n", pInfo->status);
            return;
        }

        NN_LOG("  Client %s\n", pInfo->isRegistered ? "Registered" : "Unregistered");
        clientIf = pInfo->clientIf;

        connParams.minConnectionInterval    = 12;      // 15 msec
        connParams.maxConnectionInterval    = 12;      // 15 msec
        connParams.slaveLatency             = 0;
        connParams.supervisionTimeout       = 100;      // 1000 msec
        connParams.minConnectionEventLen    = 2;
        connParams.maxConnectionEventLen    = 8;
        nn::bluetooth::SetLeDefaultConnectionParameter(connParams);

        isRegistered = pInfo->isRegistered;
    }

    void EventFromLeClientConnStateChangedCallback(const nn::bluetooth::InfoFromLeConnStateChangedCallback* pInfo)
    {
        NN_LOG("*****[%s]*****\n", __func__);
        if (pInfo->status != nn::bluetooth::BTHH_OK)
        {
            NN_LOG("ERROR. status: %d\n", pInfo->status);
            return;
        }

        switch (pInfo->connState)
        {
        case nn::bluetooth::BLE_CONN_STATE_CONNECTED:
            NN_LOG("  Connected to BLE GATT Server\n");
            NN_LOG("  Client: %d    Conn Id: %d\n", pInfo->clientIf, pInfo->connId);
            NN_LOG("  Server Address: %s\n", toHexString(pInfo->address));

            connectedDevice = pInfo->address;
            connectionId = pInfo->connId;
            serverConnected = true;

            // WA until attribute cache feature gets removed from BSA
            if (memcmp(connectedDevice.address, pInfo->address.address, 6) == 0)
            {
                discoverServiceComplete = true;
            }

            break;
        case nn::bluetooth::BLE_CONN_STATE_DISCONNECTED:
            NN_LOG("  Disconnected from BLE GATT Server\n");
            NN_LOG("  Client: %d    Conn Id: %d\n", pInfo->clientIf, pInfo->connId);
            NN_LOG("  Server Address: %s\n", toHexString(pInfo->address));
            NN_LOG("  Reason: %d\n", pInfo->reason);

            nn::bluetooth::UnregisterLeClient(clientIf);

            serverConnected = false;
            discoverServiceComplete = false;
            mtuConfigured = false;
            initTargetAttribute();

            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    void EventFromLeClientGattSrvcDiscCallback(const nn::bluetooth::InfoFromLeClientGattServiceDiscoveryCallback* pInfo)
    {
        NN_LOG("*****[%s]*****\n", __func__);
        if (pInfo->status != nn::bluetooth::BTHH_OK)
        {
            NN_LOG("ERROR. status: %d\n", pInfo->status);
            return;
        }

        NN_LOG("  Connection ID: %d\n", pInfo->connId);
        NN_LOG("  numAttributes: %d\n", pInfo->numAttributes);
        NN_LOG("  Server Address: %s\n", toHexString(pInfo->address));

        for (int i = 0; i < pInfo->numAttributes; ++i)
        {
            NN_LOG("    Handle: %d\n", pInfo->attributes[i].handle);
            if (pInfo->attributes[i].type == nn::bluetooth::GattAttributeType_Service)
            {
                NN_LOG("    Type: Service\n");
                NN_LOG("    End Group Handle: %d\n", pInfo->attributes[i].endGroupHandle);
                NN_LOG("    Is Primary Service: %s\n", pInfo->attributes[i].isPrimaryService ? "TRUE" : "FALSE");
            }
            else if (pInfo->attributes[i].type == nn::bluetooth::GattAttributeType::GattAttributeType_Characteristic)
            {
                NN_LOG("    Type: Characteristic\n");
                NN_LOG("    Property:\n");
                NN_LOG("      Broadcast: %s Read: %s, Write Cmd: %s, Write: %s\n"
                    "      Notify: %s, Indicate: %s, Signed Write: %s, Ext Prop: %s\n",
                    ((pInfo->attributes[i].property & nn::bluetooth::GattAttributeProperty_Broadcast) > 0)              ? "TRUE" : "FALSE",
                    ((pInfo->attributes[i].property & nn::bluetooth::GattAttributeProperty_Read) > 0)                   ? "TRUE" : "FALSE",
                    ((pInfo->attributes[i].property & nn::bluetooth::GattAttributeProperty_WriteWithoutResponse) > 0)   ? "TRUE" : "FALSE",
                    ((pInfo->attributes[i].property & nn::bluetooth::GattAttributeProperty_Write) > 0)                  ? "TRUE" : "FALSE",
                    ((pInfo->attributes[i].property & nn::bluetooth::GattAttributeProperty_Notify) > 0)                 ? "TRUE" : "FALSE",
                    ((pInfo->attributes[i].property & nn::bluetooth::GattAttributeProperty_Indicate) > 0)               ? "TRUE" : "FALSE",
                    ((pInfo->attributes[i].property & nn::bluetooth::GattAttributeProperty_SignedWriteCommand) > 0)     ? "TRUE" : "FALSE",
                    ((pInfo->attributes[i].property & nn::bluetooth::GattAttributeProperty_ExtendedProperty) > 0)       ? "TRUE" : "FALSE"
                );
            }
            else if (pInfo->attributes[i].type == nn::bluetooth::GattAttributeType::GattAttributeType_Descriptor)
            {
                NN_LOG("    Type: Descriptor\n");
            }
            else if (pInfo->attributes[i].type == nn::bluetooth::GattAttributeType::GattAttributeType_IncludedService)
            {
                NN_LOG("    Type: Included Service\n");
            }
            else
            {
                NN_LOG("    Type: Unknown(%d)\n", pInfo->attributes[i].type);
            }
            NN_LOG("    UUID:");


            if (pInfo->attributes[i].id.uuid.length == 2)         NN_LOG("%04X\n", pInfo->attributes[i].id.uuid.uu.uuid16);
            else if (pInfo->attributes[i].id.uuid.length == 4)    NN_LOG("%08X\n", pInfo->attributes[i].id.uuid.uu.uuid32);
            else if (pInfo->attributes[i].id.uuid.length == 16)
            {
                for (int j = 0; j < 16; ++j)
                {
                    NN_LOG("%02X ", pInfo->attributes[i].id.uuid.uu.uuid128[j]);
                }
                NN_LOG("\n");
            }

            NN_LOG("    Instant ID: %d\n", pInfo->attributes[i].id.instanceId);

            for (int j = 0; j < MAX_TARGET_ATTRIBUTE_NUM; ++j)
            {
                if (!attributeList[j].attribute)
                {
                    continue;
                }

                if (attributeList[j].attribute->id.uuid.length == pInfo->attributes[i].id.uuid.length)
                {
                    bool isSame = true;

                    int length = attributeList[j].attribute->id.uuid.length;

                    if (length == 2 && (attributeList[j].attribute->id.uuid.uu.uuid16 != pInfo->attributes[i].id.uuid.uu.uuid16))       isSame = false;
                    else if (length == 4 && (attributeList[j].attribute->id.uuid.uu.uuid32 != pInfo->attributes[i].id.uuid.uu.uuid32))  isSame = false;
                    else if (length == 16)
                    {
                        for (int k = 0; k < length; ++k)
                        {
                            if (attributeList[j].attribute->id.uuid.uu.uuid128[k] != pInfo->attributes[i].id.uuid.uu.uuid128[k])
                            {
                                isSame = false;
                                break;
                            }
                        }
                    }

                    if (isSame)
                    {
                        NN_LOG("  !! Found supported attribute %d !!\n", i);

                        attributeList[j].isSupported = true;
                        attributeList[j].attribute->handle = pInfo->attributes[i].handle;
                        attributeList[j].attribute->endGroupHandle = pInfo->attributes[i].endGroupHandle;
                        attributeList[j].attribute->type = pInfo->attributes[i].type;
                        attributeList[j].attribute->property = pInfo->attributes[i].property;
                        attributeList[j].attribute->isPrimaryService = pInfo->attributes[i].isPrimaryService;
                        attributeList[j].attribute->id.instanceId = pInfo->attributes[i].id.instanceId;

                        break;
                    }
                }
            }
        }

        discoverServiceComplete = true;
    }

    void EventFromLeClientGattOperationCallback(const nn::bluetooth::InfoFromLeGattOperationCallback* pInfo)
    {
        NN_LOG("*****[%s]*****\n", __func__);
        if (pInfo->status != nn::bluetooth::BTHH_OK)
        {
            NN_LOG("ERROR. status: %d\n", pInfo->status);
            return;
        }

        NN_LOG("  Status: %d\n", pInfo->status);
        NN_LOG("  Connection ID: %d\n", pInfo->connId);
        if (pInfo->serviceUuid.length == 2)            NN_LOG("  Service UUID: %04X\n", pInfo->serviceUuid.uu.uuid16);
        else if (pInfo->serviceUuid.length == 4)       NN_LOG("  Service UUID: %08X\n", pInfo->serviceUuid.uu.uuid32);
        else if (pInfo->serviceUuid.length == 16)      NN_LOG("  Service UUID: %s\n", toHexString(pInfo->serviceUuid.length, pInfo->serviceUuid.uu.uuid128));

        if (pInfo->charcteristicUuid.length == 2)          NN_LOG("  Characteristic UUID: %04X\n", pInfo->charcteristicUuid.uu.uuid16);
        else if (pInfo->charcteristicUuid.length == 4)     NN_LOG("  Characteristic UUID: %08X\n", pInfo->charcteristicUuid.uu.uuid32);
        else if (pInfo->charcteristicUuid.length == 16)    NN_LOG("  Characteristic UUID: %s\n", toHexString(pInfo->charcteristicUuid.length, pInfo->charcteristicUuid.uu.uuid128));

        if (pInfo->descriptorUuid.length == 2)             NN_LOG("  Descriptor UUID: %04X\n", pInfo->descriptorUuid.uu.uuid16);
        else if (pInfo->descriptorUuid.length == 4)        NN_LOG("  Descriptor UUID: %08X\n", pInfo->descriptorUuid.uu.uuid32);
        else if (pInfo->descriptorUuid.length == 16)       NN_LOG("  Descriptor UUID: %s\n", toHexString(pInfo->descriptorUuid.length, pInfo->descriptorUuid.uu.uuid128));

        int i = 0;
        switch (pInfo->operation)
        {
        case nn::bluetooth::GattOperationType_ReadCharacteristic:
        case nn::bluetooth::GattOperationType_ReadDescriptor:
            NN_LOG("  Read Length: %d bytes\n", pInfo->length);
            NN_LOG("  Read data:\n  ");
            for (i = 0; i < pInfo->length; ++i)
            {
                NN_LOG("%02X ", pInfo->value[i]);

                if (i % 16 == 15)    NN_LOG("\n  ");
            }

            if(i % 16 != 15)    NN_LOG("\n");
            break;
        case nn::bluetooth::GattOperationType_WriteCharacteristic:
        case nn::bluetooth::GattOperationType_WriteDescriptor:
            NN_LOG("  Write data complete.\n");
            break;
        case nn::bluetooth::GattOperationType_Notify:
            NN_LOG("  Notification Length: %d bytes\n", pInfo->length);
            NN_LOG("  Notification data:\n  ");
            for (i = 0; i < pInfo->length; ++i)
            {
                NN_LOG("%02X ", pInfo->value[i]);

                if (i % 16 == 15)    NN_LOG("\n  ");
            }

            if (i % 16 != 15)    NN_LOG("\n");
            break;
        case nn::bluetooth::GattOperationType_Indicate:
            NN_LOG("  Indication Length: %d bytes\n", pInfo->length);
            NN_LOG("  Indication data:\n");
            for (i = 0; i < pInfo->length; ++i)
            {
                NN_LOG("%02X ", pInfo->value[i]);

                if (i % 16 == 15)    NN_LOG("\n");
            }

            if (i % 16 != 15)    NN_LOG("\n");
            break;
        default:
            break;
        }
    }

    void EventFromLeClientMtuConfigCallback( const nn::bluetooth::InfoFromLeClientMtuConfigurationCallback* pInfo)
    {
        NN_LOG("*****[%s]*****\n", __func__);
        if (pInfo->status != nn::bluetooth::BTHH_OK)
        {
            NN_LOG("ERROR. status: %d\n", pInfo->status);
            return;
        }

        NN_LOG("  ConnId: %d\n", pInfo->connId);
        NN_LOG("  MTU: %d\n", pInfo->mtu);

        mtuConfigured = true;
        mtu = pInfo->mtu;
    }
};


static myBtClient client;

//-----------------------------------------------------------------------------
extern "C" void nnMain()
{
    NN_LOG("NBM_CLIENT NOW RUNNING \n");

    nn::bluetooth::GattAttributeUuid clientUuid;
    clientUuid.length = nn::bluetooth::GattAttributeUuidLength_16;
    clientUuid.uu.uuid16 = 0xBABE;

    BleGattAttribute smallService, smallReadChar, smallWriteReqChar, smallWriteCmdChar, smallNotifChar;
    BleGattAttribute largeService, largeReadChar, largeWriteReqChar, largeWriteCmdChar, largeNotifChar;

    getGattIdFromBytes(&smallService.id.uuid,        smallServiceUuidRawByte,        sizeof(smallServiceUuidRawByte));
    getGattIdFromBytes(&smallReadChar.id.uuid,       smallReadCharUuidRawByte,       sizeof(smallReadCharUuidRawByte));
    getGattIdFromBytes(&smallWriteReqChar.id.uuid,   smallWriteReqCharUuidRawByte,   sizeof(smallWriteReqCharUuidRawByte));
    getGattIdFromBytes(&smallWriteCmdChar.id.uuid,   smallWriteCmdCharUuidRawByte,   sizeof(smallWriteCmdCharUuidRawByte));
    getGattIdFromBytes(&smallNotifChar.id.uuid,      smallNotifCharUuidRawByte,      sizeof(smallNotifCharUuidRawByte));

    getGattIdFromBytes(&largeService.id.uuid,        largeServiceUuidRawByte,        sizeof(largeServiceUuidRawByte));
    getGattIdFromBytes(&largeReadChar.id.uuid,       largeReadCharUuidRawByte,       sizeof(largeReadCharUuidRawByte));
    getGattIdFromBytes(&largeWriteReqChar.id.uuid,   largeWriteReqCharUuidRawByte,   sizeof(largeWriteReqCharUuidRawByte));
    getGattIdFromBytes(&largeWriteCmdChar.id.uuid,   largeWriteCmdCharUuidRawByte,   sizeof(largeWriteCmdCharUuidRawByte));
    getGattIdFromBytes(&largeNotifChar.id.uuid,      largeNotifCharUuidRawByte,      sizeof(largeNotifCharUuidRawByte));

    client.startBluetooth();
    client.startBluetoothLowEnergy();

    // Use nn::bluetooth:: API to manipulate GATT attributes
    nn::bluetooth::RegisterLeHidDataPath(smallService.id.uuid);
    // Use nn::bluetooth::user API to manipulate GATT attributes
    nn::bluetooth::RegisterLeDataPath(largeService.id.uuid);

    int loopTimeMs = 0;
    int flip = 1;

    bool isSmallNoficaitionEnabled = false;
    bool isLargeNoficaitionEnabled = false;

    for(;;)
    {
        if (!client.isRegistered)
        {
            client.startBluetoothLowEnergyClient(clientUuid);

            client.registerTestAttribute(&smallService);
            client.registerTestAttribute(&smallReadChar);
            client.registerTestAttribute(&smallWriteReqChar);
            client.registerTestAttribute(&smallWriteCmdChar);
            client.registerTestAttribute(&smallNotifChar);

            client.registerTestAttribute(&largeService);
            client.registerTestAttribute(&largeReadChar);
            client.registerTestAttribute(&largeWriteReqChar);
            client.registerTestAttribute(&largeWriteCmdChar);
            client.registerTestAttribute(&largeNotifChar);
        }
        else
        {
            if (loopTimeMs == 5000)
            {
                if (!client.isScanFilterEnabled)
                {
                    NN_LOG("  [Configure LE Scan Filter]\n");

                    nn::bluetooth::AddLeScanFilterCondition(&client.scanFilter);
                }
                else if (!client.serverConnected && client.isScanFilterEnabled && !client.isScanning)
                {
                    NN_LOG("  [Start LE Scan]\n");
                    nn::bluetooth::StartLeScan();
                    client.isScanning = true;
                }
            }

            if (loopTimeMs % 1500 == 0)
            {
                if (client.serverConnected)
                {
                    if (client.discoverServiceComplete)
                    {
                        if (!client.mtuConfigured)
                        {
                            NN_LOG("  [Configure MTU] (%d Bytes)\n", mtu);
                            nn::bluetooth::LeClientConfigureMtu(client.connectionId, mtu);
                        }
                        else if (!isSmallNoficaitionEnabled && client.isAttributeSupported(&smallNotifChar))
                        {
                            NN_LOG("  [Enabled Small Notification]\n");
                            uint8_t enableNotif[2] = { 0x01, 0x00 };

                            nn::bluetooth::LeClientRegisterNotification(client.clientIf, smallService.id, bluetoothClient::PRIMARY, smallNotifChar.id);

                            nn::bluetooth::LeClientWriteDescriptor(client.connectionId, smallService.id, bluetoothClient::PRIMARY, smallNotifChar.id, cccDescriptorUuid, enableNotif, sizeof(enableNotif), bluetoothClient::AUTH_REQ_NONE);
                            isSmallNoficaitionEnabled = true;
                        }
                        else if (!isLargeNoficaitionEnabled && client.isAttributeSupported(&largeNotifChar))
                        {
                            NN_LOG("  [Enabled Large Notification]\n");
                            uint8_t enableNotif[2] = { 0x01, 0x00 };

                            nn::bluetooth::user::LeClientRegisterNotification(client.clientIf, largeService.id, bluetoothClient::PRIMARY, largeNotifChar.id);

                            nn::bluetooth::user::LeClientWriteDescriptor(client.connectionId, largeService.id, bluetoothClient::PRIMARY, largeNotifChar.id, cccDescriptorUuid, enableNotif, sizeof(enableNotif), bluetoothClient::AUTH_REQ_NONE);
                            isLargeNoficaitionEnabled = true;
                        }
                        else if (flip % 12 == 0)
                        {
                            if (client.isAttributeSupported(&largeWriteCmdChar) && client.mtu > client.LARGE_WRITE_DATA_SIZE)
                            {
                                NN_LOG("  [Write Large Characteristic without Response] (%d)\n", flip);
                                nn::bluetooth::user::LeClientWriteCharacteristic(client.connectionId, largeService.id, bluetoothClient::PRIMARY, largeWriteCmdChar.id, client.largeWriteData, client.LARGE_WRITE_DATA_SIZE, bluetoothClient::AUTH_REQ_NONE, false);
                            }
                        }
                        else if (flip % 10 == 0)
                        {
                            if (client.isAttributeSupported(&largeWriteReqChar) && client.mtu > client.LARGE_WRITE_DATA_SIZE)
                            {
                                NN_LOG("  [Write Large Characteristic with Response] (%d)\n", flip);
                                nn::bluetooth::user::LeClientWriteCharacteristic(client.connectionId, largeService.id, bluetoothClient::PRIMARY, largeWriteReqChar.id, client.largeWriteData, client.LARGE_WRITE_DATA_SIZE, bluetoothClient::AUTH_REQ_NONE, true);
                            }
                        }
                        else if (flip % 8 == 0)
                        {
                            if (client.isAttributeSupported(&largeReadChar))
                            {
                                NN_LOG("  [Read Large Characteristic] (%d)\n", flip);
                                nn::bluetooth::LeClientReadCharacteristic(client.connectionId, largeService.id, bluetoothClient::PRIMARY, largeReadChar.id, bluetoothClient::AUTH_REQ_NONE);
                            }
                        }
                        else if (flip % 6 == 0)
                        {
                            if (client.isAttributeSupported(&smallWriteCmdChar))
                            {
                                NN_LOG("  [Write Small Characteristic without Response] (%d)\n", flip);
                                nn::bluetooth::LeClientWriteCharacteristic(client.connectionId, smallService.id, bluetoothClient::PRIMARY, smallWriteCmdChar.id, client.smallWriteData, client.SMALL_WRITE_DATA_SIZE, bluetoothClient::AUTH_REQ_NONE, false);
                            }
                        }
                        else if (flip % 4 == 0)
                        {
                            if (client.isAttributeSupported(&smallWriteReqChar))
                            {
                                NN_LOG("  [Write Small Characteristic with Response] (%d)\n", flip);
                                nn::bluetooth::LeClientWriteCharacteristic(client.connectionId, smallService.id, bluetoothClient::PRIMARY, smallWriteReqChar.id, client.smallWriteData, client.SMALL_WRITE_DATA_SIZE, bluetoothClient::AUTH_REQ_NONE, true);
                            }
                        }
                        else if (flip % 2 == 0)
                        {
                            if (client.isAttributeSupported(&smallReadChar))
                            {
                                NN_LOG("  [Read Small Characteristic] (%d)\n", flip);
                                nn::bluetooth::LeClientReadCharacteristic(client.connectionId, smallService.id, bluetoothClient::PRIMARY, smallReadChar.id, bluetoothClient::AUTH_REQ_NONE);
                            }
                        }

                        flip++;
                    }
                }
            }
        }

        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(100));
        loopTimeMs += 100;
        if(loopTimeMs == 30000)
        {
           loopTimeMs=0;
        }
    }

    nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(10000));

    client.finishBluetoothLowEnergy();
    client.finishBluetooth();
} // NOLINT(impl/function_size)

