﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/gfx.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>

#include "BleSimple_ApplicationHeap.h"
#include "BleSimple_FontSystem.h"
#include "BleSimple_GraphicsSystem.h"

namespace {

const uint32_t CharCountMax = 2048;

} // namespace

FontSystem::FontSystem() NN_NOEXCEPT
    : m_pApplicationHeap(0)
    , m_pGraphicsSystem(0)
    , m_pDebugFontMemory(0)
    , m_pDebugFontMemoryPool(0)
{
}

void FontSystem::Initialize(
    ApplicationHeap* pApplicationHeap,
    GraphicsSystem* pGraphicsSystem) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pApplicationHeap);
    NN_ASSERT_NOT_NULL(pGraphicsSystem);

    m_pApplicationHeap = pApplicationHeap;

    m_pGraphicsSystem = pGraphicsSystem;

    nn::gfx::util::DebugFontTextWriterInfo info;
    info.SetDefault();
    info.SetCharCountMax(CharCountMax);
    info.SetUserMemoryPoolEnabled(true);

    nn::gfx::Device* pDevice = &m_pGraphicsSystem->GetDevice();

    const size_t memorySize =
        nn::gfx::util::DebugFontTextWriter::GetRequiredMemorySize(
            pDevice, info);

    m_pDebugFontMemory = m_pApplicationHeap->Allocate(memorySize);

    const size_t memoryPoolSize =
        nn::gfx::util::DebugFontTextWriter::GetRequiredMemoryPoolSize(
            pDevice, info);

    nn::gfx::MemoryPoolInfo memoryPoolInfo;
    memoryPoolInfo.SetDefault();
    memoryPoolInfo.SetMemoryPoolProperty(
        nn::gfx::MemoryPoolProperty_CpuUncached |
        nn::gfx::MemoryPoolProperty_GpuCached);

    m_pDebugFontMemoryPool = m_pApplicationHeap->Allocate(
        memoryPoolSize,
        nn::gfx::MemoryPool::GetPoolMemoryAlignment(pDevice,
                                                    memoryPoolInfo));

    memoryPoolInfo.SetPoolMemory(m_pDebugFontMemoryPool, memoryPoolSize);

    m_DebugFontMemoryPool.Initialize(pDevice, memoryPoolInfo);

    m_DebugFontTextWriter.Initialize(pDevice,
                                     info,
                                     m_pDebugFontMemory,
                                     memorySize,
                                     &m_DebugFontMemoryPool,
                                     0,
                                     memoryPoolSize);

    m_DebugFontTextWriter.SetDisplayWidth(
        m_pGraphicsSystem->GetWidth());

    m_DebugFontTextWriter.SetDisplayHeight(
        m_pGraphicsSystem->GetHeight());

    m_DebugFontTextWriter.SetTextureDescriptor(
        &m_pGraphicsSystem->GetTextureDescriptorPool(),
        m_pGraphicsSystem->GetTextureDescriptorBaseIndex());

    m_DebugFontTextWriter.SetSamplerDescriptor(
        &m_pGraphicsSystem->GetSamplerDescriptorPool(),
        m_pGraphicsSystem->GetSamplerDescriptorBaseIndex());
}

void FontSystem::Finalize() NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(m_pApplicationHeap);
    NN_ASSERT_NOT_NULL(m_pGraphicsSystem);

    m_DebugFontTextWriter.Finalize();

    m_DebugFontMemoryPool.Finalize(&m_pGraphicsSystem->GetDevice());

    m_pApplicationHeap->Deallocate(m_pDebugFontMemoryPool);

    m_pApplicationHeap->Deallocate(m_pDebugFontMemory);
}

nn::gfx::util::DebugFontTextWriter& FontSystem::GetDebugFontTextWriter(
    ) NN_NOEXCEPT
{
    return m_DebugFontTextWriter;
}

void FontSystem::Draw() NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(m_pGraphicsSystem);

    m_DebugFontTextWriter.Draw(&m_pGraphicsSystem->GetCommandBuffer());
}
