﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{AudioRenderer.cpp,PageSampleAudioAudioRenderer}
 *
 * @brief
 * オーディオレンダラのサンプルプログラム
 */

/**
 * @page PageSampleAudioAudioRenderer オーディオレンダラ
 * @tableofcontents
 *
 * @brief
 * オーディオレンダラのサンプルプログラムの解説です。
 *
 * @section PageSampleAudioAudioRenderer_SectionBrief 概要
 * オーディオレンダラを利用して、オーディオ再生を行うサンプルです。
 *
 * @section PageSampleAudioAudioRenderer_SectionFileStructure ファイル構成
 * 本サンプルプログラムは @link ../../../Samples/Sources/Applications/AudioRenderer
 * Samples/Sources/Applications/AudioRenderer @endlink 以下にあります。
 *
 * @section PageSampleAudioAudioRenderer_SectionNecessaryEnvironment 必要な環境
 * オーディオ出力が利用可能である必要があります。
 *
 * @section PageSampleAudioAudioRenderer_SectionHowToOperate 操作方法
 * サンプルプログラムを実行すると、自動的に BGM や SE が再生されます。
 * 再生中はキーボードや DebugPad による入力を用いて、再生される BGM や SE、サイン波を操作できます。
 *
 * <p>
 * <table>
 * <tr><th> 入力 </th><th> 動作 </th></tr>
 * <tr><td> [A]             </td><td> SampleSe0 を一度再生                           </td></tr>
 * <tr><td> [B]             </td><td> SampleSe1 を一度再生                           </td></tr>
 * <tr><td> [X]             </td><td> SampleSe2 を一度再生                           </td></tr>
 * <tr><td> [Y]             </td><td> SampleSe3 を一度再生                           </td></tr>
 * <tr><td> [L]             </td><td> Biquad filter の有効・無効を切り替え           </td></tr>
 * <tr><td> [R]             </td><td> SampleBgm0-2ch の再生状態を切り替え            </td></tr>
 * <tr><td> [AnalogStickL]  </td><td> サイン波の音量を大小操作                       </td></tr>
 * <tr><td> [Left][Right]   </td><td> SampleBgm0-2ch の左チャンネルの音量を大小操作  </td></tr>
 * <tr><td> [Up][Down]      </td><td> SampleBgm0-2ch の右チャンネルの音量を大小操作  </td></tr>
 * <tr><td> [AnalogStickR]  </td><td> サイン波のピッチを上下操作                     </td></tr>
 * <tr><td> [Start/Space]   </td><td> プログラムの終了                               </td></tr>
 * </table>
 * </p>
 *
 * @section PageSampleAudioAudioRenderer_SectionPrecaution 注意事項
 * 特にありません。
 *
 * @section PageSampleAudioAudioRenderer_SectionHowToExecute 実行手順
 * サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleAudioAudioRenderer_SectionDetail 解説
 * このサンプルプログラムは、複数の波形を用いて、オーディオレンダリングを行うものです。
 *
 * サンプルプログラムの処理の流れは以下の通りです。
 *
 * - ファイルシステムの初期化
 * - オーディオレンダラの初期化
 * - FinalMix の準備、および BufferMixer を利用したバスルーティング
 * - 出力先デバイスの設定
 * - 入力波形の準備、Voice の確保およびパラメータ設定
 * - オーディオレンダリングの開始
 * - パラメータ変更、状態取得の実施
 * - レンダリングを終了
 * - オーディオレンダラをクローズ
 * - 入力波形を破棄
 *
 * 最初にファイルシステムの初期化を行います。
 * リソースファイルはビルド時にコンバートされるものを利用します。
 * リソースディレクトリの場所は適宜変更して利用してください。
 *
 * オーディオレンダラを初期化します。
 * 初期化に先立ち、事前にパラメータの設定やワークバッファの準備を行う必要があります。
 * このワークバッファはオーディオレンダリング処理のために利用されます。
 * オーディオレンダラを利用するためには、同時に nn::audio::AudioRendererConfig を用意する必要があります。
 * この構造体はオーディオレンダラの各種パラメータを管理するために利用します。
 * nn::audio::AudioRendererConfig のためにもワークバッファを用意する必要があります。
 *
 * オーディオレンダラは最終出力の出力先を Sink で指定します。
 * このサンプルでは "MainAudioOut" という名前の DeviceSinkType を指定します
 * （現在のバージョンの SDK においては、この名前を指定することで、Windows の「規定のデバイス」から音が出力されます）。
 *
 * 次にエフェクトを用意します。このデモでは nn::audio::BufferMixerType を利用して、バスルーティングの設定を行っています。
 * エフェクトは nn::audio::FinalMixType に対して指定します。
 *
 * nn::audio::VoiceType を用いて、入力波形やパラメータの指定ができます。
 * このサンプルでは、動的に生成したサイン波、およびファイルから読み込んだ BGM, SE を再生します。
 * 入力波形は nn::audio::WaveBuffer により指定します。Voice に対して、一度に 4 つまでのバッファを指定して、
 * 連続的に再生を行うことが可能です。その他、以下のようなパラメータを指定することが可能です。
 *
 * - ボリューム
 * - ピッチ
 * - Biquad フィルタ
 * - ミックスボリューム
 *
 * nn::audio::VoiceType は、確保した後に出力先を指定する必要があります
 *
 * 各種設定は nn::audio::AudioRendererConfig に設定した後に nn::audio::RequestUpdateAudioRenderer() を
 * 呼ぶことで、実際の処理に反映されます。
 *
 * メインループでは、パラメータの変更、状態の取得、およびバッファの追加を行います。
 * このサンプルではサイン波データをストリーム再生しつつ、 BGM をループ再生します。
 * 再生中はキーボードや DebugPad 等の入力によって「操作方法」で記述した操作が可能です。
 *
 * プログラムの終了が要求されると、上記の繰り返し処理を抜けて、再生を停止し、
 * オーディオレンダラをクローズし、メモリの破棄を行います。
 */

#include <cstdlib>
#include <cmath>
#include <new>

#include <nns/nns_Log.h>
#include <nn/nn_Abort.h>
#include <nn/os.h>
#include <nn/mem.h>
#include <nn/fs.h>
#include <nn/nn_TimeSpan.h>

#include <nn/audio.h>
#include <nns/audio/audio_HidUtilities.h>
#include <nns/audio/audio_WavFormat.h>

#include <nn/hid.h>
#include <nn/hid/hid_KeyboardKey.h>
#include <nn/hid/hid_Npad.h>

#include <nn/settings/settings_DebugPad.h>

namespace {

// choose rendering engine sample rate
//const int RenderRate = 48000;
const int RenderRate = 32000;
const int RenderCount = (RenderRate / 200);

// choose number of files to play
const int BgmCount = 1;
const int SeCount = 4;

const char Title[] = "AudioRenderer";

// - and add / remove them to / from the files lists
const char* g_BgmFileNames[BgmCount] =
{
    "asset:/AudioRenderer/SampleBgm0-2ch.wav",
};

const char* g_SeFileNames[SeCount] =
{
    "asset:/AudioCommon/SampleSe0.adpcm",
    "asset:/AudioCommon/SampleSe1.adpcm",
    "asset:/AudioCommon/SampleSe2.adpcm",
    "asset:/AudioCommon/SampleSe3.adpcm",
};

NN_ALIGNAS(4096) char g_WorkBuffer[8 * 1024 * 1024];
NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char g_WaveBufferPoolMemory[14 * 1024 * 1024];

nn::mem::StandardAllocator g_Allocator;
nn::mem::StandardAllocator g_WaveBufferAllocator;

char* g_MountRomCacheBuffer = NULL;

}

void* Allocate(size_t size)
{
    return g_Allocator.Allocate(size);
}

void Deallocate(void* p, size_t size)
{
    NN_UNUSED(size);
    g_Allocator.Free(p);
}

std::size_t GenerateSineWave(void** data, int sampleRate, int frequency, int sampleCount)
{
    // g_WaveBufferAllocator が管理するメモリ領域はすべて waveBufferMemoryPool メモリプールに追加されています。
    int16_t* p = static_cast<int16_t*>(g_WaveBufferAllocator.Allocate(sampleCount * sizeof(int16_t), nn::audio::BufferAlignSize));
    NN_ABORT_UNLESS_NOT_NULL(p);
    const float Pi = 3.1415926535897932384626433f;
    for (auto i = 0; i < sampleCount; ++i)
    {
        p[i] = static_cast<int16_t>(std::numeric_limits<int16_t>::max() * sinf(2 * Pi * frequency * i / sampleRate));
    }
    *data = p;
    return sampleCount * sizeof(int16_t);
}

std::size_t ReadAdpcmFile(nn::audio::AdpcmHeaderInfo* header, void** adpcmData, const char* filename)
{
    nn::fs::FileHandle handle;
    nn::Result result = nn::fs::OpenFile(&handle, filename, nn::fs::OpenMode_Read);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    int64_t size;
    uint8_t adpcmheader[nn::audio::AdpcmHeaderSize];

    result = nn::fs::GetFileSize(&size, handle);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    *adpcmData = g_WaveBufferAllocator.Allocate(static_cast<std::size_t>(size) - sizeof(adpcmheader), nn::audio::BufferAlignSize);
    NN_ABORT_UNLESS_NOT_NULL(*adpcmData);

    result = nn::fs::ReadFile(handle, 0, adpcmheader, sizeof(adpcmheader));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    result = nn::fs::ReadFile(handle, sizeof(adpcmheader), *adpcmData, static_cast<size_t>(size) - sizeof(adpcmheader));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    nn::fs::CloseFile(handle);

    nn::audio::ParseAdpcmHeader(header, adpcmheader, sizeof(adpcmheader));

    return static_cast<std::size_t>(size) - sizeof(adpcmheader);
}

std::size_t ReadWavFile(nns::audio::WavFormat* format, void** data, const char* filename)
{
    nn::fs::FileHandle handle;
    nn::Result result = nn::fs::OpenFile(&handle, filename, nn::fs::OpenMode_Read);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    int64_t size;

    result = nn::fs::GetFileSize(&size, handle);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    *data = g_WaveBufferAllocator.Allocate(static_cast<std::size_t>(size), nn::audio::BufferAlignSize);
    NN_ABORT_UNLESS_NOT_NULL(*data);

    // DATA チャンクを読む必要がありますが、ここではそれが 1024 バイト以内に見つかると仮定しています
    const std::size_t WavHeaderDataSize = 1024;

    result = nn::fs::ReadFile(handle, 0, *data, WavHeaderDataSize);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    nns::audio::WavResult wavResult = nns::audio::ParseWavFormat(format, *data, WavHeaderDataSize);
    NN_ABORT_UNLESS_EQUAL(wavResult, nns::audio::WavResult_Success);
    NN_ABORT_UNLESS_EQUAL(format->bitsPerSample, 16);  // このサンプルでは 16bit PCM を仮定しています

    result = nn::fs::ReadFile(handle, static_cast<std::size_t>(format->dataOffset), *data, static_cast<std::size_t>(format->dataSize));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    nn::fs::CloseFile(handle);

    return static_cast<std::size_t>(format->dataSize);
}

void InitializeFileSystem()
{
    nn::fs::SetAllocator(Allocate, Deallocate);

    size_t cacheSize = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&cacheSize));
    g_MountRomCacheBuffer = new(std::nothrow) char[cacheSize];
    NN_ABORT_UNLESS_NOT_NULL(g_MountRomCacheBuffer);

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::fs::MountRom("asset", g_MountRomCacheBuffer, cacheSize)
    );
}

void FinalizeFileSystem()
{
    nn::fs::Unmount("asset");

    delete[] g_MountRomCacheBuffer;
    g_MountRomCacheBuffer = NULL;
}

void InitializeHidDevices()
{
    nn::hid::InitializeDebugPad();
    nn::hid::InitializeNpad();
    const nn::hid::NpadIdType npadIds[2] = { nn::hid::NpadId::No1, nn::hid::NpadId::Handheld };
    nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleFullKey::Mask | nn::hid::NpadStyleHandheld::Mask);
    nn::hid::SetSupportedNpadIdType(npadIds, sizeof(npadIds) / sizeof(npadIds[0]));

    //キーボードのキーを DebugPad のボタンに割り当てます。
    nn::settings::DebugPadKeyboardMap map;
    nn::settings::GetDebugPadKeyboardMap(&map);
    map.buttonA     = nn::hid::KeyboardKey::A::Index;
    map.buttonB     = nn::hid::KeyboardKey::B::Index;
    map.buttonX     = nn::hid::KeyboardKey::X::Index;
    map.buttonY     = nn::hid::KeyboardKey::Y::Index;
    map.buttonL     = nn::hid::KeyboardKey::L::Index;
    map.buttonR     = nn::hid::KeyboardKey::R::Index;
    map.buttonZL    = nn::hid::KeyboardKey::U::Index;
    map.buttonZR    = nn::hid::KeyboardKey::V::Index;
    map.buttonLeft  = nn::hid::KeyboardKey::LeftArrow::Index;
    map.buttonRight = nn::hid::KeyboardKey::RightArrow::Index;
    map.buttonUp    = nn::hid::KeyboardKey::UpArrow::Index;
    map.buttonDown  = nn::hid::KeyboardKey::DownArrow::Index;
    map.buttonStart = nn::hid::KeyboardKey::Space::Index;
    nn::settings::SetDebugPadKeyboardMap(map);
}

void PrintUsage()
{
    NNS_LOG("--------------------------------------------------------\n");
    NNS_LOG("%s Sample\n", Title);
    NNS_LOG("--------------------------------------------------------\n");
    NNS_LOG("[A]             StartSound                   (SampleSe0)\n");
    NNS_LOG("[B]             StartSound                   (SampleSe1)\n");
    NNS_LOG("[X]             StartSound                   (SampleSe2)\n");
    NNS_LOG("[Y]             StartSound                   (SampleSe3)\n");
    NNS_LOG("[L]             Toggle Biquad filter enabled (SampleBgm0-2ch)\n");
    NNS_LOG("[R]             Toggle BGM Play state        (SampleBgm0-2ch)\n");
    NNS_LOG("[AnalogStickL]  ControlVolume                (SineWave)\n");
    NNS_LOG("[Left][Right]   ControlVolume                (SineWave)\n");
    NNS_LOG("[AnalogStickR]  ControlPitch                 (SineWave)\n");
    NNS_LOG("[Up][Down]      ControlPitch                 (SineWave)\n");
    NNS_LOG("[ZLeft/U]       PanVolume (Left)             (SampleBgm0-2ch)\n");
    NNS_LOG("[ZRight/V]      PanVolume (Right)            (SampleBgm0-2ch)\n");
    NNS_LOG("[Start/Space]   Shut down sample program\n");
    NNS_LOG("-------------------------------------------------------\n");
}

extern "C" void nnMain()
{
    g_Allocator.Initialize(g_WorkBuffer, sizeof(g_WorkBuffer));
    g_WaveBufferAllocator.Initialize(g_WaveBufferPoolMemory, sizeof(g_WaveBufferPoolMemory));

    InitializeFileSystem();
    InitializeHidDevices();

    // レンダラのパラメータを指定します。
    nn::audio::AudioRendererParameter parameter;
    nn::audio::InitializeAudioRendererParameter(&parameter);
    parameter.sampleRate = RenderRate;
    parameter.sampleCount = RenderCount;
    parameter.mixBufferCount = 6 + 2; // FinalMix(6) + SubMix(2)
    parameter.voiceCount = 24;
    parameter.subMixCount = 2;
    parameter.sinkCount = 1;
    parameter.effectCount = 2;
    parameter.performanceFrameCount = 0;

    // ミックスバッファとオーディオバスの関係を定義します。
    int channelCount = 2;
    int8_t mainBus[2];
    mainBus[nn::audio::ChannelMapping_FrontLeft] = 4;
    mainBus[nn::audio::ChannelMapping_FrontRight] = 5;
    int8_t auxBusA[2];
    auxBusA[nn::audio::ChannelMapping_FrontLeft] = 0;
    auxBusA[nn::audio::ChannelMapping_FrontRight] = 1;

    // パラメータがシステムでサポートされているかどうかを確認します。
    NN_ABORT_UNLESS(
        nn::audio::IsValidAudioRendererParameter(parameter),
        "Invalid AudioRendererParameter specified."
    );

    // レンダラのワークバッファを準備します。
    size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(parameter);
    void* workBuffer = g_Allocator.Allocate(workBufferSize, nn::os::MemoryPageSize);
    NN_ABORT_UNLESS_NOT_NULL(workBuffer);

    nn::os::SystemEvent systemEvent;

    // レンダラを初期化します。
    nn::audio::AudioRendererHandle handle;
    NN_ABORT_UNLESS(
        nn::audio::OpenAudioRenderer(&handle, &systemEvent, parameter, workBuffer, workBufferSize).IsSuccess(),
        "Failed to open AudioRenderer"
    );

    // AudioRendererConfig を初期化します。
    size_t configBufferSize = nn::audio::GetAudioRendererConfigWorkBufferSize(parameter);
    void* configBuffer = g_Allocator.Allocate(configBufferSize, nn::os::MemoryPageSize);
    NN_ABORT_UNLESS_NOT_NULL(configBuffer);
    nn::audio::AudioRendererConfig config;
    nn::audio::InitializeAudioRendererConfig(&config, parameter, configBuffer, configBufferSize);

    nn::audio::FinalMixType finalMix;
    nn::audio::AcquireFinalMix(&config, &finalMix, 6);

    nn::audio::SubMixType subMix0;
    nn::audio::AcquireSubMix(&config, &subMix0, parameter.sampleRate, 1);
    nn::audio::SubMixType subMix1;
    nn::audio::AcquireSubMix(&config, &subMix1, parameter.sampleRate, 1);

    // レンダラの出力先を用意します。
    nn::audio::DeviceSinkType deviceSink;
    // オーディオ出力デバイスへの入力を設定します。
    // mainBus に指定したミックスバッファのインデックスに応じて出力チャンネルが決定されます。
    // mainBus[nn::audio::ChannelMapping_FrontLeft] が L チャンネルに、
    // mainBus[nn::audio::ChannelMapping_FrontRight] が R チャンネルにそれぞれ出力されます。
    nn::Result result = nn::audio::AddDeviceSink(&config, &deviceSink, &finalMix, mainBus, channelCount, "MainAudioOut");
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // SubMix(0) を FinalMix へ接続します
    nn::audio::SetSubMixDestination(&config, &subMix0, &finalMix);

    // SubMix(0) の 0 番目のバッファを FinalMix の mainBus[0]、mainBus[1] へ
    ///ボリューム 0.5f でミックスするように設定します
    nn::audio::SetSubMixMixVolume(&subMix0, &finalMix, 0.5f, 0, mainBus[0]);
    nn::audio::SetSubMixMixVolume(&subMix0, &finalMix, 0.5f, 0, mainBus[1]);

    // SubMix(1) を SubMix(0) へ接続します
    nn::audio::SetSubMixDestination(&config, &subMix1, &subMix0);

    // SubMix(1) の 0 番目のバッファを SubMix(0) の 0 番目のバッファへ
    // ボリューム 0.5f でミックスするように設定します
    nn::audio::SetSubMixMixVolume(&subMix1, &subMix0, 0.5f, 0, 0);

    // BufferMixer を用意し、登録します。
    nn::audio::BufferMixerType mixer1;
    result = nn::audio::AddBufferMixer(&config, &mixer1, &finalMix);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    nn::audio::SetBufferMixerInputOutput(&mixer1, auxBusA, mainBus, channelCount);

    // BufferMixer パラメータを設定します。
    nn::audio::SetBufferMixerVolume(&mixer1, 0, 1.0f);
    nn::audio::SetBufferMixerVolume(&mixer1, 1, 1.0f);

    // 設定したパラメータをレンダラに反映させます。
    result = nn::audio::RequestUpdateAudioRenderer(handle, &config);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // レンダリングを開始します。
    result = nn::audio::StartAudioRenderer(handle);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // Voice を用意します。
    nn::audio::VoiceType voiceSine;
    nn::audio::WaveBuffer waveBufferSine;
    void* dataSine;

    // WaveBuffer に追加するサンプルデータを保持するためのメモリプールを準備します。
    nn::audio::MemoryPoolType waveBufferMemoryPool;
    auto ret = nn::audio::AcquireMemoryPool(&config, &waveBufferMemoryPool, g_WaveBufferPoolMemory, sizeof(g_WaveBufferPoolMemory));
    NN_ABORT_UNLESS(ret);
    ret = nn::audio::RequestAttachMemoryPool(&waveBufferMemoryPool);
    NN_ABORT_UNLESS(ret);

    // Voice で再生する波形を準備します。
    {
        const int sineSampleRate = 32000;
        const int sineFrequency = 440;
        const int sineSampleCount = 32000;
        std::size_t size = GenerateSineWave(&dataSine, sineSampleRate, sineFrequency, sineSampleCount);

        nn::audio::AcquireVoiceSlot(&config, &voiceSine, sineSampleRate, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0);
        nn::audio::SetVoiceDestination(&config, &voiceSine, &subMix1);

        waveBufferSine.buffer = dataSine;
        waveBufferSine.size = size;
        waveBufferSine.startSampleOffset = 0;
        waveBufferSine.endSampleOffset = sineSampleCount;
        waveBufferSine.loop = 0;
        waveBufferSine.isEndOfStream = false;

        // 波形を Voice に追加します。
        nn::audio::AppendWaveBuffer(&voiceSine, &waveBufferSine);
        nn::audio::AppendWaveBuffer(&voiceSine, &waveBufferSine);
        nn::audio::AppendWaveBuffer(&voiceSine, &waveBufferSine);
        nn::audio::AppendWaveBuffer(&voiceSine, &waveBufferSine);
        nn::audio::SetVoicePlayState(&voiceSine, nn::audio::VoiceType::PlayState_Play);
        nn::audio::SetVoiceMixVolume(&voiceSine, &subMix1, 0.707f / 2, 0, 0);
    }

    // BGM
    nn::audio::VoiceType voiceBgm[BgmCount];
    nn::audio::WaveBuffer waveBufferBgm;
    void* dataBgm[BgmCount];

    for (int i = 0; i < BgmCount; ++i)
    {
        nns::audio::WavFormat format;
        std::size_t dataSize = ReadWavFile(&format, &dataBgm[i], g_BgmFileNames[i]);

        // 読み込んだ WAV 形式ファイルのチャンネル数を指定してボイスを取得します。
        // マルチチャンネルのデータを読み込んだ場合には、チャンネル数分のボイスを消費することに注意してください。
        nn::audio::AcquireVoiceSlot(&config, &voiceBgm[i], format.sampleRate, format.channelCount, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0);
        nn::audio::SetVoiceDestination(&config, &voiceBgm[i], &finalMix);

        waveBufferBgm.buffer = dataBgm[i];
        waveBufferBgm.size = dataSize;
        waveBufferBgm.startSampleOffset = 0;
        waveBufferBgm.endSampleOffset = static_cast<int32_t>(dataSize / sizeof(int16_t)) / format.channelCount;
        waveBufferBgm.loop = true;
        waveBufferBgm.isEndOfStream = false;
        waveBufferBgm.pContext = nullptr;
        waveBufferBgm.contextSize = 0;

        nn::audio::AppendWaveBuffer(&voiceBgm[i], &waveBufferBgm);
        nn::audio::SetVoicePlayState(&voiceBgm[i], nn::audio::VoiceType::PlayState_Play);
        // ボイスの 0 チャンネルを mainBus[0] へ、1 チャンネルを mainBus[1] へ出力するミックスボリュームを設定します。
        nn::audio::SetVoiceMixVolume(&voiceBgm[i], &finalMix, 0.5f, 0, mainBus[0]);
        nn::audio::SetVoiceMixVolume(&voiceBgm[i], &finalMix, 0.5f, 1, mainBus[1]);

        // カットオフ 2048Hz のローパスフィルタを設定します
        nn::audio::BiquadFilterParameter firstFilter = { true, {720, 1439, 720}, {22684, -9350} };
        nn::audio::SetVoiceBiquadFilterParameter(&voiceBgm[i], 0, firstFilter);
        // カットオフ 1024 Hz のハイパスフィルタを無効状態で設定します。
        nn::audio::BiquadFilterParameter secondFilter = { false, {14041, -28083, 14041}, {29547, -13563} };
        nn::audio::SetVoiceBiquadFilterParameter(&voiceBgm[i], 1, secondFilter);
    }

    // SE
    nn::audio::VoiceType voiceSe[SeCount];
    nn::audio::WaveBuffer waveBufferSe[SeCount];
    nn::audio::AdpcmHeaderInfo* header[SeCount];
    void* dataSe[SeCount];

    for (int i = 0; i < SeCount; ++i)
    {
        header[i] = reinterpret_cast<nn::audio::AdpcmHeaderInfo*>(g_WaveBufferAllocator.Allocate(sizeof(nn::audio::AdpcmHeaderInfo), NN_ALIGNOF(nn::audio::AdpcmHeaderInfo)));
        std::size_t dataSeSize = ReadAdpcmFile(header[i], &dataSe[i], g_SeFileNames[i]);
        nn::audio::AcquireVoiceSlot(&config, &voiceSe[i], header[i]->sampleRate, 1, nn::audio::SampleFormat_Adpcm, nn::audio::VoiceType::PriorityHighest, &header[i]->parameter, sizeof(nn::audio::AdpcmParameter));
        nn::audio::SetVoiceDestination(&config, &voiceSe[i], &finalMix);

        waveBufferSe[i].buffer = dataSe[i];
        waveBufferSe[i].size = dataSeSize;
        waveBufferSe[i].startSampleOffset = 0;
        waveBufferSe[i].endSampleOffset = header[i]->sampleCount;
        waveBufferSe[i].loop = false;
        waveBufferSe[i].isEndOfStream = false;
        waveBufferSe[i].pContext = &header[i]->loopContext;
        waveBufferSe[i].contextSize = sizeof(nn::audio::AdpcmContext);

        nn::audio::AppendWaveBuffer(&voiceSe[i], &waveBufferSe[i]);
        nn::audio::SetVoicePlayState(&voiceSe[i], nn::audio::VoiceType::PlayState_Play);
        nn::audio::SetVoiceMixVolume(&voiceSe[i], &finalMix, 0.707f / 2, 0, auxBusA[0]);
        nn::audio::SetVoiceMixVolume(&voiceSe[i], &finalMix, 0.707f / 2, 0, auxBusA[1]);
    }

    PrintUsage();

    // 波形再生が終わるまで待ちつつ、パラメータを更新します。
    for (;;)
    {
        systemEvent.Wait();

        nn::hid::NpadButtonSet npadButtonCurrent = {};
        nn::hid::NpadButtonSet npadButtonDown = {};
        nn::hid::AnalogStickState analogStickStateL = {};
        nn::hid::AnalogStickState analogStickStateR = {};

        // Npad の入力を取得します。
        if (nn::hid::GetNpadStyleSet(nn::hid::NpadId::No1).Test<nn::hid::NpadStyleFullKey>())
        {
            static nn::hid::NpadFullKeyState npadFullKeyState = {};
            nn::hid::NpadFullKeyState state;
            nn::hid::GetNpadState(&state, nn::hid::NpadId::No1);
            npadButtonCurrent |= state.buttons;
            npadButtonDown |= state.buttons & ~npadFullKeyState.buttons;
            analogStickStateL.x += state.analogStickL.x;
            analogStickStateL.y += state.analogStickL.y;
            analogStickStateR.x += state.analogStickR.x;
            analogStickStateR.y += state.analogStickR.y;
            npadFullKeyState = state;
        }
        if (nn::hid::GetNpadStyleSet(nn::hid::NpadId::Handheld).Test<nn::hid::NpadStyleHandheld>())
        {
            static nn::hid::NpadHandheldState npadHandheldState = {};
            nn::hid::NpadHandheldState state;
            nn::hid::GetNpadState(&state, nn::hid::NpadId::Handheld);
            npadButtonCurrent |= state.buttons;
            npadButtonDown |= state.buttons & ~npadHandheldState.buttons;
            analogStickStateL.x += state.analogStickL.x;
            analogStickStateL.y += state.analogStickL.y;
            analogStickStateR.x += state.analogStickR.x;
            analogStickStateR.y += state.analogStickR.y;
            npadHandheldState = state;
        }

        // DebugPad の入力を取得します。
        {
            static nn::hid::DebugPadState debugPadState = {};
            nn::hid::DebugPadButtonSet debugPadButtonCurrent = {};
            nn::hid::DebugPadButtonSet debugPadButtonDown = {};
            nn::hid::DebugPadState state;
            nn::hid::GetDebugPadState(&state);
            debugPadButtonCurrent |= state.buttons;
            debugPadButtonDown |= state.buttons & ~debugPadState.buttons;
            analogStickStateL.x += state.analogStickL.x;
            analogStickStateL.y += state.analogStickL.y;
            analogStickStateR.x += state.analogStickR.x;
            analogStickStateR.y += state.analogStickR.y;
            debugPadState = state;
            nns::audio::ConvertDebugPadButtonsToNpadButtons(&npadButtonCurrent, debugPadButtonCurrent);
            nns::audio::ConvertDebugPadButtonsToNpadButtons(&npadButtonDown, debugPadButtonDown);
        }

        //SE を再生します（同じ SE を多重再生しません）。
        for (int i = 0; i < SeCount; ++i)
        {
            // SE番号 {0, 1, 2, 3, 4, 5} が、ボタン {A, B, X, Y, L, R} に対応します。
            if(npadButtonDown.Test(i))
            {
                if (nn::audio::GetReleasedWaveBuffer(&voiceSe[i]))
                {
                    nn::audio::AppendWaveBuffer(&voiceSe[i], &waveBufferSe[i]);
                }
            }
        }

        //サイン波の音量を操作します。
        float sineVolume = nn::audio::GetVoiceVolume(&voiceSine) + 0.01f * analogStickStateL.x / (::nn::hid::AnalogStickMax + 1);

        if(npadButtonCurrent.Test< ::nn::hid::NpadButton::Right >())
        {
            sineVolume += 0.01f;
        }
        if(npadButtonCurrent.Test< ::nn::hid::NpadButton::Left >())
        {
            sineVolume -= 0.01f;
        }

        // Voice のボリューム・ミックスボリュームは nn::audio::VoiceType::GetVolumeMin() 以上 nn::audio::VoiceType::GetVolumeMax() 以下の値で指定できます。
        // このサンプルではノイズにならぬよう (0.0f, 2.0f) もしくは (0.0f, 1.0f) の範囲に制限しています。
        if (sineVolume < 2.0f && sineVolume > 0.0f)
        {
            nn::audio::SetVoiceVolume(&voiceSine, sineVolume);
        }

        for (int i = 0; i < BgmCount; ++i)
        {
            if(npadButtonDown.Test< ::nn::hid::NpadButton::L >())
            {
                // 2 つ目の Biquad フィルタの設定の有効・無効を切り替えます。
                nn::audio::BiquadFilterParameter biquadFilterParameter(nn::audio::GetVoiceBiquadFilterParameter(&voiceBgm[i], 1));
                biquadFilterParameter.enable = !biquadFilterParameter.enable;
                nn::audio::SetVoiceBiquadFilterParameter(&voiceBgm[i], 1, biquadFilterParameter);
            }

            if(npadButtonDown.Test< ::nn::hid::NpadButton::R >())
            {
                // BGM を ON/OFF します。
                switch (nn::audio::GetVoicePlayState(&voiceBgm[i]))
                {
                case nn::audio::VoiceType::PlayState_Play:
                    nn::audio::SetVoicePlayState(&voiceBgm[i], nn::audio::VoiceType::PlayState_Pause);
                    break;
                case nn::audio::VoiceType::PlayState_Pause:
                case nn::audio::VoiceType::PlayState_Stop:
                    nn::audio::SetVoicePlayState(&voiceBgm[i], nn::audio::VoiceType::PlayState_Play);
                    break;
                default:
                    NN_ABORT("Unexpected play state\n");
                    break;
                }
            }

            // 左チャンネル
            float bgmLeftVolume = nn::audio::GetVoiceMixVolume(&voiceBgm[i], &finalMix, 0, mainBus[0]);
            if(npadButtonCurrent.Test< ::nn::hid::NpadButton::ZL >())
            {
                bgmLeftVolume += 0.01f;
            }

            if(npadButtonCurrent.Test< ::nn::hid::NpadButton::ZR >())
            {
                bgmLeftVolume -= 0.01f;
            }

            if (bgmLeftVolume < 1.0f && bgmLeftVolume > 0.0f)
            {
                nn::audio::SetVoiceMixVolume(&voiceBgm[i], &finalMix, bgmLeftVolume, 0, mainBus[0]);
            }

            // 右チャンネル
            float bgmRightVolume = 1.0f - bgmLeftVolume;

            if (bgmRightVolume < 1.0f && bgmRightVolume > 0.0f)
            {
                nn::audio::SetVoiceMixVolume(&voiceBgm[i], &finalMix, bgmRightVolume, 1, mainBus[1]);
            }
        }

        //サイン波のピッチを操作します。
        float sinePitch = nn::audio::GetVoicePitch(&voiceSine) + 0.01f * analogStickStateR.x / (::nn::hid::AnalogStickMax + 1);
        if(npadButtonCurrent.Test< ::nn::hid::NpadButton::Up >())
        {
            sinePitch += 0.01f;
        }
        if(npadButtonCurrent.Test< ::nn::hid::NpadButton::Down >())
        {
            sinePitch -= 0.01f;
        }

        if (sinePitch < nn::audio::VoiceType::GetPitchMax() && sinePitch > nn::audio::VoiceType::GetPitchMin())
        {
            nn::audio::SetVoicePitch(&voiceSine, sinePitch);
        }

        while (const nn::audio::WaveBuffer* pWaveBuffer = nn::audio::GetReleasedWaveBuffer(&voiceSine))
        {
            nn::audio::AppendWaveBuffer(&voiceSine, pWaveBuffer);
        }

        if(npadButtonDown.Test< ::nn::hid::NpadButton::Plus >())
        {
            break;
        }

        result = nn::audio::RequestUpdateAudioRenderer(handle, &config);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    }

    // レンダリングを終了します。
    nn::audio::StopAudioRenderer(handle);
    nn::audio::CloseAudioRenderer(handle);
    nn::os::DestroySystemEvent(systemEvent.GetBase());

    // メモリを解放します。
    if (dataSine)
    {
        g_WaveBufferAllocator.Free(dataSine);
        dataSine = nullptr;
    }
    for (int i = 0; i < BgmCount; ++i)
    {
        if (dataBgm[i])
        {
            g_WaveBufferAllocator.Free(dataBgm[i]);
            dataBgm[i] = nullptr;
        }
    }
    for (int i = 0; i < SeCount; ++i)
    {
        if (dataSe[i])
        {
            g_WaveBufferAllocator.Free(dataSe[i]);
            dataSe[i] = nullptr;
            g_WaveBufferAllocator.Free(header[i]);
            header[i] = nullptr;
        }
    }
    if (configBuffer)
    {
        g_Allocator.Free(configBuffer);
        configBuffer = nullptr;
    }
    if (workBuffer)
    {
        g_Allocator.Free(workBuffer);
        workBuffer = nullptr;
    }

    FinalizeFileSystem();
}  // NOLINT(readability/fn_size)
