﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{AudioPerformanceMetrics.cpp,PageSampleAudioAudioPerformanceMetrics}
 *
 * @brief
 * オーディオレンダラのパフォーマンス情報取得のサンプルプログラムです。
 */

/**
 * @page PageSampleAudioAudioPerformanceMetrics オーディオレンダラのパフォーマンス情報取得
 * @tableofcontents
 *
 * @brief
 * オーディオレンダラのパフォーマンス情報取得方法について解説します。
 *
 * @section PageSampleAudioAudioPerformanceMetrics_SectionBrief 概要
 * オーディオレンダラによるレンダリング処理のパフォーマンス情報を取得するサンプルです。
 *
 * @section PageSampleAudioAudioPerformanceMetrics_SectionFileStructure ファイル構成
 * 本サンプルプログラムは @link ../../../Samples/Sources/Applications/AudioPerformanceMetrics
 * Samples/Sources/Applications/AudioPerformanceMetrics @endlink 以下にあります。
 *
 * @section PageSampleAudioAudioPerformanceMetrics_SectionNecessaryEnvironment 必要な環境
 * オーディオ出力が利用可能である必要があります。
 *
 * @section PageSampleAudioAudioPerformanceMetrics_SectionHowToOperate 操作方法
 * 一定の時間 BGM や SE を再生し、自動的に終了します。
 * その間オーディオレンダラのレンダリング処理に掛かる負荷情報を定期的にログ出力します。
 *
 * <p>
 * <table>
 * <tr><th> 入力 </th><th> 動作 </th></tr>
 * <tr><td> [A]             </td><td> SampleSe0 を一度再生                 </td></tr>
 * <tr><td> [B]             </td><td> SampleSe1 を一度再生                 </td></tr>
 * <tr><td> [X]             </td><td> SampleSe2 を一度再生                 </td></tr>
 * <tr><td> [Y]             </td><td> SampleSe3 を一度再生                 </td></tr>
 * <tr><td> [L]             </td><td> Delay の有効・無効を切り替え         </td></tr>
 * <tr><td> [Start/Space]   </td><td> プログラムの終了                     </td></tr>
 * </table>
 * </p>
 *
 * @section PageSampleAudioAudioPerformanceMetrics_SectionPrecaution 注意事項
 * 本サンプルで解説する機能を用いて得られた負荷情報は、現在のリリースにおける値です。
 * これらの処理負荷は今後のリリースで改善される可能性があります。
 *
 * @section PageSampleAudioAudioPerformanceMetrics_SectionHowToExecute 実行手順
 * サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleAudioAudioPerformanceMetrics_SectionDetail 解説
 * オーディオレンダラは波形の整形処理、複数の波形のミックス処理、エフェクト処理などを行いオーディオデバイスに出力します。@n
 * これらの処理を「レンダリング処理」と呼びます。
 * またオーディオレンダラは、nn::audio::AudioRendererParameter.sampleCount で指定されたサンプル数を 1 周期とする時間間隔でレンダリング処理を繰り返します。@n
 * この時間周期を「オーディオフレーム」と呼びます。
 * 途切れない音声の再生を実現するためには、オーディオフレーム内に収まるようにレンダリング処理を完結させる必要があります。@n
 *
 * このサンプルでは、オーディオレンダラがオーディオフレーム内でどれほどの時間を利用しているか、@n
 * つまりオーディオレンダラの負荷情報を取得する方法を解説します。
 * 以降この負荷情報をパフォーマンス情報と呼びます。
 *
 * パフォーマンス情報は所定の初期化を行うことで、オーディオレンダラが自動的に記録し続けます。@n
 * ユーザーはその情報を定期的に取得することで、オーディオレンダラのパフォーマンス情報を確認することができます。@n
 *
 * 取得に必要な大まかな手順は以下の通りです。
 * - オーディオレンダラの初期化時に、パフォーマンス情報を保持する期間を指定して初期化する
 * - パフォーマンス情報を書き込むバッファを指定してオーディオレンダラを更新する
 * - パフォーマンス情報が書き込まれたバッファを取得し、バッファの内容をパースする
 */


#include <cstdlib>
#include <cmath>

#include <nns/nns_Log.h>
#include <nn/nn_Abort.h>
#include <nn/os.h>
#include <nn/mem.h>
#include <nn/fs.h>
#include <nn/nn_TimeSpan.h>

#include <nn/audio.h>
#include <nns/audio/audio_HidUtilities.h>
#include <nns/audio/audio_WavFormat.h>
#include <nn/util/util_BytePtr.h>
#include <nn/util/util_PlacementArray.h>

#include <nn/hid.h>
#include <nn/hid/hid_KeyboardKey.h>
#include <nn/hid/hid_Npad.h>

#include <nn/settings/settings_DebugPad.h>

namespace {

// choose rendering engine sample rate
//const int RenderRate = 48000;
const int RenderRate = 32000;
const int RenderCount = (RenderRate / 200);

// choose number of files to play
const int BgmCount = 1;
const int SeCount = 4;

const char Title[] = "AudioPerformanceMetrics";

// - and add / remove them to / from the files lists
const char* g_BgmFileNames[BgmCount] =
{
    "asset:/AudioPerformanceMetrics/SampleBgm0-2ch.wav",
};

const char* g_SeFileNames[SeCount] =
{
    "asset:/AudioCommon/SampleSe0.adpcm",
    "asset:/AudioCommon/SampleSe1.adpcm",
    "asset:/AudioCommon/SampleSe2.adpcm",
    "asset:/AudioCommon/SampleSe3.adpcm",
};

NN_ALIGNAS(4096) char g_WorkBuffer[8 * 1024 * 1024];
NN_ALIGNAS(4096) char g_WaveBufferPoolMemory[14 * 1024 * 1024];
NN_ALIGNAS(4096) char g_EffectBufferPoolMemory[12 * 1024 * 1024];

nn::mem::StandardAllocator g_Allocator;
nn::mem::StandardAllocator g_WaveBufferAllocator;
nn::mem::StandardAllocator g_EffectBufferAllocator;

char* g_MountRomCacheBuffer = NULL;

}


void* Allocate(size_t size)
{
    return g_Allocator.Allocate(size);
}

void Deallocate(void* p, size_t size)
{
    NN_UNUSED(size);
    g_Allocator.Free(p);
}


std::size_t ReadAdpcmFile(nn::audio::AdpcmHeaderInfo* header, void** adpcmData, const char* filename)
{
    nn::fs::FileHandle handle;
    nn::Result result = nn::fs::OpenFile(&handle, filename, nn::fs::OpenMode_Read);
    NN_ABORT_UNLESS(result.IsSuccess());

    int64_t size;
    uint8_t adpcmheader[nn::audio::AdpcmHeaderSize];

    result = nn::fs::GetFileSize(&size, handle);
    NN_ABORT_UNLESS(result.IsSuccess());

    *adpcmData = g_WaveBufferAllocator.Allocate(static_cast<std::size_t>(size) - sizeof(adpcmheader), nn::audio::BufferAlignSize);
    NN_ABORT_UNLESS_NOT_NULL(*adpcmData);

    result = nn::fs::ReadFile(handle, 0, adpcmheader, sizeof(adpcmheader));
    NN_ABORT_UNLESS(result.IsSuccess());
    result = nn::fs::ReadFile(handle, sizeof(adpcmheader), *adpcmData, static_cast<size_t>(size) - sizeof(adpcmheader));
    NN_ABORT_UNLESS(result.IsSuccess());
    nn::fs::CloseFile(handle);

    nn::audio::ParseAdpcmHeader(header, adpcmheader, sizeof(adpcmheader));

    return static_cast<std::size_t>(size) - sizeof(adpcmheader);
}

std::size_t ReadWavFile(nns::audio::WavFormat* format, void** data, const char* filename)
{
    nn::fs::FileHandle handle;
    nn::Result result = nn::fs::OpenFile(&handle, filename, nn::fs::OpenMode_Read);
    NN_ABORT_UNLESS(result.IsSuccess());

    int64_t size;

    result = nn::fs::GetFileSize(&size, handle);
    NN_ABORT_UNLESS(result.IsSuccess());

    *data = g_WaveBufferAllocator.Allocate(static_cast<std::size_t>(size), nn::audio::BufferAlignSize);
    NN_ABORT_UNLESS_NOT_NULL(*data);

    // DATA チャンクを読む必要がありますが、ここではそれが 1024 バイト以内に見つかると仮定しています
    const std::size_t WavHeaderDataSize = 1024;

    result = nn::fs::ReadFile(handle, 0, *data, WavHeaderDataSize);
    NN_ABORT_UNLESS(result.IsSuccess());

    nns::audio::WavResult wavResult = nns::audio::ParseWavFormat(format, *data, WavHeaderDataSize);
    NN_ABORT_UNLESS_EQUAL(wavResult, nns::audio::WavResult_Success);
    NN_ABORT_UNLESS_EQUAL(format->channelCount, 2);    // このサンプルでは 2ch を仮定しています
    NN_ABORT_UNLESS_EQUAL(format->bitsPerSample, 16);  // このサンプルでは 16bit PCM を仮定しています

    result = nn::fs::ReadFile(handle, static_cast<std::size_t>(format->dataOffset), *data, static_cast<std::size_t>(format->dataSize));
    NN_ABORT_UNLESS(result.IsSuccess());
    nn::fs::CloseFile(handle);

    return static_cast<std::size_t>(format->dataSize);
}

void InitializeFileSystem()
{
    nn::fs::SetAllocator(Allocate, Deallocate);

    size_t cacheSize = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&cacheSize));
    g_MountRomCacheBuffer = new(std::nothrow) char[cacheSize];
    NN_ABORT_UNLESS_NOT_NULL(g_MountRomCacheBuffer);

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::fs::MountRom("asset", g_MountRomCacheBuffer, cacheSize)
    );
}

void FinalizeFileSystem()
{
    nn::fs::Unmount("asset");

    delete[] g_MountRomCacheBuffer;
    g_MountRomCacheBuffer = NULL;
}

void InitializeHidDevices()
{
    nn::hid::InitializeDebugPad();
    nn::hid::InitializeNpad();
    const nn::hid::NpadIdType npadIds[2] = { nn::hid::NpadId::No1, nn::hid::NpadId::Handheld };
    nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleFullKey::Mask | nn::hid::NpadStyleHandheld::Mask);
    nn::hid::SetSupportedNpadIdType(npadIds, sizeof(npadIds) / sizeof(npadIds[0]));

    //キーボードのキーを DebugPad のボタンに割り当てます。
    nn::settings::DebugPadKeyboardMap map;
    nn::settings::GetDebugPadKeyboardMap(&map);
    map.buttonA     = nn::hid::KeyboardKey::A::Index;
    map.buttonB     = nn::hid::KeyboardKey::B::Index;
    map.buttonX     = nn::hid::KeyboardKey::X::Index;
    map.buttonY     = nn::hid::KeyboardKey::Y::Index;
    map.buttonL     = nn::hid::KeyboardKey::L::Index;
    map.buttonR     = nn::hid::KeyboardKey::R::Index;
    map.buttonLeft  = nn::hid::KeyboardKey::LeftArrow::Index;
    map.buttonRight = nn::hid::KeyboardKey::RightArrow::Index;
    map.buttonUp    = nn::hid::KeyboardKey::UpArrow::Index;
    map.buttonDown  = nn::hid::KeyboardKey::DownArrow::Index;
    map.buttonStart = nn::hid::KeyboardKey::Space::Index;
    nn::settings::SetDebugPadKeyboardMap(map);
}

void PrintUsage()
{
    NNS_LOG("----------------------------------------------\n");
    NNS_LOG("%s Sample\n", Title);
    NNS_LOG("----------------------------------------------\n");
    NNS_LOG("[A]             StartSound    (SampleSe0)\n");
    NNS_LOG("[B]             StartSound    (SampleSe1)\n");
    NNS_LOG("[X]             StartSound    (SampleSe2)\n");
    NNS_LOG("[Y]             StartSound    (SampleSe3)\n");
    NNS_LOG("[L]             Toggle Delay effect enabled\n");
    NNS_LOG("[Start/space]   Finish sample\n");
    NNS_LOG("----------------------------------------------\n");
}

extern "C" void nnMain()
{
    g_Allocator.Initialize(g_WorkBuffer, sizeof(g_WorkBuffer));
    g_WaveBufferAllocator.Initialize(g_WaveBufferPoolMemory, sizeof(g_WaveBufferPoolMemory));
    g_EffectBufferAllocator.Initialize(g_EffectBufferPoolMemory, sizeof(g_EffectBufferPoolMemory));

    InitializeFileSystem();
    InitializeHidDevices();

    // オーディオレンダラの初期化時に、パフォーマンス情報を保持する期間を指定して初期化する。
    // パフォーマンス情報はオーディオフレーム毎に記録されます。この 1 オーディオフレーム分のパフォーマンス情報を「パフォーマンスフレーム」と呼びます。@n
    // オーディオレンダラはその動作中にパフォーマンスフレームを逐次記録していきます。
    // オーディオレンダラ内で、いくつのパフォーマンスフレームを保持するかを、nn::audio::AudioRendererParameter.performanceFrameCount で指定します。
    // パフォーマンス情報を取得するためには nn::audio::AudioRendererParameter.performanceFrameCount に 2 以上の数を指定する必要があります。
    //
    // オーディオレンダラ内では performanceFrameCount で指定したオーディオフレーム分の履歴を保管します。
    // パフォーマンス情報は nn::audio::RequestUpdateAudioRenderer() の呼び出し時にオーディオレンダラから取り出されるため、
    // performanceFrameCount で指定した数以上のオーディオフレームが経過すると、古い情報から順に破棄されます。
    // パフォーマンスフレームを欠落なく取得するには nn::audio::RequestUpdateAudioRenderer() の呼び出し間隔に含まれるオーディオフレーム以上の数を指定することを推奨します。
    // 初めての利用の場合は、想定されるオーディオフレーム数よりも大きな値を指定し、取得できるオーディオフレーム数を見極めたのちに数を削減する、といった調整を検討ください。
    // 取得したパフォーマンス情報に含まれるパフォーマンスフレームの個数の確認方法は、後述します。
    nn::audio::AudioRendererParameter parameter;
    nn::audio::InitializeAudioRendererParameter(&parameter);
    parameter.sampleRate = RenderRate;
    parameter.sampleCount = RenderCount;
    parameter.mixBufferCount = 6 + 2; // FinalMix(6) + SubMix(2)
    parameter.voiceCount = 24;
    parameter.subMixCount = 2;
    parameter.sinkCount = 1;
    parameter.effectCount = 3;
    parameter.performanceFrameCount = 5;

    // @link ../../../Samples/Sources/Applications/AudioRenderer
    // Samples/Sources/Applications/AudioRenderer @endlink サンプルと同様の構成でオーディオレンダラを初期化します。
    // オーディオレンダラ自身の詳しい解説はリンク先を確認してください。
    int channelCount = 2;
    int8_t mainBus[2];
    mainBus[nn::audio::ChannelMapping_FrontLeft] = 4;
    mainBus[nn::audio::ChannelMapping_FrontRight] = 5;
    int8_t auxBusA[2];
    auxBusA[nn::audio::ChannelMapping_FrontLeft] = 0;
    auxBusA[nn::audio::ChannelMapping_FrontRight] = 1;
    int8_t auxBusB[2];
    auxBusB[nn::audio::ChannelMapping_FrontLeft] = 2;
    auxBusB[nn::audio::ChannelMapping_FrontRight] = 3;
    NN_ABORT_UNLESS(
        nn::audio::IsValidAudioRendererParameter(parameter),
        "Invalid AudioRendererParameter specified."
    );
    size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(parameter);
    void* workBuffer = g_Allocator.Allocate(workBufferSize, nn::os::MemoryPageSize);
    NN_ABORT_UNLESS_NOT_NULL(workBuffer);

    nn::os::SystemEvent systemEvent;
    nn::audio::AudioRendererHandle handle;
    NN_ABORT_UNLESS(
        nn::audio::OpenAudioRenderer(&handle, &systemEvent, parameter, workBuffer, workBufferSize).IsSuccess(),
        "Failed to open AudioRenderer"
    );
    size_t configBufferSize = nn::audio::GetAudioRendererConfigWorkBufferSize(parameter);
    void* configBuffer = g_Allocator.Allocate(configBufferSize, nn::os::MemoryPageSize);
    NN_ABORT_UNLESS_NOT_NULL(configBuffer);
    nn::audio::AudioRendererConfig config;
    nn::audio::InitializeAudioRendererConfig(&config, parameter, configBuffer, configBufferSize);

    nn::audio::FinalMixType finalMix;
    nn::audio::AcquireFinalMix(&config, &finalMix, 6);
    nn::audio::SubMixType subMix0;
    nn::audio::AcquireSubMix(&config, &subMix0, parameter.sampleRate, 1);
    nn::audio::SubMixType subMix1;
    nn::audio::AcquireSubMix(&config, &subMix1, parameter.sampleRate, 1);

    nn::audio::DeviceSinkType deviceSink;
    nn::Result result = nn::audio::AddDeviceSink(&config, &deviceSink, &finalMix, mainBus, channelCount, "MainAudioOut");
    NN_ABORT_UNLESS(result.IsSuccess());
    nn::audio::SetSubMixDestination(&config, &subMix0, &finalMix);

    nn::audio::SetSubMixMixVolume(&subMix0, &finalMix, 0.5f, 0, mainBus[nn::audio::ChannelMapping_FrontLeft]);
    nn::audio::SetSubMixMixVolume(&subMix0, &finalMix, 0.5f, 0, mainBus[nn::audio::ChannelMapping_FrontRight]);
    nn::audio::SetSubMixDestination(&config, &subMix1, &subMix0);
    nn::audio::SetSubMixMixVolume(&subMix1, &subMix0, 0.5f, 0, 0);

    nn::audio::MemoryPoolType effectBufferPool;
    bool rt = nn::audio::AcquireMemoryPool(&config, &effectBufferPool, g_EffectBufferPoolMemory, sizeof(g_EffectBufferPoolMemory));
    NN_ABORT_UNLESS(rt);
    rt = nn::audio::RequestAttachMemoryPool(&effectBufferPool);
    NN_ABORT_UNLESS(rt);

    nn::audio::DelayType delay;
    const nn::TimeSpan delayTimeMax = nn::TimeSpan::FromSeconds(1); // delay エフェクトで利用しうる最大のディレイ時間を指定します。
    size_t delaySize = nn::audio::GetRequiredBufferSizeForDelay(delayTimeMax, parameter.sampleRate, 2);
    void* delayBuffer = g_EffectBufferAllocator.Allocate(delaySize, nn::audio::BufferAlignSize);
    nn::audio::BufferMixerType mixer0;
    nn::audio::BufferMixerType mixer1;

    result = nn::audio::AddDelay(&config, &delay, delayBuffer, delaySize, &finalMix, delayTimeMax, 2);
    NN_ABORT_UNLESS(result.IsSuccess());
    nn::audio::SetDelayInputOutput(&delay, auxBusA, auxBusA, channelCount);
    result = nn::audio::AddBufferMixer(&config, &mixer0, &finalMix);
    NN_ABORT_UNLESS(result.IsSuccess());
    nn::audio::SetBufferMixerInputOutput(&mixer0, auxBusB, auxBusA, channelCount);
    result = nn::audio::AddBufferMixer(&config, &mixer1, &finalMix);
    NN_ABORT_UNLESS(result.IsSuccess());
    nn::audio::SetBufferMixerInputOutput(&mixer1, auxBusA, mainBus, channelCount);

    nn::audio::SetDelayEnabled(&delay, true);
    nn::audio::SetDelayTime(&delay, nn::TimeSpan::FromMilliSeconds(100));
    nn::audio::SetDelayFeedbackGain(&delay, 0.7f);
    nn::audio::SetDelayChannelSpread(&delay, 0.256f);
    nn::audio::SetDelayLowPassAmount(&delay, 0.9f);
    nn::audio::SetDelayDryGain(&delay, 0.5f);
    nn::audio::SetDelayInGain(&delay, 0.4f);

    nn::audio::SetBufferMixerVolume(&mixer0, 0, 1.0f);
    nn::audio::SetBufferMixerVolume(&mixer0, 1, 1.0f);
    nn::audio::SetBufferMixerVolume(&mixer1, 0, 1.0f);
    nn::audio::SetBufferMixerVolume(&mixer1, 1, 1.0f);

    result = nn::audio::RequestUpdateAudioRenderer(handle, &config);
    NN_ABORT_UNLESS(result.IsSuccess());

    result = nn::audio::StartAudioRenderer(handle);
    NN_ABORT_UNLESS(result.IsSuccess());

    nn::audio::MemoryPoolType waveBufferMemoryPool;
    rt = nn::audio::AcquireMemoryPool(&config, &waveBufferMemoryPool, g_WaveBufferPoolMemory, sizeof(g_WaveBufferPoolMemory));
    NN_ABORT_UNLESS(rt);
    rt = nn::audio::RequestAttachMemoryPool(&waveBufferMemoryPool);
    NN_ABORT_UNLESS(rt);

    nn::audio::VoiceType voiceBgm[BgmCount];
    nn::audio::WaveBuffer waveBufferBgm[BgmCount];
    void* dataBgm[BgmCount];

    for (int i = 0; i < BgmCount; ++i)
    {
        const int bgmChannelCount = 2;
        nns::audio::WavFormat format;
        std::size_t dataSize = ReadWavFile(&format, &dataBgm[i], g_BgmFileNames[i]);
        nn::audio::AcquireVoiceSlot(&config, &voiceBgm[i], format.sampleRate, bgmChannelCount, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0);
        nn::audio::SetVoiceDestination(&config, &voiceBgm[i], &finalMix);

        waveBufferBgm[i].buffer = dataBgm[i];
        waveBufferBgm[i].size = dataSize;
        waveBufferBgm[i].startSampleOffset = 0;
        waveBufferBgm[i].endSampleOffset = static_cast<int32_t>(dataSize / sizeof(int16_t)) / bgmChannelCount;
        waveBufferBgm[i].loop = true;
        waveBufferBgm[i].isEndOfStream = false;
        waveBufferBgm[i].pContext = nullptr;
        waveBufferBgm[i].contextSize = 0;

        nn::audio::AppendWaveBuffer(&voiceBgm[i], &waveBufferBgm[i]);
        nn::audio::SetVoicePlayState(&voiceBgm[i], nn::audio::VoiceType::PlayState_Play);
        nn::audio::SetVoiceMixVolume(&voiceBgm[i], &finalMix, 0.707f / 2, 0, auxBusA[nn::audio::ChannelMapping_FrontLeft]);
        nn::audio::SetVoiceMixVolume(&voiceBgm[i], &finalMix, 0.707f / 2, 1, auxBusA[nn::audio::ChannelMapping_FrontRight]);

        // カットオフ 2048Hz のローパスフィルタを設定します
        nn::audio::BiquadFilterParameter firstFilter = { true, {720, 1439, 720}, {22684, -9350} };
        nn::audio::SetVoiceBiquadFilterParameter(&voiceBgm[i], 0, firstFilter);
        // カットオフ 1024 Hz のハイパスフィルタを無効状態で設定します。
        nn::audio::BiquadFilterParameter secondFilter = { false, {14041, -28083, 14041}, {29547, -13563} };
        nn::audio::SetVoiceBiquadFilterParameter(&voiceBgm[i], 1, secondFilter);
    }

    // SE
    nn::audio::VoiceType voiceSe[SeCount];
    nn::audio::WaveBuffer waveBufferSe[SeCount];
    nn::audio::AdpcmHeaderInfo* header[SeCount];
    void* dataSe[SeCount];

    for (int i = 0; i < SeCount; ++i)
    {
        header[i] = reinterpret_cast<nn::audio::AdpcmHeaderInfo*>(g_WaveBufferAllocator.Allocate(sizeof(nn::audio::AdpcmHeaderInfo), NN_ALIGNOF(nn::audio::AdpcmHeaderInfo)));
        std::size_t dataSeSize = ReadAdpcmFile(header[i], &dataSe[i], g_SeFileNames[i]);
        nn::audio::AcquireVoiceSlot(&config, &voiceSe[i], header[i]->sampleRate, 1, nn::audio::SampleFormat_Adpcm, nn::audio::VoiceType::PriorityHighest, &header[i]->parameter, sizeof(nn::audio::AdpcmParameter));
        nn::audio::SetVoiceDestination(&config, &voiceSe[i], &finalMix);

        waveBufferSe[i].buffer = dataSe[i];
        waveBufferSe[i].size = dataSeSize;
        waveBufferSe[i].startSampleOffset = 0;
        waveBufferSe[i].endSampleOffset = header[i]->sampleCount;
        waveBufferSe[i].loop = false;
        waveBufferSe[i].isEndOfStream = false;
        waveBufferSe[i].pContext = &header[i]->loopContext;
        waveBufferSe[i].contextSize = sizeof(nn::audio::AdpcmContext);

        nn::audio::AppendWaveBuffer(&voiceSe[i], &waveBufferSe[i]);
        nn::audio::SetVoicePlayState(&voiceSe[i], nn::audio::VoiceType::PlayState_Play);
        nn::audio::SetVoiceMixVolume(&voiceSe[i], &finalMix, 0.707f / 2, 0, auxBusA[nn::audio::ChannelMapping_FrontLeft]);
        nn::audio::SetVoiceMixVolume(&voiceSe[i], &finalMix, 0.707f / 2, 0, auxBusA[nn::audio::ChannelMapping_FrontRight]);
    }

    // 波形再生が終わるまで待ちつつ、パラメータを更新します。
    size_t perfBufferSize = nn::audio::GetRequiredBufferSizeForPerformanceFrames(parameter);
    void* perfBuffer[2];
    perfBuffer[0] = g_Allocator.Allocate(perfBufferSize);
    NN_ABORT_UNLESS_NOT_NULL(perfBuffer[0]);
    perfBuffer[1] = g_Allocator.Allocate(perfBufferSize);
    NN_ABORT_UNLESS_NOT_NULL(perfBuffer[1]);
    int perfBufferIndex = 0;

    PrintUsage();
    int dumpCount = 0;

    for (;;)
    {
        systemEvent.Wait();

        nn::hid::NpadButtonSet npadButtonDown = {};

        // Npad の入力を取得します。
        if (nn::hid::GetNpadStyleSet(nn::hid::NpadId::No1).Test<nn::hid::NpadStyleFullKey>())
        {
            static nn::hid::NpadFullKeyState npadFullKeyState = {};
            nn::hid::NpadFullKeyState state;
            nn::hid::GetNpadState(&state, nn::hid::NpadId::No1);
            npadButtonDown |= state.buttons & ~npadFullKeyState.buttons;
            npadFullKeyState = state;
        }
        if (nn::hid::GetNpadStyleSet(nn::hid::NpadId::Handheld).Test<nn::hid::NpadStyleHandheld>())
        {
            static nn::hid::NpadHandheldState npadHandheldState = {};
            nn::hid::NpadHandheldState state;
            nn::hid::GetNpadState(&state, nn::hid::NpadId::Handheld);
            npadButtonDown |= state.buttons & ~npadHandheldState.buttons;
            npadHandheldState = state;
        }

        // DebugPad の入力を取得します。
        {
            static nn::hid::DebugPadState debugPadState = {};
            nn::hid::DebugPadButtonSet debugPadButtonDown = {};
            nn::hid::DebugPadState state;
            nn::hid::GetDebugPadState(&state);
            debugPadButtonDown |= state.buttons & ~debugPadState.buttons;
            debugPadState = state;
            nns::audio::ConvertDebugPadButtonsToNpadButtons(&npadButtonDown, debugPadButtonDown);
        }

        //SE を再生します（同じ SE を多重再生しません）。
        for (int i = 0; i < SeCount; ++i)
        {
            // SE番号 {0, 1, 2, 3, 4, 5} が、ボタン {A, B, X, Y, L, R} に対応します。
            if(npadButtonDown.Test(i))
            {
                if (nn::audio::GetReleasedWaveBuffer(&voiceSe[i]))
                {
                    nn::audio::AppendWaveBuffer(&voiceSe[i], &waveBufferSe[i]);
                }
            }
        }

        // Delay エフェクトを ON/OFF します。
        if(npadButtonDown.Test< ::nn::hid::NpadButton::L >())
        {
            bool enabled = !nn::audio::IsDelayEnabled(&delay);
            nn::audio::SetDelayEnabled(&delay, enabled);
            NNS_LOG("set Delay %s\n", enabled ? "enable" : "disable");
        }

        // サンプルを終了します。
        if(npadButtonDown.Test< ::nn::hid::NpadButton::Plus >())
        {
            break;
        }

        // 詳細なパフォーマンス情報を取得する対象を設定します。
        // このブロックでは対象の設定方法を示します。
        // この設定の結果得られる nn::audio::PerformanceDetail で得られる情報と、その取り出し方法は後述するブロックで説明します。
        if ( npadButtonDown.Test< ::nn::hid::NpadButton::Up>() || npadButtonDown.Test<::nn::hid::NpadButton::Down>() )
        {
            enum DetailTarget
            {
                DetailTarget_None,
                DetailTarget_VoiceBgm,
                DetailTarget_VoiceSe,
                DetailTarget_SubMix,
                DetailTarget_FinalMix,
                DetailTarget_Count,
            };
            static DetailTarget target = DetailTarget_None;

            if (npadButtonDown.Test< ::nn::hid::NpadButton::Down>())
            {
                target = static_cast<DetailTarget>(target + 1);
                target = (target >= DetailTarget_Count) ? DetailTarget_None : target;
            }
            else if (npadButtonDown.Test< ::nn::hid::NpadButton::Up>())
            {
                target = static_cast<DetailTarget>(target - 1);
                target = (target < DetailTarget_None) ? DetailTarget_FinalMix : target;
            }

            // 詳細なパフォーマンス情報を取得する対象を target に応じて設定します。
            // 設定には nn::audio::SetPerformanceDetailTarget() を利用します。
            // 詳細なパフォーマンス情報が取得できるのは、 Voice / SubMix / FinalMix の中から同時に 1 つのみで、
            // 同時に複数個の詳細なパフォーマンス情報を取得することはできません。
            // 複数回にわたって nn::audio::SetPerformanceDetailTarget() を呼び出した場合は、
            // 最後に nn::audio::SetPerformanceDetailTarget() で指定した対象の詳細なパフォーマンス情報が取得されます。
            const char* targetName = nullptr;
            switch (target)
            {
            case DetailTarget_VoiceBgm:
                targetName = "voiceBgm";
                nn::audio::SetPerformanceDetailTarget(&config, &voiceBgm[0]);
                break;
            case DetailTarget_VoiceSe:
                targetName = "voiceSe[1]";
                nn::audio::SetPerformanceDetailTarget(&config, &voiceSe[1]);
                break;
            case DetailTarget_SubMix:
                targetName = "subMix0";
                nn::audio::SetPerformanceDetailTarget(&config, &subMix0);
                break;
            case DetailTarget_FinalMix:
                targetName = "finalMix";
                nn::audio::SetPerformanceDetailTarget(&config, &finalMix);
                break;
            case DetailTarget_None:
            default:
                targetName = "no target";
                // 詳細なパフォーマンス情報を取得を停止するには nn::audio::ClearPerformanceDetailTarget() を呼び出します。
                nn::audio::ClearPerformanceDetailTarget(&config);
                break;
            }
            NNS_LOG("Set nn::audio::PerformanceDetail target to %s\n", targetName);
        }

        // nn::audio::SetPerformanceDetailTarget() および nn::audio::ClearPerformanceDetailTarget() の設定は、
        // それらの呼び出し後の nn::audio::RequestUpdateAudioRenderer() によって、オーディオレンダラに反映されます。
        // よって上記したブロックでの nn::audio::SetPerformanceDetailTarget() の設定は、この nn::audio::RequestUpdateAudioRenderer() で反映されます。
        // また後述する nn::audio::SetPerformanceFrameBuffer() によってパフォーマンス情報を書き込むバッファが設定されていれば、
        // この nn::audio::RequestUpdateAudioRenderer() 呼び出し中に、パフォーマンス情報が書き込まれます。
        result = nn::audio::RequestUpdateAudioRenderer(handle, &config);
        NN_ABORT_UNLESS(result.IsSuccess());

        // パフォーマンス情報をオーディオレンダラから取得します。@n
        // パフォーマンス情報は nn::audio::RequestUpdateAudioRenderer() の呼び出し時に@n
        // nn::audio::SetPerformanceFrameBuffer() で指定したバッファに書き込まれます。@n
        // そこでまずはパフォーマンス情報取得用のバッファを設定する必要があります。
        //
        // ここでは lastBuffer としてパフォーマンス情報が書き込まれたバッファを取得し、@n
        // 次にパフォーマンス情報を書き込むバッファを perfBuffer[perfBufferIndex] として設定しています。@n
        // 初めて nn::audio::SetPerformanceFrameBuffer() を呼ぶ際、もしくは最後の設定が nullptr だった場合は、この返り値には nullptr が返ります。
        void* lastBuffer = nn::audio::SetPerformanceFrameBuffer(&config, perfBuffer[perfBufferIndex], perfBufferSize);
        perfBufferIndex++;
        perfBufferIndex %= 2;


        // パフォーマンス情報をバッファから取り出します。
        // nn::audio::PerformanceInfo は nn::audio::SetPerformanceFrameBuffer() で得られたパフォーマンス情報が書き込まれたバッファから@n
        // パフォーマンスフレーム単位で情報をパースするためのユーティリティクラスです。
        // nn::audio::PerformanceInfo::SetBuffer() で初期化します。
        // 渡されたバッファにパフォーマンス情報が含まれていると true を返します。
        nn::audio::PerformanceInfo performanceInfo;
        if (lastBuffer == nullptr || performanceInfo.SetBuffer(lastBuffer, perfBufferSize) == false)
        {
            continue;
        }

        // パフォーマンス情報はオーディオフレーム毎に記録されています。
        // このオーディオフレーム単位のパフォーマンス情報を「パフォーマンスフレーム」と呼びます。@n
        // パフォーマンスフレームには、Voice, SubMix, FinalMix, Sink 単位で処理負荷が記録されています。@n
        // これら各々単位のパフォーマンス情報を「パフォーマンスエントリー( nn::audio::PerformanceEtnry )」と呼びます。@n
        do
        {
            // このサンプルでは 100 オーディオフレームおきにのその瞬間のパフォーマンス値をログ出力します。
            const int logPrintInterval = 100;
            if (++dumpCount == logPrintInterval)
            {
                // パフォーマンスフレームに含まれるパフォーマンスエントリー列へのポインタを取得します。
                // パフォーマンスエントリーには以下の情報が含まれています。
                // - id : この nn::audio::PerformanceEntry が対象とする VoiceType / SubMixType / FinalMixType の NodeId
                // - startTime : オーディオフレームの処理開始時間を 0 とした相対的な処理開始時間 [MicroSeconds]
                // - processingTime : 処理にかかった時間 [MicroSeconds]
                // - entryType : エントリーの種類
                // NodeId は Voice, SubMix, FinalMix をオーディオレンダラ内で一意に特定するための識別子です。
                // nn::audio::GetVoiceNodeId(), nn::audio::GetSubMixNodeId(), nn::audio::GetFinalMixNodeId() でそれぞれ取得することができます。
                int entryCount = 0;
                const nn::audio::PerformanceEntry* entries = performanceInfo.GetEntries(&entryCount);
                for (int i = 0; i < entryCount; ++i)
                {
                    const char* name = nullptr;
                    switch (entries[i].entryType)
                    {
                    case nn::audio::PerformanceEntryType_Voice:       name = "Voice   "; break;
                    case nn::audio::PerformanceEntryType_SubMix:      name = "SubMix  "; break;
                    case nn::audio::PerformanceEntryType_FinalMix:    name = "FinalMix"; break;
                    case nn::audio::PerformanceEntryType_Sink:        name = "Sink    "; break;
                    case nn::audio::PerformanceEntryType_Unknown:     name = "None    "; break;
                    default:                                          name = "Unknown "; break;
                    }

                    NNS_LOG("App:[Entry |%s|%08x]:%10d [usec]\t%10d [usec]\n", name, entries[i].id, entries[i].processingTime, entries[i].startTime);
                }

                // パフォーマンスフレームに含まれるパフォーマンスディテール列へのポインタを取得します。
                // パフォーマンスディテールには以下の情報が含まれています。
                // - parentId : この PerformanceDetail に紐づく VoiceType / SubMixType / FinalMixType の NodeId
                // - startTime : オーディオフレームの処理開始時間を 0 とした相対的な処理開始時間 [MicroSeconds]
                // - processingTime : 処理にかかった時間 [MicroSeconds]
                // - detailType : この PerformanceDetail が保持している情報の種類
                // - parentEntryType : この PerformanceDetail に紐づく VoiceType / SubMixType / FinalMixType の種類 (nn::audio::PerformanceEntryType を保持)
                // parentId に保持される NodeId は Voice, SubMix, FinalMix をオーディオレンダラ内で一意に特定するための識別子です。
                // nn::audio::GetVoiceNodeId(), nn::audio::GetSubMixNodeId(), nn::audio::GetFinalMixNodeId() でそれぞれ取得することができます。
                int detailCount = 0;
                const nn::audio::PerformanceDetail* details = performanceInfo.GetDetails(&detailCount);
                for (int i = 0; i < detailCount; ++i)
                {
                    const char* name = nullptr;
                    switch (details[i].detailType)
                    {
                    case nn::audio::PerformanceDetailType_PcmInt16DataSource: name = "PcmInt16"; break;
                    case nn::audio::PerformanceDetailType_AdpcmDataSource:    name = "Adpcm   "; break;
                    case nn::audio::PerformanceDetailType_BiquadFilter:       name = "Biquad  "; break;
                    case nn::audio::PerformanceDetailType_Volume:             name = "Volume  "; break;
                    case nn::audio::PerformanceDetailType_Mix:                name = "Mix     "; break;
                    case nn::audio::PerformanceDetailType_Delay:              name = "Delay   "; break;
                    case nn::audio::PerformanceDetailType_Reverb:             name = "Reverb  "; break;
                    case nn::audio::PerformanceDetailType_Aux:                name = "Aux     "; break;
                    default:                                                  name = "Unknown "; break;
                    }

                    NNS_LOG("App:[Detail|%s|%08x]:%10d [usec]\t%10d [usec]\n", name, details[i].parentId, details[i].processingTime, details[i].startTime);

                }
            }
            else if (dumpCount > logPrintInterval)
            {
                NNS_LOG("App:[Total ]:%d[usec] [RenderingTimeLimitExceeded]:%s [VoiceDropCount]:%d StartTime:%lld (FrameIndex:%d)\n",
                        performanceInfo.GetTotalProcessingTime(),
                        performanceInfo.IsRenderingTimeLimitExceeded() ? "True" : "False",
                        performanceInfo.GetVoiceDropCount(),
                        performanceInfo.GetStartTime(),
                        performanceInfo.GetFrameIndex());
                dumpCount = 0;
            }

            // 次のパフォーマンスフレームに進みます。
            // 取得されたバッファにこれ以上のパフォーマンスフレームが含まれない場合は false を返しループを抜けます。
        } while (performanceInfo.MoveToNextFrame());
    }

    // レンダリングを終了します。
    nn::audio::StopAudioRenderer(handle);
    nn::audio::CloseAudioRenderer(handle);
    nn::os::DestroySystemEvent(systemEvent.GetBase());

    // メモリを解放します。
    if (delayBuffer)
    {
        g_EffectBufferAllocator.Free(delayBuffer);
        delayBuffer = nullptr;
    }
    for (int i = 0; i < BgmCount; ++i)
    {
        if (dataBgm[i])
        {
            g_WaveBufferAllocator.Free(dataBgm[i]);
            dataBgm[i] = nullptr;
        }
    }
    for (int i = 0; i < SeCount; ++i)
    {
        if (dataSe[i])
        {
            g_WaveBufferAllocator.Free(dataSe[i]);
            dataSe[i] = nullptr;
        }
    }
    if (configBuffer)
    {
        g_Allocator.Free(configBuffer);
        configBuffer = nullptr;
    }
    if (workBuffer)
    {
        g_Allocator.Free(workBuffer);
        workBuffer = nullptr;
    }

    FinalizeFileSystem();
}  // NOLINT(readability/fn_size)
