﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{AudioDevice.cpp,PageSampleAudioAudioDevice}
 *
 * @brief
 * 音声出力デバイス関連の諸設定を扱う nn::audio のサンプル
 */

/**
 * @page PageSampleAudioAudioDevice 音声出力デバイス関連の諸設定を扱う nn::audio のサンプル
 * @tableofcontents
 *
 * @brief
 * 音声出力デバイス固有の設定や、アプリケーション独自のダウンミックス処理の設定について解説するサンプルです。
 *
 * @section PageSampleAudioAudioDevice_SectionBrief 概要
 * このサンプルは以下の機能についてその利用方法を説明することを目的としています。
 * #1 出力先デバイス、出力チャンネル数、およびその切り替えタイミングの取得
 * #2 出力先デバイス毎のボリューム設定
 * #3 アプリケーション独自のダウンミックス処理の設定
 *
 * @section PageSampleAudioAudioDevice_SectionFileStructure ファイル構成
 * 本サンプルプログラムは @link ../../../Samples/Sources/Applications/AudioDevice
 * Samples/Sources/Applications/AudioDevice @endlink 以下にあります。
 *
 * @section PageSampleAudioAudioDevice_SectionNecessaryEnvironment 必要な環境
 * オーディオ出力が利用可能である必要があります。
 *
 * @section PageSampleAudioAudioDevice_SectionHowToOperate 操作方法
 * サンプルプログラムを実行すると BGM の再生を開始します。またボタン操作に応じて SE を再生します。
 * <p>
 * <table>
 * <tr><th> 入力 </th><th> 動作 </th></tr>
 * <tr><td> [A]             </td><td> SE を再生する                        </td></tr>
 * <tr><td> [B]             </td><td> SE を再生する                        </td></tr>
 * <tr><td> [X]             </td><td> SE を再生する                        </td></tr>
 * <tr><td> [Y]             </td><td> SE を再生する                        </td></tr>
 * <tr><td> [L]             </td><td> SE を再生する                        </td></tr>
 * <tr><td> [Left/Right]    </td><td> ボリュームを設定するデバイスの選択       </td></tr>
 * <tr><td> [Up/Down]       </td><td> 選択されたデバイスのボリュームを調整      </td></tr>
 * <tr><td> [Select/Minus]  </td><td> サンプルの説明を表示                   </td></tr>
 * <tr><td> [Start/Space]   </td><td> プログラムの終了                      </td></tr>
 * </table>
 * </p>
 *
 * @section PageSampleAudioAudioDevice_SectionPrecaution 注意事項
 * 基本となるオーディオレンダラの使い方を事前に理解していることを前提としています。
 * オーディオレンダラの詳しい利用例は @link ../../../Samples/Sources/Applications/AudioRenderer
 * Samples/Sources/Applications/AudioRenderer @endlink を参照してください。
 *
 * @section PageSampleAudioAudioDevice_SectionHowToExecute 実行手順
 * サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleAudioAudioDevice_SectionDetail 解説
 * 下記するサンプルコードの中では、以下の機能についてそれぞれ解説しています。
 *
 * #1 出力先デバイス、出力チャンネル数、およびその切り替えタイミングの取得
 *    現在の出力中のデバイス名は nn::audio::GetActiveAudioDeviceName() にて取得することが可能です。
 *    また、現在の出力チャンネル数は nn::audio::GetActiveAudioDeviceChannelCountForOutput() にて取得することが可能です。
 *    この出力先デバイス、出力チャンネル数はプラットフォームの動作状況に応じて、動的に変化しますが、
 *    変化するタイミングは nn::audio::AcquireAudioDeviceNotificationForOutput() で取得されるシステムイベントを通じて取得することができます。
 *    コード中に #1-1 および #1-2 として関連するコードにコメントが付与されています。
 *
 * #2 出力先デバイス毎のボリューム設定
 *    各出力先デバイスは、それぞれ個別に出力ボリュームを設定することができます。
 *    このボリュームは nn::audio::SetAudioDeviceOutputVolume() で設定が可能です。
 *    コード中に #2-1 および #2-2 として関連するコードにコメントが付与されています。
 *
 * #3 アプリケーション独自のダウンミックス処理の設定
 *    ダウンミックス処理は「システムデフォルトのダウンミックス処理」と「アプリケーション独自のダウンミックス処理」とが存在します。
 *    後者はオーディオレンダラでのみ利用可能な機能であり、このサンプルでは後者について解説しています。
 *    前者も含めたダウンミックス処理についての包括的な説明は@confluencelink{166500119, リンク先ドキュメントの「ダウンミックス処理について」を参照してください。}
 *    「アプリケーション独自のダウンミックス処理」は nn::audio::SetDownMixParameter() にてパラメータを設定した後、
 *    nn::audio::SetDownMixParameterEnabled() にて有効化することで利用できます。
 *    コード中に #3-1 および #3-2 として関連するコードにコメントが付与されています。
 */

#include <cstdlib>
#include <cmath>
#include <new>

#include <nns/nns_Log.h>
#include <nn/nn_Abort.h>
#include <nn/os.h>
#include <nn/mem.h>
#include <nn/fs.h>
#include <nn/nn_TimeSpan.h>

#include <nn/audio.h>
#include <nns/audio/audio_HidUtilities.h>
#include <nns/audio/audio_WavFormat.h>

#include <nn/hid.h>
#include <nn/hid/hid_KeyboardKey.h>
#include <nn/hid/hid_Npad.h>

#include <nn/settings/settings_DebugPad.h>

namespace {

// choose rendering engine sample rate
//const int RenderRate = 48000;
const int RenderRate = 32000;
const int RenderCount = (RenderRate / 200);

// choose number of files to play
const int BgmCount = 1;
const int SeCount = 4;

const char Title[] = "AudioDevice";

// - and add / remove them to / from the files lists
const char* g_BgmFileNames[BgmCount] =
{
    "asset:/AudioDevice/SampleBgm0-2ch.wav",
};

const char* g_SeFileNames[SeCount] =
{
    "asset:/AudioCommon/SampleSe0.adpcm",
    "asset:/AudioCommon/SampleSe1.adpcm",
    "asset:/AudioCommon/SampleSe2.adpcm",
    "asset:/AudioCommon/SampleSe3.adpcm",
};

NN_ALIGNAS(4096) char g_WorkBuffer[8 * 1024 * 1024];
NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char g_WaveBufferPoolMemory[14 * 1024 * 1024];
NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char g_EffectBufferPoolMemory[12 * 1024 * 1024];

nn::mem::StandardAllocator g_Allocator;
nn::mem::StandardAllocator g_WaveBufferAllocator;
nn::mem::StandardAllocator g_EffectBufferAllocator;

char* g_MountRomCacheBuffer = NULL;

}

void* Allocate(size_t size)
{
    return g_Allocator.Allocate(size);
}

void Deallocate(void* p, size_t size)
{
    NN_UNUSED(size);
    g_Allocator.Free(p);
}

std::size_t GenerateSineWave(void** data, int sampleRate, int frequency, int sampleCount)
{
    // g_WaveBufferAllocator が管理するメモリ領域はすべて waveBufferMemoryPool メモリプールに追加されています。
    int16_t* p = static_cast<int16_t*>(g_WaveBufferAllocator.Allocate(sampleCount * sizeof(int16_t), nn::audio::BufferAlignSize));
    NN_ABORT_UNLESS_NOT_NULL(p);
    const float Pi = 3.1415926535897932384626433f;
    for (auto i = 0; i < sampleCount; ++i)
    {
        p[i] = static_cast<int16_t>(std::numeric_limits<int16_t>::max() * sinf(2 * Pi * frequency * i / sampleRate));
    }
    *data = p;
    return sampleCount * sizeof(int16_t);
}

std::size_t ReadAdpcmFile(nn::audio::AdpcmHeaderInfo* header, void** adpcmData, const char* filename)
{
    nn::fs::FileHandle handle;
    nn::Result result = nn::fs::OpenFile(&handle, filename, nn::fs::OpenMode_Read);
    NN_ABORT_UNLESS(result.IsSuccess());

    int64_t size;
    uint8_t adpcmheader[nn::audio::AdpcmHeaderSize];

    result = nn::fs::GetFileSize(&size, handle);
    NN_ABORT_UNLESS(result.IsSuccess());

    *adpcmData = g_WaveBufferAllocator.Allocate(static_cast<std::size_t>(size) - sizeof(adpcmheader), nn::audio::BufferAlignSize);
    NN_ABORT_UNLESS_NOT_NULL(*adpcmData);

    result = nn::fs::ReadFile(handle, 0, adpcmheader, sizeof(adpcmheader));
    NN_ABORT_UNLESS(result.IsSuccess());
    result = nn::fs::ReadFile(handle, sizeof(adpcmheader), *adpcmData, static_cast<size_t>(size) - sizeof(adpcmheader));
    NN_ABORT_UNLESS(result.IsSuccess());
    nn::fs::CloseFile(handle);

    nn::audio::ParseAdpcmHeader(header, adpcmheader, sizeof(adpcmheader));

    return static_cast<std::size_t>(size) - sizeof(adpcmheader);
}

std::size_t ReadWavFile(nns::audio::WavFormat* format, void** data, const char* filename)
{
    nn::fs::FileHandle handle;
    nn::Result result = nn::fs::OpenFile(&handle, filename, nn::fs::OpenMode_Read);
    NN_ABORT_UNLESS(result.IsSuccess());

    int64_t size;

    result = nn::fs::GetFileSize(&size, handle);
    NN_ABORT_UNLESS(result.IsSuccess());

    *data = g_WaveBufferAllocator.Allocate(static_cast<std::size_t>(size), nn::audio::BufferAlignSize);
    NN_ABORT_UNLESS_NOT_NULL(*data);

    // DATA チャンクを読む必要がありますが、ここではそれが 1024 バイト以内に見つかると仮定しています
    const std::size_t WavHeaderDataSize = 1024;

    result = nn::fs::ReadFile(handle, 0, *data, WavHeaderDataSize);
    NN_ABORT_UNLESS(result.IsSuccess());

    nns::audio::WavResult wavResult = nns::audio::ParseWavFormat(format, *data, WavHeaderDataSize);
    NN_ABORT_UNLESS_EQUAL(wavResult, nns::audio::WavResult_Success);
    NN_ABORT_UNLESS_EQUAL(format->bitsPerSample, 16);  // このサンプルでは 16bit PCM を仮定しています

    result = nn::fs::ReadFile(handle, static_cast<std::size_t>(format->dataOffset), *data, static_cast<std::size_t>(format->dataSize));
    NN_ABORT_UNLESS(result.IsSuccess());
    nn::fs::CloseFile(handle);

    return static_cast<std::size_t>(format->dataSize);
}

void InitializeFileSystem()
{
    nn::fs::SetAllocator(Allocate, Deallocate);

    size_t cacheSize = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&cacheSize));
    g_MountRomCacheBuffer = new(std::nothrow) char[cacheSize];
    NN_ABORT_UNLESS_NOT_NULL(g_MountRomCacheBuffer);

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::fs::MountRom("asset", g_MountRomCacheBuffer, cacheSize)
    );
}

void FinalizeFileSystem()
{
    nn::fs::Unmount("asset");

    delete[] g_MountRomCacheBuffer;
    g_MountRomCacheBuffer = NULL;
}

void InitializeHidDevices()
{
    nn::hid::InitializeDebugPad();
    nn::hid::InitializeNpad();
    const nn::hid::NpadIdType npadIds[2] = { nn::hid::NpadId::No1, nn::hid::NpadId::Handheld };
    nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleFullKey::Mask | nn::hid::NpadStyleHandheld::Mask);
    nn::hid::SetSupportedNpadIdType(npadIds, sizeof(npadIds) / sizeof(npadIds[0]));

    //キーボードのキーを DebugPad のボタンに割り当てます。
    nn::settings::DebugPadKeyboardMap map;
    nn::settings::GetDebugPadKeyboardMap(&map);
    map.buttonA     = nn::hid::KeyboardKey::A::Index;
    map.buttonB     = nn::hid::KeyboardKey::B::Index;
    map.buttonX     = nn::hid::KeyboardKey::X::Index;
    map.buttonY     = nn::hid::KeyboardKey::Y::Index;
    map.buttonL     = nn::hid::KeyboardKey::L::Index;
    map.buttonR     = nn::hid::KeyboardKey::R::Index;
    map.buttonZL    = nn::hid::KeyboardKey::U::Index;
    map.buttonZR    = nn::hid::KeyboardKey::V::Index;
    map.buttonLeft  = nn::hid::KeyboardKey::LeftArrow::Index;
    map.buttonRight = nn::hid::KeyboardKey::RightArrow::Index;
    map.buttonUp    = nn::hid::KeyboardKey::UpArrow::Index;
    map.buttonDown  = nn::hid::KeyboardKey::DownArrow::Index;
    map.buttonStart = nn::hid::KeyboardKey::Space::Index;
    map.buttonSelect = nn::hid::KeyboardKey::Minus::Index;
    nn::settings::SetDebugPadKeyboardMap(map);
}


nn::hid::NpadButtonSet GetButtonDown()
{
    nn::hid::NpadButtonSet npadButtonDown = {};

    // Npad の入力を取得します。
    if (nn::hid::GetNpadStyleSet(nn::hid::NpadId::No1).Test<nn::hid::NpadStyleFullKey>())
    {
        static nn::hid::NpadFullKeyState npadFullKeyState = {};
        nn::hid::NpadFullKeyState state;
        nn::hid::GetNpadState(&state, nn::hid::NpadId::No1);
        npadButtonDown |= state.buttons & ~npadFullKeyState.buttons;
        npadFullKeyState = state;
    }
    if (nn::hid::GetNpadStyleSet(nn::hid::NpadId::Handheld).Test<nn::hid::NpadStyleHandheld>())
    {
        static nn::hid::NpadHandheldState npadHandheldState = {};
        nn::hid::NpadHandheldState state;
        nn::hid::GetNpadState(&state, nn::hid::NpadId::Handheld);
        npadButtonDown |= state.buttons & ~npadHandheldState.buttons;
        npadHandheldState = state;
    }

    // DebugPad の入力を取得します。
    {
        static nn::hid::DebugPadState debugPadState = {};
        nn::hid::DebugPadButtonSet debugPadButtonDown = {};
        nn::hid::DebugPadState state;
        nn::hid::GetDebugPadState(&state);
        debugPadButtonDown |= state.buttons & ~debugPadState.buttons;
        debugPadState = state;
        nns::audio::ConvertDebugPadButtonsToNpadButtons(&npadButtonDown, debugPadButtonDown);
    }

    return npadButtonDown;
}

void PrintUsage()
{
    NNS_LOG("--------------------------------------------------------\n");
    NNS_LOG("%s Sample\n", Title);
    NNS_LOG("--------------------------------------------------------\n");
    NNS_LOG("[A]             StartSound                   (SampleSe0)\n");
    NNS_LOG("[B]             StartSound                   (SampleSe1)\n");
    NNS_LOG("[X]             StartSound                   (SampleSe2)\n");
    NNS_LOG("[Y]             StartSound                   (SampleSe3)\n");
    NNS_LOG("[L]             Toggle Application Specific DownMix     \n");
    NNS_LOG("[Left][Right]   Select Device                           \n");
    NNS_LOG("[Up][Down]      Control Selected Device Volume          \n");
    NNS_LOG("[Select/-]      Print Usage                             \n");
    NNS_LOG("[Start/Space]   Shut down sample program                \n");
    NNS_LOG("--------------------------------------------------------\n");
}

void SetDefaultParameter(nn::audio::AudioRendererParameter &parameter)
{
    nn::audio::InitializeAudioRendererParameter(&parameter);
    parameter.sampleRate = RenderRate;
    parameter.sampleCount = RenderCount;
    parameter.mixBufferCount = 6 + 2; // FinalMix(6) + SubMix(2)
    parameter.voiceCount = 24;
    parameter.subMixCount = 2;
    parameter.sinkCount = 1;
    parameter.effectCount = 3;
    parameter.performanceFrameCount = 0;
}

void SetupBgmVoices(
    nn::audio::VoiceType* pOutVoices,
    nn::audio::WaveBuffer* pOutWaveBuffers,
    void** ppDataBuffer,
    nn::audio::AudioRendererConfig& config)
{
    for (int i = 0; i < BgmCount; ++i)
    {
        nns::audio::WavFormat format;
        std::size_t dataSize = ReadWavFile(&format, &ppDataBuffer[i], g_BgmFileNames[i]);
        nn::audio::AcquireVoiceSlot(&config, &pOutVoices[i], format.sampleRate, format.channelCount, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0);
        nn::audio::SetVoicePlayState(&pOutVoices[i], nn::audio::VoiceType::PlayState_Play);

        pOutWaveBuffers[i].buffer = ppDataBuffer[i];
        pOutWaveBuffers[i].size = dataSize;
        pOutWaveBuffers[i].startSampleOffset = 0;
        pOutWaveBuffers[i].endSampleOffset = static_cast<int32_t>(dataSize / sizeof(int16_t)) / format.channelCount;
        pOutWaveBuffers[i].loop = true;
        pOutWaveBuffers[i].isEndOfStream = false;
        pOutWaveBuffers[i].pContext = nullptr;
        pOutWaveBuffers[i].contextSize = 0;
        nn::audio::AppendWaveBuffer(&pOutVoices[i], &pOutWaveBuffers[i]);
    }
}

void SetupSeVoices(
    nn::audio::VoiceType* pOutVoices,
    nn::audio::AdpcmHeaderInfo** pOutHeaders,
    nn::audio::WaveBuffer * pOutWaveBuffers,
    void** pOutDataBuffer,
    nn::audio::AudioRendererConfig config)
{
    for (int i = 0; i < SeCount; ++i)
    {
        pOutHeaders[i] = reinterpret_cast<nn::audio::AdpcmHeaderInfo*>(g_WaveBufferAllocator.Allocate(sizeof(nn::audio::AdpcmHeaderInfo), NN_ALIGNOF(nn::audio::AdpcmHeaderInfo)));
        std::size_t dataSeSize = ReadAdpcmFile(pOutHeaders[i], &pOutDataBuffer[i], g_SeFileNames[i]);
        nn::audio::AcquireVoiceSlot(&config, &pOutVoices[i], pOutHeaders[i]->sampleRate, 1, nn::audio::SampleFormat_Adpcm, nn::audio::VoiceType::PriorityHighest, &pOutHeaders[i]->parameter, sizeof(nn::audio::AdpcmParameter));
        nn::audio::SetVoicePlayState(&pOutVoices[i], nn::audio::VoiceType::PlayState_Play);

        pOutWaveBuffers[i].buffer = pOutDataBuffer[i];
        pOutWaveBuffers[i].size = dataSeSize;
        pOutWaveBuffers[i].startSampleOffset = 0;
        pOutWaveBuffers[i].endSampleOffset = pOutHeaders[i]->sampleCount;
        pOutWaveBuffers[i].loop = false;
        pOutWaveBuffers[i].isEndOfStream = false;
        pOutWaveBuffers[i].pContext = &pOutHeaders[i]->loopContext;
        pOutWaveBuffers[i].contextSize = sizeof(nn::audio::AdpcmContext);
        nn::audio::AppendWaveBuffer(&pOutVoices[i], &pOutWaveBuffers[i]);
    }
}

int FindDevice(nn::audio::AudioDeviceName* deviceNames, int deviceNameCount, const char* targetName)
{
    int index = -1;
    for (auto i = 0; i < deviceNameCount; ++i)
    {
        if (strncmp(deviceNames[i].name, targetName, nn::audio::AudioDeviceName::NameLength) == 0)
        {
            index = i;
            break;
        }
    }
    NN_ABORT_UNLESS(index > 0);
    return index;
}


void UpdateDeviceIndexPamrameter(int* outDeviceIndex, nn::hid::NpadButtonSet &buttonDown, int listedDeviceCount)
{
    int index = *outDeviceIndex;

    if (buttonDown.Test< ::nn::hid::NpadButton::Left >())
    {
        --index;
        if (index < 0)
        {
            index = listedDeviceCount - 1;
        }
    }
    else if (buttonDown.Test< ::nn::hid::NpadButton::Right >())
    {
        ++index;
        if (index >= listedDeviceCount)
        {
            index = 0;
        }
    }

    *outDeviceIndex = index;
}

void UpdateDeviceVolumeParameter(float* outVolume, nn::hid::NpadButtonSet &buttonDown)
{
    float vol = *outVolume;

    if (buttonDown.Test< ::nn::hid::NpadButton::Up>())
    {
        vol += 0.1f;
        if (vol >= 1.0f)
        {
            vol = 1.0f;
        }
    }
    else if (buttonDown.Test< ::nn::hid::NpadButton::Down>())
    {
        vol -= 0.1f;
        if (vol <= 0.0f)
        {
            vol = 0.0f;
        }
    }

    *outVolume = vol;
}


extern "C" void nnMain()
{
    g_Allocator.Initialize(g_WorkBuffer, sizeof(g_WorkBuffer));
    g_WaveBufferAllocator.Initialize(g_WaveBufferPoolMemory, sizeof(g_WaveBufferPoolMemory));
    g_EffectBufferAllocator.Initialize(g_EffectBufferPoolMemory, sizeof(g_EffectBufferPoolMemory));

    InitializeFileSystem();
    InitializeHidDevices();

    //////////////////////////////////////////////////////////////////////////
    // オーディオレンダラの基本的な設定を行います。オーディオレンダラ自身の説明については、
    // オーディオレンダラの詳しい利用例は @link ../../../Samples/Sources/Applications/AudioRenderer
    // Samples/Sources/Applications/AudioRenderer @endlink を参照してください。
    //////////////////////////////////////////////////////////////////////////
    nn::audio::AudioRendererParameter parameter;
    SetDefaultParameter(parameter);
    NN_ABORT_UNLESS(
        nn::audio::IsValidAudioRendererParameter(parameter),
        "Invalid AudioRendererParameter specified."
    );

    const int channelCount = 6;
    int8_t mainBus[channelCount];
    mainBus[nn::audio::ChannelMapping_FrontLeft] = 0;
    mainBus[nn::audio::ChannelMapping_FrontRight] = 1;
    mainBus[nn::audio::ChannelMapping_FrontCenter] = 2;
    mainBus[nn::audio::ChannelMapping_LowFrequency] = 3;
    mainBus[nn::audio::ChannelMapping_RearLeft] = 4;
    mainBus[nn::audio::ChannelMapping_RearRight] = 5;

    size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(parameter);
    void* workBuffer = g_Allocator.Allocate(workBufferSize, nn::os::MemoryPageSize);
    NN_ABORT_UNLESS_NOT_NULL(workBuffer);

    nn::audio::AudioRendererHandle handle;
    nn::os::SystemEvent systemEvent;
    NN_ABORT_UNLESS(
        nn::audio::OpenAudioRenderer(&handle, &systemEvent, parameter, workBuffer, workBufferSize).IsSuccess(),
        "Failed to open AudioRenderer"
    );

    size_t configBufferSize = nn::audio::GetAudioRendererConfigWorkBufferSize(parameter);
    void* configBuffer = g_Allocator.Allocate(configBufferSize, nn::os::MemoryPageSize);
    NN_ABORT_UNLESS_NOT_NULL(configBuffer);
    nn::audio::AudioRendererConfig config;
    nn::audio::InitializeAudioRendererConfig(&config, parameter, configBuffer, configBufferSize);

    nn::audio::FinalMixType finalMix;
    nn::audio::AcquireFinalMix(&config, &finalMix, 6);
    nn::audio::DeviceSinkType deviceSink;
    NN_ABORT_UNLESS(nn::audio::AddDeviceSink(&config, &deviceSink, &finalMix, mainBus, 6, "MainAudioOut").IsSuccess());
    NN_ABORT_UNLESS(nn::audio::RequestUpdateAudioRenderer(handle, &config).IsSuccess());
    NN_ABORT_UNLESS(nn::audio::StartAudioRenderer(handle).IsSuccess());

    nn::audio::MemoryPoolType waveBufferMemoryPool;
    NN_ABORT_UNLESS(nn::audio::AcquireMemoryPool(&config, &waveBufferMemoryPool, g_WaveBufferPoolMemory, sizeof(g_WaveBufferPoolMemory)));
    NN_ABORT_UNLESS(nn::audio::RequestAttachMemoryPool(&waveBufferMemoryPool));

    nn::audio::VoiceType voiceBgm[BgmCount];
    nn::audio::WaveBuffer waveBufferBgm[BgmCount];
    void* dataBgm[BgmCount];
    SetupBgmVoices(voiceBgm, waveBufferBgm, dataBgm, config);

    // BGM の出力先を finalMix に指定し、出力チャンネルを Left および Right に設定します。
    for (auto i = 0; i < BgmCount; ++i)
    {
        nn::audio::SetVoiceDestination(&config, &voiceBgm[i], &finalMix);
        nn::audio::SetVoiceMixVolume(&voiceBgm[i], &finalMix, 0.5f, 0, mainBus[nn::audio::ChannelMapping_FrontLeft]);
        nn::audio::SetVoiceMixVolume(&voiceBgm[i], &finalMix, 0.5f, 1, mainBus[nn::audio::ChannelMapping_FrontRight]);
    }

    nn::audio::VoiceType voiceSe[SeCount];
    nn::audio::WaveBuffer waveBufferSe[SeCount];
    nn::audio::AdpcmHeaderInfo* header[SeCount];
    void* dataSe[SeCount];
    SetupSeVoices(voiceSe, header, waveBufferSe, dataSe, config);

    // SE の出力先を finalMix に指定し、出力チャンネルを各 SE 毎にそれぞれ Center, Lfe, RearLeft, RearRight にそれぞれ振り分けます。
    for (auto i = 0; i < SeCount; ++i)
    {
        nn::audio::SetVoiceDestination(&config, &voiceSe[i], &finalMix);
    }
    nn::audio::SetVoiceMixVolume(&voiceSe[0], &finalMix, 0.707f, 0, mainBus[nn::audio::ChannelMapping_FrontCenter]);
    nn::audio::SetVoiceMixVolume(&voiceSe[1], &finalMix, 0.707f, 0, mainBus[nn::audio::ChannelMapping_LowFrequency]);
    nn::audio::SetVoiceMixVolume(&voiceSe[2], &finalMix, 0.707f, 0, mainBus[nn::audio::ChannelMapping_RearLeft]);
    nn::audio::SetVoiceMixVolume(&voiceSe[3], &finalMix, 0.707f, 0, mainBus[nn::audio::ChannelMapping_RearRight]);


    //////////////////////////////////////////////////////////////////////////
    // #1-1 出力先デバイス、およびその切り替えタイミングの取得
    //
    // NX プラットフォームでは @confluencelink{166500119, リンク先ドキュメントの「出力先デバイスの設定」}に記載のある通り、
    // 動作状況に応じて音声出力デバイスを「実際のオーディオデバイス」の中から動的に選択し切り替えます。
    // ここでは、切り替えが発生したタイミングの取得手段としてシステムイベントの登録を行います。
    // またイベントの通知とは無関係に、現在の出力先デバイスは nn::audio::GetActiveAudioDeviceName() を用いて任意のタイミングで取得可能です。
    //////////////////////////////////////////////////////////////////////////
    nn::os::SystemEvent deviceEvent;
    nn::audio::AcquireAudioDeviceNotificationForOutput(&deviceEvent);


    //////////////////////////////////////////////////////////////////////////
    // #2-1 出力先デバイス毎のボリューム設定
    //
    // 音声出力先デバイスの一覧は nn::audio::ListAudioDeviceName() で取得します。
    // 以下の 4 種類が取得されます。
    // - "AudioTvOutput"
    // - "AudioStereoJackOutput"
    // - "AudioBuiltInSpeakerOutput"
    // - "AudioUsbDeviceOutput"
    // @confluencelink{166500119, 取得されるデバイス説明は「出力先デバイスの設定」に記載されています。}
    //////////////////////////////////////////////////////////////////////////
    const int deviceNameCount = 8;
    nn::audio::AudioDeviceName deviceNames[deviceNameCount];
    int listedDeviceCount = nn::audio::ListAudioDeviceName(deviceNames, deviceNameCount);
    NNS_LOG(";; AudioDeviceList\n");
    for (auto i = 0; i < listedDeviceCount; ++i)
    {
        NNS_LOG("[%d] %s\n", i, deviceNames[i].name);
    }
    int deviceIndex = 0;
    float deviceVolume = 0.0f;


    //////////////////////////////////////////////////////////////////////////
    // #3-1 アプリケーション独自のダウンミックス処理の設定
    //
    // ここではダウンミックス処理に用いるパラメータを更新し、アプリケーション独自のダウンミックス処理を無効化状態にしています。
    // アプリケーション独自のダウンミックス処理の「有効・無効の切り替え」および「ダウンミックス処理に用いるパラメータ」は任意のタイミングで設定可能ですが、
    // これらの設定関数はスレッドセーフではありませんので、その点のみご注意ください。
    // ダウンミックス処理の詳細については@confluencelink{166500119, リンク先ドキュメントの「ダウンミックス処理について」も併せて参照してください。}
    //
    // ここではサンプル用の極端な例として FrontLeft, FrontRight 以外のチャンネルをカットするダウンミックス処理のパラメータを設定しています。
    // やや上のコードで voiceBgm を FrontLeft, FrontRight に、voiceSe をそれ以外のチャンネルに割り振っていますので、
    // アプリケーション独自のダウンミックス処理の有効にすると、 SE の音がすべてカットされます。
    //////////////////////////////////////////////////////////////////////////
    nn::audio::DeviceSinkType::DownMixParameter downMixParam;
    downMixParam.coeff[0] = 1.f;
    downMixParam.coeff[1] = .0f;
    downMixParam.coeff[2] = .0f;
    downMixParam.coeff[3] = .0f;
    nn::audio::SetDownMixParameter(&deviceSink, &downMixParam);
    nn::audio::SetDownMixParameterEnabled(&deviceSink, false);


    PrintUsage();
    for (;;)
    {
        systemEvent.Wait();
        auto buttonDown = GetButtonDown();
        const auto PlusKeyMask =
            nn::hid::NpadButton::Up::Mask |
            nn::hid::NpadButton::Down::Mask |
            nn::hid::NpadButton::Left::Mask |
            nn::hid::NpadButton::Right::Mask;


        //////////////////////////////////////////////////////////////////////////
        // #1-2 出力先デバイス、出力チャンネル数、およびその切り替えタイミングの取得
        // nn::audio::AcquireAudioDeviceNotificationForOutput() で取得したシステムイベントにより、
        // 音声出力先デバイスに関するイベントが発生したかどうかを確認します。
        // NX プラットフォーム上では、以下のイベント発生を検知することが出来ます。
        // - 音声出力先の切り替わり
        // - 音声出力チャンネル数の変更
        //
        // このイベントは nn::os::EventClearMode_AutoClear で初期化されています。
        //////////////////////////////////////////////////////////////////////////
        if (deviceEvent.TryWait())
        {
            nn::audio::AudioDeviceName deviceName;
            nn::audio::GetActiveAudioDeviceName(&deviceName);
            const auto ChannelCount = nn::audio::GetActiveAudioDeviceChannelCountForOutput();
            NNS_LOG("Active DeviceName:%s Channel Count:%d\n", deviceName.name, ChannelCount);
        }


        //////////////////////////////////////////////////////////////////////////
        // #2-2 出力先デバイス毎のボリューム設定
        //
        // ボリュームの設定を行います。
        // 各出力先デバイスのボリュームは、独立に設定が可能です。
        // ここでは十字ボタンの操作に応じて選択したデバイスの音量を 0.0f ~ 1.0f の範囲で操作します。
        // nn::audio::SetAudioDeviceOutputVolume() で設定可能なボリュームの上限は 128.0f まで許容されますが、
        // 過剰なボリューム設定は音割れの原因となりますので、ご注意ください。
        //////////////////////////////////////////////////////////////////////////
        if ((buttonDown & PlusKeyMask).IsAnyOn())
        {
            UpdateDeviceIndexPamrameter(&deviceIndex, buttonDown, listedDeviceCount);
            // ボリュームの現在値を取得
            deviceVolume = nn::audio::GetAudioDeviceOutputVolume(&deviceNames[deviceIndex]);
            UpdateDeviceVolumeParameter(&deviceVolume, buttonDown);
            NNS_LOG("OutputVolumeControl: %s -> %f\n", deviceNames[deviceIndex].name, deviceVolume);
            nn::audio::SetAudioDeviceOutputVolume(&deviceNames[deviceIndex], deviceVolume);
        }


        //////////////////////////////////////////////////////////////////////////
        // #3-2 アプリケーション独自のダウンミックス処理の設定
        //
        // アプリケーション独自のダウンミックス処理の有効無効を切り替えます。
        // nn::audio::SetDownMixParameterEnabled() によって、有効無効が設定可能です。
        // このサンプルでは nn::audio::DeviceSinkType::DownMixParameter に、
        // FrontLeft および FrontRight 以外のチャンネルをカットする設定を施してありますので、
        // アプリケーション独自のダウンミックス処理を有効化すると BGM 以外の音が発音されなくなります。
        //////////////////////////////////////////////////////////////////////////
        if (buttonDown.Test< ::nn::hid::NpadButton::L >())
        {
            static bool enabled = true;
            nn::audio::SetDownMixParameterEnabled(&deviceSink, enabled);
            enabled = !enabled;
        }


        for (int i = 0; i < SeCount; ++i)
        {
            if(buttonDown.Test(i))
            {
                if (nn::audio::GetReleasedWaveBuffer(&voiceSe[i]))
                {
                    nn::audio::AppendWaveBuffer(&voiceSe[i], &waveBufferSe[i]);
                }
            }
        }

        if(buttonDown.Test< ::nn::hid::NpadButton::Plus >())
        {
            break;
        }

        if (buttonDown.Test< ::nn::hid::NpadButton::Minus >())
        {
            PrintUsage();
        }

        NN_ABORT_UNLESS(nn::audio::RequestUpdateAudioRenderer(handle, &config).IsSuccess());
    }

    // レンダリングを終了します。
    nn::audio::StopAudioRenderer(handle);
    nn::audio::CloseAudioRenderer(handle);
    nn::os::DestroySystemEvent(systemEvent.GetBase());

    // メモリを解放します。
    for (int i = 0; i < BgmCount; ++i)
    {
        if (dataBgm[i])
        {
            g_WaveBufferAllocator.Free(dataBgm[i]);
            dataBgm[i] = nullptr;
        }
    }
    for (int i = 0; i < SeCount; ++i)
    {
        if (dataSe[i])
        {
            g_WaveBufferAllocator.Free(dataSe[i]);
            dataSe[i] = nullptr;
            g_WaveBufferAllocator.Free(header[i]);
            header[i] = nullptr;
        }
    }
    if (configBuffer)
    {
        g_Allocator.Free(configBuffer);
        configBuffer = nullptr;
    }
    if (workBuffer)
    {
        g_Allocator.Free(workBuffer);
        workBuffer = nullptr;
    }

    FinalizeFileSystem();
}  // NOLINT(readability/fn_size)
