﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{AtkSubMix.cpp,PageSampleAtkSubMix}
 *
 * @brief
 *  nn::atk ライブラリでサブミックス構成をユーザが定義してサウンドを再生するサンプルプログラム
 */

/**
 * @page PageSampleAtkSubMix サブミックス構成を定義した再生
 * @tableofcontents
 *
 * @brief
 *  nn::atk ライブラリでサブミックス構成をユーザが定義してサウンドを再生するサンプルプログラムの解説です。
 *
 * @section PageSampleAtkSubMix_SectionBrief 概要
 *  nn::atk ライブラリでサブミックス構成をユーザが定義してサウンドを再生するサンプルです。
 *
 * @section PageSampleAtkSubMix_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/AtkSubMix Samples/Sources/Applications/AtkSubMix @endlink 以下にあります。
 *
 * @section PageSampleAtkSubMix_SectionNecessaryEnvironment 必要な環境
 *  とくになし
 *
 * @section PageSampleAtkSubMix_SectionHowToOperate 操作方法
 *  コンソールに操作方法が表示されます。
 *
 * @section PageSampleAtkSubMix_SectionPrecaution 注意事項
 *  実行中のログが、画面上とコンソールに表示され、
 *  起動直後のログとして操作方法が表示されます。
 *
 *  PC 版ではウィンドウは立ち上がらず、コンソール表示のみになる点にご注意ください。
 *
 * @section PageSampleAtkSubMix_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleAtkSubMix_SectionDetail 解説
 *
 * @subsection PageSampleAtkSubMix_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  AtkSubMix.cpp
 *  @includelineno AtkSubMix.cpp
 *
 * @subsection PageSampleAtkSubMix_SectionSampleDetail サンプルプログラムの解説
 *  先のサンプルプログラムの Atk に関連している処理の流れは以下の通りです(AtkSimple と違う部分には追加要素と書かれています)。
 *
 *  - アロケータの初期化
 *    - 内部で２種類（メモリプール管理対象用とそれ以外）のアロケータを初期化。
 *  - Atk ライブラリの初期化 (InitializeAtk())
 *    - SoundSystem の初期化パラメータの設定 (追加要素)
 *    - SoundSystem の初期化
 *    - SubMix の初期化 (InitializeSubMix()) (追加要素)
 *    - SoundHeap の初期化
 *    - SoundArchive をオープンし、パラメータ情報をロード
 *    - SoundDataManager の初期化
 *    - SoundArchivePlayer で用いるストリームバッファをメモリプール管理対象用アロケータから確保し初期化
 *    - メモリプール管理対象のメモリをメモリプールにアタッチ
 *    - SoundArchivePlayer の初期化
 *    - SubMix にエフェクトを追加 (追加要素)
 *  - SoundHeap にウェーブサウンドとシーケンスサウンドの再生に必要な情報をロード (LoadData())
 *  - メインループでの処理
 *    - キー操作による処理
 *      - サウンドの再生・停止処理を実行
 *        - サウンドの再生時に使用するサブミックスを設定 (追加要素)
 *      - 使用するサブミックスの選択 (追加要素)
 *    - SoundArchivePlayer の更新
 *  - Atk ライブラリの終了処理 (FinalizeAtk())
 *    - SubMix からエフェクトを削除 (追加要素)
 *    - SoundArchivePlayer の 終了処理
 *    - メモリプール管理対象のメモリをメモリプールからデタッチ
 *    - その他の終了処理
 *    - SubMix の終了処理 (追加要素)
 *    - SoundSystemの終了処理
 *    - メモリ解放
 *  - アロケータの終了処理
 *
 * SubMix の初期化・終了処理は、それぞれ SoundSystem の初期化直後・終了処理直前に実行する必要があります。
 * それ以外のタイミングでは安全に処理できない点にご注意ください。
 *
 */

#include "Common.fsid"

#include <nn/atk.h>

#include <nns/atk/atk_SampleCommon.h>

#include <nns/nns_Log.h>

namespace
{
    const char Title[] = "AtkSubMix";
    const char ArchiveRelativePath[] = "Common.bfsar";

    const int SoundHeapSize = 4 * 1024 * 1024;

    nn::atk::SoundHeap          g_SoundHeap;
    nn::atk::FsSoundArchive     g_SoundArchive;
    nn::atk::SoundArchivePlayer g_SoundArchivePlayer;
    nn::atk::SoundDataManager   g_SoundDataManager;

    nn::audio::MemoryPoolType   g_MemoryPool;

    void* g_pMemoryForSoundSystem;
    void* g_pMemoryForSoundHeap;
    void* g_pMemoryForInfoBlock;
    void* g_pMemoryForSoundDataManager;
    void* g_pMemoryForSoundArchivePlayer;
    void* g_pMemoryForStreamBuffer;

    nn::atk::SoundHandle        g_SoundHandle;
    nn::atk::SoundHandle        g_SoundHandleHold;

    enum SubMixIndex
    {
        SubMixIndex_A,
        SubMixIndex_B,
        SubMixIndex_C,
        SubMixIndex_Count,
    };
    const int AppendEffectBusIndex = 1;
    const int SubMixChannelCount = 6;
    const int SubMixBusCount[SubMixIndex_Count] = { 2, 1, 1 };

    nn::atk::SubMix             g_SubMix[SubMixIndex_Count];
    void*                       g_pMemoryForSubMix[SubMixIndex_Count];

    nn::atk::EffectReverb       g_EffectReverb;
    void*                       g_pMemoryForEffectReverb;

    int                         g_TargetSubMixIndex;
}

void Initialize()
{
    nns::atk::InitializeHeap();
    nns::atk::InitializeFileSystem();
    nns::atk::InitializeHidDevices();
}

void Finalize()
{
    nns::atk::FinalizeHidDevices();
    nns::atk::FinalizeFileSystem();
    nns::atk::FinalizeHeap();
}

void InitializeSubMix()
{
    // SubMix[A] --> SubMix[B] --+
    //                           +--> FinalMix
    // SubMix[C] ----------------+
    //
    // SubMix[A] : バス数 2, チャンネル数 6 (2 番目のバスには EffectReverb を追加)
    // SubMix[B] : バス数 1, チャンネル数 6
    // SubMix[C] : バス数 1, チャンネル数 6
    //
    // というサブミックスを構成します

    // SubMix[A] の初期化
    std::size_t memSizeForSubMixA = nn::atk::SubMix::GetRequiredMemorySize(
        SubMixBusCount[SubMixIndex_A],
        SubMixChannelCount,
        SubMixBusCount[SubMixIndex_B],
        SubMixChannelCount
    );
    g_pMemoryForSubMix[SubMixIndex_A] = nns::atk::Allocate( memSizeForSubMixA );
    g_SubMix[SubMixIndex_A].Initialize(
        SubMixBusCount[SubMixIndex_A],
        SubMixChannelCount,
        SubMixBusCount[SubMixIndex_B],
        SubMixChannelCount,
        g_pMemoryForSubMix[SubMixIndex_A],
        memSizeForSubMixA
    );

    // SubMix[B] の初期化
    nn::atk::FinalMix& finalMix = nn::atk::SoundSystem::GetFinalMix();
    std::size_t memSizeForSubMixB = nn::atk::SubMix::GetRequiredMemorySize(
        SubMixBusCount[SubMixIndex_B],
        SubMixChannelCount,
        finalMix.GetBusCount(),
        finalMix.GetChannelCount()
    );
    g_pMemoryForSubMix[SubMixIndex_B] = nns::atk::Allocate( memSizeForSubMixB );
    g_SubMix[SubMixIndex_B].Initialize(
        SubMixBusCount[SubMixIndex_B],
        SubMixChannelCount,
        finalMix.GetBusCount(),
        finalMix.GetChannelCount(),
        g_pMemoryForSubMix[SubMixIndex_B],
        memSizeForSubMixB
    );

    // SubMix[C] の初期化
    std::size_t memSizeForSubMixC = nn::atk::SubMix::GetRequiredMemorySize(
        SubMixBusCount[SubMixIndex_C],
        SubMixChannelCount,
        finalMix.GetBusCount(),
        finalMix.GetChannelCount()
    );
    g_pMemoryForSubMix[SubMixIndex_C] = nns::atk::Allocate( memSizeForSubMixC );
    g_SubMix[SubMixIndex_C].Initialize(
        SubMixBusCount[SubMixIndex_C],
        SubMixChannelCount,
        finalMix.GetBusCount(),
        finalMix.GetChannelCount(),
        g_pMemoryForSubMix[SubMixIndex_C],
        memSizeForSubMixC
    );

    // SubMix のそれぞれを接続
    g_SubMix[SubMixIndex_A].SetDestination( &g_SubMix[SubMixIndex_B] );
    g_SubMix[SubMixIndex_B].SetDestination( &finalMix );
    g_SubMix[SubMixIndex_C].SetDestination( &finalMix );

    // SubMix の音量設定
    g_SubMix[SubMixIndex_C].SetSubMixVolume( 0.2f, 0 ); // SubMix[C] から再生されている事が確認できるよう音量を下げます
    // SubMix[A] のエフェクトバスを SubMix[B] の先頭バスに出力するよう設定します。
    // その他のバスのセンド量は、同じインデクスのバスについては初期状態でセンド量が設定されているため、設定の必要はありません。
    g_SubMix[SubMixIndex_A].SetSend( AppendEffectBusIndex, 0, 1.0f );

    g_TargetSubMixIndex = SubMixIndex_A;
}

void FinalizeSubMix()
{
    const auto WaitTime = nn::TimeSpan::FromMilliSeconds( 5 );

    // SubMix は接続元から終了処理する必要があります。
    // つまり、この例では SubMix[B] よりも先に SubMix[A] を終了処理する必要があります。
    while( !g_SubMix[SubMixIndex_A].IsFinalizable() )
    {
        nn::os::SleepThread( WaitTime );
    }
    g_SubMix[SubMixIndex_A].Finalize();

    while( !g_SubMix[SubMixIndex_B].IsFinalizable() )
    {
        nn::os::SleepThread( WaitTime );
    }
    g_SubMix[SubMixIndex_B].Finalize();

    while( !g_SubMix[SubMixIndex_C].IsFinalizable() )
    {
        nn::os::SleepThread( WaitTime );
    }
    g_SubMix[SubMixIndex_C].Finalize();
}

void InitializeAtk()
{
    bool isSuccess = true;

    // SubMix を使用するための設定
    int totalBusCount = 0;
    for(int i = 0; i < SubMixIndex_Count; i++)
    {
        totalBusCount += SubMixBusCount[i];
    }

    nn::atk::SoundSystem::SoundSystemParam param;
    param.enableCustomSubMix      = true;
    param.subMixTotalChannelCount = totalBusCount * SubMixChannelCount;
    param.subMixCount             = SubMixIndex_Count;

    // SoundSystem の初期化
    std::size_t memSizeForSoundSystem = nn::atk::SoundSystem::GetRequiredMemSize( param );
    g_pMemoryForSoundSystem = nns::atk::Allocate( memSizeForSoundSystem, nn::atk::SoundSystem::WorkMemoryAlignSize );
    isSuccess = nn::atk::SoundSystem::Initialize(
        param,
        reinterpret_cast<uintptr_t>( g_pMemoryForSoundSystem ),
        memSizeForSoundSystem );
    NN_ABORT_UNLESS( isSuccess, "cannot initialize SoundSystem" );

    // SubMix の初期化
    InitializeSubMix();

    // SoundHeap の初期化
    g_pMemoryForSoundHeap = nns::atk::Allocate( SoundHeapSize );
    isSuccess = g_SoundHeap.Create( g_pMemoryForSoundHeap, SoundHeapSize );
    NN_ABORT_UNLESS( isSuccess, "cannot create SoundHeap" );

    // SoundArchive の初期化
    const char* archiveAbsolutePath = nns::atk::GetAbsolutePath(ArchiveRelativePath);
    isSuccess = g_SoundArchive.Open(archiveAbsolutePath);
    NN_ABORT_UNLESS( isSuccess, "cannot open SoundArchive(%s)\n", archiveAbsolutePath );

    // SoundArchive のパラメータ情報をメモリにロード
    std::size_t infoBlockSize = g_SoundArchive.GetHeaderSize();
    g_pMemoryForInfoBlock = nns::atk::Allocate(  infoBlockSize, nn::atk::FsSoundArchive::BufferAlignSize );
    isSuccess = g_SoundArchive.LoadHeader( g_pMemoryForInfoBlock, infoBlockSize );
    NN_ABORT_UNLESS( isSuccess, "cannot load InfoBlock" );

    // SoundDataManager の初期化
    std::size_t memSizeForSoundDataManager = g_SoundDataManager.GetRequiredMemSize( &g_SoundArchive );
    g_pMemoryForSoundDataManager = nns::atk::Allocate( memSizeForSoundDataManager, nn::atk::SoundDataManager::BufferAlignSize );
    isSuccess = g_SoundDataManager.Initialize(
        &g_SoundArchive,
        g_pMemoryForSoundDataManager,
        memSizeForSoundDataManager );
    NN_ABORT_UNLESS( isSuccess, "cannot initialize SoundDataManager" );

    // SoundArchivePlayer で用いるストリームバッファの初期化
    // ストリームバッファはメモリプール管理されているヒープから確保する必要があります。
    std::size_t memSizeForStreamBuffer = g_SoundArchivePlayer.GetRequiredStreamBufferSize( &g_SoundArchive );
    g_pMemoryForStreamBuffer = nns::atk::AllocateForMemoryPool(memSizeForStreamBuffer);

    // 専用のヒープをメモリプールにアタッチ
    nn::atk::SoundSystem::AttachMemoryPool(&g_MemoryPool, nns::atk::GetPoolHeapAddress(), nns::atk::GetPoolHeapSize());

    // SoundArchivePlayer の初期化
    std::size_t memSizeForSoundArchivePlayer = g_SoundArchivePlayer.GetRequiredMemSize( &g_SoundArchive );
    g_pMemoryForSoundArchivePlayer = nns::atk::Allocate( memSizeForSoundArchivePlayer, nn::atk::SoundArchivePlayer::BufferAlignSize);
    isSuccess = g_SoundArchivePlayer.Initialize(
        &g_SoundArchive,
        &g_SoundDataManager,
        g_pMemoryForSoundArchivePlayer, memSizeForSoundArchivePlayer,
        g_pMemoryForStreamBuffer, memSizeForStreamBuffer );
    NN_ABORT_UNLESS( isSuccess, "cannot initialize SoundArchivePlayer" );

    // EffectReverb をカスタムサブミックスに追加
    {
        g_EffectReverb.SetSampleRate(nn::atk::EffectBase::SampleRate_48000);
        g_EffectReverb.SetChannelMode(nn::atk::EffectBase::ChannelMode_2Ch);
        g_EffectReverb.SetEarlyMode(nn::atk::EffectReverb::EarlyMode_Hall);
        g_EffectReverb.SetLateMode(nn::atk::EffectReverb::LateMode_Hall);
        g_EffectReverb.SetLateGain(0.6f);
        g_EffectReverb.SetDecayTime(nn::TimeSpan::FromMilliSeconds(3000));

        // SubMix[A] に接続してバスのセンドの確認に用います。
        std::size_t memSizeForEffectReverb = g_EffectReverb.GetRequiredMemSize();
        g_pMemoryForEffectReverb = nns::atk::AllocateForMemoryPool( memSizeForEffectReverb, nn::audio::BufferAlignSize );
        g_EffectReverb.SetEnabled( true );

        // SubMix[A] の AppendEffectBusIndex バスにエフェクトを追加
        g_SubMix[SubMixIndex_A].AppendEffect( &g_EffectReverb, AppendEffectBusIndex, g_pMemoryForEffectReverb, memSizeForEffectReverb );
    }
}

void FinalizeAtk()
{
    // Effect をカスタムサブミックスから削除
    {
        const auto WaitTime = nn::TimeSpan::FromMilliSeconds( 5 );
        g_EffectReverb.SetEnabled( false );
        while( !g_EffectReverb.IsRemovable() )
        {
            nn::os::SleepThread( WaitTime );
        }
        g_SubMix[SubMixIndex_A].ClearEffect( AppendEffectBusIndex );
    }

    g_SoundArchivePlayer.Finalize();

    // 専用のヒープをメモリプールからデタッチ
    nn::atk::SoundSystem::DetachMemoryPool(&g_MemoryPool);

    g_SoundDataManager.Finalize();
    g_SoundArchive.Close();
    g_SoundHeap.Destroy();

    // SubMix の終了処理
    FinalizeSubMix();

    nn::atk::SoundSystem::Finalize();

    nns::atk::FreeForMemoryPool(g_pMemoryForStreamBuffer);
    nns::atk::FreeForMemoryPool(g_pMemoryForEffectReverb);
    nns::atk::Free(g_pMemoryForSoundArchivePlayer);
    nns::atk::Free(g_pMemoryForSoundDataManager);
    nns::atk::Free(g_pMemoryForInfoBlock);
    nns::atk::Free(g_pMemoryForSoundHeap);
    nns::atk::Free(g_pMemoryForSubMix[SubMixIndex_C]);
    nns::atk::Free(g_pMemoryForSubMix[SubMixIndex_B]);
    nns::atk::Free(g_pMemoryForSubMix[SubMixIndex_A]);
    nns::atk::Free(g_pMemoryForSoundSystem);
}

void LoadData()
{
    bool isSuccess = true;

    isSuccess = g_SoundDataManager.LoadData( SEQ_MARIOKART, &g_SoundHeap );
    NN_ABORT_UNLESS( isSuccess, "LoadData(SEQ_MARIOKART) failed." );

    isSuccess = g_SoundDataManager.LoadData( SE_YOSHI, &g_SoundHeap );
    NN_ABORT_UNLESS( isSuccess, "LoadData(SE_YOSHI) failed." );
}

void PlayWithStartSound(nn::atk::SoundArchive::ItemId soundId, const char* debugLabelName)
{
    g_SoundHandle.Stop( 0 );

    nn::atk::SoundArchivePlayer::StartInfo info;
    info.enableFlag |= nn::atk::SoundArchivePlayer::StartInfo::EnableFlagBit_OutputReceiver;
    info.pOutputReceiver = &g_SubMix[g_TargetSubMixIndex]; // 選択中のサブミックスで再生します

    bool result = g_SoundArchivePlayer.StartSound( &g_SoundHandle, soundId, &info ).IsSuccess();
    NNS_LOG("StartSound(%s) on SubMix[%d] ... (%d)\n", debugLabelName, g_TargetSubMixIndex, result);

    // SubMix[A] のエフェクトが確認できるようにセンドを設定します
    g_SoundHandle.SetSend(AppendEffectBusIndex,  0.5f);
    // SoundMaker によりセンドMainにデフォルト値が設定されているため、そのセンド量を減らします
    g_SoundHandle.SetSend(0, -0.5f);
}

void PrintTargetSubMixState()
{
    NNS_LOG( "Target SubMix Index: %d\n", g_TargetSubMixIndex );
}

void PrintUsage()
{
    NNS_LOG("--------------------------------------------------\n");
    NNS_LOG("%s Sample\n", Title);
    NNS_LOG("--------------------------------------------------\n");
    NNS_LOG("[A]              StartSound SEQ  (SEQ_MARIOKART)\n");
    NNS_LOG("[X]              StartSound WSD  (SE_YOSHI)\n");
    NNS_LOG("[Y]              StartSound STRM (STRM_MARIOKART)\n");
    NNS_LOG("[B]              Stop Sound\n");
    NNS_LOG("[L]              Print Usage\n");
    NNS_LOG("[LEFT/RIGHT]     Change Targe SubMix Index\n");
    NNS_LOG("[+/Start][Space] Exit Application\n");
    NNS_LOG("--------------------------------------------------\n");

    PrintTargetSubMixState();
}

bool UpdateAtk()
{
    // StartSound / Stop
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::A >() )
    {
        PlayWithStartSound(SEQ_MARIOKART, "SEQ_MARIOKART");
    }
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::X >() )
    {
        PlayWithStartSound(SE_YOSHI, "SE_YOSHI");
    }
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Y >() )
    {
        PlayWithStartSound(STRM_MARIOKART, "STRM_MARIOKART");
    }
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::B >() )
    {
        g_SoundHandle.Stop( 0 );
    }

    bool isTargetSubMixChanged = false;
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Left >() || nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Right >() )
    {
        g_TargetSubMixIndex += nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Left >() ? -1 : 1;
        if( g_TargetSubMixIndex < 0 )
        {
            g_TargetSubMixIndex = SubMixIndex_Count - 1;
        }
        if( g_TargetSubMixIndex >= SubMixIndex_Count )
        {
            g_TargetSubMixIndex = 0;
        }
        isTargetSubMixChanged = true;
    }

    // Print Usage
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::L >() )
    {
        PrintUsage();
    }

    if( isTargetSubMixChanged )
    {
        PrintTargetSubMixState();
    }

    // Exit
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Start >() )
    {
        return false;
    }

    g_SoundArchivePlayer.Update();

    return true;
} //NOLINT(impl/function_size)

extern "C" void nnMain()
{
    Initialize();
    InitializeAtk();

    LoadData();

    PrintUsage();

    for ( ;; )
    {
        nns::atk::UpdateHidDevices();

        if ( !UpdateAtk() )
        {
            break;
        }

        // Vsync の代わり
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
    }

    FinalizeAtk();
    Finalize();
}

