﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

// ----------------------------------------------------------------------------------------
// VoiceCommand 版は AtkSandboxVcmd のほうのソリューションを開くことで実行できます。
// また、AtkSandboxVcmd 版のプロジェクトの設定で NN_ATK_ENABLE_VOICE_COMMAND_SAMPLE
// も定義されます。
// ----------------------------------------------------------------------------------------

#include "test.fsid"

#include <nn/atk.h>
#include <nn/htcs.h>
#include <nn/profiler.h>
#include <nn/nn_Assert.h>
#include <nn/util/util_Arithmetic.h>

#include <nn/atk/viewer/atk_SoundEdit.h>
#include "SpyToolProfiler.h"

#include "AtkProfiler.h"

#include "nns/atk/atk_SampleCommon.h"
#include "FlagList.h"

class CommonObject
{
public:
    class InitializeParam
    {
    public:
        InitializeParam() NN_NOEXCEPT
        : m_SoundSystemParam()
        , m_pFlagList( nullptr )
        {
            if ( GetGlobalFlagList().IsFlagEnabled(GlobalFlagType_DisableSoundThread) )
            {
#if defined(NN_ATK_ENABLE_VOICE_COMMAND_SAMPLE)
                m_SoundSystemParam.enableSoundThread = false;
#endif
            }

            m_SoundSystemParam.enableProfiler = true;
        }

        nn::atk::SoundSystem::SoundSystemParam& GetSoundSystemParam() NN_NOEXCEPT;
        const nn::atk::SoundSystem::SoundSystemParam& GetSoundSystemParam() const NN_NOEXCEPT;

        void SetFlagList(FlagList* pFlagList) NN_NOEXCEPT;
        FlagList* GetFlagList() NN_NOEXCEPT;
        const FlagList* GetFlagList() const NN_NOEXCEPT;

    private:
        nn::atk::SoundSystem::SoundSystemParam m_SoundSystemParam;
        FlagList* m_pFlagList;
    };
    struct StartParam{
        const nn::atk::SoundStartable::StartInfo* pStartInfo;

        //  StartSound の前に SoundHandle の Stop を呼ぶか
        bool isSoundStopEnabled;

        //  SoundHandle の Stop を呼ぶときの fadeFrames の値
        int stopFadeFrames;

        StartParam() NN_NOEXCEPT
        : pStartInfo( nullptr )
        , isSoundStopEnabled( true )
        , stopFadeFrames( 0 )
        {}
    };

    CommonObject() NN_NOEXCEPT;
    virtual ~CommonObject() NN_NOEXCEPT;

    virtual void Initialize() NN_NOEXCEPT;
    virtual void Initialize(const CommonObject::InitializeParam& param) NN_NOEXCEPT;
    virtual void Finalize() NN_NOEXCEPT;
    virtual void LoadData() NN_NOEXCEPT;
    virtual void Update() NN_NOEXCEPT;
#if defined( NN_ATK_ENABLE_GFX_VIEWING )
    virtual void UpdateDraw(const char* moduleName) NN_NOEXCEPT;
#endif
    virtual void PrintUsage() NN_NOEXCEPT;
    virtual int UpdateInput() NN_NOEXCEPT;
    virtual void PlayWithStartSound(nn::atk::SoundArchive::ItemId soundId, const char* debugLabelName) NN_NOEXCEPT;
    virtual void PlayWithStartSound(nn::atk::SoundArchive::ItemId soundId, const char* debugLabelName, const StartParam& startParam) NN_NOEXCEPT;

    bool CheckCacheState(nn::atk::SoundArchive::ItemId soundId) NN_NOEXCEPT;

    nn::atk::SoundArchivePlayer& GetSoundArchivePlayer() NN_NOEXCEPT
    {
        return m_SoundArchivePlayer;
    }

    nn::atk::SoundArchive& GetSoundArchive() NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(m_pSoundArchive);
        return *m_pSoundArchive;
    }

    nn::atk::SoundDataManager& GetSoundDataManager() NN_NOEXCEPT
    {
        return m_SoundDataManager;
    }

    nn::atk::SoundHandle& GetSoundHandle() NN_NOEXCEPT
    {
        return m_SoundHandle;
    }

    nn::atk::SoundHeap& GetSoundHeap() NN_NOEXCEPT
    {
        return m_SoundHeap;
    }

    AtkProfiler& GetAtkProfiler() NN_NOEXCEPT
    {
        return m_AtkProfiler;
    }

    SpyToolProfiler& GetSpyToolProfiler() NN_NOEXCEPT
    {
        return m_SpyToolProfiler;
    }

protected:
    virtual void OnPostStartSound() NN_NOEXCEPT;
    virtual void OnPreFinalizeSoundSystem() NN_NOEXCEPT;

private:
#if defined( NN_ATK_ENABLE_GFX_VIEWING )
    void DrawSoundCommonInfo() NN_NOEXCEPT;
    void DrawSoundHandleInfo() NN_NOEXCEPT;
#endif

    nn::atk::SoundHeap          m_SoundHeap;
    nn::atk::MemorySoundArchive m_MemorySoundArchive;
    nn::atk::FsSoundArchive     m_FsSoundArchive;
    nn::atk::SoundArchive*      m_pSoundArchive;
    nn::atk::SoundArchivePlayer m_SoundArchivePlayer;
    nn::atk::SoundDataManager   m_SoundDataManager;

    void* m_pMemoryForSoundSystem;
    void* m_pMemoryForSoundHeap;
    void* m_pMemoryForInfoBlock;
    void* m_pMemoryForSoundDataManager;
    void* m_pMemoryForSoundArchivePlayer;
    void* m_pMemoryForStreamBuffer;
    void* m_pMemoryForStreamCacheBuffer;
    void* m_pMemoryForStreamInstanceBuffer;
    void* m_pMemoryForMemorySoundArchive;
    void* m_pMemoryForSoundSystemMemoryPool;
    void* m_pMemoryForSoundSystemCircularBufferSink;

    nn::atk::SoundHandle        m_SoundHandle;
    bool m_IsPause;
    bool m_IsRendererSuspended;

#if defined(NN_ATK_CONFIG_ENABLE_DEV)
    nn::atk::viewer::SoundEdit m_SoundEdit;
    void* m_pMemoryForSoundEdit;
    void* m_pMemoryForLabelData;
    bool m_IsConnected;
#endif // NN_ATK_CONFIG_ENABLE_DEV

    SpyToolProfiler m_SpyToolProfiler;
    void* m_pMemoryForSpyToolProfiler;

    AtkProfiler m_AtkProfiler;
    void* m_pMemoryForAtkProfiler;
    static const int TaskThreadProfileCount = 32;
    static const int SoundThreadUpdateProfileCount = 32;
};

class UserEffectAux : public nn::atk::EffectAux
{
public:
    UserEffectAux() NN_NOEXCEPT
        : m_SamplePosition( 0 ), m_SampleRate( 32000 ), m_IsInitialized( false )
    {
    }
    bool SetSampleRate( int sampleRate ) NN_NOEXCEPT
    {
        if ( !m_IsInitialized )
        {
            m_SampleRate = sampleRate;
            return true;
        }
        return false;
    }
protected:
    virtual bool Initialize() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_LOG( "  EffectVolumeDown: Initialize().\n" );
        m_SamplePosition = 0;
        m_IsInitialized = true;
        return true;
    }
    virtual void Finalize() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_LOG( "  EffectVolumeDown: Finalize().\n" );
        m_IsInitialized = false;
    }
    virtual void UpdateSamples( int32_t* pSamples, const UpdateSamplesArg& arg ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_ABORT_UNLESS_EQUAL( arg.sampleRate, m_SampleRate );
        NN_ABORT_UNLESS_EQUAL( arg.readSampleCount % (arg.sampleCountPerAudioFrame * arg.channelCount), 0 );
        int readAudioFrameCount = arg.readSampleCount / (arg.sampleCountPerAudioFrame * arg.channelCount);
        NN_ABORT_UNLESS_LESS_EQUAL( readAudioFrameCount, arg.audioFrameCount );

        for ( auto frame = 0; frame < readAudioFrameCount; ++frame )
        {
            int32_t* bufferBase = pSamples + frame * arg.sampleCountPerAudioFrame * arg.channelCount;
            for ( auto ch = 0; ch < arg.channelCount; ++ch )
            {
                int32_t* pTargetSample = bufferBase + ch * arg.sampleCountPerAudioFrame;
                for ( auto i = 0; i < arg.sampleCountPerAudioFrame; ++i )
                {
                    // FL はサイン波形に置き換え、それ以外のチャンネルはゲインを下げる
                    if ( ch == 0 )
                    {
                        pTargetSample[i] = GenerateSineWaveSample( m_SamplePosition, m_SampleRate );
                        m_SamplePosition++;
                    }
                    else
                    {
                        pTargetSample[i] /= 4;
                    }
                }
            }
        }
    }
private:
    int16_t GenerateSineWaveSample( int samplePosition, int sampleRate ) NN_NOEXCEPT
    {
        const float Pi = 3.1415926535897932384626433f;
        const int Frequency = 440;
        return static_cast<int16_t>(std::numeric_limits<int16_t>::max() * nn::util::SinEst( 2 * Pi * Frequency * samplePosition / sampleRate ));
    }
    int m_SamplePosition;
    int m_SampleRate;
    bool m_IsInitialized;
};

