﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{AtkEffect.cpp,PageSampleAtkEffect}
 *
 * @brief
 *  nn::atk ライブラリのエフェクトの使い方を示すサンプルプログラム
 */

/**
 * @page PageSampleAtkEffect エフェクトを適用した再生
 * @tableofcontents
 *
 * @brief
 *  nn::atk ライブラリのエフェクトのサンプルプログラムの解説です。
 *
 * @section PageSampleAtkEffect_SectionBrief 概要
 *  Atk ライブラリのエフェクトを適用して、音声を再生するサンプルです。
 *  Delay や Reverb のエフェクト、またユーザ定義のエフェクトを適用することができます。
 *
 * @section PageSampleAtkSEffect_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/AtkEffect Samples/Sources/Applications/AtkEffect @endlink 以下にあります。
 *
 * @section PageSampleAtkEffect_SectionNecessaryEnvironment 必要な環境
 *  とくになし
 *
 * @section PageSampleAtkEffect_SectionHowToOperate 操作方法
 *  コンソールに操作方法が表示されます。
 *
 * @section PageSampleAtkEffect_SectionPrecaution 注意事項
 *  実行中のログが、画面上とコンソールに表示され、
 *  起動直後のログとして操作方法が表示されます。
 *
 *  PC 版ではウィンドウは立ち上がらず、コンソール表示のみになる点にご注意ください。
 *
 * @section PageSampleAtkEffect_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleAtkEffect_SectionDetail 解説
 *
 * @subsection PageSampleAtkEffect_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  AtkEffect.cpp
 *  @includelineno AtkEffect.cpp
 *
 * @subsection PageSampleAtkEffect_SectionSampleDetail サンプルプログラムの解説
 *  先のサンプルプログラムの Atk に関連している処理の流れは以下の通りです(AtkSimple と違う部分には追加要素と書かれています)。
 *
 *  - Atk ライブラリの初期化 (InitializeAtk())
 *    - AtkSimple で実行している初期化処理
 *  - SoundHeap にウェーブサウンドとシーケンスサウンドの再生に必要な情報をロード (LoadData())
 *  - メインループでの処理
 *    - キー操作による処理
 *      - サウンドの再生・停止処理を実行
 *        - エフェクト・メインのセンド量の設定(追加要素)
 *      - 適用するエフェクトの種類の変更(追加要素)
 *        - 適用するエフェクトの追加処理(AppendEffect())と削除処理(ClearEffect())(追加要素)
 *    - SoundArchivePlayer の更新
 *  - Atk ライブラリの終了処理 (FinalizeAtk())
 *
 *  キー操作を行うことで、サウンドに適用するエフェクトを変更することができます。
 *  適用できるエフェクトはディレイ(左右チャンネル, 右チャンネルのみの 2 種類)およびリバーブ、
 *  I3DL2 リバーブ、ユーザ定義のエフェクトの 5 種類となっています。
 *
 *  エフェクトを削除する場合は、まずエフェクトを無効化した後に(PrepareEffectRemove())
 *  エフェクトが削除可能である事を確認した上で(CheckEffectRemovable())、削除を行う必要があります。
 */

#include "Common.fsid"

#include <nn/atk.h>

#include <nns/atk/atk_SampleCommon.h>

#include <nns/nns_Log.h>

namespace
{
    const char Title[] = "AtkEffect";
    const char ArchiveRelativePath[] = "Common.bfsar";

    const int SoundHeapSize = 4 * 1024 * 1024;

    nn::atk::SoundHeap          g_SoundHeap;
    nn::atk::FsSoundArchive     g_SoundArchive;
    nn::atk::SoundArchivePlayer g_SoundArchivePlayer;
    nn::atk::SoundDataManager   g_SoundDataManager;

    nn::audio::MemoryPoolType   g_MemoryPool;

    void* g_pMemoryForSoundSystem;
    void* g_pMemoryForSoundHeap;
    void* g_pMemoryForInfoBlock;
    void* g_pMemoryForSoundDataManager;
    void* g_pMemoryForSoundArchivePlayer;
    void* g_pMemoryForStreamBuffer;

    nn::atk::SoundHandle        g_SoundHandle;

    nn::atk::EffectReverb       g_Reverb;
    nn::atk::EffectI3dl2Reverb  g_I3dl2Reverb;
    nn::atk::EffectDelay        g_Delay;
    nn::atk::EffectDelay        g_MonoDelay;
    nn::atk::EffectAux*         g_pAux;

    void* g_pMemoryForEffectDelay;
    void* g_pMemoryForEffectMonoDelay;
    void* g_pMemoryForEffectReverb;
    void* g_pMemoryForEffectI3dl2Reverb;
    void* g_pMemoryForEffectAux;

    enum EffectType
    {
        EffectType_None,
        EffectType_Reverb,
        EffectType_I3dl2Reverb,
        EffectType_Delay,
        EffectType_MonoRightDelay,
        EffectType_Aux,
        EffectType_Max = EffectType_Aux
    };

    class EffectVolumeDown : public nn::atk::EffectAux
    {
    public:
        virtual bool Initialize() NN_NOEXCEPT NN_OVERRIDE
        {
            NNS_LOG( "EffectVolumeDown: Initialize().\n" );
            return true;
        }
        virtual void Finalize() NN_NOEXCEPT NN_OVERRIDE
        {
            NNS_LOG( "EffectVolumeDown: Finalize().\n" );
        }
    protected:
        virtual void UpdateSamples( int32_t* pSamples, const UpdateSamplesArg& arg ) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_ABORT_UNLESS_EQUAL(arg.readSampleCount % (arg.sampleCountPerAudioFrame * arg.channelCount), 0);
            int readAudioFrameCount = arg.readSampleCount / (arg.sampleCountPerAudioFrame * arg.channelCount);
            NN_ABORT_UNLESS_LESS_EQUAL(readAudioFrameCount, arg.audioFrameCount);

            for (auto frame = 0; frame < readAudioFrameCount; ++frame)
            {
                int32_t* bufferBase = pSamples + frame * arg.sampleCountPerAudioFrame * arg.channelCount;
                for (auto ch = 0; ch < arg.channelCount; ++ch)
                {
                    int32_t* pTargetSample = bufferBase + ch * arg.sampleCountPerAudioFrame;
                    for (auto i = 0; i < arg.sampleCountPerAudioFrame; ++i)
                    {
                        pTargetSample[i] /= 4;
                    }
                }
            }
        }
    };

    int g_CurrentEffectType;
    int g_NextEffectType;
    bool g_IsEffectClearing;
    bool g_IsEffectChanging;
    EffectVolumeDown g_EffectVolumeDown;
}

void AppendEffect( int effectType )
{
    switch ( effectType )
    {
    case EffectType_Delay:
        {
            g_Delay.SetSampleRate(nn::atk::EffectBase::SampleRate_48000);
            g_Delay.SetChannelMode(nn::atk::EffectBase::ChannelMode_2Ch);
            g_Delay.SetDelayTimeMax(nn::TimeSpan::FromMilliSeconds(3000));
            g_Delay.SetFeedbackGain(0.6f);
            g_Delay.SetEnabled(true);

            size_t delayBufferSize = g_Delay.GetRequiredMemSize();
            // エフェクト用のバッファはメモリプール管理されているヒープから確保する必要があります。
            g_pMemoryForEffectDelay = nns::atk::AllocateForMemoryPool(delayBufferSize, nn::audio::BufferAlignSize);
            NN_ABORT_UNLESS_NOT_NULL(g_pMemoryForEffectDelay);

            bool isSuccess = nn::atk::SoundSystem::AppendEffect(nn::atk::AuxBus_A, &g_Delay, g_pMemoryForEffectDelay, delayBufferSize);
            NNS_LOG( "Delay  %d\n", isSuccess );
        }
        break;
    case EffectType_MonoRightDelay:
        {
            g_MonoDelay.SetSampleRate(nn::atk::EffectBase::SampleRate_48000);
            const nn::atk::ChannelIndex MonoDelayChannel[] = {nn::atk::ChannelIndex_FrontRight};
            g_MonoDelay.SetChannelIndex(MonoDelayChannel, nn::atk::EffectBase::ChannelMode_1Ch);
            g_MonoDelay.SetDelayTimeMax(nn::TimeSpan::FromMilliSeconds(3000));
            g_MonoDelay.SetDelayTime(nn::TimeSpan::FromMilliSeconds(400));
            g_MonoDelay.SetFeedbackGain(0.6f);
            g_MonoDelay.SetEnabled(true);

            size_t monoDelayBufferSize = g_MonoDelay.GetRequiredMemSize();
            // エフェクト用のバッファはメモリプール管理されているヒープから確保する必要があります。
            g_pMemoryForEffectMonoDelay = nns::atk::AllocateForMemoryPool(monoDelayBufferSize, nn::audio::BufferAlignSize);
            NN_ABORT_UNLESS_NOT_NULL(g_pMemoryForEffectMonoDelay);

            bool isSuccess = nn::atk::SoundSystem::AppendEffect(nn::atk::AuxBus_A, &g_MonoDelay, g_pMemoryForEffectMonoDelay, monoDelayBufferSize);
            NNS_LOG( "Delay(Mono Right)  %d\n", isSuccess );
        }
        break;
    case EffectType_Reverb:
        {
            g_Reverb.SetSampleRate(nn::atk::EffectBase::SampleRate_48000);
            g_Reverb.SetChannelMode(nn::atk::EffectBase::ChannelMode_2Ch);
            g_Reverb.SetEarlyMode(nn::atk::EffectReverb::EarlyMode_Hall);
            g_Reverb.SetLateMode(nn::atk::EffectReverb::LateMode_Hall);
            g_Reverb.SetLateGain(0.6f);
            g_Reverb.SetDecayTime(nn::TimeSpan::FromMilliSeconds(3000));
            g_Reverb.SetEnabled(true);

            size_t reverbBufferSize = g_Reverb.GetRequiredMemSize();
            // エフェクト用のバッファはメモリプール管理されているヒープから確保する必要があります。
            g_pMemoryForEffectReverb = nns::atk::AllocateForMemoryPool(reverbBufferSize, nn::audio::BufferAlignSize);
            NN_ABORT_UNLESS_NOT_NULL(g_pMemoryForEffectReverb);

            bool isSuccess = nn::atk::SoundSystem::AppendEffect(nn::atk::AuxBus_A, &g_Reverb, g_pMemoryForEffectReverb, reverbBufferSize);
            NNS_LOG( "Reverb %d\n", isSuccess );
        }
        break;
    case EffectType_I3dl2Reverb:
        {
            g_I3dl2Reverb.SetSampleRate(nn::atk::EffectBase::SampleRate_48000);
            g_I3dl2Reverb.SetChannelMode(nn::atk::EffectBase::ChannelMode_6Ch);
            g_I3dl2Reverb.SetEnabled(true);

            size_t i3dl2ReverbBufferSize = g_I3dl2Reverb.GetRequiredMemSize();
            // エフェクト用のバッファはメモリプール管理されているヒープから確保する必要があります。
            g_pMemoryForEffectI3dl2Reverb = nns::atk::AllocateForMemoryPool(i3dl2ReverbBufferSize, nn::audio::BufferAlignSize);
            NN_ABORT_UNLESS_NOT_NULL(g_pMemoryForEffectI3dl2Reverb);
            bool isSuccess = nn::atk::SoundSystem::AppendEffect(nn::atk::AuxBus_A, &g_I3dl2Reverb, g_pMemoryForEffectI3dl2Reverb, i3dl2ReverbBufferSize);
            NNS_LOG( "I3DL2 Reverb %d\n", isSuccess );
        }
        break;
    case EffectType_Aux:
        {
            g_EffectVolumeDown.SetAudioFrameCount(5);
            g_EffectVolumeDown.SetChannelCount(2);
            g_EffectVolumeDown.SetEnabled(true);
            g_pAux = &g_EffectVolumeDown;

            size_t auxBufferSize = nn::atk::SoundSystem::GetRequiredEffectAuxBufferSize(g_pAux);
            // エフェクト用のバッファはメモリプール管理されているヒープから確保する必要があります。
            g_pMemoryForEffectAux = nns::atk::AllocateForMemoryPool(auxBufferSize, nn::audio::BufferAlignSize);
            NN_ABORT_UNLESS_NOT_NULL(g_pMemoryForEffectAux);

            bool isSuccess = nn::atk::SoundSystem::AppendEffect(nn::atk::AuxBus_A, g_pAux, g_pMemoryForEffectAux, auxBufferSize);
            NNS_LOG("Aux    %d (bufferSize = %zu)\n", isSuccess, auxBufferSize);
        }
        break;
    case EffectType_None:
        NNS_LOG("None\n");
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
        break;
    }
}

void ClearEffect( int effectType )
{
    switch ( effectType )
    {
    case EffectType_Delay:
        nn::atk::SoundSystem::ClearEffect(nn::atk::AuxBus_A);
        nns::atk::FreeForMemoryPool(g_pMemoryForEffectDelay);
        break;
    case EffectType_MonoRightDelay:
        nn::atk::SoundSystem::ClearEffect(nn::atk::AuxBus_A);
        nns::atk::FreeForMemoryPool(g_pMemoryForEffectMonoDelay);
        break;
    case EffectType_Reverb:
        nn::atk::SoundSystem::ClearEffect(nn::atk::AuxBus_A);
        nns::atk::FreeForMemoryPool(g_pMemoryForEffectReverb);
        break;
    case EffectType_I3dl2Reverb:
        nn::atk::SoundSystem::ClearEffect(nn::atk::AuxBus_A);
        nns::atk::FreeForMemoryPool(g_pMemoryForEffectI3dl2Reverb);
        break;
    case EffectType_Aux:
        nn::atk::SoundSystem::ClearEffect(nn::atk::AuxBus_A);
        nns::atk::FreeForMemoryPool(g_pMemoryForEffectAux);
        g_pAux = nullptr;
        break;
    case EffectType_None:
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
        break;
    }
}

bool PrepareEffectRemove( int effectType )
{
    switch ( effectType )
    {
    case EffectType_Delay:
        g_Delay.SetEnabled(false);
        return true;
    case EffectType_MonoRightDelay:
        g_MonoDelay.SetEnabled(false);
        return true;
    case EffectType_Reverb:
        g_Reverb.SetEnabled(false);
        return true;
    case EffectType_I3dl2Reverb:
        g_I3dl2Reverb.SetEnabled(false);
        return true;
    case EffectType_Aux:
        NN_ABORT_UNLESS_NOT_NULL(g_pAux);
        g_pAux->SetEnabled(false);
        return true;
    case EffectType_None:
        return false;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

bool CheckEffectRemovable( int effectType )
{
    switch ( effectType )
    {
    case EffectType_Delay:
        return g_Delay.IsRemovable();
    case EffectType_MonoRightDelay:
        return g_MonoDelay.IsRemovable();
    case EffectType_Reverb:
        return g_Reverb.IsRemovable();
    case EffectType_I3dl2Reverb:
        return g_I3dl2Reverb.IsRemovable();
    case EffectType_Aux:
        NN_ABORT_UNLESS_NOT_NULL(g_pAux);
        return g_pAux->IsRemovable();
    case EffectType_None:
        return true;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

void Initialize()
{
    nns::atk::InitializeHeap();
    nns::atk::InitializeFileSystem();
    nns::atk::InitializeHidDevices();
}

void Finalize()
{
    nns::atk::FinalizeHidDevices();
    nns::atk::FinalizeFileSystem();
    nns::atk::FinalizeHeap();
}

void InitializeAtk()
{
    bool isSuccess = true;

    // SoundSystem の初期化
    nn::atk::SoundSystem::SoundSystemParam param;
    std::size_t memSizeForSoundSystem = nn::atk::SoundSystem::GetRequiredMemSize( param );
    g_pMemoryForSoundSystem = nns::atk::Allocate( memSizeForSoundSystem, nn::atk::SoundSystem::WorkMemoryAlignSize );
    nn::atk::SoundSystem::Initialize(
        param,
        reinterpret_cast<uintptr_t>( g_pMemoryForSoundSystem ),
        memSizeForSoundSystem );

    // SoundHeap の初期化
    g_pMemoryForSoundHeap = nns::atk::Allocate( SoundHeapSize );
    isSuccess = g_SoundHeap.Create( g_pMemoryForSoundHeap, SoundHeapSize );
    NN_ABORT_UNLESS( isSuccess, "cannot create SoundHeap" );

    // SoundArchive の初期化
    const char* archiveAbsolutePath = nns::atk::GetAbsolutePath(ArchiveRelativePath);
    isSuccess = g_SoundArchive.Open(archiveAbsolutePath);
    NN_ABORT_UNLESS( isSuccess, "cannot open SoundArchive(%s)\n", archiveAbsolutePath );

    // SoundArchive のパラメータ情報をメモリにロード
    std::size_t infoBlockSize = g_SoundArchive.GetHeaderSize();
    g_pMemoryForInfoBlock = nns::atk::Allocate( infoBlockSize, nn::atk::FsSoundArchive::BufferAlignSize );
    isSuccess = g_SoundArchive.LoadHeader( g_pMemoryForInfoBlock, infoBlockSize );
    NN_ABORT_UNLESS( isSuccess, "cannot load InfoBlock" );

    // SoundDataManager の初期化
    std::size_t memSizeForSoundDataManager = g_SoundDataManager.GetRequiredMemSize( &g_SoundArchive );
    g_pMemoryForSoundDataManager = nns::atk::Allocate( memSizeForSoundDataManager, nn::atk::SoundDataManager::BufferAlignSize );
    isSuccess = g_SoundDataManager.Initialize(
        &g_SoundArchive,
        g_pMemoryForSoundDataManager,
        memSizeForSoundDataManager );
    NN_ABORT_UNLESS( isSuccess, "cannot initialize SoundDataManager" );

    // SoundArchivePlayer で用いるストリームバッファの初期化
    // ストリームバッファはメモリプール管理されているヒープから確保する必要があります。
    std::size_t memSizeForStreamBuffer = g_SoundArchivePlayer.GetRequiredStreamBufferSize( &g_SoundArchive );
    g_pMemoryForStreamBuffer = nns::atk::AllocateForMemoryPool(memSizeForStreamBuffer);

    // 専用のヒープをメモリプールにアタッチ
    nn::atk::SoundSystem::AttachMemoryPool(&g_MemoryPool, nns::atk::GetPoolHeapAddress(), nns::atk::GetPoolHeapSize());

    // SoundArchivePlayer の初期化
    std::size_t memSizeForSoundArchivePlayer = g_SoundArchivePlayer.GetRequiredMemSize( &g_SoundArchive );
    g_pMemoryForSoundArchivePlayer = nns::atk::Allocate( memSizeForSoundArchivePlayer, nn::atk::SoundArchivePlayer::BufferAlignSize );
    isSuccess = g_SoundArchivePlayer.Initialize(
        &g_SoundArchive,
        &g_SoundDataManager,
        g_pMemoryForSoundArchivePlayer, memSizeForSoundArchivePlayer,
        g_pMemoryForStreamBuffer, memSizeForStreamBuffer );
    NN_ABORT_UNLESS( isSuccess, "cannot initialize SoundArchivePlayer" );

    g_CurrentEffectType = EffectType_None;
    g_NextEffectType = EffectType_None;
    g_IsEffectClearing = false;
    g_IsEffectChanging = false;
}

void FinalizeAtk()
{
    // エフェクトの解放処理
    if ( !g_IsEffectClearing )
    {
        g_IsEffectClearing = PrepareEffectRemove( g_CurrentEffectType );
    }
    if ( g_IsEffectClearing )
    {
        // エフェクトが削除可能になるまで待つ
        for ( ;; )
        {
            if ( CheckEffectRemovable( g_CurrentEffectType ) )
            {
                ClearEffect( g_CurrentEffectType );
                g_IsEffectClearing = false;
                break;
            }
            g_SoundArchivePlayer.Update();
        }
    }

    g_SoundArchivePlayer.Finalize();

    // 専用のヒープをメモリプールからデタッチ
    nn::atk::SoundSystem::DetachMemoryPool(&g_MemoryPool);

    g_SoundDataManager.Finalize();
    g_SoundArchive.Close();
    g_SoundHeap.Destroy();
    nn::atk::SoundSystem::Finalize();

    nns::atk::FreeForMemoryPool(g_pMemoryForStreamBuffer);
    nns::atk::Free(g_pMemoryForSoundArchivePlayer);
    nns::atk::Free(g_pMemoryForSoundDataManager);
    nns::atk::Free(g_pMemoryForInfoBlock);
    nns::atk::Free(g_pMemoryForSoundHeap);
    nns::atk::Free(g_pMemoryForSoundSystem);
}

void LoadData()
{
    bool isSuccess = true;

    isSuccess = g_SoundDataManager.LoadData( SEQ_MARIOKART, &g_SoundHeap );
    NN_ABORT_UNLESS( isSuccess, "LoadData(SEQ_MARIOKART) failed." );

    isSuccess = g_SoundDataManager.LoadData( SE_YOSHI, &g_SoundHeap );
    NN_ABORT_UNLESS( isSuccess, "LoadData(SE_YOSHI) failed." );
}

void PlayWithStartSound(nn::atk::SoundArchive::ItemId soundId, const char* debugLabelName)
{
    g_SoundHandle.Stop( 0 );

    bool result = g_SoundArchivePlayer.StartSound( &g_SoundHandle, soundId ).IsSuccess();
    NNS_LOG("StartSound(%s) ... (%d)\n", debugLabelName, result);

    g_SoundHandle.SetEffectSend(nn::atk::AuxBus_A, 0.8f);
    g_SoundHandle.SetEffectSend(nn::atk::AuxBus_B, 0.0f);
    g_SoundHandle.SetEffectSend(nn::atk::AuxBus_C, 0.0f);
    g_SoundHandle.SetMainSend(-0.8f);
}

void PrintUsage()
{
    NNS_LOG("--------------------------------------------------\n");
    NNS_LOG("%s Sample\n", Title);
    NNS_LOG("--------------------------------------------------\n");
    NNS_LOG("[A]              StartSound SEQ  (SEQ_MARIOKART)\n");
    NNS_LOG("[X]              StartSound WSD  (SE_YOSHI)\n");
    NNS_LOG("[Y]              StartSound STRM (STRM_MARIOKART)\n");
    NNS_LOG("[B]              Stop Sound\n");
    NNS_LOG("[RIGHT/LEFT]     Change EffectType\n");
    NNS_LOG("[L]              Print Usage\n");
    NNS_LOG("[+/Start][Space] Exit Application\n");
    NNS_LOG("--------------------------------------------------\n");
}

bool UpdateAtk()
{
    // StartSound / Stop
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::A >() )
    {
        PlayWithStartSound(SEQ_MARIOKART, "SEQ_MARIOKART");
    }
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::X >() )
    {
        PlayWithStartSound(SE_YOSHI, "SE_YOSHI");
    }
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Y >() )
    {
        PlayWithStartSound(STRM_MARIOKART, "STRM_MARIOKART");
    }
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::B >() )
    {
        g_SoundHandle.Stop( 0 );
    }

    // Change EffectType
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Left >() )
    {
        if ( !g_IsEffectClearing )
        {
            g_IsEffectClearing = PrepareEffectRemove( g_CurrentEffectType );

            g_NextEffectType -= 1;
            if ( g_NextEffectType < 0 )
            {
                g_NextEffectType = EffectType_Max;
            }
            g_IsEffectChanging = true;
        }
    }
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Right >() )
    {
        if ( !g_IsEffectClearing )
        {
            g_IsEffectClearing = PrepareEffectRemove( g_CurrentEffectType );

            g_NextEffectType += 1;
            if ( g_NextEffectType > EffectType_Max )
            {
                g_NextEffectType = 0;
            }
            g_IsEffectChanging = true;
        }
    }

    // Print Usage
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::L >() )
    {
        PrintUsage();
    }

    // Exit
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Start >() )
    {
        return false;
    }

    g_SoundArchivePlayer.Update();

    // エフェクトが削除可能かをチェック
    if ( g_IsEffectClearing )
    {
        if ( CheckEffectRemovable( g_CurrentEffectType ) )
        {
            ClearEffect( g_CurrentEffectType );
            g_IsEffectClearing = false;
        }
    }

    // 次のエフェクトに変更する
    if ( g_IsEffectChanging && !g_IsEffectClearing )
    {
        AppendEffect( g_NextEffectType );
        g_CurrentEffectType = g_NextEffectType;
        g_IsEffectChanging = false;
    }

    return true;
}

extern "C" void nnMain()
{
    Initialize();
    InitializeAtk();

    LoadData();

    PrintUsage();

    for ( ;; )
    {
        nns::atk::UpdateHidDevices();

        if ( !UpdateAtk() )
        {
            break;
        }

        // Vsync の代わり
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
    }

    FinalizeAtk();
    Finalize();
}

