﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_FormatString.h>
#include "DebugViewer.h"
#if defined( NN_BUILD_CONFIG_OS_WIN )
#define     WIN32_LEAN_AND_MEAN //  nn_Windows.h の警告対策
#define     NOMINMAX            //  nn_Windows.h の警告対策
#include <nn/nn_Windows.h>
#endif

const float DebugViewer::DefaultPrintScaleX = 0.6f;
const float DebugViewer::DefaultPrintScaleY = 0.8f;

NN_DEFINE_STATIC_CONSTANT( const int DebugViewer::DefaultDrawBarWidth );
NN_DEFINE_STATIC_CONSTANT( const int DebugViewer::DefaultDrawBarHeight );

namespace
{
    //  ディスプレイの幅と高さ
    const int DisplayWidth = 1280;
    const int DisplayHeight = 720;

    //  GfxContext に割り当てるメモリとそのサイズ
    const size_t GfxContextMemorySize = 16 * 1024 * 1024;
    char g_GfxContextMemory[GfxContextMemorySize];

    //  DebugPrimitiveViewer のフォント倍率
    const float DebugPrimitiveViewerFontScale = 0.75f;

    //  DebugPrintViewer のデフォルトの文字列描画位置
    const float DebugPrintViewerDefaultPositionX = 720;
    //  DebugPrintViewer の Y 方向の位置調整量
    const float DebugPrintViewerDefaultPositionDown= 27;
    //  DebugPrintViewer の文字の色
    const nn::util::Unorm8x4 DebugPrintViewerFontColor = { { 255, 255, 255, 255 } };
    //  DebugPrintViewer の文字の背景色
    const nn::util::Uint8x4 DebugPrintViewerBackColor = { { 16, 16, 16, 255 } };
    //  DebugPrintViewer のフォント倍率
    const float DebugPrintViewerFontScale = 1.25f;
    //  DebugPrintViewer のフォーマット指定文字列の最大の長さ
    const int FormatStringMaxLength = 512;
}

//  class DebugPrimitiveViewer
DebugPrimitiveViewer::DebugPrimitiveViewer() NN_NOEXCEPT
    : m_DataChangeMutex( false )
{
}
void DebugPrimitiveViewer::Initialize() NN_NOEXCEPT
{
    Reset();
}
void DebugPrimitiveViewer::Erase(const char* tag) NN_NOEXCEPT
{
    //  SoundThread, TaskThread から呼ばれてもいいように排他制御を行います
    {
        nn::atk::detail::fnd::ScopedLock<nn::os::Mutex> lock( m_DataChangeMutex );
        m_PrimitiveData.erase(tag);
    }
}
void DebugPrimitiveViewer::Reset() NN_NOEXCEPT
{
    //  SoundThread, TaskThread から呼ばれてもいいように排他制御を行います
    {
        nn::atk::detail::fnd::ScopedLock<nn::os::Mutex> lock( m_DataChangeMutex );
        m_PrimitiveData.clear();
    }
}
void DebugPrimitiveViewer::Draw(GfxContext& gfxContext) NN_NOEXCEPT
{
    auto& fontRenderer = gfxContext.GetFontRenderer();
    float centerX = 0, centerY = 0;

    fontRenderer.SetScale( DebugPrimitiveViewerFontScale, DebugPrimitiveViewerFontScale );

    //  SoundThread, TaskThread から呼ばれてもいいように排他制御を行います
    {
        nn::atk::detail::fnd::ScopedLock<nn::os::Mutex> lock( m_DataChangeMutex );
        for(auto itr = m_PrimitiveData.begin(); itr != m_PrimitiveData.end(); ++itr)
        {
            const auto& data = itr->second;
            switch( data.type )
            {
                case PrimitiveType::PrimitiveType_Quad:
                {
                    const auto& quad = data.quad;
                    gfxContext.DrawQuad( quad.left, quad.top, quad.right, quad.bottom, data.color );
                    centerX = ( quad.left + quad.right ) * 0.5f;
                    centerY = ( quad.top + quad.bottom ) * 0.5f;
                    break;
                }
                case PrimitiveType::PrimitiveType_Line:
                {
                    const auto& line = data.line;
                    gfxContext.DrawLine( line.x1, line.y1, line.x2, line.y2, data.color );
                    centerX = ( line.x1 + line.x2 ) * 0.5f;
                    centerY = ( line.y1 + line.y2 ) * 0.5f;
                    break;
                }
                case PrimitiveType::PrimitiveType_Circle:
                {
                    const auto& circle = data.circle;
                    gfxContext.DrawCircle( circle.x, circle.y, circle.r, data.color );
                    centerX = circle.x;
                    centerY = circle.y;
                    break;
                }
                default: NN_UNEXPECTED_DEFAULT;
            }

            const float width = fontRenderer.CalculateWidth( data.tag.c_str() );
            const float height = fontRenderer.CalculateHeight( data.tag.c_str() );
            fontRenderer.SetPosition( centerX - 0.5f * width, centerY - 0.5f * height );
            fontRenderer.Print( data.tag.c_str() );
        }
    }
    fontRenderer.SetScale( 1.0f, 1.0f );
}
void DebugPrimitiveViewer::DrawQuad(const char* tag, float left, float top, float right, float bottom, uint32_t color) NN_NOEXCEPT
{
    PrimitiveData data;
    data.type = PrimitiveType::PrimitiveType_Quad;
    data.color = color;
    data.tag = tag;
    data.quad.left = left;
    data.quad.top = top;
    data.quad.right = right;
    data.quad.bottom = bottom;

    //  SoundThread, TaskThread から呼ばれてもいいように排他制御を行います
    {
        nn::atk::detail::fnd::ScopedLock<nn::os::Mutex> lock( m_DataChangeMutex );
        m_PrimitiveData[data.tag] = data;
    }
}
void DebugPrimitiveViewer::DrawLine(const char* tag, float x1, float y1, float x2, float y2, uint32_t color) NN_NOEXCEPT
{
    PrimitiveData data;
    data.type = PrimitiveType::PrimitiveType_Line;
    data.color = color;
    data.tag = tag;
    data.line.x1 = x1;
    data.line.y1 = y1;
    data.line.x2 = x2;
    data.line.y2 = y2;

    //  SoundThread, TaskThread から呼ばれてもいいように排他制御を行います
    {
        nn::atk::detail::fnd::ScopedLock<nn::os::Mutex> lock( m_DataChangeMutex );
        m_PrimitiveData[data.tag] = data;
    }
}
void DebugPrimitiveViewer::DrawCircle(const char* tag, float x, float y, float r, uint32_t color) NN_NOEXCEPT
{
    PrimitiveData data;
    data.type = PrimitiveType::PrimitiveType_Circle;
    data.color = color;
    data.tag = tag;
    data.circle.x = x;
    data.circle.y = y;
    data.circle.r = r;

    //  SoundThread, TaskThread から呼ばれてもいいように排他制御を行います
    {
        nn::atk::detail::fnd::ScopedLock<nn::os::Mutex> lock( m_DataChangeMutex );
        m_PrimitiveData[data.tag] = data;
    }
}


//  class DebugPrintViewer
DebugPrintViewer::DebugPrintViewer() NN_NOEXCEPT
    : m_DataChangeMutex( true )
{
}
void DebugPrintViewer::Initialize() NN_NOEXCEPT
{
    Reset();
}
void DebugPrintViewer::Reset() NN_NOEXCEPT
{
    //  SoundThread, TaskThread から呼ばれてもいいように排他制御を行います
    {
        nn::atk::detail::fnd::ScopedLock<nn::os::Mutex> lock( m_DataChangeMutex );
        m_PrintData.clear();
    }
    m_PrintPositionX = DebugPrintViewerDefaultPositionX;
    m_PrintPositionY = 0;
}
void DebugPrintViewer::Draw(GfxContext& gfxContext) NN_NOEXCEPT
{
    auto& fontRenderer = gfxContext.GetFontRenderer();

    //  SoundThread, TaskThread から呼ばれてもいいように排他制御を行います
    {
        nn::atk::detail::fnd::ScopedLock<nn::os::Mutex> lock( m_DataChangeMutex );
        for(auto itr = m_PrintData.begin(); itr != m_PrintData.end(); itr++)
        {
            const auto& data = itr->second;
            fontRenderer.SetScale( data.scaleX, data.scaleY );
            gfxContext.DrawQuad( data.x, data.y, data.x + fontRenderer.CalculateWidth( data.str.c_str() ), data.y + fontRenderer.CalculateHeight( data.str.c_str() ), data.backColor );
            fontRenderer.SetColor( data.fontColor );
            fontRenderer.SetPosition( data.x, data.y );
            fontRenderer.Print( data.str.c_str() );
        }
    }

    fontRenderer.SetScale( 1, 1 );
}
void DebugPrintViewer::Print(const char* tag, const char* fmt, va_list args) NN_NOEXCEPT
{
    nn::atk::detail::fnd::ScopedLock<nn::os::Mutex> lock( m_DataChangeMutex );
    const auto itr = m_PrintData.find( tag );
    float x, y;

    if( itr == m_PrintData.end() )
    {
        x = m_PrintPositionX;
        y = m_PrintPositionY;
        m_PrintPositionY += DebugPrintViewerDefaultPositionDown;
    }
    else
    {
        x = itr->second.x;
        y = itr->second.y;
    }

    Print( tag, x, y, fmt, args );
}
void DebugPrintViewer::Print(const char* tag, float x, float y, const char* fmt, va_list args) NN_NOEXCEPT
{
    Print(tag, x, y, DebugPrintViewerFontScale, DebugPrintViewerFontScale, fmt, args);
}
void DebugPrintViewer::Print(const char* tag, float x, float y, float scaleX, float scaleY, const char* fmt, va_list args) NN_NOEXCEPT
{
    Print(tag, x, y, scaleX, scaleY, DebugPrintViewerFontColor, DebugPrintViewerBackColor, fmt, args);
}
void DebugPrintViewer::Print(const char* tag, float x, float y, float scaleX, float scaleY, const nn::util::Unorm8x4& fontColor, const nn::util::Uint8x4& backColor, const char* fmt, va_list args) NN_NOEXCEPT
{
    const std::string tagStr = tag;
    char buf[FormatStringMaxLength];
    const int length = nn::util::VSNPrintf( buf, FormatStringMaxLength, fmt, args );

    PrintData data;
    if( length < FormatStringMaxLength )
    {
        data.str = tagStr + ": " + buf;
    }
    else
    {
        data.str = tagStr + ": buffer over";
    }

    data.x = x;
    data.y = y;
    data.scaleX = scaleX;
    data.scaleY = scaleY;
    data.fontColor = fontColor;
    data.backColor = backColor;

    //  SoundThread, TaskThread から呼ばれてもいいように排他制御を行います
    {
        nn::atk::detail::fnd::ScopedLock<nn::os::Mutex> lock( m_DataChangeMutex );
        m_PrintData[tagStr] = data;
    }
}


//  class DebugViewer
DebugViewer::DebugViewer() NN_NOEXCEPT
{
}
void DebugViewer::Initialize() NN_NOEXCEPT
{
    m_GfxContext.Initialize( g_GfxContextMemory, GfxContextMemorySize, DisplayWidth, DisplayHeight );
    m_PrimitiveViewer.Initialize();
    m_PrintViewer.Initialize();

#if defined( NN_BUILD_CONFIG_OS_WIN )
    //  作成されたウィンドウよりもコンソールが前に表示されるようにします
    HWND hWnd = GetConsoleWindow();
    if( hWnd != NULL )
    {
        SetWindowPos( hWnd, HWND_TOP, 0, 0, 0, 0, SWP_NOSIZE | SWP_NOMOVE );
    }
#endif

    Present();
}
void DebugViewer::Finalize() NN_NOEXCEPT
{
    m_GfxContext.Finalize();
}
void DebugViewer::ErasePrimitive(const char* tag) NN_NOEXCEPT
{
    m_PrimitiveViewer.Erase(tag);
}
void DebugViewer::Reset() NN_NOEXCEPT
{
    m_PrimitiveViewer.Reset();
    m_PrintViewer.Reset();
}
void DebugViewer::Update() NN_NOEXCEPT
{
}
void DebugViewer::Present() NN_NOEXCEPT
{
    m_GfxContext.Begin();
    m_PrimitiveViewer.Draw( m_GfxContext );
    m_PrintViewer.Draw( m_GfxContext );
    m_GfxContext.End();
}
void DebugViewer::Print(const char* tag, const char* fmt, ...) NN_NOEXCEPT
{
    va_list args;
    va_start( args, fmt );
    PrintImpl( tag, fmt, args );
    va_end( args );
}
void DebugViewer::Print(const char* tag, int x, int y, const char* fmt, ...) NN_NOEXCEPT
{
    va_list args;
    va_start( args, fmt );
    PrintImpl( tag, x, y, fmt, args );
    va_end( args );
}
void DebugViewer::Print(const char* tag, int x, int y, float scaleX, float scaleY, const char* fmt, ...) NN_NOEXCEPT
{
    va_list args;
    va_start( args, fmt );
    PrintImpl( tag, x, y, scaleX, scaleY, fmt, args );
    va_end( args );
}
void DebugViewer::Print(const char* tag, int x, int y, float scaleX, float scaleY, const nn::util::Unorm8x4& fontColor, const nn::util::Uint8x4& backColor, const char* fmt, ...) NN_NOEXCEPT
{
    va_list args;
    va_start( args, fmt );
    PrintImpl( tag, x, y, scaleX, scaleY, fontColor, backColor, fmt, args );
    va_end( args );
}
void DebugViewer::PrintImpl(const char* tag, const char* fmt, va_list args) NN_NOEXCEPT
{
    m_PrintViewer.Print( tag, fmt, args );
}
void DebugViewer::PrintImpl(const char* tag, int x, int y, const char* fmt, va_list args) NN_NOEXCEPT
{
    m_PrintViewer.Print( tag, static_cast<float>(x), static_cast<float>(y), fmt, args );
}
void DebugViewer::PrintImpl(const char* tag, int x, int y, float scaleX, float scaleY, const char* fmt, va_list args) NN_NOEXCEPT
{
    m_PrintViewer.Print( tag, static_cast<float>(x), static_cast<float>(y), scaleX, scaleY, fmt, args );
}
void DebugViewer::PrintImpl(const char* tag, int x, int y, float scaleX, float scaleY, const nn::util::Unorm8x4& fontColor, const nn::util::Uint8x4& backColor, const char* fmt, va_list args) NN_NOEXCEPT
{
    m_PrintViewer.Print( tag, static_cast<float>(x), static_cast<float>(y), scaleX, scaleY, fontColor, backColor, fmt, args );
}
void DebugViewer::DrawQuad(const char* tag, int left, int top, int right, int bottom, uint32_t color) NN_NOEXCEPT
{
    m_PrimitiveViewer.DrawQuad( tag, static_cast<float>(left), static_cast<float>(top), static_cast<float>(right), static_cast<float>(bottom), color );
}
void DebugViewer::DrawLine(const char* tag, int x1, int y1, int x2, int y2, uint32_t color) NN_NOEXCEPT
{
    m_PrimitiveViewer.DrawLine( tag, static_cast<float>(x1), static_cast<float>(y1), static_cast<float>(x2), static_cast<float>(y2), color );
}
void DebugViewer::DrawCircle(const char* tag, int x, int y, int r, uint32_t color) NN_NOEXCEPT
{
    m_PrimitiveViewer.DrawCircle( tag, static_cast<float>(x), static_cast<float>(y), static_cast<float>(r), color );
}
DebugViewer& DebugViewer::GetInstance() NN_NOEXCEPT
{
    static DebugViewer instance;
    return instance;
}


void DebugPrint(const char* tag, const char* fmt, ...) NN_NOEXCEPT
{
    va_list args;
    va_start( args, fmt );
    DebugViewer::GetInstance().PrintImpl( tag, fmt, args );
    va_end( args );
}
void DebugPrint(const char* tag, int x, int y, const char* fmt, ...) NN_NOEXCEPT
{
    va_list args;
    va_start( args, fmt );
    DebugViewer::GetInstance().PrintImpl( tag, x, y, fmt, args );
    va_end( args );
}
void DebugPrint(const char* tag, int x, int y, float scaleX, float scaleY, const char* fmt, ...) NN_NOEXCEPT
{
    va_list args;
    va_start( args, fmt );
    DebugViewer::GetInstance().PrintImpl( tag, x, y, scaleX, scaleY, fmt, args );
    va_end( args );
}
void DebugPrint(const char* tag, int x, int y, float scaleX, float scaleY, const nn::util::Unorm8x4& fontColor, const nn::util::Uint8x4& backColor, const char* fmt, ...) NN_NOEXCEPT
{
    va_list args;
    va_start( args, fmt );
    DebugViewer::GetInstance().PrintImpl( tag, x, y, scaleX, scaleY, fontColor, backColor, fmt, args );
    va_end( args );
}

void DebugDrawQuad(const char* tag, int left, int top, int right, int bottom, uint32_t color) NN_NOEXCEPT
{
    DebugViewer::GetInstance().DrawQuad( tag, left, top, right, bottom, color );
}
void DebugDrawLine(const char* tag, int x1, int y1, int x2, int y2, uint32_t color) NN_NOEXCEPT
{
    DebugViewer::GetInstance().DrawLine( tag, x1, y1, x2, y2, color );
}
void DebugDrawCircle(const char* tag, int x, int y, int r, uint32_t color) NN_NOEXCEPT
{
    DebugViewer::GetInstance().DrawCircle( tag, x, y, r, color );
}

uint32_t GetDebugColor(uint8_t r, uint8_t g, uint8_t b) NN_NOEXCEPT
{
    return GetDebugColor( r, g, b, 255 );
}
uint32_t GetDebugColor(uint8_t r, uint8_t g, uint8_t b, uint8_t a) NN_NOEXCEPT
{
    return ( a << 24 ) | ( r << 16 ) | ( g << 8 ) | b;
}
