﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Atk2SandboxCommon.h"

#include "FlagList.h"
#include "GfxCode/DebugViewer.h"

namespace
{
    nn::audio::MemoryPoolType g_MemoryPool;

    // モジュール名表示のデフォルトの文字列描画位置
    const int DefaultPrintModuleNamePosX = 800;
    const int DefaultPrintModuleNamePosY = 0;
    // 操作説明のデフォルトの文字列描画位置
    const int DefaultPrintUsagePosX = 800;
    const int DefaultPrintUsagePosY = 20;
    // オーディオエンジン情報のデフォルトの文字列描画位置
    const int DefaultPrintAudioEngineInfoPosX = 800;
    const int DefaultPrintAudioEngineInfoPosY = 600;

    void DrawGeneralUsage() NN_NOEXCEPT
    {
        DebugPrint("Common_Usage", DefaultPrintUsagePosX, DefaultPrintUsagePosY, DebugViewer::DefaultPrintScaleX, DebugViewer::DefaultPrintScaleY,
            "\n"
        );
    }
}

CommonObject::InitArg::InitArg() NN_NOEXCEPT
: m_AudioEngineInitArg()
, m_AudioEnginePlatformInitArg()
{
}

nn::atk2::AudioEngine::InitArg& CommonObject::InitArg::GetAudioEngineInitArg() NN_NOEXCEPT
{
    return m_AudioEngineInitArg;
}

const nn::atk2::AudioEngine::InitArg& CommonObject::InitArg::GetAudioEngineInitArg() const NN_NOEXCEPT
{
    return m_AudioEngineInitArg;
}

nn::atk2::AudioEngine::PlatformInitArg& CommonObject::InitArg::GetAudioEnginePlatformInitArg() NN_NOEXCEPT
{
    return m_AudioEnginePlatformInitArg;
}

const nn::atk2::AudioEngine::PlatformInitArg& CommonObject::InitArg::GetAudioEnginePlatformInitArg() const NN_NOEXCEPT
{
    return m_AudioEnginePlatformInitArg;
}

CommonObject::CommonObject() NN_NOEXCEPT
{
}

CommonObject::~CommonObject() NN_NOEXCEPT
{
}

void CommonObject::Initialize()
{
    CommonObject::InitArg arg;
    m_AudioEngine.GetDefaultInitArg(arg.GetAudioEngineInitArg(), arg.GetAudioEnginePlatformInitArg());
    Initialize(arg);
}

void CommonObject::Initialize(const CommonObject::InitArg& arg) NN_NOEXCEPT
{
    // AudioEngine の初期化
    nn::atk2::AudioEngine::InitArg initArg = arg.GetAudioEngineInitArg();
    nn::atk2::AudioEngine::PlatformInitArg platformInitArg = arg.GetAudioEnginePlatformInitArg();

    if ( GetGlobalFlagList().IsFlagEnabled(GlobalFlagType_DisableAudioThread) )
    {
        initArg.isAudioThreadEnabled = false;
    }
    if ( GetGlobalFlagList().IsFlagEnabled(GlobalFlagType_ManualAudioFrameUpdate) )
    {
        initArg.isAudioThreadEnabled = false;
        initArg.isManualAudioFrameUpdateEnabled = true;
    }

    const size_t bufferSizeForAudioEngine = m_AudioEngine.GetRequiredBufferSize(initArg, platformInitArg);
    m_BufferForAudioEngine = nns::atk::Allocate( bufferSizeForAudioEngine, nn::os::MemoryPageSize );
    initArg.workBuffer = m_BufferForAudioEngine;
    initArg.workBufferSize = bufferSizeForAudioEngine;

    const size_t platformBufferSizeForAudioEngine = m_AudioEngine.GetRequiredPlatformBufferSize(initArg, platformInitArg);
    m_PlatformBufferForAudioEngine = nns::atk::Allocate( platformBufferSizeForAudioEngine );
    platformInitArg.workBuffer = m_PlatformBufferForAudioEngine;
    platformInitArg.workBufferSize = platformBufferSizeForAudioEngine;

    const size_t bufferSizeMemoryPoolForAudioEngine = m_AudioEngine.GetRequiredBufferSizeForMemoryPool(initArg, platformInitArg);
    m_BufferMemoryPoolForAudioEngine = nns::atk::Allocate( bufferSizeMemoryPoolForAudioEngine, nn::audio::MemoryPoolType::AddressAlignment );
    platformInitArg.workBufferForMemoryPool = m_BufferMemoryPoolForAudioEngine;
    platformInitArg.workBufferSizeForMemoryPool = bufferSizeMemoryPoolForAudioEngine;

    nn::atk2::AudioEngine::Result result = m_AudioEngine.Initialize(initArg, platformInitArg);
    if (result != nn::atk2::AudioEngine::Result_Success)
    {
        NN_ABORT_UNLESS( false, "Cannot initialize AudioEngine" );
    }

    // 専用のヒープをメモリプールにアタッチ
    nn::audio::AcquireMemoryPool( &m_AudioEngine.GetRendererManager().GetAudioRendererConfig(), &g_MemoryPool, nns::atk::GetPoolHeapAddress(), nns::atk::GetPoolHeapSize() );
    nn::audio::RequestAttachMemoryPool(&g_MemoryPool);

} // NOLINT(impl/function_size)

void CommonObject::Finalize() NN_NOEXCEPT
{
    m_AudioEngine.Finalize();
    nns::atk::Free( m_BufferForAudioEngine );
}

void CommonObject::LoadData() NN_NOEXCEPT
{
}

void CommonObject::Update() NN_NOEXCEPT
{
    nn::atk2::AudioEngine::Result result = m_AudioEngine.Update();
    if (result != nn::atk2::AudioEngine::Result_Success)
    {
        NN_ASSERT(false, "Failed to update Audio Engine.");
    }
    if ( GetGlobalFlagList().IsFlagEnabled(GlobalFlagType_ManualAudioFrameUpdate) )
    {
        if (m_AudioEngine.GetAudioFrameUpdater().Update() != nn::atk2::detail::AudioFrameUpdater::Result_Success)
        {
            NN_ASSERT(false, "Failed to update Audio Frame Updater.");
        }
    }
}

void CommonObject::UpdateDraw(const char* moduleName) NN_NOEXCEPT
{
    // モジュール名の表示
    DebugPrint("ModuleName", DefaultPrintModuleNamePosX, DefaultPrintModuleNamePosY, DebugViewer::DefaultPrintScaleX, DebugViewer::DefaultPrintScaleY, moduleName);

    // 操作説明の表示
    DrawGeneralUsage();

    DebugPrint("AudioEngine", DefaultPrintAudioEngineInfoPosX, DefaultPrintAudioEngineInfoPosY, DebugViewer::DefaultPrintScaleX, DebugViewer::DefaultPrintScaleY,
        "\n"
        "  MaxVoiceCount:        %8d\n"
        "  UsingVoiceCount:      %8d\n"
        "  VirtualVoiceCount:    %8d\n"
        "  UnReleasedVoiceCount: %8d\n"
        , m_AudioEngine.GetRendererVoiceManager().GetVoiceCount()
        , m_AudioEngine.GetRendererVoiceManager().GetUsingVoiceCount()
        , m_AudioEngine.GetVirtualVoiceManager().GetAllocatedVirtualVoiceCount()
        , m_AudioEngine.GetVirtualVoiceManager().GetUnreleasedRendererVoiceCount()
    );
}

void CommonObject::PrintUsage() NN_NOEXCEPT
{
}

int CommonObject::UpdateInput() NN_NOEXCEPT
{
    int processCount = 0;

    if (nns::atk::IsTrigger< ::nn::hid::DebugPadButton::A >())
    {
        if (nns::atk::IsHold< ::nn::hid::DebugPadButton::R >())
        {
            NN_LOG("Pressed button A with R.\n");
            processCount++;
        }
        else
        {
            NN_LOG("Pressed button A.\n");
            processCount++;
        }
    }
    return processCount;
}

nn::atk2::AudioEngine& CommonObject::GetAudioEngine() NN_NOEXCEPT
{
    return m_AudioEngine;
}
