﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
    @examplesource{AlbumMovieMakerSimple.cpp,PageSampleAlbumMovieMakerSimple}

    @brief @copybrief PageSampleAlbumMovieMakerSimple
 */

/**
    @page PageSampleAlbumMovieMakerSimple アプリケーションによる動画撮影処理のサンプル

    @tableofcontents

    @brief アプリケーション自身が動画ファイルを作成し、保存します。

    @section PageSampleAlbumMovieMakerSimple_SectionBrief 概要
    アルバムライブラリにより、アプリケーション自身が描画した画と指定した音声で動画ファイルを作成し、保存します。

    @section PageSampleAlbumMovieMakerSimple_SectionFileStructure ファイル構成
    本サンプルプログラムは @link ../../../Samples/Sources/Applications/AlbumMovieMakerSimple
    Samples/Sources/Applications/AlbumMovieMakerSimple @endlink 以下にあります。

    @section PageSampleAlbumMovieMakerSimple_SectionNecessaryEnvironment 必要な環境
    動画ファイルの作成および保存には micro SD カードが必要です。
    Windows 環境はサポートされていません。

    @section PageSampleAlbumMovieMakerSimple_SectionHowToOperate 操作方法
    一般的なサンプルプログラムと同様に本プログラムをビルドし、実行してください。
    特に操作は必要ありません。

    @section PageSampleAlbumMovieMakerSimple_SectionDetail 解説
    このサンプルプログラムは次の順序で処理を行います。

    - グラフィックスライブラリの初期化
    - 動画撮影用と画面出力用の画を描画するための準備
    - AudioOut の準備
    - 音声データの作成
    - 動画撮影の設定
    - 動画撮影の事前チェック
    - 動画撮影開始
    - 以下を指定した時間だけ繰り返し
        - 動画撮影用の画を描画
        - 画面出力用の画を描画
        - 音の再生
        - 動画撮影用の音を入力
        - エラーチェック
    - 動画撮影の終了
    - アプリケーションの各種終了処理

 */

#include <cstdlib>

#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/init.h>
#include <nn/audio.h>
#include <nn/vi.h>
#include <nn/err.h>
#include <nn/mem/mem_StandardAllocator.h>
#include <nn/util/util_ScopeExit.h>

#include <nn/album/album_Result.h>
#include <nn/album/album_MovieMaker.h>

#include <nv/nv_MemoryManagement.h>

#include <nvn/nvn.h>
#include <nvn/nvn_FuncPtrInline.h>

extern "C" PFNNVNGENERICFUNCPTRPROC NVNAPIENTRY nvnBootstrapLoader(const char *name);

namespace
{
    const size_t GraphicsHeapSize = 16 * 1024 * 1024;
    NN_ALIGNAS(4096) nn::Bit8 g_GraphicsHeap[GraphicsHeapSize];

    const size_t MovieWorkMemorySize = 128 * 1024 * 1024;
    NN_ALIGNAS(4096) nn::Bit8 g_MovieWorkMemory[MovieWorkMemorySize];

    const size_t ApplicationHeapSize = 256 * 1024 * 1024;
    NN_ALIGNAS(4096) nn::Bit8 g_ApplicationHeap[ApplicationHeapSize];

    const size_t ControlMemorySize = 4 * 4096;
    const size_t CommandPoolMemorySize = 10 * 4096;
    const size_t FramePoolMemorySize = 16 * 1024 * 1024;

    const int FrameBufferWidth = 1280;
    const int FrameBufferHeight = 720;

    nn::vi::Display* m_pDisplay;
    nn::vi::Layer*   m_pLayer;

    // video および audio 入力のフレームレート
    const int InputDataFrameRate = 30;

    // 動画エンコード時のフレームレート
    const int CapturedFrameRate = 30;

    // 保存する動画の長さ（秒）
    const int CapturedDuration = 10;

    // 動画エンコーダーに指定するビットレート (bps)
    const int CapturedBitRate = 5000000;

    const int FrameCountLimit = CapturedDuration * CapturedFrameRate;

    // audio buffer
    const int AudioSampleRate = 48000;
    const int AudioChannelCount = 2;

    void InitializeGraphics()
    {
        nv::InitializeGraphics(g_GraphicsHeap, GraphicsHeapSize);

        nv::SetGraphicsAllocator(
            [](size_t size, size_t alignment, void*) -> void*
            {
                return std::aligned_alloc(alignment, size);
            },
            [](void* p, void*)
            {
                std::free(p);
            },
            [](void* p, size_t size, void*)
            {
                return std::realloc(p, size);
            },
            nullptr
        );
    }

    void FinalizeGraphics()
    {
        nv::FinalizeGraphics();
    }

    void InitializeNvnAndDevice(NVNdevice& device) NN_NOEXCEPT
    {
        PFNNVNDEVICEGETPROCADDRESSPROC getProcAddress = (PFNNVNDEVICEGETPROCADDRESSPROC)((nvnBootstrapLoader)("nvnDeviceGetProcAddress"));
        nvnLoadCProcs(nullptr, getProcAddress);

        int flags =
            NVN_DEVICE_FLAG_DEBUG_ENABLE_BIT |
            NVN_DEVICE_FLAG_DEBUG_ENABLE_LEVEL_0_BIT |
            NVN_DEVICE_FLAG_DEBUG_ENABLE_LEVEL_1_BIT |
            NVN_DEVICE_FLAG_DEBUG_ENABLE_LEVEL_2_BIT |
            NVN_DEVICE_FLAG_DEBUG_ENABLE_LEVEL_3_BIT |
            NVN_DEVICE_FLAG_DEBUG_ENABLE_LEVEL_4_BIT;
        NVNdeviceBuilder builder;
        nvnDeviceBuilderSetDefaults(&builder);
        nvnDeviceBuilderSetFlags(&builder, flags);
        NN_ABORT_UNLESS(nvnDeviceInitialize(&device, &builder));

        // load procs
        nvnLoadCProcs(&device, getProcAddress);

        // check version
        int majorVersion;
        int minorVersion;
        nvnDeviceGetInteger(&device, NVN_DEVICE_INFO_API_MAJOR_VERSION, &majorVersion);
        nvnDeviceGetInteger(&device, NVN_DEVICE_INFO_API_MINOR_VERSION, &minorVersion);
        NN_ABORT_UNLESS_EQUAL(majorVersion, NVN_API_MAJOR_VERSION);
        NN_ABORT_UNLESS_GREATER_EQUAL(minorVersion, NVN_API_MINOR_VERSION);

        nvnDeviceSetWindowOriginMode(&device, NVN_WINDOW_ORIGIN_MODE_UPPER_LEFT);
    }

    void InitializeCommandMemoryPool(NVNmemoryPool& commandPool, NVNdevice& device, void* pMemory, size_t size) NN_NOEXCEPT
    {
        NVNmemoryPoolBuilder builder;
        nvnMemoryPoolBuilderSetDevice(&builder, &device);
        nvnMemoryPoolBuilderSetDefaults(&builder);
        nvnMemoryPoolBuilderSetFlags(&builder, NVN_MEMORY_POOL_FLAGS_CPU_UNCACHED_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT);
        nvnMemoryPoolBuilderSetStorage(&builder, pMemory, size);
        NN_ABORT_UNLESS(nvnMemoryPoolInitialize(&commandPool, &builder));
    }

    void InitializeTextureMemoryPool(NVNmemoryPool& texturePool, NVNdevice& device, void* pMemory, size_t size) NN_NOEXCEPT
    {
        NVNmemoryPoolBuilder builder;
        nvnMemoryPoolBuilderSetDevice(&builder, &device);
        nvnMemoryPoolBuilderSetDefaults(&builder);
        nvnMemoryPoolBuilderSetFlags(&builder, NVN_MEMORY_POOL_FLAGS_CPU_NO_ACCESS_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT | NVN_MEMORY_POOL_FLAGS_COMPRESSIBLE_BIT);
        nvnMemoryPoolBuilderSetStorage(&builder, pMemory, size);
        NN_ABORT_UNLESS(nvnMemoryPoolInitialize(&texturePool, &builder));
    }

    void InitializeQueue(NVNqueue& queue, NVNdevice& device) NN_NOEXCEPT
    {
        NVNqueueBuilder builder;
        nvnQueueBuilderSetDefaults(&builder);
        nvnQueueBuilderSetDevice(&builder, &device);
        NN_ABORT_UNLESS(nvnQueueInitialize(&queue, &builder));
    }

    void InitializeCommandBuffer(NVNcommandBuffer& commandBuffer, NVNdevice& device) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(nvnCommandBufferInitialize(&commandBuffer, &device));
    }

    void InitializeFrameBufferTexture(NVNtexture* pTexutures, int count, int width, int height, NVNdevice& device, NVNmemoryPool& texturePool, ptrdiff_t offset, size_t size) NN_NOEXCEPT
    {
        ptrdiff_t pos = offset;
        ptrdiff_t posEnd = offset + size;
        NVNtextureBuilder builder;
        nvnTextureBuilderSetDevice(&builder, &device);
        nvnTextureBuilderSetDefaults(&builder);
        nvnTextureBuilderSetSize2D(&builder, width, height);
        nvnTextureBuilderSetFlags(&builder, NVN_TEXTURE_FLAGS_COMPRESSIBLE_BIT | NVN_TEXTURE_FLAGS_DISPLAY_BIT | NVN_TEXTURE_FLAGS_IMAGE_BIT);
        nvnTextureBuilderSetFormat(&builder, NVN_FORMAT_RGBA8_SRGB);
        nvnTextureBuilderSetTarget(&builder, NVN_TEXTURE_TARGET_2D);

        size_t texSize = nvnTextureBuilderGetStorageSize(&builder);
        size_t texAlign = nvnTextureBuilderGetStorageAlignment(&builder);

        pos = ((pos + texAlign - 1) / texAlign) * texAlign;
        NN_ABORT_UNLESS_GREATER_EQUAL(posEnd - pos, texSize * count);

        for (int i = 0; i < count; i++)
        {
            nvnTextureBuilderSetStorage(&builder, &texturePool, pos);
            NN_ABORT_UNLESS(nvnTextureInitialize(&pTexutures[i], &builder));
            pos += texSize;
        }
    }

    void InitializeWindow(NVNwindow& window, NVNtexture*const* pTextureList, int count, NVNnativeWindow nativeWindow, NVNdevice& device) NN_NOEXCEPT
    {
        NVNwindowBuilder builder;
        nvnWindowBuilderSetDevice(&builder, &device);
        nvnWindowBuilderSetDefaults(&builder);
        nvnWindowBuilderSetNativeWindow(&builder, nativeWindow);
        nvnWindowBuilderSetTextures(&builder, count, pTextureList);
        NN_ABORT_UNLESS(nvnWindowInitialize(&window, &builder));

    }

    void InitializeTextureViewForFrameBuffer(NVNtextureView& texView) NN_NOEXCEPT
    {
        nvnTextureViewSetDefaults(&texView);
        nvnTextureViewSetFormat(&texView, NVN_FORMAT_RGBA8_SRGB);
        nvnTextureViewSetTarget(&texView, NVN_TEXTURE_TARGET_2D);
    }

    void GenerateSquareWaveInt16(void* buffer, int channelCount, int sampleRate, int sampleCount, int amplitude)
    {
        static int s_TotalSampleCount[6] = { 0 };
        const int frequencies[6] = { 415, 698, 554, 104, 349, 277 };

        int16_t* buf = reinterpret_cast<int16_t*>(buffer);
        for (int ch = 0; ch < channelCount; ch++)
        {
            int waveLength = sampleRate / frequencies[ch]; // 1周期分の波形の長さ (サンプル数単位)

            for (int sample = 0; sample < sampleCount; sample++)
            {
                int16_t value = static_cast<int16_t>(s_TotalSampleCount[ch] < (waveLength / 2) ? amplitude : -amplitude);
                buf[sample*channelCount + ch] = value;
                s_TotalSampleCount[ch]++;
                if (s_TotalSampleCount[ch] == waveLength)
                {
                    s_TotalSampleCount[ch] = 0;
                }
            }
        }
    }

    void FillDisplay(NVNcommandBuffer* pCommandBuffer, int frame)
    {
        // 画面を4分割し、それぞれのクリアカラーを1秒ごとに切り替えていく
        int seconds = frame / InputDataFrameRate;
        float ratePerSecond = static_cast<float>(frame % InputDataFrameRate) / InputDataFrameRate;

        int halfWidth = FrameBufferWidth * 0.5f;
        int halfHeight = FrameBufferHeight * 0.5f;

        float blackToRed[4]  = {ratePerSecond, 0.0f, 0.0f, 1.0f};
        float redToGreen[4]  = {1.0f - ratePerSecond, ratePerSecond, 0.0f, 1.0f};
        float greenToBlue[4] = {0.0f, 1.0f - ratePerSecond, ratePerSecond, 1.0f};
        float blueToBlack[4] = {0.0f, 0.0f, 1.0f - ratePerSecond, 1.0f};

        float* clearColors[4] = {blackToRed, redToGreen, greenToBlue, blueToBlack};

        float* pColorLowerLeft = clearColors[seconds % 4];
        float* pColorLowerRight = clearColors[(seconds + 1) % 4];
        float* pColorUpperRight = clearColors[(seconds + 2) % 4];
        float* pColorUpperLeft = clearColors[(seconds + 3) % 4];

        nvnCommandBufferSetScissor(pCommandBuffer, 0, 0, halfWidth, halfHeight);
        nvnCommandBufferClearColor(pCommandBuffer, 0, pColorLowerLeft, NVN_CLEAR_COLOR_MASK_RGBA);

        nvnCommandBufferSetScissor(pCommandBuffer, halfWidth, 0, FrameBufferWidth - halfWidth, halfHeight);
        nvnCommandBufferClearColor(pCommandBuffer, 0, pColorLowerRight, NVN_CLEAR_COLOR_MASK_RGBA);

        nvnCommandBufferSetScissor(pCommandBuffer, halfWidth, halfHeight, FrameBufferWidth - halfWidth, FrameBufferHeight - halfHeight);
        nvnCommandBufferClearColor(pCommandBuffer, 0, pColorUpperRight, NVN_CLEAR_COLOR_MASK_RGBA);

        nvnCommandBufferSetScissor(pCommandBuffer, 0, halfHeight, halfWidth, FrameBufferHeight - halfHeight);
        nvnCommandBufferClearColor(pCommandBuffer, 0, pColorUpperLeft, NVN_CLEAR_COLOR_MASK_RGBA);
    }

    void DrawPseudoProgressBar(NVNcommandBuffer* pCommandBuffer, int frame)
    {
        // 徐々にクリア領域を横方向へ延ばしていく（録画処理の進捗状況を表現）
        float progressRate = static_cast<float>( frame ) / FrameCountLimit;
        float white[4] = { 1.0f, 1.0f, 1.0f, 1.0f };

        nvnCommandBufferSetScissor(pCommandBuffer, 0, 0, FrameBufferWidth * progressRate, FrameBufferHeight);
        nvnCommandBufferClearColor(pCommandBuffer, 0, white, NVN_CLEAR_COLOR_MASK_RGBA);
    }
}

extern "C" void nnMain() NN_NOEXCEPT
{
    nn::mem::StandardAllocator memoryAllocator;

    InitializeGraphics();

    nn::album::Initialize();
    nn::vi::Initialize();

    memoryAllocator.Initialize(g_ApplicationHeap, ApplicationHeapSize);

    // MovieMaker 初期化
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::album::InitializeMovieMaker(g_MovieWorkMemory, MovieWorkMemorySize));

    // 通常出力用 native window の取得
    nn::vi::NativeWindowHandle defaultNativeWindow;
    nn::vi::OpenDefaultDisplay(&m_pDisplay);
    nn::vi::CreateLayer(&m_pLayer, m_pDisplay);
    nn::vi::SetLayerScalingMode(m_pLayer, nn::vi::ScalingMode_FitToLayer);
    nn::vi::GetNativeWindow(&defaultNativeWindow, m_pLayer);

    // MovieMaker 用 native window の取得
    nn::vi::NativeWindowHandle videoNativeWindow;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::album::GetMovieMakerNativeWindow(&videoNativeWindow));

    // nvn 初期化
    auto pControlMemory = aligned_alloc(4096, ControlMemorySize);
    auto pCommandPoolMemory = aligned_alloc(4096, CommandPoolMemorySize);
    auto pFramePoolMemory = aligned_alloc(4096, FramePoolMemorySize);
    NN_ABORT_UNLESS_NOT_NULL(pControlMemory);
    NN_ABORT_UNLESS_NOT_NULL(pCommandPoolMemory);
    NN_ABORT_UNLESS_NOT_NULL(pFramePoolMemory);

    NVNdevice device;
    InitializeNvnAndDevice(device);

    NVNmemoryPool commandPool;
    InitializeCommandMemoryPool(commandPool, device, pCommandPoolMemory, CommandPoolMemorySize);

    NVNqueue queue;
    InitializeQueue(queue, device);

    NVNcommandBuffer commandBuffer;
    InitializeCommandBuffer(commandBuffer, device);

    NVNmemoryPool framePool;
    InitializeTextureMemoryPool(framePool, device, pFramePoolMemory, FramePoolMemorySize);

    NVNtexture frameTexs[4];
    InitializeFrameBufferTexture(frameTexs, 4, FrameBufferWidth, FrameBufferHeight, device, framePool, 0, FramePoolMemorySize);

    // 動画撮影用 NVNwindow
    NVNwindow videoWindow;
    NVNtexture* videoWindowTextureList[2];
    for (int i = 0; i < 2; i++)
    {
        videoWindowTextureList[i] = &frameTexs[i];
    }
    InitializeWindow(videoWindow, videoWindowTextureList, 2, videoNativeWindow, device);

    // ディスプレイ出力用 NVNwindow
    NVNwindow defaultWindow;
    NVNtexture* defaultWindowTextureList[2];
    for (int i = 0; i < 2; i++)
    {
        defaultWindowTextureList[i] = &frameTexs[i + 2];
    }
    InitializeWindow(defaultWindow, defaultWindowTextureList, 2, defaultNativeWindow, device);

    NVNtextureView texView;
    InitializeTextureViewForFrameBuffer(texView);

    // AudioOut 準備
    nn::audio::AudioOut audioOut;

    nn::os::SystemEvent systemEvent;
    nn::audio::AudioOutParameter parameter;
    nn::audio::InitializeAudioOutParameter(&parameter);
    parameter.sampleRate = AudioSampleRate;
    parameter.channelCount = AudioChannelCount;
    NN_ABORT_UNLESS(nn::audio::OpenDefaultAudioOut(&audioOut, &systemEvent, parameter).IsSuccess(), "Failed to open AudioOut");

    nn::audio::SampleFormat sampleFormat = nn::audio::GetAudioOutSampleFormat(&audioOut);
    NN_ASSERT(sampleFormat == nn::audio::SampleFormat_PcmInt16);

    // 再生用の音声データ生成
    const int frameSampleCount = AudioSampleRate / InputDataFrameRate;
    const size_t dataSize = frameSampleCount * AudioChannelCount * nn::audio::GetSampleByteSize(sampleFormat);
    const size_t bufferSize = nn::util::align_up(dataSize, nn::audio::AudioOutBuffer::SizeGranularity);
    const int bufferCount = 4;
    const int amplitude = std::numeric_limits<int16_t>::max() / 16;

    nn::audio::AudioOutBuffer audioOutBuffer[bufferCount];
    void* outBuffer[bufferCount];
    for (int i = 0; i < bufferCount; ++i)
    {
        outBuffer[i] = memoryAllocator.Allocate(bufferSize, nn::audio::AudioOutBuffer::AddressAlignment);
        NN_ASSERT(outBuffer[i]);
        GenerateSquareWaveInt16(outBuffer[i], AudioChannelCount, AudioSampleRate, frameSampleCount, amplitude);
        nn::audio::SetAudioOutBufferInfo(&audioOutBuffer[i], outBuffer[i], bufferSize, dataSize);
        nn::audio::AppendAudioOutBuffer(&audioOut, &audioOutBuffer[i]);
    }

    NN_ABORT_UNLESS(nn::audio::StartAudioOut(&audioOut).IsSuccess(), "Failed to start audio playback.");

    // 画面出力対象および録画対象処理
    auto frameFunction = [&](int frame) -> nn::Result
    {
        NVNsync acqSync;
        nvnSyncInitialize(&acqSync, &device);
        NN_UTIL_SCOPE_EXIT{ nvnSyncFinalize(&acqSync); };

        int texIdx = -1;
        NN_ABORT_UNLESS_EQUAL(nvnWindowAcquireTexture(&videoWindow, &acqSync, &texIdx), NVN_WINDOW_ACQUIRE_TEXTURE_RESULT_SUCCESS);

        int defTexIdx = -1;
        NN_ABORT_UNLESS_EQUAL(nvnWindowAcquireTexture(&defaultWindow, &acqSync, &defTexIdx), NVN_WINDOW_ACQUIRE_TEXTURE_RESULT_SUCCESS);

        nvnCommandBufferAddCommandMemory(&commandBuffer, &commandPool, 0, CommandPoolMemorySize);
        nvnCommandBufferAddControlMemory(&commandBuffer, pControlMemory, ControlMemorySize);
        nvnCommandBufferBeginRecording(&commandBuffer);

        // ここで描画コマンドを積んでもよい

        // 録画対象の画を描画
        nvnCommandBufferSetRenderTargets(&commandBuffer, 1, &videoWindowTextureList[texIdx], NULL, NULL, NULL);
        FillDisplay(&commandBuffer, frame);

        // ディスプレイ出力用の画を描画
        nvnCommandBufferSetRenderTargets(&commandBuffer, 1, &defaultWindowTextureList[texIdx], NULL, NULL, NULL);
        DrawPseudoProgressBar(&commandBuffer, frame);

        // wait display
        nvnCommandBufferWaitSync(&commandBuffer, &acqSync);

        auto hCommand = nvnCommandBufferEndRecording(&commandBuffer);
        nvnQueueSubmitCommands(&queue, 1, &hCommand);
        nvnQueuePresentTexture(&queue, &videoWindow, texIdx);
        nvnQueuePresentTexture(&queue, &defaultWindow, texIdx);
        nvnQueueFinish(&queue);

        // オーディオ
        systemEvent.Wait();

        nn::audio::AudioOutBuffer* pAudioOutBuffer = nullptr;
        pAudioOutBuffer = nn::audio::GetReleasedAudioOutBuffer(&audioOut);

        while (pAudioOutBuffer)
        {
            void* pOutBuffer = nn::audio::GetAudioOutBufferDataPointer(pAudioOutBuffer);
            size_t outSize = nn::audio::GetAudioOutBufferDataSize(pAudioOutBuffer);

            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::album::EncodeMovieMakerAudioSample(pOutBuffer, outSize));

            GenerateSquareWaveInt16(pOutBuffer, AudioChannelCount, AudioSampleRate, frameSampleCount, amplitude);
            nn::audio::AppendAudioOutBuffer(&audioOut, pAudioOutBuffer);

            pAudioOutBuffer = nn::audio::GetReleasedAudioOutBuffer(&audioOut);
        }

        return nn::album::CheckMovieMakerError();
    };

    // 録画設定
    nn::album::MovieMakerMovieParameter mmParam = nn::album::MovieMakerMovieParameter::GetDefaultValue();
    mmParam.SetVideoFrameRate(CapturedFrameRate);
    mmParam.SetVideoBitRate(CapturedBitRate);
    mmParam.SetVideoWidth(FrameBufferWidth);
    mmParam.SetVideoHeight(FrameBufferHeight);

    // 指定した尺の動画を保存するために必要なファイルサイズの概算（多めに見積もる）
    uint64_t neededFileSize = CapturedBitRate * CapturedDuration + 1 * 1024 * 1024;

    // 事前チェック
    nn::Result result;
    result = nn::album::PrecheckToStartMovieMaker(neededFileSize);
    if (result.IsSuccess())
    {
        // 録画開始
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::album::StartMovieMaker(mmParam));

        // 指定時間分の動画保存を開始
        for (int frame = 0; frame < FrameCountLimit; frame++)
        {
            result = frameFunction(frame);
            if (result.IsFailure())
            {
                break;
            }
        }

        // 録画終了処理
        if (result.IsSuccess())
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::album::FinishMovieMaker());
            NN_LOG("MovieMaker created a movie file.\n");
        }
        else
        {
            nn::album::AbortMovieMaker();
            NN_LOG("MovieMaker could not create a movie file.\n");
        }
    }
    else
    {
        nn::err::ShowError(result);
        NN_LOG("PrecheckToStartMovieMaker failed.\n");
    }

    // アプリケーションの終了処理
    nn::audio::StopAudioOut(&audioOut);
    nn::audio::CloseAudioOut(&audioOut);
    nn::os::DestroySystemEvent(systemEvent.GetBase());

    for (int i = 0; i < bufferCount; ++i)
    {
        memoryAllocator.Free(outBuffer[i]);
    }

    nvnWindowFinalize(&videoWindow);
    for (auto& e : frameTexs)
    {
        nvnTextureFinalize(&e);
    }
    nvnMemoryPoolFinalize(&framePool);
    nvnCommandBufferFinalize(&commandBuffer);
    nvnQueueFinalize(&queue);
    nvnMemoryPoolFinalize(&commandPool);
    nvnDeviceFinalize(&device);

    free(pControlMemory);
    free(pCommandPoolMemory);
    free(pFramePoolMemory);

    nn::vi::DestroyLayer(m_pLayer);
    nn::vi::CloseDisplay(m_pDisplay);

    nn::album::FinalizeMovieMaker();

    memoryAllocator.Finalize();

    // アルバムライブラリを終了します。
    nn::album::Finalize();

    FinalizeGraphics();
}//NOLINT(impl/function_size)
