﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
    @examplesource{AccountManagement.cpp,PageSampleAccountManagement}

    @brief
    ユーザーアカウントの管理のためのサンプルプログラム
 */

/**
    @page PageSampleAccountManagement ユーザーアカウント管理
    @tableofcontents

    @brief
    アカウントライブラリを使用した、ユーザーアカウント管理のためのサンプルプログラムです。

    @section PageSampleAccountManagement_SectionBrief 概要
    このプログラムは、アカウントライブラリを利用したユーザーアカウント管理の方法を例示するためのサンプルプログラムです。
    アカウントライブラリがもつ種々の機能のうち、ユーザーアカウントの作成や削除、プロフィールなどの属性情報の変更を対象に、アプリケーションでの利用方法を紹介します。

    @section PageSampleAccountManagement_SectionFileStructure ファイル構成
    本サンプルプログラムは @link ../../../Samples/Sources/Applications/AccountManagement
    Samples/Sources/Applications/AccountManagement @endlink 以下にあります。

    @section PageSampleAccountManagement_SectionNecessaryEnvironment 必要な環境
    このサンプルプログラムの動作のために特別な環境は必要ありません。

    @section PageSampleAccountManagement_SectionHowToOperate 操作方法
    本プログラムは操作可能ではありません。

    @section PageSampleAccountManagement_SectionPrecaution 注意事項
    特記すべき注意事項はありません。

    @section PageSampleAccountManagement_SectionHowToExecute 実行手順
    サンプルプログラムをビルドし、実行してください。

    @section PageSampleAccountManagement_SectionDetail 解説
    このサンプルプログラムは、アカウントライブラリを使用したユーザーアカウント管理の方法を例示します。
    ユーザーアカウント管理には次の操作が含まれます。

    - ユーザーアカウントリストの取得および並び替え
    - ユーザーアカウントの作成、削除およびプロフィールの編集

    サンプルプログラムの処理の流れは以下の通りです。
    それぞれの処理の間に随時、ユーザーアカウントの列挙処理を行います。

    - ユーザーアカウントの新規作成
    - プロフィール情報を伴うユーザーアカウントの新規作成
    - ユーザーアカウントのプロフィール情報の編集
    - ユーザーアカウントリストの並び替え
    - ユーザーアカウントの削除

    まずもっとも単純な、プロフィールをもたないユーザーアカウントを作成します。
    このユーザーアカウントはニックネームもプロフィール画像ももたずに作成されます。

    次に、作成の完了時点でプロフィールが設定された状態となるユーザーアカウントを作成方法を示します。
    これはユーザーアカウントの作成中、その完了前にプロフィールを設定することで実現します。
    このとき入力値の異常などによりプロフィールの設定に失敗した場合は、ユーザーアカウントの作成を明示的にキャンセルしてください。

    一度作成したユーザーアカウントは、任意のタイミングでプロフィール情報を変更することができます。
    プロフィール情報の変更には、ニックネームや任意データの変更のみを行う方法とプロフィール画像を同時に変更する方法があります。
    プロフィール画像の更新がない場合には前者の方法を採用してください。処理が高速に完了します。

    最後に、ユーザーアカウントのリスト中での表示順序の変更と、ユーザーアカウントの削除の方法を例示します。
    一度削除したユーザーアカウントはいかなる手段でも復旧することはできないことに留意してください。

    このサンプルの実行結果を以下に示します。
    ただし、実行結果中の "<...>" は実行する環境によって表示が異なります。

    @verbinclude AccountManagement_ExampleOutput.txt
 */

#define NN_LOG_USE_DEFAULT_LOCALE_CHARSET

#include <cstring>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/init.h>
#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForAdministrators.h>
#include <nn/account/account_Result.h>
#include <nn/account/account_ResultForAdministrators.h>
#include <nn/util/util_StringUtil.h>

namespace
{

void ListUserAccounts()
{
    // Uid 格納用の配列
    nn::account::Uid users[nn::account::UserCountMax];

    // 実際のユーザー数
    int actualCount;
    auto result = nn::account::ListAllUsers(&actualCount, users, sizeof(users) / sizeof(users[0]));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    NN_LOG("Users:\n");
    for (int i = 0; i < actualCount; ++ i)
    {
        NN_ASSERT(users[i]);

        nn::account::Nickname nickname;
        result = nn::account::GetNickname(&nickname, users[i]);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        NN_LOG(" [%d] \"%s\"\n", i, nickname.name);
    }
}

// プロフィール情報を持たないユーザーアカウントの作成のための関数
nn::account::Uid CreateUserAccount()
{
    nn::account::Uid userAccount;

    // 作成処理の開始
    auto result = nn::account::BeginUserRegistration(&userAccount);
    if (nn::account::ResultUserRegistryFull::Includes(result))
    {
        // 登録可能なユーザー数の上限に達しています。
        NN_LOG("Error: Up-to %d user accounts can be registred\n", nn::account::UserCountMax);
        return nn::account::InvalidUid;
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_ASSERT(userAccount);

    // 作成処理の完了
    result = nn::account::CompleteUserRegistration(userAccount);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    return userAccount;
}

// プロフィール情報を与えてユーザーアカウントを作成するための関数
nn::account::Uid CreateUserAccountWithProfile(
    const char* nicknameBuffer, size_t nicknameBufferBytes,
    const void* profileImage, size_t profileImageBytes,
    const char* userDataBuffer, size_t userDataBufferBytes)
{
    nn::account::Uid userAccount;

    // 作成処理の開始
    auto result = nn::account::BeginUserRegistration(&userAccount);
    if (nn::account::ResultUserRegistryFull::Includes(result))
    {
        // 登録可能なユーザー数の上限に達しています。
        NN_LOG("Error: Up-to %d user accounts can be registred\n", nn::account::UserCountMax);
        return nn::account::InvalidUid;
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_ASSERT(userAccount);

    // プロフィール情報は作成中にも設定可能です
    nn::account::ProfileEditor profileEditor;
    result = nn::account::GetProfileEditor(&profileEditor, userAccount);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // Nickname 型の値の作成
    NN_ASSERT(strnlen(nicknameBuffer, nicknameBufferBytes) <= nn::account::NicknameBytesMax);
    nn::account::Nickname nickname;
    nn::util::Strlcpy(nickname.name, nicknameBuffer, static_cast<int>(nicknameBufferBytes));

    // ニックネームの設定
    profileEditor.SetNickname(nickname);
    // 任意データの設定
    profileEditor.SetUserData(userDataBuffer, userDataBufferBytes);
    // 画像データを指定してプロフィールの反映
    result = profileEditor.FlushWithImage(profileImage, profileImageBytes);
    if (!result.IsSuccess())
    {
        // プロフィールの反映に失敗しユーザー作成を中断する際は、 CancelUserRegistration を呼びます。
        result = nn::account::CancelUserRegistration(userAccount);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        return nn::account::InvalidUid;
    }

    // 作成処理の完了
    result = nn::account::CompleteUserRegistration(userAccount);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    return userAccount;
}

// ユーザーのニックネームを変更するための関数
void ModifyUserNickname(
    const nn::account::Uid& userAccount,
    const char* nicknameBuffer, size_t nicknameBufferBytes)
{
    // プロフィールエディタの取得
    nn::account::ProfileEditor profileEditor;
    auto result = nn::account::GetProfileEditor(&profileEditor, userAccount);
    // 指定されたユーザーが存在しません。
    NN_ASSERT(!nn::account::ResultUserNotExist::Includes(result), "No such user");
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // Nickname 型の値の作成
    NN_ASSERT(strnlen(nicknameBuffer, nicknameBufferBytes) <= nn::account::NicknameBytesMax);
    nn::account::Nickname nickname;
    nn::util::Strlcpy(nickname.name, nicknameBuffer, static_cast<int>(nicknameBufferBytes));

    // ニックネームの設定
    profileEditor.SetNickname(nickname);
    // 画像データの更新はないため、 Flush() を使用する。
    result = profileEditor.Flush();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}

} // 無名名前空間終わり


// アロケータ用のバッファ
NN_ALIGNAS(4096) uint8_t  g_MallocBuffer[1 * 1024 * 1024];
extern "C" void nninitStartup()
{
    nn::init::InitializeAllocator(g_MallocBuffer, sizeof(g_MallocBuffer));
}

extern "C" void nnMain()
{
    // ファイルシステムの初期化

    // アカウントシステムの、管理者権限での初期化
    nn::account::InitializeForAdministrator();

    // 起動時点でのアカウントの登録状況を表示します。
    ListUserAccounts();

    // プロフィール情報を持たないユーザーアカウントの作成
    NN_LOG("> Create new user account\n");
    auto user0 = CreateUserAccount();
    NN_ASSERT(user0);

    // プロフィール情報を持つユーザーアカウントの作成
    NN_LOG("> Create new user account with profile\n");
    char profileImageRaw[128] = {}; // 任意のデータを設定します。
    void* profileImage = profileImageRaw;
    char userData[nn::account::UserDataBytesMax] = {}; // 任意のデータを設定します。
    auto user1 = CreateUserAccountWithProfile(
        "User1", sizeof("User1"),
        profileImage, sizeof(profileImageRaw),
        userData, sizeof(userData));
    NN_ASSERT(user1);

    // 現時点でのアカウントの登録状況を表示します。
    ListUserAccounts();

    // ユーザーのニックネームの変更
    NN_LOG("> Modify user nickname\n");
    ModifyUserNickname(user1, "*User1*", sizeof("*User1*"));

    // ユーザー1を先頭に配置します。
    NN_LOG("> Reorder user accounts\n");
    auto result = nn::account::SetUserPosition(user1, 0);
    NN_ASSERT(!nn::account::ResultUserNotExist::Includes(result), "No such user");
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // 現時点でのアカウントの登録状況を表示します。
    ListUserAccounts();

    // アカウントを削除します。
    NN_LOG("> Delete user accounts\n");
    result = nn::account::DeleteUser(user0);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    result = nn::account::DeleteUser(user1);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // 現時点でのアカウントの登録状況を表示します。
    ListUserAccounts();

    return;
}
