﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "3dSampleViewer_Renderer.h"
#include "3dSampleViewer.h"
#include "3dSampleViewer_Menu.h"
#include "3dSampleViewer_Viewer.h"

namespace
{
    //----------------------------------------------------------------------
    //      Uniform Blocks
    //----------------------------------------------------------------------

    template<typename T, int BufferingCount>
    class UniformBlock
    {
    public:
        using UniformBlockType = T;
        void Initialize(const UniformBlockType& initValue, nn::gfx::Device* pDevice, nn::gfx::MemoryPool* pMemoryPool, nn::gfx::util::MemoryPoolAllocator* pAllocator) NN_NOEXCEPT
        {
            m_pAllocator = pAllocator;

            // バッファ生成
            {
                size_t size = sizeof(UniformBlockType);
                nn::gfx::Buffer::InfoType info;
                {
                    info.SetDefault();
                    info.SetSize(size);
                    info.SetGpuAccessFlags(nn::gfx::GpuAccess_ConstantBuffer);
                }
                size_t alignment = nn::gfx::Buffer::GetBufferAlignment(pDevice, info);

                for (int bufferIndex = 0; bufferIndex < BufferingCount; ++bufferIndex)
                {
                    m_MemoryPoolOffset[bufferIndex] = m_pAllocator->Allocate(size, alignment);
                    NN_ASSERT_NOT_EQUAL(m_MemoryPoolOffset[bufferIndex], nn::gfx::util::MemoryPoolAllocator::InvalidOffset);
                    m_Buffer[bufferIndex].Initialize(pDevice, info, pMemoryPool, m_MemoryPoolOffset[bufferIndex], size);

                    // 初期値をコピー
                    WriteValue(initValue, 0, bufferIndex);
                }
            }
        }

        void Finalize(nn::gfx::Device* pDevice) NN_NOEXCEPT
        {
            for (int bufferIndex = 0; bufferIndex < BufferingCount; ++bufferIndex)
            {
                m_Buffer[bufferIndex].Finalize(pDevice);
                m_pAllocator->Free(m_MemoryPoolOffset[bufferIndex]);
            }
        }

        template<typename ValueType>
        void WriteValue(const ValueType& value, ptrdiff_t offset, int bufferIndex) NN_NOEXCEPT
        {
            NN_ASSERT_RANGE(bufferIndex, 0, BufferingCount);
            NN_ASSERT_RANGE(offset, 0, static_cast<ptrdiff_t>(sizeof(UniformBlockType)));

            nn::util::BytePtr ptr(m_Buffer[bufferIndex].Map(), offset);
            memcpy(ptr.Get(), &value, sizeof(ValueType));
            m_Buffer[bufferIndex].FlushMappedRange(0, sizeof(ValueType));
            m_Buffer[bufferIndex].Unmap();
        }

        const nn::gfx::Buffer* GetBuffer(int bufferIndex) const NN_NOEXCEPT
        {
            NN_ASSERT_RANGE(bufferIndex, 0, BufferingCount);
            return &m_Buffer[bufferIndex];
        }

    private:
        nn::gfx::util::MemoryPoolAllocator* m_pAllocator;
        nn::gfx::Buffer m_Buffer[BufferingCount];
        ptrdiff_t m_MemoryPoolOffset[BufferingCount];
    };

    struct TownContext
    {
        static const int CascadeCount = 4;
        nn::util::FloatColumnMajor4x4 lightViewProj[CascadeCount];
        nn::util::Float4 cascadeDepth;
        nn::util::Float2 tanFov;
        nn::util::Float2 nearFar;
        float diffIntensity;
        nn::util::Float3 padding;
    };
    UniformBlock<TownContext, 1> g_TownContext;

    struct SelectedInfo
    {
        int smoothSkinningOffset;
        int rigidSkininingOffset;
    };
    UniformBlock<SelectedInfo, 2> g_SelectedInfo;

    //----------------------------------------------------------------------
    //      Renderer
    //----------------------------------------------------------------------

    nns::gfx::PrimitiveRenderer::Renderer*  g_pRenderer = nullptr;
    g3ddemo::BoundingRenderer               g_BoundingRenderer;
    GridRenderer                            g_MainGridRenderer;
    GridRenderer                            g_SubGridRenderer;
    SkeletonRenderer                        g_SkeletonRenderer;

    nn::gfx::RasterizerState                g_WireframeRasterizerState;
    nn::gfx::BlendState                     g_WireframeBlendState;

    int g_ViewIndex = 0;

    nn::util::Matrix4x3fType g_ViewMatrix;
    nn::util::Matrix4x4fType g_ProjectionMatrix;

    void InitializeGridRenderer() NN_NOEXCEPT
    {
        const float width = 10000.0f;
        // main grid
        {
            g_MainGridRenderer.Initialize(g_pRenderer);
            g_MainGridRenderer.SetInterval(1000.0f);
            g_MainGridRenderer.SetWidth(width);
            g_MainGridRenderer.SetLineWidth(1.6f);
            g_SubGridRenderer.SetColor(nn::util::Color4u8::White());
        }

        // sub grid
        {
            g_SubGridRenderer.Initialize(g_pRenderer);
            g_SubGridRenderer.SetInterval(100.0f);
            g_SubGridRenderer.SetWidth(width);
            nn::util::Color4u8 darkGray(64, 64, 64, 255);
            g_SubGridRenderer.SetColor(darkGray);
        }
    }

    void FinalizeGridRenderer() NN_NOEXCEPT
    {
        g_SubGridRenderer.Finalize();
        g_MainGridRenderer.Finalize();
    }

    void InitializeSkeletonRenderer() NN_NOEXCEPT
    {
        g_SkeletonRenderer.Initialize(g_pRenderer);
        g_SkeletonRenderer.SetJointRadius(10.0f);

        nn::util::Color4u8 lightGreen(67, 255, 163, 255);
        g_SkeletonRenderer.SetSelectedJointColor(lightGreen);

        nn::util::Color4u8 lightGray(160, 160, 160, 255);
        g_SkeletonRenderer.SetDefaultJointColor(lightGray);
    }

    void FinalizeSkeletonRenderer() NN_NOEXCEPT
    {
        g_SkeletonRenderer.Finalize();
    }

    void InitializeWireframeRenderState(nn::gfx::Device* pDevice) NN_NOEXCEPT
    {
        // rasterizer state
        {
            nn::gfx::RasterizerState::InfoType info;
            info.SetDefault();

            // bias = depthBias + slopeScaledDepthBias * max(abs(ddx(depth)), abs(ddy(depth)))
            info.SetDepthBias(-1);
            info.SetSlopeScaledDepthBias(-.5f);
            info.SetFillMode(nn::gfx::FillMode_Wireframe);
            g_WireframeRasterizerState.Initialize(pDevice, info);
        }

        // blend state
        {
            nn::gfx::BlendState::InfoType info;
            info.SetDefault();

            nn::gfx::BlendTargetStateInfo targetStateInfo;
            {
                targetStateInfo.SetChannelMask(nn::gfx::ChannelMask_All);

                targetStateInfo.SetBlendEnabled(true);
                targetStateInfo.SetColorBlendFunction(nn::gfx::BlendFunction_ReverseSubtract);
                targetStateInfo.SetSourceColorBlendFactor(nn::gfx::BlendFactor_Zero);
                targetStateInfo.SetDestinationColorBlendFactor(nn::gfx::BlendFactor_Zero);

                targetStateInfo.SetAlphaBlendFunction(nn::gfx::BlendFunction_Add);
                targetStateInfo.SetSourceAlphaBlendFactor(nn::gfx::BlendFactor_Zero);
                targetStateInfo.SetDestinationAlphaBlendFactor(nn::gfx::BlendFactor_One);
            }
            info.SetBlendTargetStateInfoArray(&targetStateInfo, 1);
            size_t size = nn::gfx::BlendState::GetRequiredMemorySize(info);
            if (size > 0)
            {
                void* pBuffer = g3ddemo::AllocateMemory(size, nn::gfx::BlendState::RequiredMemoryInfo_Alignment);
                g_WireframeBlendState.SetMemory(pBuffer, size);
            }
            g_WireframeBlendState.Initialize(pDevice, info);
        }
    }

    void FinalizeWireframeRenderState(nn::gfx::Device* pDevice) NN_NOEXCEPT
    {
        void* pBuffer = g_WireframeBlendState.GetMemory();
        if (pBuffer)
        {
            g3ddemo::FreeMemory(pBuffer);
        }
        g_WireframeBlendState.Finalize(pDevice);
        g_WireframeRasterizerState.Finalize(pDevice);
    }
}

void InitializeRenderer() NN_NOEXCEPT
{
    nns::gfx::DebugGraphicsFramework* pFramework = g3ddemo::GetGfxFramework();
    nn::gfx::Device* pDevice = pFramework->GetDevice();

    g_pRenderer = pFramework->GetPrimitiveRenderer();
    g_BoundingRenderer.Initialize(pDevice);
    InitializeGridRenderer();
    InitializeSkeletonRenderer();
    InitializeWireframeRenderState(pDevice);
}

void FinalizeRenderer() NN_NOEXCEPT
{
    nns::gfx::DebugGraphicsFramework* pFramework = g3ddemo::GetGfxFramework();
    nn::gfx::Device* pDevice = pFramework->GetDevice();

    FinalizeWireframeRenderState(pDevice);
    FinalizeSkeletonRenderer();
    FinalizeGridRenderer();

    g_BoundingRenderer.Finalize(pDevice);
    g_pRenderer = nullptr;
}

void UpdateRenderer(nns::g3d::RenderView* pView, int viewIndex) NN_NOEXCEPT
{
    g_ViewIndex = viewIndex;
    g_ViewMatrix = pView->GetViewMtx(g_ViewIndex);
    g_ProjectionMatrix = pView->GetProjectionMtx(g_ViewIndex);
}

void InitializeUniformBlocks() NN_NOEXCEPT
{
    nns::gfx::DebugGraphicsFramework* pFramework = g3ddemo::GetGfxFramework();
    nn::gfx::Device* pDevice = pFramework->GetDevice();
    nns::gfx::GraphicsFramework::MemoryPoolType type = nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer;
    nn::gfx::MemoryPool* pMemoryPool = pFramework->GetMemoryPool(type);
    nn::gfx::util::MemoryPoolAllocator* pMemoryPoolAllocator = pFramework->GetMemoryPoolAllocator(type);

    const nn::gfx::Buffer* pBufferArray[BufferingCount];

    // TownContext
    {
        TownContext context;
        context.diffIntensity = 1.0f;
        g_TownContext.Initialize(context, pDevice, pMemoryPool, pMemoryPoolAllocator);

        for (const nn::gfx::Buffer*& pBuffer : pBufferArray)
        {
            pBuffer = g_TownContext.GetBuffer(0);
        }
    }
    SetUniformBlock(UniformBlockType_TownContext, pBufferArray, BufferingCount);

    // SelectedInfo
    {
        SelectedInfo info = { -1,-1 };
        g_SelectedInfo.Initialize(info, pDevice, pMemoryPool, pMemoryPoolAllocator);

        for (int bufferIndex = 0; bufferIndex < BufferingCount; ++bufferIndex)
        {
            pBufferArray[bufferIndex] = g_SelectedInfo.GetBuffer(bufferIndex);
        }
    }
    SetUniformBlock(UniformBlockType_SelectedInfo, pBufferArray, BufferingCount);
}

void FinalizeUniformBlocks() NN_NOEXCEPT
{
    nn::gfx::Device* pDevice = g3ddemo::GetGfxFramework()->GetDevice();
    g_SelectedInfo.Finalize(pDevice);
    g_TownContext.Finalize(pDevice);
}

void UpdateUniformBlock(int bufferIndex) NN_NOEXCEPT
{
    // 選択情報の更新
    SelectedInfo info = { -1,-1 };

    const nn::g3d::ModelObj* pSelectedModelObj = GetSelectedModelObj();
    if (!pSelectedModelObj)
    {
        return;
    }
    int selectedBoneIndex = GetMenuContext()->GetSelectedIndex(MenuContext::SelectedInfo_Bone);
    const nn::g3d::ResSkeleton* pResSkeleton = pSelectedModelObj->GetSkeleton()->GetResource();

    const int16_t* pMtxToBoneTable = pResSkeleton->ToData().pMtxToBoneTable.Get();

    int offsetIndex = 0;

    // スムーススキニング
    for (int smoothSkinigMatrixCount = pResSkeleton->GetSmoothMtxCount(); offsetIndex < smoothSkinigMatrixCount; ++offsetIndex)
    {
        int boneIndex = pMtxToBoneTable[offsetIndex];
        if (boneIndex != selectedBoneIndex)
        {
            continue;
        }
        info.smoothSkinningOffset = offsetIndex;
    }

    // リジッドスキニング
    for (int skinningCount = pResSkeleton->GetMtxCount(); offsetIndex < skinningCount; ++offsetIndex)
    {
        int boneIndex = pMtxToBoneTable[offsetIndex];
        if (boneIndex != selectedBoneIndex)
        {
            continue;
        }
        info.rigidSkininingOffset = offsetIndex;
    }

    g_SelectedInfo.WriteValue(info, 0, bufferIndex);
}

void MakeModelDrawCommand(nn::gfx::CommandBuffer* pCommandBuffer, int bufferIndex) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pCommandBuffer);

    g3ddemo::ResourceHolder* pHolder = GetResourceHolder();
    const MenuContext* pContext = GetMenuContext();

    int selectedModelIndex = pContext->GetSelectedIndex(MenuContext::SelectedInfo_Model);
    int selectedShapeIndex = pContext->GetSelectedIndex(MenuContext::SelectedInfo_Shape);
    int selectedSubMesheIndex = pContext->GetSelectedIndex(MenuContext::SelectedInfo_SubMesh);
    bool isSingleShapeEnabled = pContext->IsDrawConfigEnabled(MenuContext::DrawConfigType_SigleShape);
    bool isSingleSubMeshEnabled = pContext->IsDrawConfigEnabled(MenuContext::DrawConfigType_SingleSubMesh);

    for (int modelIndex = 0, modelCount = pHolder->renderModelObjs.GetCount(); modelIndex < modelCount; ++modelIndex)
    {
        nns::g3d::RenderModelObj* pRenderModelObj = pHolder->renderModelObjs[modelIndex];

        // 一度、単体描画機能を無効化
        pRenderModelObj->SetDrawSingleShapeDisabled();
        for (int shapeIndex = 0, shapeCount = pRenderModelObj->GetModelObj()->GetShapeCount(); shapeIndex < shapeCount; ++shapeIndex)
        {
            pRenderModelObj->GetRenderUnitObj(shapeIndex)->SetDrawSingleSubmeshDisabled();
        }

        if (selectedModelIndex != modelIndex)
        {
            continue;
        }

        // シェイプ単体描画
        {
            if (!isSingleShapeEnabled || selectedShapeIndex == MenuContext::InvalidIndex)
            {
                continue;
            }

            pRenderModelObj->SetDrawSingleShapeEnabled(selectedShapeIndex);
        }

        // サブメッシュ単体描画
        {
            if (!isSingleSubMeshEnabled)
            {
                continue;
            }

            nns::g3d::RenderUnitObj* pUnitObj = pRenderModelObj->GetRenderUnitObj(selectedShapeIndex);
            pUnitObj->SetDrawSingleSubmeshEnabled(selectedSubMesheIndex);
        }
    }

    for (nns::g3d::RenderModelObj* pRenderModelObj : pHolder->renderModelObjs)
    {
        pRenderModelObj->Draw(pCommandBuffer, g_ViewIndex, bufferIndex);
    }

    // デバッグ描画対象モデルを描画
    if (selectedModelIndex != MenuContext::InvalidIndex)
    {
        nns::g3d::RenderModelObj* pRenderModelObj = pHolder->renderModelObjs[selectedModelIndex];
        pRenderModelObj->Draw(pCommandBuffer, DrawPassType_Visualize, g_ViewIndex, bufferIndex);
    }
}

void AABBToSphere(nn::g3d::Sphere* pSphere, const nn::g3d::Aabb& aabb) NN_NOEXCEPT
{
    nn::util::VectorAdd(&pSphere->center, aabb.min, aabb.max);
    nn::util::VectorMultiply(&pSphere->center, pSphere->center, .5f);
    nn::util::Vector3f diff;
    nn::util::VectorSubtract(&diff, aabb.max, pSphere->center);
    pSphere->radius = nn::util::VectorLength(diff);
}

void MakeBoundingSphereDrawCommand(nn::gfx::CommandBuffer* pCommandBuffer) NN_NOEXCEPT
{
    g3ddemo::ResourceHolder* pHolder = GetResourceHolder();
    const MenuContext* pContext = GetMenuContext();
    MenuContext::SelectedInfo selectedInfo = GetSelectedInfo();
    int index = pContext->GetSelectedIndex(selectedInfo);
    switch (selectedInfo)
    {
    default:
    case MenuContext::SelectedInfo_Model:
        {
            int modelIndex = pContext->GetSelectedIndex(MenuContext::SelectedInfo_Model);

            // env モデルは無視する
            if (modelIndex == MenuContext::InvalidIndex)
            {
                break;
            }
            nn::g3d::ModelObj* pModelObj = pHolder->modelAnimObjs[modelIndex]->GetModelObj();
            g_BoundingRenderer.DrawSphere(pModelObj, pCommandBuffer, 0);
        }
        break;
    case MenuContext::SelectedInfo_Shape:
        {
            int modelIndex = pContext->GetSelectedIndex(MenuContext::SelectedInfo_Model);
            nn::g3d::ModelObj* pModelObj = pHolder->modelAnimObjs[modelIndex]->GetModelObj();
            g_BoundingRenderer.DrawSphere(pModelObj->GetShape(index)->GetBounding(), pCommandBuffer);
        }
        break;
    case MenuContext::SelectedInfo_SubMesh:
        {
            int modelIndex = pContext->GetSelectedIndex(MenuContext::SelectedInfo_Model);
            nn::g3d::ModelObj* pModelObj = pHolder->modelAnimObjs[modelIndex]->GetModelObj();
            int shapeIndex = pContext->GetSelectedIndex(MenuContext::SelectedInfo_Shape);

            //NOTE: 剛体でなければサブメッシュのバウンディングは更新されない
            nn::g3d::ShapeObj* pShapeObj = pModelObj->GetShape(shapeIndex);
            if (pShapeObj->IsRigidBody())
            {
                nn::g3d::Sphere sphere;
                AABBToSphere(&sphere, pShapeObj->GetSubMeshBoundingArray()[index]);
                g_BoundingRenderer.DrawSphere(&sphere, pCommandBuffer);
            }
            else if (pShapeObj->GetSubMeshCount() == 1)
            {
                g_BoundingRenderer.DrawSphere(pShapeObj->GetBounding(), pCommandBuffer);
            }
        }
        break;
    }
}

void MakeBoundingBoxDrawCommand(nn::gfx::CommandBuffer* pCommandBuffer) NN_NOEXCEPT
{
    g3ddemo::ResourceHolder* pHolder = GetResourceHolder();
    const MenuContext* pContext = GetMenuContext();
    MenuContext::SelectedInfo selectedInfo = GetSelectedInfo();
    int index = pContext->GetSelectedIndex(selectedInfo);
    switch (selectedInfo)
    {
    default:
    case MenuContext::SelectedInfo_Model:
        {
            int modelIndex = pContext->GetSelectedIndex(MenuContext::SelectedInfo_Model);

            // env モデルは無視する
            if (modelIndex == MenuContext::InvalidIndex)
            {
                break;
            }
            nn::g3d::ModelObj* pModelObj = pHolder->modelAnimObjs[modelIndex]->GetModelObj();
            for (int shapeIndex = 0, shapeCount = pModelObj->GetShapeCount(); shapeIndex < shapeCount; ++shapeIndex)
            {
                g_BoundingRenderer.DrawBox(pModelObj, pCommandBuffer, shapeIndex, 0);
            }
        }
        break;
    case MenuContext::SelectedInfo_Shape:
        {
            int modelIndex = pContext->GetSelectedIndex(MenuContext::SelectedInfo_Model);
            nn::g3d::ModelObj* pModelObj = pHolder->modelAnimObjs[modelIndex]->GetModelObj();
            g_BoundingRenderer.DrawBox(pModelObj, pCommandBuffer, index, 0);
        }
        break;
    case MenuContext::SelectedInfo_SubMesh:
        {
            int modelIndex = pContext->GetSelectedIndex(MenuContext::SelectedInfo_Model);
            nn::g3d::ModelObj* pModelObj = pHolder->modelAnimObjs[modelIndex]->GetModelObj();
            int shapeIndex = pContext->GetSelectedIndex(MenuContext::SelectedInfo_Shape);
            g_BoundingRenderer.DrawBox(pModelObj, pCommandBuffer, shapeIndex, 0, index);
        }
        break;
    }
}

void MakeBoundingDrawCommand(nn::gfx::CommandBuffer* pCommandBuffer, int bufferIndex) NN_NOEXCEPT
{
    NN_UNUSED(bufferIndex);
    const MenuContext* pContext = GetMenuContext();
    g_BoundingRenderer.Update(pCommandBuffer, &g_ViewMatrix, &g_ProjectionMatrix);

    if (pContext->IsDrawConfigEnabled(MenuContext::DrawConfigType_BoundingSphere))
    {
        MakeBoundingSphereDrawCommand(pCommandBuffer);
    }
    if (pContext->IsDrawConfigEnabled(MenuContext::DrawConfigType_BoundingBox))
    {
        MakeBoundingBoxDrawCommand(pCommandBuffer);
    }
}

void MakeGridDrawCommand(nn::gfx::CommandBuffer* pCommandBuffer, int bufferIndex) NN_NOEXCEPT
{
    NN_UNUSED(bufferIndex);

    g_SubGridRenderer.Draw(pCommandBuffer);
    g_MainGridRenderer.Draw(pCommandBuffer);
}

void MakeSkeletonDrawCommand(nn::gfx::CommandBuffer* pCommandBuffer, int bufferIndex) NN_NOEXCEPT
{
    NN_UNUSED(bufferIndex);

    const MenuContext* pContext = GetMenuContext();
    if (!pContext->IsDrawConfigEnabled(MenuContext::DrawConfigType_Skeleton))
    {
        return;
    }

    int modelIndex = pContext->GetSelectedIndex(MenuContext::SelectedInfo_Model);
    if (modelIndex == MenuContext::InvalidIndex)
    {
        return;
    }

    g3ddemo::ResourceHolder* pHolder = GetResourceHolder();
    nn::g3d::ModelObj* pModelObj = pHolder->modelAnimObjs[modelIndex]->GetModelObj();

    MenuContext::SelectedInfo selectedInfo = GetSelectedInfo();
    if (selectedInfo == MenuContext::SelectedInfo_Bone)
    {
        int boneIndex = pContext->GetSelectedIndex(MenuContext::SelectedInfo_Bone);
        g_SkeletonRenderer.SetSelectedBoneIndex(boneIndex);
    }
    else
    {
        g_SkeletonRenderer.SetSelectedBoneIndex(SkeletonRenderer::InvalidBoneIndex);
    }
    g_SkeletonRenderer.Draw(pCommandBuffer, pModelObj);
}

void MakeWireframeDrawCommand(nn::gfx::CommandBuffer* pCommandBuffer, float lineWidth, int bufferIndex) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pCommandBuffer);

    const MenuContext* pContext = GetMenuContext();
    int modelIndex = pContext->GetSelectedIndex(MenuContext::SelectedInfo_Model);
    if (modelIndex == MenuContext::InvalidIndex)
    {
        return;
    }

    if (!pContext->IsDrawConfigEnabled(MenuContext::DrawConfigType_Wireframe))
    {
        return;
    }

    g3ddemo::ResourceHolder* pHolder = GetResourceHolder();
    nns::g3d::RenderModelObj* pRenderModelObj = pHolder->renderModelObjs[modelIndex];

    nns::g3d::RenderModel* pRenderModel = pRenderModelObj->GetRenderModel();
    int materialCount = pRenderModel->GetResModel()->GetMaterialCount();

    ScopedDynamicArray<const nn::gfx::BlendState*> pOldBlendStateArray(materialCount);
    ScopedDynamicArray<const nn::gfx::RasterizerState*> pOldRasterizerState(materialCount);
    for (int materialIndex = 0; materialIndex < materialCount; ++materialIndex)
    {
        nns::g3d::RenderMaterial* pRenderMaterial = pRenderModel->GetRenderMaterial(materialIndex);

        pOldBlendStateArray[materialIndex] = pRenderMaterial->GetBlendState();
        pRenderMaterial->SetBlendState(&g_WireframeBlendState);

        pOldRasterizerState[materialIndex] = pRenderMaterial->GetRasterizerState();
        pRenderMaterial->SetRasterizerState(&g_WireframeRasterizerState);
    }

    pCommandBuffer->SetLineWidth(lineWidth);

    pRenderModelObj->Draw(pCommandBuffer, ViewType_Camera, bufferIndex);

    for (int materialIndex = 0; materialIndex < materialCount; ++materialIndex)
    {
        nns::g3d::RenderMaterial* pRenderMaterial = pRenderModel->GetRenderMaterial(materialIndex);
        pRenderMaterial->SetBlendState(pOldBlendStateArray[materialIndex]);
        pRenderMaterial->SetRasterizerState(pOldRasterizerState[materialIndex]);
    }

    // とりあえず、デフォルト値に設定しなおす.
    pCommandBuffer->SetLineWidth(1.0f);
}

void GridRenderer::Initialize(nns::gfx::PrimitiveRenderer::Renderer* pRenderer) NN_NOEXCEPT
{
    m_pRenderer = pRenderer;
}

void GridRenderer::Finalize() NN_NOEXCEPT
{
    m_pRenderer = NULL;
}

void GridRenderer::Draw(nn::gfx::CommandBuffer* pCommandBuffer) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(g_pRenderer);
    NN_ASSERT_NOT_NULL(pCommandBuffer);

    m_pRenderer->SetDefaultParameters();
    m_pRenderer->SetViewMatrix(&g_ViewMatrix);
    m_pRenderer->SetProjectionMatrix(&g_ProjectionMatrix);
    m_pRenderer->SetColor(m_Color);
    m_pRenderer->SetDepthStencilState(pCommandBuffer, nns::gfx::PrimitiveRenderer::DepthStencilType_DepthTest);

    pCommandBuffer->SetLineWidth(m_LineWidth);

    for (float width = .0f; width <= m_Width; width += m_Interval)
    {
        // x 軸
        {
            nn::util::Vector3fType begin = NN_UTIL_VECTOR_3F_INITIALIZER(width - m_Width * .5f, .0f, -m_Width * .5f);
            nn::util::Vector3fType end = NN_UTIL_VECTOR_3F_INITIALIZER(width - m_Width * .5f, .0f, m_Width * .5f);
            m_pRenderer->DrawLine(pCommandBuffer, begin, end);
        }

        // z 軸
        {
            nn::util::Vector3fType begin = NN_UTIL_VECTOR_3F_INITIALIZER(-m_Width * .5f, .0f, width - m_Width* .5f);
            nn::util::Vector3fType end = NN_UTIL_VECTOR_3F_INITIALIZER(m_Width * .5f, .0f, width - m_Width* .5f);
            m_pRenderer->DrawLine(pCommandBuffer, begin, end);
        }
    }

    pCommandBuffer->SetLineWidth(1.0f);
}

void SkeletonRenderer::Initialize(nns::gfx::PrimitiveRenderer::Renderer* pRenderer) NN_NOEXCEPT
{
    m_pRenderer = pRenderer;
}

void SkeletonRenderer::Finalize() NN_NOEXCEPT
{
    m_pRenderer = NULL;
}

void SkeletonRenderer::Draw(nn::gfx::CommandBuffer* pCommandBuffer, nn::g3d::ModelObj* pModelObj) NN_NOEXCEPT
{
    const nn::util::Matrix4x3fType* worldMtxArray = pModelObj->GetSkeleton()->GetWorldMtxArray();
    m_pRenderer->SetDefaultParameters();
    m_pRenderer->SetDepthStencilState(pCommandBuffer, nns::gfx::PrimitiveRenderer::DepthStencilType_DepthNoWriteTest);
    m_pRenderer->SetViewMatrix(&g_ViewMatrix);
    m_pRenderer->SetProjectionMatrix(&g_ProjectionMatrix);

    for (int boneIndex = 0, boneCount = pModelObj->GetSkeleton()->GetBoneCount(); boneIndex < boneCount; ++boneIndex)
    {
        m_pRenderer->SetColor(m_SelectedBoneIndex == boneIndex ? m_SelectedJointColor : m_DefaultJointColor);

        nn::util::Matrix4x3fType boneWorldMtx = worldMtxArray[boneIndex];
        float radius = 2.0f;
        nn::util::Vector3fType position;
        nn::util::MatrixGetAxisW(&position, boneWorldMtx);
        m_pRenderer->SetModelMatrix(&nn::util::Matrix4x3f::Identity());
        m_pRenderer->DrawSphere(pCommandBuffer, nns::gfx::PrimitiveRenderer::Surface_Wired, nns::gfx::PrimitiveRenderer::Subdiv_Normal, position, radius * 2.0f);
        nn::util::Vector3fType axisX, axisY, axisZ;
        {
            nn::util::MatrixGetAxisX(&axisX, boneWorldMtx);
            nn::util::MatrixGetAxisY(&axisY, boneWorldMtx);
            nn::util::MatrixGetAxisZ(&axisZ, boneWorldMtx);
            nn::util::Vector3f rot;

            nn::util::VectorMultiply(&rot, axisX, radius * 2.0f);
            nn::util::VectorAdd(&rot, position, rot);
            m_pRenderer->SetColor(nn::util::Color4u8::Red());
            m_pRenderer->DrawLine(pCommandBuffer, position, rot);

            nn::util::VectorMultiply(&rot, axisY, radius * 2.0f);
            nn::util::VectorAdd(&rot, position, rot);
            m_pRenderer->SetColor(nn::util::Color4u8::Green());
            m_pRenderer->DrawLine(pCommandBuffer, position, rot);

            nn::util::VectorMultiply(&rot, axisZ, radius * 2.0f);
            nn::util::VectorAdd(&rot, position, rot);
            m_pRenderer->SetColor(nn::util::Color4u8::Blue());
            m_pRenderer->DrawLine(pCommandBuffer, position, rot);
        }

        {
            int parentIndex = pModelObj->GetSkeleton()->GetResBone(boneIndex)->GetParentIndex();
            if (parentIndex == nn::g3d::ResBone::InvalidBoneIndex)
            {
                continue;
            }

            m_pRenderer->SetColor(m_SelectedBoneIndex == parentIndex ? m_SelectedJointColor : m_DefaultJointColor);

            nn::util::Matrix4x3fType parentBoneMtx = worldMtxArray[parentIndex];
            nn::util::Vector3fType parentPosition;
            nn::util::MatrixGetAxisW(&parentPosition, parentBoneMtx);
            nn::util::Vector3fType diff;
            nn::util::VectorSubtract(&diff, position, parentPosition);
            float height = nn::util::VectorLength(diff);
            nn::util::VectorNormalize(&axisY, diff);
            nn::util::VectorLoad(&axisX, NN_UTIL_FLOAT_3_INITIALIZER(1.0f, .0f, .001f));
            nn::util::VectorCross(&axisZ, axisX, axisY);
            nn::util::VectorNormalize(&axisZ, axisZ);
            nn::util::VectorCross(&axisX, axisY, axisZ);
            nn::util::VectorNormalize(&axisX, axisX);
            nn::util::Matrix4x3fType rotMtx;
            nn::util::MatrixIdentity(&rotMtx);
            nn::util::MatrixSetAxisX(&rotMtx, axisX);
            nn::util::MatrixSetAxisY(&rotMtx, axisY);
            nn::util::MatrixSetAxisZ(&rotMtx, axisZ);
            nn::util::Vector3fType offset;
            nn::util::VectorMultiply(&offset, axisY, radius);
            nn::util::VectorAdd(&position, parentPosition, offset);
            nn::util::MatrixSetAxisW(&rotMtx, position);

            m_pRenderer->SetModelMatrix(&rotMtx);
            m_pRenderer->DrawCone(pCommandBuffer, nns::gfx::PrimitiveRenderer::Surface_Wired, NN_UTIL_VECTOR_3F_INITIALIZER(.0f, .0f, .0f), radius, height - 2.0f * radius);
        }
    }
}
