﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <nn/nn_Assert.h>
#include <nn/util/util_Vector.h>

namespace nn { namespace g3d { namespace demo {

class Menu;
class Page;
class Group;
class Content;

// 複数のページを管理します。
class Menu
{
public:
    Menu() NN_NOEXCEPT
        : m_pName(nullptr)
        , m_pPages(nullptr)
        , m_PageCount(0)
        , m_SelectedPageIndex(0)
        , m_IsInitialized(false)
        , m_DrawCallback(nullptr)
        , m_pDrawUserData(nullptr)
    {
    }

    // 初期化処理を行います。
    void Initialize(const char* name, Page** pPageArray, int pageCount) NN_NOEXCEPT
    {
        NN_ASSERT(!IsInitialized());
        m_pName = name;
        m_pPages = pPageArray;
        m_PageCount = pageCount;
        m_IsInitialized = true;
    }

    // 終了処理を行います。
    void Finalize() NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        m_pName = nullptr;
        m_pPages = nullptr;
        m_PageCount = 0;
        m_IsInitialized = false;
    }

    // 次のページに切り替えます。
    void ChangeNextPage() NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        m_SelectedPageIndex = (m_SelectedPageIndex + 1) % m_PageCount;
    }

    // 前のページに切り替えます。
    void ChangePreviousPage() NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        m_SelectedPageIndex = (m_SelectedPageIndex + (m_PageCount - 1)) % m_PageCount;
    }

    // 選択されているページを更新します。
    void Update() NN_NOEXCEPT;

    // 選択されているページの描画処理を行います。
    void Draw() NN_NOEXCEPT;

    // 選択されているコンテンツを取得します。
    Content* GetSelectedContent() NN_NOEXCEPT;
    const Content* GetSelectedContent() const NN_NOEXCEPT;

    // 強制的に選択しているコンテンツを変更します。
    bool SelectContent(Content* pSelectContent) NN_NOEXCEPT;

    // 与えたコンテンツを保持しているかを取得します。
    bool IsContentContained(const Content* pContent) const NN_NOEXCEPT;

    typedef void(*DrawCallback)(Menu* pMenu, void* pUserData);
    void SetDrawCallback(DrawCallback pDrawCallback, void* pUserData)
    {
        NN_ASSERT(IsInitialized());
        m_DrawCallback = pDrawCallback;
        m_pDrawUserData = pUserData;
    }

    // コンテンツの選択可否を設定します。
    bool SetContentSelectability(Content* pContent, bool isSelectable) NN_NOEXCEPT;

    // グループの選択可否を設定します。
    bool SetGroupSelectability(Group* pGroup, bool isSelectable) NN_NOEXCEPT;

    // 名前を取得します。
    const char* GetName() const NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        return m_pName;
    }

    // 初期化済みであるかどうかを取得します。
    bool IsInitialized() const NN_NOEXCEPT
    {
        return m_IsInitialized;
    }

private:
    const char* m_pName;
    Page**      m_pPages;
    int         m_PageCount;
    int         m_SelectedPageIndex;
    bool        m_IsInitialized;

    DrawCallback m_DrawCallback;
    void* m_pDrawUserData;
};

// 複数のグループを管理します。
class Page
{
public:
    Page() NN_NOEXCEPT
        : m_pName(nullptr)
        , m_pGroups(nullptr)
        , m_GroupCount(0)
        , m_SelectedGroup(0)
        , m_IsInitialized(false)
        , m_DrawCallback(nullptr)
        , m_pDrawUserData(nullptr)
    {
    }

    // 初期化処理を行います。
    void Initialize(const char* name, Group** pGroupArray, int groupCount) NN_NOEXCEPT;

    // 終了処理を行います。
    void Finalize() NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        m_pName = nullptr;
        m_pGroups = nullptr;
        m_GroupCount = m_SelectedGroup = 0;
        m_DrawCallback = nullptr;
        m_pDrawUserData = nullptr;
        m_IsInitialized = false;
    }

    // グループを更新します。
    void Update() NN_NOEXCEPT;

    // グループを描画します。
    void Draw() NN_NOEXCEPT;

    // 次のコンテンツを選択します。
    void ChangeNextContent() NN_NOEXCEPT;

    // 前のコンテンツを選択します。
    void ChangePreviousContent() NN_NOEXCEPT;

    // 選択されているコンテンツを取得します。
    Content* GetSelectedContent() NN_NOEXCEPT;
    const Content* GetSelectedContent() const NN_NOEXCEPT;

    Group* GetGroup(int index) NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        NN_ASSERT_RANGE(index, 0, m_GroupCount);
        return m_pGroups[index];
    }

    const Group* GetGroup(int index) const NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        NN_ASSERT_RANGE(index, 0, m_GroupCount);
        return m_pGroups[index];
    }

    // 与えたコンテンツを保持しているかを取得します。
    bool IsContentContained(const Content* pContent) const NN_NOEXCEPT;

    // コンテンツの選択可否を設定します。
    bool SetContentSelectability(Content* pContent, bool isSelectable) NN_NOEXCEPT;

    // グループの選択可否を設定します。
    bool SetGroupSelectability(Group* pGroup, bool isSelectable) NN_NOEXCEPT;

    typedef void(*DrawCallback)(Page* pPage, void* pUserData);
    void SetDrawCallback(DrawCallback pDrawCallback, void* pUserData)
    {
        m_DrawCallback = pDrawCallback;
        m_pDrawUserData = pUserData;
    }

    // 強制的に選択しているコンテンツを変更します。
    bool SelectContent(Content* pSelectContent) NN_NOEXCEPT;

    // 選択情報をクリアします。
    void ClearIsSelectedInfo() NN_NOEXCEPT;

    // 初期化済みであるかどうかを取得します。
    bool IsInitialized() const NN_NOEXCEPT
    {
        return m_IsInitialized;
    }

private:
    int GetNextGroupIndex() const NN_NOEXCEPT;
    int GetPreviousGroupIndex() const NN_NOEXCEPT;

private:
    const char* m_pName;
    Group**     m_pGroups;
    int         m_GroupCount;
    int         m_SelectedGroup;
    bool        m_IsInitialized;

    DrawCallback m_DrawCallback;
    void* m_pDrawUserData;
};

// 複数のコンテンツを管理します。
class Group
{
public:
    Group() NN_NOEXCEPT
        : m_pName(nullptr)
        , m_pContents(nullptr)
        , m_ContentCount(0)
        , m_SelectedContent(0)
        , m_IsInitialized(false)
        , m_DrawCallback(nullptr)
        , m_pDrawUserData(nullptr)
    {
    }

    // 初期化処理を行います。
    void Initialize(const char* name, Content** pContents, int contentCount) NN_NOEXCEPT;

    // 終了処理を行います。
    void Finalize() NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        m_pName = nullptr;
        m_pContents = nullptr;
        m_ContentCount = m_SelectedContent = 0;
        m_DrawCallback = nullptr;
        m_pDrawUserData = nullptr;
        m_IsInitialized = false;
    }

    // コンテンツを更新します。
    void Update() NN_NOEXCEPT;

    // コンテンツを描画します。
    void Draw() NN_NOEXCEPT;

    // 選択されているコンテンツを取得します。
    Content* GetSelectedContent() NN_NOEXCEPT;
    const Content* GetSelectedContent() const NN_NOEXCEPT;

    Content* GetContent(int index) NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        NN_ASSERT_RANGE(index, 0, m_ContentCount);
        return m_pContents[index];
    }

    const Content* GetContent(int index) const NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        NN_ASSERT_RANGE(index, 0, m_ContentCount);
        return m_pContents[index];
    }

    int GetContentCount() const NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        return m_ContentCount;
    }

    const char* GetName() const NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        return m_pName;
    }

    typedef void(*DrawCallback)(Group* pGroup, void* pUserData);
    void SetDrawCallback(DrawCallback pDrawCallback, void* pUserData)
    {
        m_DrawCallback = pDrawCallback;
        m_pDrawUserData = pUserData;
    }

    // 次にコンテンツが存在するかを取得します。
    bool HasNextContent() const NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        return GetNextContentIndex() != InvalidIndex;
    }

    // 前にコンテンツが存在するかを取得します。
    bool HasPreviousContent() const NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        return GetPreviousContentIndex() != InvalidIndex;
    }

    // 与えたコンテンツを保持しているかを取得します。
    bool IsContentContained(const Content* pContent) const NN_NOEXCEPT;

    // 選択可能なコンテンツを保持しているかを取得します。
    bool HasSelectableContent() const NN_NOEXCEPT;

    // 次のコンテンツを選択します。
    void ChangeNextContent() NN_NOEXCEPT;

    // 前のコンテンツを選択します。
    void ChangePreviousContent() NN_NOEXCEPT;

    // 一番前のコンテンツを選択します。
    void SelectFirstContent() NN_NOEXCEPT;

    // 一番後ろのコンテンツを選択します。
    void SelectLastContent() NN_NOEXCEPT;

    // コンテンツの選択情報をクリアします。
    void ClearIsSelectedInfo() NN_NOEXCEPT;

    enum SetSelectabilityResultFlag
    {
        SetSelectabilityResult_NotFound,
        SetSelectabilityResult_Success,
        SetSelectabilityResult_ChangeGroup
    };
    SetSelectabilityResultFlag SetContentSelectability(Content* pContent, bool isSelectable) NN_NOEXCEPT;

    // 強制的に選択しているコンテンツを変更します。
    bool SelectContent(Content* pSelectContent) NN_NOEXCEPT;

    // 初期化済みであるかどうかを取得します。
    bool IsInitialized() const NN_NOEXCEPT
    {
        return m_IsInitialized;
    }

private:
    static const int InvalidIndex = -1;
    int GetNextContentIndex() const NN_NOEXCEPT;
    int GetPreviousContentIndex() const NN_NOEXCEPT;

private:
    const char* m_pName;
    Content**   m_pContents;
    int         m_ContentCount;
    int         m_SelectedContent;
    bool        m_IsInitialized;

    DrawCallback m_DrawCallback;
    void* m_pDrawUserData;
};

class Content
{
    NN_DISALLOW_COPY(Content);
public:
    typedef void(*UpdateCallback)(Content* pContent, void* pUserData);
    typedef void(*DrawCallback)(Content* pContent, void* pUserData);
    typedef void(*ValueChangedCallback)(Content* pContent, void* pUserData);
public:

    enum ValueTypeInfo
    {
        ValueTypeInfo_Void,
        ValueTypeInfo_Bool,
        ValueTypeInfo_Int,
        ValueTypeInfo_Float,
        ValueTypeInfo_Vector3,
        ValueTypeInfo_Unkown
    };

    enum UpdateFrequency
    {
        UpdateFrequency_Always,
        UpdateFrequency_Selected
    };

    Content() NN_NOEXCEPT
        : m_pName(nullptr)
        , m_pValue(nullptr)
        , m_ValueSize(0)
        , m_TypeInfo(ValueTypeInfo_Unkown)
        , m_IsInitialized(false)
        , m_IsSelectable(true)
        , m_IsSelected(false)
        , m_UpdateCallback(nullptr)
        , m_pUpdateUserData(nullptr)
        , m_UpdateFrequency(UpdateFrequency_Always)
        , m_DrawCallback(nullptr)
        , m_pDrawUserData(nullptr)
    {
    }

    // 初期化処理を行います。
    void Initialize(const char* name, void* pValue, size_t valueSize, ValueTypeInfo typeInfo) NN_NOEXCEPT
    {
        NN_ASSERT(!IsInitialized());
        m_pName = name;
        m_pValue = pValue;
        m_ValueSize = valueSize;
        m_TypeInfo = typeInfo;
        m_IsInitialized = true;
    }

    // 終了処理を行います。
    void Finalize() NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        m_pName = nullptr;
        m_pValue = nullptr;
        m_ValueSize = 0;
        m_TypeInfo = ValueTypeInfo_Unkown;
        m_IsInitialized = false;
    }

    template<typename T>
    void Initialize(const char* name, T* pValue) NN_NOEXCEPT
    {
        Initialize(name, pValue, sizeof(T), ValueTypeInfo_Unkown);
    }

    template<>
    void Initialize(const char* name, bool* pValue) NN_NOEXCEPT
    {
        Initialize(name, pValue, sizeof(bool), ValueTypeInfo_Bool);
    }

    template<>
    void Initialize(const char* name, int* pValue) NN_NOEXCEPT
    {
        Initialize(name, pValue, sizeof(int), ValueTypeInfo_Int);
    }

    template<>
    void Initialize(const char* name, float* pValue) NN_NOEXCEPT
    {
        Initialize(name, pValue, sizeof(float), ValueTypeInfo_Float);
    }

    template<>
    void Initialize(const char* name, nn::util::Vector3fType* pValue) NN_NOEXCEPT
    {
        Initialize(name, pValue, sizeof(nn::util::Vector3fType), ValueTypeInfo_Vector3);
    }

    // 更新します。
    void Update() NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        if (m_UpdateCallback && ShouldUpdate())
        {
            m_UpdateCallback(this, m_pUpdateUserData);
        }
    }

    // 描画します。
    void Draw() NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        if (m_DrawCallback)
        {
            m_DrawCallback(this, m_pDrawUserData);
        }
    }

    const char* GetName() const NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        return m_pName;
    }

    const void* GetValue() const NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        return m_pValue;
    }

    template<typename T>
    T GetValue() const NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        NN_ASSERT_EQUAL(m_ValueSize, sizeof(T));
        return *static_cast<const T*>(m_pValue);
    }

    void SetValue(const void* pValue, size_t size) NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        NN_ASSERT_EQUAL(m_ValueSize, size);
        memcpy(m_pValue, pValue, size);
        if (m_ValueChangedCallback)
        {
            m_ValueChangedCallback(this, m_pValueChangedUserData);
        }
    }

    template<typename T>
    void SetValue(const T& value) NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        SetValue(&value, sizeof(T));
    }

    ValueTypeInfo GetTypeInfo() const NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        return m_TypeInfo;
    }

    void SetTypeInfo(ValueTypeInfo type) NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        m_TypeInfo = type;
    }

    bool IsSelectable() const NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        return m_IsSelectable;
    }

    bool IsSelected() const NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        return m_IsSelected;
    }

    void SetUpdateCallback(UpdateCallback pUpdateCallback, void* pUserData) NN_NOEXCEPT
    {
        SetUpdateCallback(pUpdateCallback, pUserData, UpdateFrequency_Always);
    }

    void SetUpdateCallback(UpdateCallback pUpdateCallback, void* pUserData, UpdateFrequency updateFrequency) NN_NOEXCEPT
    {
        m_UpdateCallback = pUpdateCallback;
        m_pUpdateUserData = pUserData;
        m_UpdateFrequency = updateFrequency;
    }

    void SetDrawCallback(DrawCallback pDrawCallback, void* pUserData) NN_NOEXCEPT
    {
        m_DrawCallback = pDrawCallback;
        m_pDrawUserData = pUserData;
    }

    void SetValueChangedCallback(ValueChangedCallback pValueChangedCallback, void* pUserData) NN_NOEXCEPT
    {
        m_ValueChangedCallback = pValueChangedCallback;
        m_pValueChangedUserData = pUserData;
    }

    bool IsInitialized() const NN_NOEXCEPT
    {
        return m_IsInitialized;
    }

private:

    bool ShouldUpdate() const NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        switch (m_UpdateFrequency)
        {
        case UpdateFrequency_Always:
            return true;
        case UpdateFrequency_Selected:
            return IsSelected();
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    friend void Group::ChangeNextContent() NN_NOEXCEPT;
    friend void Group::ChangePreviousContent() NN_NOEXCEPT;
    friend void Group::ClearIsSelectedInfo() NN_NOEXCEPT;
    friend void Group::SelectFirstContent() NN_NOEXCEPT;
    friend void Group::SelectLastContent() NN_NOEXCEPT;
    friend Group::SetSelectabilityResultFlag Group::SetContentSelectability(Content* pContent, bool isSelectable) NN_NOEXCEPT;
    friend bool Group::SelectContent(Content* pSelectContent) NN_NOEXCEPT;

private:
    const char*             m_pName;
    void*                   m_pValue;
    size_t                  m_ValueSize;
    ValueTypeInfo           m_TypeInfo;
    bool                    m_IsInitialized;

    bool                    m_IsSelectable;
    bool                    m_IsSelected;

    UpdateCallback          m_UpdateCallback;
    void*                   m_pUpdateUserData;
    UpdateFrequency         m_UpdateFrequency;

    DrawCallback            m_DrawCallback;
    void*                   m_pDrawUserData;

    ValueChangedCallback    m_ValueChangedCallback;
    void*                   m_pValueChangedUserData;
};

template<typename T>
class ContentHolder
{
    NN_DISALLOW_COPY(ContentHolder);
public:
    typedef T ValueType;

    ContentHolder() NN_NOEXCEPT
    {
    }

    void Initialize(const char* name, const ValueType& initValue) NN_NOEXCEPT
    {
        m_Value = initValue;
        m_Content.Initialize<ValueType>(name, &m_Value);
    }

    const ValueType* GetValue() const NN_NOEXCEPT
    {
        return &m_Value;
    }

    void SetValue(ValueType* pValue) NN_NOEXCEPT
    {
        m_Content.SetValue(pValue, sizeof(ValueType));
    }

    Content* GetContent() NN_NOEXCEPT
    {
        return &m_Content;
    }

    const Content* GetContent() const NN_NOEXCEPT
    {
        return &m_Content;
    }

private:
    Content     m_Content;
    ValueType   m_Value;
};

template<>
class ContentHolder<void>
{
    NN_DISALLOW_COPY(ContentHolder);
public:
    ContentHolder() NN_NOEXCEPT
    {
    }

    void Initialize(const char* name) NN_NOEXCEPT
    {
        m_Content.Initialize(name, nullptr, 0, Content::ValueTypeInfo_Void);
    }

    Content* GetContent() NN_NOEXCEPT
    {
        return &m_Content;
    }

    const Content* GetContent() const NN_NOEXCEPT
    {
        return &m_Content;
    }

private:
    Content m_Content;
};

template<int TContentCount>
class GroupHolder
{
public:

    void Initialize(const char* name, Content** pContents, int contentCount) NN_NOEXCEPT
    {
        NN_ASSERT_EQUAL(ContentCount, contentCount);
        for (int contentIndex = 0; contentIndex < contentCount; ++contentIndex)
        {
            m_Value[contentIndex] = pContents[contentIndex];
        }
        m_Group.Initialize(name, m_Value, contentCount);
    }

    Group* GetGroup() NN_NOEXCEPT
    {
        return &m_Group;
    }

    const Group* GetGroup() const NN_NOEXCEPT
    {
        return &m_Group;
    }

    static const int ContentCount = TContentCount;

private:
    NN_STATIC_ASSERT(ContentCount > 0);

private:
    Group       m_Group;
    Content*    m_Value[ContentCount];
};

template<int TGroupCount>
class PageHolder
{
public:
    PageHolder() NN_NOEXCEPT
    {
    }

    void Initialize(const char* name, Group** pGroupArray, int groupCount) NN_NOEXCEPT
    {
        NN_ASSERT_EQUAL(GroupCount, groupCount);

        for (int groupIndex = 0; groupIndex < groupCount; ++groupIndex)
        {
            m_pValue[groupIndex] = pGroupArray[groupIndex];
        }
        m_Page.Initialize(name, m_pValue, groupCount);
    }

    Page* GetPage() NN_NOEXCEPT
    {
        return &m_Page;
    }

    const Page* GetPage() const NN_NOEXCEPT
    {
        return &m_Page;
    }

    static const int GroupCount = TGroupCount;

private:
    NN_STATIC_ASSERT(GroupCount > 0);

private:
    Page    m_Page;
    Group*  m_pValue[GroupCount];
};

template<int TPageCount>
class MenuHolder
{
public:
    MenuHolder() NN_NOEXCEPT
    {
    }

    void Initialize(const char* name, Page** pPageArray, int pageCount) NN_NOEXCEPT
    {
        NN_ASSERT_EQUAL(PageCount, pageCount);
        for (int pageIndex = 0; pageIndex < pageCount; ++pageIndex)
        {
            m_pValue[pageIndex] = pPageArray[pageIndex];
        }
        m_Menu.Initialize(name, m_pValue, pageCount);
    }

    Menu* GetMenu() NN_NOEXCEPT
    {
        return &m_Menu;
    }

    const Menu* GetMenu() const NN_NOEXCEPT
    {
        return &m_Menu;
    }

    static const int PageCount = TPageCount;

private:
    NN_STATIC_ASSERT(PageCount > 0);

private:
    Menu m_Menu;
    Page* m_pValue[PageCount];
};

} } }
