﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "3dSampleViewer_Menu.h"
#include "3dSampleViewer.h"
#include "3dSampleViewer_MenuUtility.h"

namespace
{

struct ShaderOptionSetting
{
    const char* id;                             // シェーダーオプション変数の id
    MenuContext::DrawConfigType configType;     // 結びつく描画設定の種類
};

ShaderOptionSetting g_ShaderOptionSettings[] = {
    {"display_vertex_normal", MenuContext::DrawConfigType_VertexNormal},
    {"display_bone_weight" , MenuContext::DrawConfigType_BoneWeight},

    // TODO: 追加
    //{"display_submesh"},
};

g3ddemo::MenuHolder<1> g_Menu;

enum GroupType
{
    GroupType_HowTo,
    GroupType_CameraInfo,
    GroupType_SelectedInfo,
    GroupType_DrawConfig,
    GroupType_Count
};

g3ddemo::PageHolder<GroupType_Count> g_Page;

g3ddemo::GroupHolder<1> g_HowToGroup;
g3ddemo::ContentHolder<const char*> g_HowToMessage;

g3ddemo::GroupHolder<2> g_CameraInfoGroup;
g3ddemo::ContentHolder<nn::util::Vector3fType> g_CameraPosition;
g3ddemo::ContentHolder<nn::util::Vector3fType> g_CameraTarget;

g3ddemo::GroupHolder<MenuContext::SelectedInfo_Count> g_SelectedInfoGroup;
g3ddemo::Content g_SelectedIndeces[MenuContext::SelectedInfo_Count];

g3ddemo::GroupHolder<MenuContext::DrawConfigType_Count> g_DrawConfigGroup;
g3ddemo::Content g_DrawConfig[MenuContext::DrawConfigType_Count];

MenuContext g_Context;

nns::gfx::PrimitiveRenderer::Renderer* g_pRenderer;
g3ddemo::ScreenInfo g_ScreenInfo;

template<void (g3ddemo::CameraController::*GetValueFunc)(nn::util::Vector3fType*) const>
void CopyCameraParamater(g3ddemo::Content* pContent, void* pUserPtr)
{
    g3ddemo::CameraController* pController = static_cast<g3ddemo::CameraController*>(pUserPtr);
    nn::util::Vector3fType value;
    (pController->*GetValueFunc)(&value);
    pContent->SetValue(value);
}

void PrintCameraInfo(g3ddemo::Content* pContent, void*) NN_NOEXCEPT
{
    nn::util::Vector3fType value = pContent->GetValue<nn::util::Vector3fType>();
    g_ScreenInfo.Print("  %s: (%.2f, %.2f, %.2f)\n", pContent->GetName(), nn::util::VectorGetX(value), nn::util::VectorGetY(value), nn::util::VectorGetZ(value));
}

void InitializeCameraInfoContents() NN_NOEXCEPT
{
    // Position
    {
        g_CameraPosition.Initialize("Position", NN_UTIL_VECTOR_3F_INITIALIZER(.0f, .0f, .0f));
        g_CameraPosition.GetContent()->SetUpdateCallback(CopyCameraParamater<&g3ddemo::CameraController::GetPos>, GetCameraController());
        g_CameraPosition.GetContent()->SetDrawCallback(PrintCameraInfo, nullptr);
    }

    // Target
    {
        g_CameraTarget.Initialize("Target", NN_UTIL_VECTOR_3F_INITIALIZER(.0f, .0f, .0f));
        g_CameraTarget.GetContent()->SetUpdateCallback(CopyCameraParamater<&g3ddemo::CameraController::GetLookAtPos>, GetCameraController());
        g_CameraTarget.GetContent()->SetDrawCallback(PrintCameraInfo, nullptr);
    }
}

int UpdateIndex(int value, int count, int offset) NN_NOEXCEPT
{
    count -= offset;
    using Pad = g3ddemo::Pad;
    Pad& pad = g3ddemo::GetPad();
    if (pad.IsTriggered(Pad::BUTTON_LEFT))
    {
        value = ((value - offset) + (count - 1)) % count;
        value += offset;
    }
    else if (pad.IsTriggered(Pad::BUTTON_RIGHT))
    {
        value = ((value - offset) + 1) % count;
        value += offset;
    }

    return value;
}

int UpdateIndex(int value, int count) NN_NOEXCEPT
{
    return UpdateIndex(value, count, 0);
}

void SetMenuTextColor(g3ddemo::Content* pContent)
{
    if (pContent->IsSelected())
    {
        const nn::util::Color4u8 lightGreen(0, 150, 50, 255);
        g_ScreenInfo.SetTextColor(lightGreen.GetR(), lightGreen.GetG(), lightGreen.GetB(), lightGreen.GetA());
    }
    else
    {
        const nn::util::Color4u8& color = pContent->IsSelectable() ? nn::util::Color4u8::White() : nn::util::Color4u8::Gray();
        g_ScreenInfo.SetTextColor(color.GetR(), color.GetG(), color.GetB(), color.GetA());
    }
}

void SetSelectedModelInfoCallback() NN_NOEXCEPT
{
    g_SelectedIndeces[MenuContext::SelectedInfo_Model].SetUpdateCallback(
        [](g3ddemo::Content* pContent, void*)
    {
        g3ddemo::ResourceHolder* pHolder = GetResourceHolder();
        g3ddemo::Vector<nns::g3d::ModelAnimObj*>& modelAnimObjs = pHolder->modelAnimObjs;
        int modelCount = modelAnimObjs.GetCount();
        int selectedIndex = pContent->GetValue<int>();
        int oldIndex = selectedIndex;
        selectedIndex = UpdateIndex(oldIndex, modelCount, 1);
        if (oldIndex != selectedIndex)
        {
            pContent->SetValue(selectedIndex);
        }
    }
    , nullptr, g3ddemo::Content::UpdateFrequency_Selected);

    g_SelectedIndeces[MenuContext::SelectedInfo_Model].SetValueChangedCallback(
        [](g3ddemo::Content* pContent, void*)
    {
        int selectedIndex = pContent->GetValue<int>();
        bool isModelSelected = selectedIndex > 0;
        g_Menu.GetMenu()->SetContentSelectability(pContent, isModelSelected);

        // モデルに依存するパラメータを更新
        // モデル内にシェイプが存在しないケースは Shape の値が更新された際のコールバックにて対応します。
        g_SelectedIndeces[MenuContext::SelectedInfo_Shape].SetValue(isModelSelected ? 0 : MenuContext::InvalidIndex);
        g_SelectedIndeces[MenuContext::SelectedInfo_Bone].SetValue(isModelSelected ? 0 : MenuContext::InvalidIndex);
    }, nullptr);

    g_SelectedIndeces[MenuContext::SelectedInfo_Model].SetDrawCallback(
        [](g3ddemo::Content* pContent, void*)
    {
        SetMenuTextColor(pContent);

        int selectedIndex = pContent->GetValue<int>();
        bool isModelSelected = selectedIndex > 0;
        if (!isModelSelected)
        {
            g_ScreenInfo.Print("  %s: None\n", pContent->GetName());
            return;
        }

        const g3ddemo::Vector<nns::g3d::ModelAnimObj*>& modelAnimObjs = GetResourceHolder()->modelAnimObjs;
        const nn::g3d::ModelObj* pModelObj = modelAnimObjs[selectedIndex]->GetModelObj();
        const char* modelName = pModelObj->GetName();
        g_ScreenInfo.Print("  %s: %s (%d/%d)\n", pContent->GetName(), modelName, selectedIndex, modelAnimObjs.GetCount() - 1);
    }
    , nullptr);
}

void SetSelectedShapeInfoCallback() NN_NOEXCEPT
{
    g_SelectedIndeces[MenuContext::SelectedInfo_Shape].SetUpdateCallback(
        [](g3ddemo::Content* pContent, void*)
    {
        const nn::g3d::ModelObj* pModelObj = GetSelectedModelObj();
        const int shapeCount = pModelObj->GetShapeCount();

        int selectedIndex = pContent->GetValue<int>();
        int oldIndex = selectedIndex;
        selectedIndex = UpdateIndex(oldIndex, shapeCount);
        if (oldIndex != selectedIndex)
        {
            pContent->SetValue(selectedIndex);
        }
    }
    , nullptr, g3ddemo::Content::UpdateFrequency_Selected);

    g_SelectedIndeces[MenuContext::SelectedInfo_Shape].SetValueChangedCallback(
        [](g3ddemo::Content* pContent, void*)
    {
        int selectedIndex = pContent->GetValue<int>();
        if (selectedIndex != MenuContext::InvalidIndex)
        {
            // 本当にシェイプが存在するかチェック
            const nn::g3d::ModelObj* pModelObj = GetSelectedModelObj();
            if (pModelObj)
            {
                const int shapeCount = pModelObj->GetShapeCount();
                if (shapeCount > 0)
                {
                    g_Menu.GetMenu()->SetContentSelectability(pContent, true);

                    // サブメッシュはシェイプに依存するので値を更新
                    g_SelectedIndeces[MenuContext::SelectedInfo_SubMesh].SetValue(0);
                    return;
                }
            }
            pContent->SetValue(MenuContext::InvalidIndex);
        }
        g_Menu.GetMenu()->SetContentSelectability(pContent, false);

        // サブメッシュはシェイプに依存するので値を更新
        g_SelectedIndeces[MenuContext::SelectedInfo_SubMesh].SetValue(MenuContext::InvalidIndex);
    }
    , nullptr);

    g_SelectedIndeces[MenuContext::SelectedInfo_Shape].SetDrawCallback(
        [](g3ddemo::Content* pContent, void*)
    {
        SetMenuTextColor(pContent);
        const nn::g3d::ModelObj* pModelObj = GetSelectedModelObj();
        int selectedIndex = pContent->GetValue<int>();
        if (selectedIndex == MenuContext::InvalidIndex || !pModelObj)
        {
            g_ScreenInfo.Print("  %s: None\n", pContent->GetName());
            return;
        }

        const char* shapeName = pModelObj->GetShapeName(selectedIndex);
        g_ScreenInfo.Print("  %s: %s(%d/%d)\n", pContent->GetName(), shapeName, selectedIndex + 1, pModelObj->GetShapeCount());
    }
    , nullptr);
}

void SetSelectedSubMeshCallback() NN_NOEXCEPT
{
    g_SelectedIndeces[MenuContext::SelectedInfo_SubMesh].SetUpdateCallback(
        [](g3ddemo::Content* pContent, void*)
    {
        const nn::g3d::ModelObj* pModelObj = GetSelectedModelObj();
        int shapeIndex = g_Context.GetSelectedIndex(MenuContext::SelectedInfo_Shape);
        const nn::g3d::ShapeObj* pShapeObj = pModelObj->GetShape(shapeIndex);
        const int subMeshCount = pShapeObj->GetSubMeshCount();

        int oldIndex = pContent->GetValue<int>();
        int selectedIndex = UpdateIndex(oldIndex, subMeshCount);
        if (oldIndex != selectedIndex)
        {
            pContent->SetValue(selectedIndex);
        }
    }
    , nullptr, g3ddemo::Content::UpdateFrequency_Selected);

    g_SelectedIndeces[MenuContext::SelectedInfo_SubMesh].SetValueChangedCallback(
        [](g3ddemo::Content* pContent, void*)
    {
        int selectedIndex = pContent->GetValue<int>();
        g_Menu.GetMenu()->SetContentSelectability(pContent, selectedIndex != MenuContext::InvalidIndex);
    }
    , nullptr);

    g_SelectedIndeces[MenuContext::SelectedInfo_SubMesh].SetDrawCallback(
        [](g3ddemo::Content* pContent, void*)
    {
        SetMenuTextColor(pContent);
        const nn::g3d::ModelObj* pModelObj = GetSelectedModelObj();
        int shapeIndex = g_Context.GetSelectedIndex(MenuContext::SelectedInfo_Shape);
        int selectedIndex = pContent->GetValue<int>();
        if (!pModelObj || shapeIndex == MenuContext::InvalidIndex || selectedIndex == MenuContext::InvalidIndex)
        {
            g_ScreenInfo.Print("  %s: None\n", pContent->GetName());
            return;
        }

        const nn::g3d::ShapeObj* pShapeObj = pModelObj->GetShape(shapeIndex);
        g_ScreenInfo.Print("  %s: (%d/%d)\n", pContent->GetName(), selectedIndex + 1, pShapeObj->GetSubMeshCount());
    }
    , nullptr);
}

void SetSelectedBoneCallback() NN_NOEXCEPT
{
    g_SelectedIndeces[MenuContext::SelectedInfo_Bone].SetUpdateCallback(
        [](g3ddemo::Content* pContent, void*)
    {
        int selectedIndex = pContent->GetValue<int>();
        if (selectedIndex == MenuContext::InvalidIndex)
        {
            g_Menu.GetMenu()->SetContentSelectability(pContent, false);
            return;
        }

        const nn::g3d::ModelObj* pModelObj = GetSelectedModelObj();
        const nn::g3d::SkeletonObj* pSkeletonObj = pModelObj->GetSkeleton();
        const int boneCount = pSkeletonObj->GetBoneCount();

        selectedIndex = UpdateIndex(selectedIndex, boneCount);
        pContent->SetValue(selectedIndex);
    }
    , nullptr, g3ddemo::Content::UpdateFrequency_Selected);

    g_SelectedIndeces[MenuContext::SelectedInfo_Bone].SetValueChangedCallback(
        [](g3ddemo::Content* pContent, void*)
    {
        int selectedBoneIndex = pContent->GetValue<int>();
        g_Menu.GetMenu()->SetContentSelectability(pContent, selectedBoneIndex != MenuContext::InvalidIndex);
    }
    , nullptr);

    g_SelectedIndeces[MenuContext::SelectedInfo_Bone].SetDrawCallback(
        [](g3ddemo::Content* pContent, void*)
    {
        SetMenuTextColor(pContent);
        const nn::g3d::ModelObj* pModelObj = GetSelectedModelObj();
        int selectedIndex = pContent->GetValue<int>();
        if (!pModelObj || selectedIndex == MenuContext::InvalidIndex)
        {
            g_ScreenInfo.Print("  %s: None\n", pContent->GetName());
            return;
        }

        const nn::g3d::SkeletonObj* pSkeletonObj = pModelObj->GetSkeleton();
        const char* boneName = pSkeletonObj->GetBoneName(selectedIndex);
        g_ScreenInfo.Print("  %s: %s (%d/%d)\n", pContent->GetName(), boneName, selectedIndex + 1, pSkeletonObj->GetBoneCount());
    }
    , nullptr);
}

void InitializeSelectedIndexContents() NN_NOEXCEPT
{
    const char* names[] = {
        {"Model"},
        {"Shape"},
        {"SubMesh"},
        {"Bone"}
    };
    NN_STATIC_ASSERT(NN_ARRAY_SIZE(names) == MenuContext::SelectedInfo_Count);

    int* pSelectedIndexArray = g_Context.GetSelectedIndeciesArray();
    for (int index = 0; index < MenuContext::SelectedInfo_Count; ++index)
    {
        g_SelectedIndeces[index].Initialize(names[index], &pSelectedIndexArray[index]);
    }

    SetSelectedModelInfoCallback();
    SetSelectedShapeInfoCallback();
    SetSelectedSubMeshCallback();
    SetSelectedBoneCallback();
}

void FlipDrawConfig(g3ddemo::Content* pContent, void* ) NN_NOEXCEPT
{
    using Pad = g3ddemo::Pad;
    Pad& pad = g3ddemo::GetPad();
    nn::Bit32 value = pContent->GetValue<nn::Bit32>();

    for (int index = 0; index < MenuContext::DrawConfigType_Count; ++index)
    {
        if (pContent != &g_DrawConfig[index])
        {
            continue;
        }

        if (pad.IsTriggered(Pad::BUTTON_LEFT) || pad.IsTriggered(Pad::BUTTON_RIGHT))
        {
            pContent->SetValue(value ^ (1 << index));
        }
    }
}

void DrawDrawConfig(g3ddemo::Content* pContent, void*) NN_NOEXCEPT
{
    SetMenuTextColor(pContent);
    for (int configIndex = 0; configIndex < MenuContext::DrawConfigType_Count; ++configIndex)
    {
        if (pContent != &g_DrawConfig[configIndex])
        {
            continue;
        }

        bool isEnabled = g_Context.IsDrawConfigEnabled(static_cast<MenuContext::DrawConfigType>(configIndex));
        g_ScreenInfo.Print("  %s :%-12s\n", pContent->GetName(), isEnabled ? "[Enable]" : "[Disable]");
    }
}

void WriteShaderOption(g3ddemo::Content*, void*) NN_NOEXCEPT
{
    g3ddemo::ResourceHolder* pHolder = GetResourceHolder();
    for (nns::g3d::RenderModelObj* pRenderModelObj : pHolder->renderModelObjs)
    {
        ::WriteVisializeInfoKey(pRenderModelObj);
    }
}

void InitializeDrawConfigContents() NN_NOEXCEPT
{
    const char* names[] = {
        {"Wireframe"},
        {"Skeleton"},
        {"Bounding Sphere"},
        {"Bounding Box"},
        {"Single Shape"},
        {"Single SubMesh"},
        {"Bone Weight"},
        {"Vertex Normal"}
    };
    NN_STATIC_ASSERT(NN_ARRAY_SIZE(names) == MenuContext::DrawConfigType_Count);

    nn::Bit32* pDrawConfig = g_Context.GetDrawConfig();
    for (int index = 0; index < MenuContext::DrawConfigType_Count; ++index)
    {
        g_DrawConfig[index].Initialize(names[index], pDrawConfig);
        g_DrawConfig[index].SetUpdateCallback(FlipDrawConfig, nullptr, g3ddemo::Content::UpdateFrequency_Selected);
        g_DrawConfig[index].SetDrawCallback(DrawDrawConfig, nullptr);
    }

    g_DrawConfig[MenuContext::DrawConfigType_SigleShape].SetValueChangedCallback(
        [](g3ddemo::Content*, void*)
    {
        nn::Bit32* value = g_Context.GetDrawConfig();
        bool isDisabled = (*value & (1 << MenuContext::DrawConfigType_SigleShape)) == 0;
        if (isDisabled)
        {
            nn::Bit32 mask = ~(1U << MenuContext::DrawConfigType_SingleSubMesh);
            *value &= mask;
        }
    },
    nullptr);

    g_DrawConfig[MenuContext::DrawConfigType_SingleSubMesh].SetValueChangedCallback(
        [](g3ddemo::Content*, void*)
    {
        nn::Bit32* value = g_Context.GetDrawConfig();
        bool isEnabled = (*value & (1 << MenuContext::DrawConfigType_SingleSubMesh)) != 0;
        if (isEnabled)
        {
            *value |= (1 << MenuContext::DrawConfigType_SigleShape);
        }
    }
    , nullptr);

    g_DrawConfig[MenuContext::DrawConfigType_BoneWeight].SetValueChangedCallback(WriteShaderOption, nullptr);
    g_DrawConfig[MenuContext::DrawConfigType_VertexNormal].SetValueChangedCallback(WriteShaderOption, nullptr);
    // TODO: コンフィグを増やしたらこの辺を for 分に･･･
}

void InitializeContents() NN_NOEXCEPT
{
    // TODO: Windows 環境なら F でフォーカスのほうがみな、慣れ親しんでいるのでは?
    g_HowToMessage.Initialize("Move Focus to Selected", "R");
    {
        g3ddemo::Content* pContent = g_HowToMessage.GetContent();
        pContent->SetDrawCallback(
            [](g3ddemo::Content* pContent, void*)
        {
            const nn::util::Color4u8& white = nn::util::Color4u8::White();
            g_ScreenInfo.SetTextColor(white.GetR(), white.GetG(), white.GetB(), white.GetA());
            g_ScreenInfo.Print("%s :%s\n", pContent->GetName(), pContent->GetValue<const char*>());
        }, nullptr);
    }

    // CameraInfo の初期化
    InitializeCameraInfoContents();

    // SelectedInfo の初期化
    InitializeSelectedIndexContents();

    // DrawConfig の初期化
    InitializeDrawConfigContents();
}

void PrintGroupName(g3ddemo::Group* pGroup, void*) NN_NOEXCEPT
{
    g_ScreenInfo.Print("%s\n", pGroup->GetName());
}

}

void InitializeMenu() NN_NOEXCEPT
{
    // PrimitiveRenderer
    g_pRenderer = g3ddemo::GetGfxFramework()->GetPrimitiveRenderer();
    NN_ASSERT_NOT_NULL(g_pRenderer);

    // ScreenInfo
    {
        g_ScreenInfo.Initialize(g3ddemo::GetGfxFramework()->GetDevice());
        g_ScreenInfo.SetWindowPosition(16.0f, 16.0f);
        g_ScreenInfo.SetFontSize(20.0f);
    }

    // Contents
    InitializeContents();

    // Group
    {
        // How To
        {
            g3ddemo::Content* pContent = g_HowToMessage.GetContent();
            g_HowToGroup.Initialize("How to", &pContent, decltype(g_HowToGroup)::ContentCount);
        }

        // Camera Info
        {
            g3ddemo::Content* pContents[] = { g_CameraPosition.GetContent(),g_CameraTarget.GetContent() };
            g_CameraInfoGroup.Initialize("Camera Information", pContents, NN_ARRAY_SIZE(pContents));
            g_CameraInfoGroup.GetGroup()->SetDrawCallback(PrintGroupName, nullptr);
        }

        // Selected Info
        {
            const int arraySize = NN_ARRAY_SIZE(g_SelectedIndeces);
            g3ddemo::Content* pContents[arraySize];
            for (int index = 0; index < arraySize; ++index)
            {
                pContents[index] = &g_SelectedIndeces[index];
            }
            g_SelectedInfoGroup.Initialize("Selected Information", pContents, arraySize);
            g_SelectedInfoGroup.GetGroup()->SetDrawCallback(PrintGroupName, nullptr);
        }

        // Draw Config
        {
            const int arraySize = NN_ARRAY_SIZE(g_DrawConfig);
            g3ddemo::Content* pContents[arraySize];
            for (int index = 0; index < arraySize; ++index)
            {
                pContents[index] = &g_DrawConfig[index];
            }
            g_DrawConfigGroup.Initialize("Draw Config", pContents, arraySize);
            g_DrawConfigGroup.GetGroup()->SetDrawCallback([](g3ddemo::Group* pGroup, void*) {
                const nn::util::Color4u8& white = nn::util::Color4u8::White();
                g_ScreenInfo.SetTextColor(white.GetR(), white.GetG(), white.GetB(), white.GetA());
                g_ScreenInfo.Print("%s\n", pGroup->GetName());
            }, nullptr);
        }
    }

    // Page
    {
        g3ddemo::Group* pGroupArray[] = {
            g_HowToGroup.GetGroup(),
            g_CameraInfoGroup.GetGroup(),
            g_SelectedInfoGroup.GetGroup(),
            g_DrawConfigGroup.GetGroup()
        };
        g_Page.Initialize(nullptr, pGroupArray, NN_ARRAY_SIZE(pGroupArray));
    }

    // Menu
    {
        g3ddemo::Page* pPage = g_Page.GetPage();
        g_Menu.Initialize("3d sample viewer", &pPage, decltype(g_Menu)::PageCount);
        g_Menu.GetMenu()->SetDrawCallback(
            [](g3ddemo::Menu* pMenu, void*)
        {
            const nn::util::Color4u8& white = nn::util::Color4u8::White();
            g_ScreenInfo.SetTextColor(white.GetR(), white.GetG(), white.GetB(), white.GetA());
            g_ScreenInfo.Print("[%s]\n", pMenu->GetName());
        }, nullptr);

        g_Menu.GetMenu()->SetGroupSelectability(g_HowToGroup.GetGroup(), false);
        g_Menu.GetMenu()->SetGroupSelectability(g_CameraInfoGroup.GetGroup(), false);
        g_Menu.GetMenu()->SetGroupSelectability(g_SelectedInfoGroup.GetGroup(), false);
    }
}

void UpdateMenu() NN_NOEXCEPT
{
    using Pad = g3ddemo::Pad;
    Pad& pad = g3ddemo::GetPad();

    if (pad.IsTriggered(Pad::BUTTON_UP))
    {
        g_Page.GetPage()->ChangePreviousContent();
    }
    else if (pad.IsTriggered(Pad::BUTTON_DOWN))
    {
        g_Page.GetPage()->ChangeNextContent();
    }

    g_Menu.GetMenu()->Update();
}

void DrawMenu(nn::gfx::CommandBuffer* pCommandBuffer) NN_NOEXCEPT
{
    g_ScreenInfo.StartPrint();
    g_Menu.GetMenu()->Draw();
    g_ScreenInfo.EndPrint();
    g_ScreenInfo.AdjustWindowSize();
    g_ScreenInfo.Draw(pCommandBuffer);
}

const MenuContext* GetMenuContext() NN_NOEXCEPT
{
    return &g_Context;
}

void SetSelectedIndex(MenuContext::SelectedInfo info, int value, bool isFourceBind) NN_NOEXCEPT
{
    int oldValue = g_SelectedIndeces[info].GetValue<int>();
    if (oldValue != value || isFourceBind)
    {
        g_SelectedIndeces[info].SetValue(value);
    }
}

void SetSelectedIndex(MenuContext::SelectedInfo info, int value) NN_NOEXCEPT
{
    SetSelectedIndex(info, value, false);
}

void SelectInfo(MenuContext::SelectedInfo info) NN_NOEXCEPT
{
    bool isSelected = g_Menu.GetMenu()->SelectContent(&g_SelectedIndeces[static_cast<int>(info)]);
    NN_ASSERT(isSelected);
}

MenuContext::SelectedInfo GetSelectedInfo() NN_NOEXCEPT
{
    g3ddemo::Content* pContent = g_Menu.GetMenu()->GetSelectedContent();
    NN_ASSERT_NOT_NULL(pContent);

    for (int infoIndex = 0; infoIndex < MenuContext::SelectedInfo_Count; ++infoIndex)
    {
        if (&g_SelectedIndeces[infoIndex] == pContent)
        {
            return static_cast<MenuContext::SelectedInfo>(infoIndex);
        }
    }
    return MenuContext::SelectedInfo_Model;
}

void WriteVisializeInfoKey(nns::g3d::RenderModelObj* pRenderModelObj) NN_NOEXCEPT
{
    int passCount = pRenderModelObj->GetRenderModel()->GetPassCount();
    if (passCount < DrawPassType_Count)
    {
        return;
    }

    const nn::g3d::ModelObj* pModelObj = pRenderModelObj->GetModelObj();
    for (int shapeIndex = 0, shapeCount = pModelObj->GetShapeCount(); shapeIndex < shapeCount; ++shapeIndex)
    {
        nns::g3d::RenderUnitObj* pUnitObj = pRenderModelObj->GetRenderUnitObj(shapeIndex);
        nn::g3d::ShaderSelector* pSelector = pUnitObj->GetShaderSelector(DrawPassType_Visualize);

        for (int optionIndex = 0; optionIndex < NN_ARRAY_SIZE(g_ShaderOptionSettings); ++optionIndex)
        {
            int index = pSelector->FindDynamicOptionIndex(g_ShaderOptionSettings[optionIndex].id);
            NN_ASSERT_NOT_EQUAL(index, nn::util::ResDic::Npos);

            const nn::g3d::ResShaderOption* pResShaderOption = pSelector->GetDynamicOption(index);
            int choiceIndex = pResShaderOption->FindChoiceIndex(g_Context.IsDrawConfigEnabled(g_ShaderOptionSettings[optionIndex].configType) ? "1" : "0");
            pSelector->WriteDynamicKey(index, choiceIndex);
        }
    }
}
