﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_DEMO_PAD_WIN_H_
#define NW_DEMO_PAD_WIN_H_

#include <nw/demo/pad/demo_Pad.h>
#include <nw/demo/pad/win/demo_KeyboardMouseDeviceWin.h>
#include <nw/demo/pad/win/demo_JoyPadDeviceWin.h>

namespace nw
{
namespace demo
{

//---------------------------------------------------------------------------
//! @brief        PC 用デモパッドクラスです。
//!
//! @details :category     入力デバイス
//---------------------------------------------------------------------------
class PadWin : public Pad
{
public:
    NW_UT_RUNTIME_TYPEINFO(Pad)

    //! @brief デフォルト値です。
    enum
    {
        STICK_CLAMP_MIN_DEFAULT = 0x1000,   //!< アナログスティックのクランプの最小値のデフォルトです。
        STICK_CLAMP_MAX_DEFAULT = 0x7fff,   //!< アナログスティックのクランプの最大値のデフォルトです。
        STICK_CLAMP_VALUE_MIN = 0x1,        //!< アナログスティックのクランプの最小値です。
        STICK_CLAMP_VALUE_MAX = 0x7fff,     //!< アナログスティックのクランプの最大値です。
        JOY_PAD_NUMBER_INVALID = JoyPadDeviceWin::NUM_MAX   //!< 無効なジョイパッドの番号です。
    };


    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //!
    //! @param[in]    joyPadNumber 参照するジョイパッドの番号です。
    //---------------------------------------------------------------------------
    explicit PadWin( int joyPadNumber );

    //---------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //---------------------------------------------------------------------------
    virtual ~PadWin() {}


    //----------------------------------------
    //! @name 仮想キーコードの割り当て
    //  @{

    //---------------------------------------------------------------------------
    //! @brief        パッドのボタンをどの仮想キーコードに割り当てるかを設定します。
    //!
    //! @param[in]    keyConfig ボタンの割り当てです。
    //---------------------------------------------------------------------------
    void SetVKeyConfig(const u8* keyConfig);

    //---------------------------------------------------------------------------
    //! @brief        パッドのボタンをどの仮想キーコードに割り当てるかを設定します。
    //!
    //! @param[in]    bit       設定するパッドのビット番号です。
    //! @param[in]    vkey      割り当てる仮想キーコードです。
    //---------------------------------------------------------------------------
    void SetVKeyConfig(PadIdx bit, u8 vkey);

    //---------------------------------------------------------------------------
    //! @brief        パッドのボタンをどの仮想キーコードに割り当てるかを、デフォルト値に設定します。
    //---------------------------------------------------------------------------
    void SetVKeyConfigDefault();

    //  @}

    //----------------------------------------
    //! @name ジョイパッドの設定・情報取得
    //  @{

    //---------------------------------------------------------------------------
    //! @brief        このパッドが参照しているジョイパッドの番号を取得します。
    //!
    //! @return       参照しているジョイパッドの番号を返します。
    //---------------------------------------------------------------------------
    int GetJoyPadNumber() const { return m_JoyPadNumber; }

    //---------------------------------------------------------------------------
    //! @brief        このコントローラが参照するジョイパッドの番号を設定します。
    //!
    //! @param[in]    number    ジョイパッドの番号です。
    //---------------------------------------------------------------------------
    void SetJoyPadNumber(int number) { m_JoyPadNumber = number; }

    //---------------------------------------------------------------------------
    //! @brief        参照しているジョイパッドが有効か否かを取得します。
    //!
    //! @return       有効な場合、 true を返します。
    //---------------------------------------------------------------------------
    bool IsJoyPadEnable() const;

    //---------------------------------------------------------------------------
    //! @brief        アナログスティックのクランプ値を取得します。
    //!
    //! @param[out]   min       最小値を受け取るポインタです。
    //! @param[out]   max       最大値を受け取るポインタです。
    //---------------------------------------------------------------------------
    void GetStickClamp(int* min, int* max) const;

    //---------------------------------------------------------------------------
    //! @brief        アナログスティックのクランプ値を設定します。
    //!
    //! @param[in]    min       クランプの最小値です。
    //! @param[in]    max       クランプの最大値です。
    //---------------------------------------------------------------------------
    void SetStickClamp(int min, int max);

    //---------------------------------------------------------------------------
    //! @brief        ジョイパッドのボタンをパッドのボタンのどれに割り当てるかを設定します。
    //!
    //! @param[in]    joyPadConfig ボタンの割り当てです。
    //---------------------------------------------------------------------------
    void SetJoyPadConfig(const u8* joyPadConfig);

    //---------------------------------------------------------------------------
    //! @brief        パッドのボタンをどの仮想キーコードに割り当てるかを設定します。
    //!
    //! @param[in]    bit       設定するパッドのビット番号です。
    //! @param[in]    no        割り当てるジョイパッドのボタン番号です。
    //---------------------------------------------------------------------------
    void SetJoyPadConfig(PadIdx bit, u8 no);

    //---------------------------------------------------------------------------
    //! @brief        パッドのボタンをジョイパッドのどのボタンに割り当てるかを、デフォルト値に設定します。
    //---------------------------------------------------------------------------
    void SetJoyPadConfigDefault();

    //---------------------------------------------------------------------------
    //! @brief        ウィンドウが非アクティブな場合に、 JoyPad を有効にするかを返します。
    //!
    //! @return       有効にする場合、 true を返します。
    //---------------------------------------------------------------------------
    bool IsJoyPadEnableWhenWindowInactive() const { return m_Flag.IsMaskOn( MASK_IS_JOY_PAD_ENABLE_WHEN_WINDOW_INACTIVE ); }

    //---------------------------------------------------------------------------
    //! @brief        ウィンドウが非アクティブな場合に、 JoyPad を有効にするかを設定します。
    //!
    //! @param[in]    active    有効にする場合、 true を指定します。
    //---------------------------------------------------------------------------
    void SetJoyPadEnableWhenWindowInactive(bool active) { m_Flag.ChangeMask( MASK_IS_JOY_PAD_ENABLE_WHEN_WINDOW_INACTIVE, active ); }

    //---------------------------------------------------------------------------
    //! @brief        アナログスティックの入力を円形に正規化するか ( length が 1 以上にならないようにするか) の設定を取得します。
    //!
    //! @return       正規化を行う場合、 true を返します。
    //---------------------------------------------------------------------------
    bool IsEnableNormalizeAnalogStickAsCircle() const { return m_Flag.IsMaskOn( MASK_NORMALIZE_ANALOG_STICK_AS_CIRCLE ); }

    //---------------------------------------------------------------------------
    //! @brief        アナログスティックの入力を円形に正規化するか ( length が 1 以上にならないようにするか ) を設定します。
    //!
    //! @param[in]    enable    正規化を行う場合は true を指定します。
    //---------------------------------------------------------------------------
    void SetEnableNormalizeAnalogStickAsCircle(bool enable) { m_Flag.ChangeMask( MASK_NORMALIZE_ANALOG_STICK_AS_CIRCLE, enable ); }

    //  @}

    //---------------------------------------------------------------------------
    //! @brief        キーボード・マウスデバイスを設定します。
    //!
    //! @param[in]    device    設定するデバイスへのポインタです。
    //---------------------------------------------------------------------------
    void SetKeyboardMouseDevice( KeyboardMouseDeviceWin* device ){ m_KeyboardMouseDevice = device; }

    //---------------------------------------------------------------------------
    //! @brief        ジョイパッドデバイスを設定します。
    //!
    //! @param[in]    device    設定するデバイスへのポインタです。
    //---------------------------------------------------------------------------
    void SetJoyPadDevice( JoyPadDeviceWin* device ){ m_JoyPadDevice = device; }

    //---------------------------------------------------------------------------
    //! @brief        プラットフォームに依存した形式でパッドの状態を取得します。
    //!
    //! @param[out]   pad        出力するパッド状態へのポインタです。
    //---------------------------------------------------------------------------
    virtual void GetPadStatus( RawPadStatus* pad );


protected:
    enum FlagMask
    {
        MASK_IS_JOY_PAD_ENABLE_WHEN_WINDOW_INACTIVE = 1,      //!< ウィンドウがアクティブでないときにジョイパッドを有効にするかのフラグ値です。
        MASK_NORMALIZE_ANALOG_STICK_AS_CIRCLE       = 1 << 1, //!< アナログスティックの入力を円形に正規化するかのフラグ値です。
    };


    //! @brief 更新処理の実装です。
    virtual void UpdateImpl();


    int m_JoyPadNumber;             //!< 参照するジョイパッドの番号です。
    int m_StickClampMin;            //!< アナログスティックのクランプの下限値です。
    int m_StickClampMax;            //!< アナログスティックのクランプの上限値です。

    u8 m_VKeyConfig[PAD_IDX_MAX];   //!< マウスとキーボードで使われる仮想キーコードの割り当てです。
    u8 m_JoyPadConfig[PAD_IDX_MAX]; //!< ジョイパッドの割り当てです。

    nw::ut::BitFlag32 m_Flag;
    static const u8 DEFAULT_V_KEY_CONFIG[PAD_IDX_MAX];      //!< デフォルトの仮想キーコードの割り当てです。
    static const u8 DEFAULT_JOY_PAD_CONFIG[PAD_IDX_MAX];    //!< デフォルトのジョイパッドの割り当てです。

    KeyboardMouseDeviceWin* m_KeyboardMouseDevice;      //!< 使用するキーボード・マウスデバイスです。
    JoyPadDeviceWin* m_JoyPadDevice;                    //!< 使用するジョイパッドデバイスです。
};


} // namespace demo
} // namespace nw

#endif // NW_DEMO_PAD_WIN_H_
