﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_DEMO_PAD_H_
#define NW_DEMO_PAD_H_

#include <nw/types.h>
#include <nw/demo/pad/demo_PadBase.h>

#if defined( NW_USE_NINTENDO_SDK )
  #include <nn/hid.h>
#elif defined( NW_PLATFORM_CAFE )
  #include <cafe/pad.h>
#elif defined( NW_PLATFORM_WIN32 )
  #include <winext/cafe/pad.h>
#endif

namespace nw
{
namespace demo
{

#if defined( NW_USE_NINTENDO_SDK )
//! プラットフォームのパッド状態です。
typedef nn::hid::DebugPadState RawPadStatus;
#elif defined( NW_PLATFORM_WIN32 )
//! プラットフォームのパッド状態です。
typedef nw::internal::winext::PADStatus RawPadStatus;
#elif defined( NW_PLATFORM_CAFE )
//! プラットフォームのパッド状態です。
typedef PADStatus RawPadStatus;
#else
#error "not implemented"
#endif

//---------------------------------------------------------------------------
//! @brief パッド状態を表す構造体です。
//!
//! @details :category     入力デバイス
//---------------------------------------------------------------------------
struct PadStatus
{
    //! パッドの最大数です。
#if defined( NW_USE_NINTENDO_SDK )
    static const int MAX_CONTROLLERS = 1;
#elif defined( NW_PLATFORM_WIN32 )
    static const int MAX_CONTROLLERS = NW_WINEXT_PAD_MAX_CONTROLLERS;
#elif defined( NW_PLATFORM_CAFE )
    static const int MAX_CONTROLLERS = PAD_MAX_CONTROLLERS;
#else
    #error "not implemented"
#endif

    RawPadStatus padStatuses[ MAX_CONTROLLERS ];    //!< プラットフォームのパッド状態です。
    bool        isPadEnable;                        //!< パッドが有効かを表します。
    u8*         keyStatus;                          //!< キー状態です。 NULL を指定した場合は使用しません。
    bool        isKeyEnable;                        //!< キーが有効かを表します。

    //! @brief コンストラクタです。
    PadStatus()
      : isPadEnable( false ),
        keyStatus( NULL ),
        isKeyEnable( false )
    {
        for ( int i = 0; i < MAX_CONTROLLERS; ++i )
        {
#if defined( NW_USE_NINTENDO_SDK )
            padStatuses[i].samplingNumber = 0;
            padStatuses[i].buttons.state.storage = 0;
            padStatuses[i].analogStickL.x = 0;
            padStatuses[i].analogStickL.y = 0;
            padStatuses[i].analogStickR.x = 0;
            padStatuses[i].analogStickR.y = 0;
#else
            padStatuses[i].button = 0;
            padStatuses[i].stickX = 0;
            padStatuses[i].stickY = 0;
            padStatuses[i].substickX = 0;
            padStatuses[i].substickY = 0;
            padStatuses[i].triggerLeft = 0;
            padStatuses[i].triggerRight = 0;
            padStatuses[i].analogA = 0;
            padStatuses[i].analogB = 0;
#if defined( NW_PLATFORM_WIN32 )
            padStatuses[i].err = NW_WINEXT_PAD_ERR_NO_CONTROLLER;
#else // defined( NW_PLATFORM_CAFE )
            padStatuses[i].err = PAD_ERR_NO_CONTROLLER;
#endif
#endif
        }
    }
};

//---------------------------------------------------------------------------
//! @brief        demoPad クラスです。
//!
//! @details :category     入力デバイス
//---------------------------------------------------------------------------
class Pad : public PadBase
{
public:
    NW_UT_RUNTIME_TYPEINFO(PadBase)

    //! @brief パッドのビット番号による定義です。
    enum PadIdx
    {
        IDX_A = 0,
        IDX_B,
        IDX_C,
        IDX_X,
        IDX_Y,
        IDX_Z,
        IDX_2,
        IDX_1,
        IDX_HOME,
        IDX_MINUS,
        IDX_PLUS,
        IDX_START,
        IDX_SELECT,
        IDX_ZL,
        IDX_ZR,
        IDX_L,
        IDX_R,
        IDX_TOUCH,
        IDX_UP,
        IDX_DOWN,
        IDX_LEFT,
        IDX_RIGHT,
        IDX_LS_UP,
        IDX_LS_DOWN,
        IDX_LS_LEFT,
        IDX_LS_RIGHT,
        IDX_RS_UP,
        IDX_RS_DOWN,
        IDX_RS_LEFT,
        IDX_RS_RIGHT,
        PAD_IDX_MAX
    };

    //! @brief パッドのマスクによる定義です。
    enum PadMask
    {
        MASK_A          = 1 << IDX_A,
        MASK_B          = 1 << IDX_B,
        MASK_C          = 1 << IDX_C,
        MASK_X          = 1 << IDX_X,
        MASK_Y          = 1 << IDX_Y,
        MASK_Z          = 1 << IDX_Z,
        MASK_2          = 1 << IDX_2,
        MASK_1          = 1 << IDX_1,
        MASK_HOME       = 1 << IDX_HOME,
        MASK_MINUS      = 1 << IDX_MINUS,
        MASK_PLUS       = 1 << IDX_PLUS,
        MASK_START      = 1 << IDX_START,
        MASK_SELECT     = 1 << IDX_SELECT,
        MASK_ZL         = 1 << IDX_ZL,
        MASK_ZR         = 1 << IDX_ZR,
        MASK_L          = 1 << IDX_L,
        MASK_R          = 1 << IDX_R,
        MASK_TOUCH      = 1 << IDX_TOUCH,
        MASK_UP         = 1 << IDX_UP,
        MASK_DOWN       = 1 << IDX_DOWN,
        MASK_LEFT       = 1 << IDX_LEFT,
        MASK_RIGHT      = 1 << IDX_RIGHT,
        MASK_LS_UP      = 1 << IDX_LS_UP,
        MASK_LS_DOWN    = 1 << IDX_LS_DOWN,
        MASK_LS_LEFT    = 1 << IDX_LS_LEFT,
        MASK_LS_RIGHT   = 1 << IDX_LS_RIGHT,
        MASK_RS_UP      = 1 << IDX_RS_UP,
        MASK_RS_DOWN    = 1 << IDX_RS_DOWN,
        MASK_RS_LEFT    = 1 << IDX_RS_LEFT,
        MASK_RS_RIGHT   = 1 << IDX_RS_RIGHT
    };

    enum
    {
        //! 十字キー全てのマスク
        MASK_CROSS = MASK_UP | MASK_DOWN | MASK_LEFT | MASK_RIGHT,
        //! 左スティックを十字キーとみなした時の全てのマスク
        MASK_CROSS_LS = MASK_LS_UP | MASK_LS_DOWN | MASK_LS_LEFT | MASK_LS_RIGHT,
        //! 右スティックを十字キーとみなした時の全てのマスク
        MASK_CROSS_RS = MASK_RS_UP | MASK_RS_DOWN | MASK_RS_LEFT | MASK_RS_RIGHT,
        //! 全てのボタンのマスク
        MASK_ALL = 0x0fffffff
    };


    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    Pad();

    //---------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //---------------------------------------------------------------------------
    virtual ~Pad() {}


    //---------------------------------------------------------------------------
    //! @brief        パッドを更新します。
    //!
    //!               設定された PadStatus またはデバイスを用いてパッドを更新します。
    //!               PadStatus を優先して使用します。
    //---------------------------------------------------------------------------
    virtual void Update();

    //---------------------------------------------------------------------------
    //! @brief        プラットフォームに依存した形式でパッドの状態を取得します。
    //!
    //! @param[out]   pad        出力するパッド状態へのポインタです。
    //---------------------------------------------------------------------------
    virtual void GetPadStatus( RawPadStatus* pad ) { NW_UNUSED_VARIABLE(pad); }

    //---------------------------------------------------------------------------
    //! @brief        パッド状態を設定します。
    //!
    //! @param[in]    status    設定するパッド状態です。
    //---------------------------------------------------------------------------
    void SetPadStatus( PadStatus* status ) { m_PadStatus = status; }

    //---------------------------------------------------------------------------
    //! @brief        パッド状態を取得します。
    //!
    //! @return       パッド状態を返します。
    //---------------------------------------------------------------------------
    const PadStatus* GetMouseStatus() const { return m_PadStatus; }


protected:
    //! @brief パッド更新の実装です。
    virtual void UpdateImpl() = 0;

    PadStatus* m_PadStatus;
};

} // namespace demo
} // namespace nw

#endif // NW_DEMO_PAD_H_
