﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_DEMO_SYSTEM_WIN_GL_DRC_H_
#define NW_DEMO_SYSTEM_WIN_GL_DRC_H_

#include <nw/demo/gl/demo_FrameBufferGL.h>
#include <nw/demo/gl/demo_TextureFrameBufferGL.h>
#include <nw/demo/demo_TestUtility.h>
#include <nw/demo/demo_Utility.h>
#include <nw/demo/pad/win/demo_PadWin.h>
#include <nw/demo/pad/win/demo_WPadWin.h>
#include <nw/demo/pad/win/demo_VPadWin.h>
#include <nw/demo/pad/win/demo_MouseWin.h>
#include <nw/dev/dev_MeterDrawer.h>
#include <nw/dev/dev_FileDeviceManager.h>
#include <nw/ut/ut_Color.h>
#include <nw/ut/os/ut_Time.h>

#if defined(NW_USE_NINTENDO_SDK)
typedef int BOOL;
#endif

namespace nw
{
namespace demo
{

//---------------------------------------------------------------------------
//! @brief        DRC ウィンドウ付き GL 用デモシステムクラスです。
//!
//!   - このクラスを使う場合、winmm.libをリンクする必要があります。
//!   - 複数のインスタンスを作ることはできません。
//---------------------------------------------------------------------------
class SystemWinGLDRC
{
public:
    //! @brief コンストラクタに渡す生成パラメータです。
    struct CreateArg
    {
        nw::ut::IAllocator* allocator;                      //!< デモシステムで用いるアロケータです。
        s32                 width;                          //!< ウインドウおよびフレームバッファの幅です。
        s32                 height;                         //!< ウインドウおよびフレームバッファの高さです。
        s32                 widthDRC;                       //!< DRC ウインドウおよびフレームバッファの幅です。
        s32                 heightDRC;                      //!< DRC ウインドウおよびフレームバッファの高さです。
        DisplayRotateState  rotateDRC;                      //!< DRC ウインドウの回転状態です。
        u32                 waitVBlank;                     //!< 一回の更新で VBlank を待つ回数です。（通常、1 なら 60 fps 、2 なら 30fps のアプリになります）0 のとき VBlank と同期しません。
        f32                 fps;                            //!< waitVBlank が 0 か、ハード的に VBlank 待ちが行えない場合に、何 fps でタイマー駆動するかを指定します。
        nw::ut::Color4u8    clearColor;                     //!< 画面クリア色です。
        f32                 clearDepth;                     //!< 画面クリアデプスです。
        bool                createDefaultFrameBuffer;       //!< フレームバッファを内部で生成するかどうかを指定します。
        u8*                 fontBinary;                     //!< デバッグ文字描画に用いるフォントバイナリへのポインタです。
        u32                 fontBinarySize;                 //!< デバッグ文字描画に用いるフォントバイナリのサイズです。
        const char*         fontPath;                       //!< デバッグ文字描画に用いるフォントのパスを指定します。（バイナリを直接設定せずファイルからロードする場合に指定します。デモファイルシステムが初期化されている必要があります。）
        bool                drawMeter;                      //!< 負荷メーターを表示するか指定します。
        bool                usePointerCursor;               //!< ポインタカーソルを有効にするかを指定します。
        bool                primitiveRendererInitialize;    //!< PrimitiveRenderer を初期化するかを指定します。
        bool                fileDeviceManagerInitialize;    //!< デモファイルシステムを初期化するかを指定します。
        bool                isSRGBWrite;                    //!< フレームバッファに書き込む際にリニアからsRGBに変換する処理を有効にするかを指定します。

        //! @brief コンストラクタです。
        CreateArg() :
            allocator( NULL ),
            width( 640 ),
            height( 480 ),
            widthDRC( 854 ),
            heightDRC( 480 ),
            rotateDRC( DISPLAY_ROTATE_NONE ),
            waitVBlank( 1 ),
            fps( 60.0 ),
            clearColor( SRGBToLinear(nw::ut::Color4u8::GRAY) ),
            clearDepth( 1.f ),
            createDefaultFrameBuffer( true ),
            fontBinary( NULL ),
            fontBinarySize( 0 ),
            fontPath( NULL ),
            drawMeter( false ),
            usePointerCursor( false ),
            primitiveRendererInitialize( false ),
            fileDeviceManagerInitialize( false ),
            isSRGBWrite( true )
        {}
    };

    //! @brief clear メソッドに渡すビットマスクです。
    enum ClearFlag
    {
        CLEAR_FLAG_NONE     = 0,        //!< 何も指定しない。
        CLEAR_FLAG_COLOR    = 1 << 0,   //!< 色要素をクリアする。
        CLEAR_FLAG_DEPTH    = 1 << 1,   //!< 深度要素をクリアする。
        CLEAR_FLAG_STENCIL  = 1 << 2    //!< ステンシル要素をクリアする。
    };

    //! @brief 画面表示状態です。
    enum DisplayState
    {
        HIDE,       //!< 非表示状態です。
        READY,      //!< 表示準備状態(次のVSyncで表示)です。
        SHOW        //!< 表示中です。
    };

    //! @brief SetProcessPriority に渡すパラメータです。
    enum ProcessPriority
    {
        IDLE,       //!< 低優先度です。
        NORMAL,     //!< 普通優先度です。
        HIGH,       //!< 高優先度です。
        REAL_TIME   //!< 最高優先度です。
    };


    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //!
    //! @param[in]    arg      生成パラメータです。
    //---------------------------------------------------------------------------
    /* ctor */ explicit SystemWinGLDRC( const CreateArg& arg );

    //---------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //---------------------------------------------------------------------------
    virtual ~SystemWinGLDRC() { s_Instance = NULL; }


    //---------------------------------------------------------------------------
    //! @brief        初期化処理です。
    //---------------------------------------------------------------------------
    void Initialize();

    //---------------------------------------------------------------------------
    //! @brief        終了処理です。
    //---------------------------------------------------------------------------
    void Finalize();

    //---------------------------------------------------------------------------
    //! @brief        グラフィックエンジンの初期化を行います。
    //!
    //!   CreateArgをもとにして、以下の処理を行います。
    //!   - Graphicsのインスタンス生成。
    //!   - 引数に指定したサイズで仮想フレームバッファを作成。
    //!   - VBlank待ち回数の設定。
    //---------------------------------------------------------------------------
    void InitializeGraphicsSystem( int argc = 0, char** argv = NULL );

    //---------------------------------------------------------------------------
    //! @brief        グラフィックスエンジンの終了処理を行います。
    //---------------------------------------------------------------------------
    void FinalizeGraphicsSystem();


    //---------------------------------------------------------------------------
    //! @brief        Window ハンドルを取得します。
    //!
    //! @return       Window ハンドルを返します。
    //---------------------------------------------------------------------------
    void* GetWindowHandle() const { return m_HWnd; }

    //---------------------------------------------------------------------------
    //! @brief        デバイスコンテキストハンドルを取得します。
    //!
    //! @return       デバイスコンテキストハンドルを返します。
    //---------------------------------------------------------------------------
    void* GetHDC() const { return m_HDC; }

    //---------------------------------------------------------------------------
    //! @brief        GL コンテキストハンドルを取得します。
    //!
    //! @return       GL コンテキストハンドルを返します。
    //---------------------------------------------------------------------------
    void* GetHGLRC() const { return m_HGLRC; }

    //---------------------------------------------------------------------------
    //! @brief        独自のメッセージプロシージャを設定します。
    //!
    //!   ここに関数を設定すると、MsgProcImpl の最初で呼ばれます。
    //!   設定した関数の返り値が 0 以外の場合、MsgProcImpl の続きは実行されず、その値を返します。
    //!   NULL を設定すると、独自の関数は呼ばれなくなります。
    //!
    //! @param[in]    userProcFunc 独自のメッセージプロシージャを指定します。
    //---------------------------------------------------------------------------
    void SetMsgProcCallback( void* userProcFunc ) { m_MsgProcCallback = userProcFunc; }

    //---------------------------------------------------------------------------
    //! @brief        設定されている独自のメッセージプロシージャを取得します。
    //!
    //! @return       メッセージプロシージャを返します。
    //---------------------------------------------------------------------------
    void* GetMsgProcCallback() const { return m_MsgProcCallback; }

    //---------------------------------------------------------------------------
    //! @brief        ウインドウのタイトルを設定します。
    //!
    //! @param[in] title    タイトルに設定する文字列です。
    //!
    //! @returns
    //! 自身への参照を返します。
    //---------------------------------------------------------------------------
    SystemWinGLDRC& SetWindowTitle(const char* title);

    //---------------------------------------------------------------------------
    //! @brief        VBlank 待ち回数を変更します。
    //!
    //! @param[in]    interval VBlank 待ち回数です。
    //!
    //! @return       設定に成功した場合、 true を返します。
    //---------------------------------------------------------------------------
    bool SetWaitVBlankInterval( u32 interval );

    //---------------------------------------------------------------------------
    //! @brief        fps を設定します。
    //!
    //!   VBlank 待ち設定が 0 でないときは無視されます。
    //!
    //! @param[in]    fps      設定する fps です。
    //---------------------------------------------------------------------------
    void SetFps( f32 fps );

    //---------------------------------------------------------------------------
    //! @brief        DefaultFrameBuffer を返します。
    //!
    //! @return       DefaultFrameBuffer です。
    //---------------------------------------------------------------------------
    const FrameBuffer* GetDefaultFrameBuffer() const { return m_DefaultFrameBuffer; }

    //---------------------------------------------------------------------------
    //! @brief        前回更新からの経過時間から、現在のフレームレートを計算して取得します。
    //!
    //! @return       現在のフレームレートを返します。
    //---------------------------------------------------------------------------
    f32 CalcFps();

    //---------------------------------------------------------------------------
    //! @brief        タイトルバーに文字列を表示します。
    //!
    //! @param[in]    caption   表示する文字列です。
    //---------------------------------------------------------------------------
    void SetCaption( const char16* caption );

    //---------------------------------------------------------------------------
    //! @brief        プロセスの動作優先度を変更します。
    //!
    //! @param[in]    priority 設定する優先度です。
    //!
    //! @return       設定に成功した場合、 true を返します。
    //---------------------------------------------------------------------------
     bool SetProcessPriority( ProcessPriority priority );

    //---------------------------------------------------------------------------
    //! @brief        設定されているアロケータを取得します。
    //!
    //! @return       アロケータを返します。
    //---------------------------------------------------------------------------
    nw::ut::IAllocator* GetAllocator(){ return m_Arg.allocator; }

    //---------------------------------------------------------------------------
    //! @brief        ウインドウの幅（フレームバッファの幅）を取得します。
    //!
    //! @return       ウインドウの幅を返します。
    //---------------------------------------------------------------------------
    s32 GetWidth(){ return m_Arg.width; }

    //---------------------------------------------------------------------------
    //! @brief        ウインドウの高さ（フレームバッファの高さ）を取得します。
    //!
    //! @return       ウインドウの高さを返します。
    //---------------------------------------------------------------------------
    s32 GetHeight(){ return m_Arg.height; }

    //---------------------------------------------------------------------------
    //! @brief        DRC 画面の横幅を取得します。
    //!
    //! @return       DRC 画面の横幅を返します。
    //---------------------------------------------------------------------------
    s32 GetDRCWidth(){ return m_Arg.widthDRC; }

    //---------------------------------------------------------------------------
    //! @brief        DRC 画面の縦幅を取得します。
    //!
    //! @return       DRC 画面の縦幅を返します。
    //---------------------------------------------------------------------------
    s32 GetDRCHeight(){ return m_Arg.heightDRC; }

    //---------------------------------------------------------------------------
    //! @brief        画面表示状態を取得します。
    //!
    //! @return       画面表示状態を返します。
    //---------------------------------------------------------------------------
    DisplayState GetDisplayState() const { return m_DisplayState; }

    //---------------------------------------------------------------------------
    //! @brief        終了判定を取得します。
    //!
    //! @return       終了判定フラグを返します。
    //---------------------------------------------------------------------------
    bool IsExiting() const
    {
        // メモリーリーク検出が有効な場合、パッド操作で終了判定を true にします。
#if defined(NW_DEBUG_CHECK_MEMORY_LEAK)
        if( m_Pad->IsHoldAll( nw::demo::Pad::MASK_L | nw::demo::Pad::MASK_R | nw::demo::Pad::MASK_X ) )
        {
            return true;
        }
#endif //defined(NW_DEBUG_CHECK_MEMORY_LEAK)

        // 普通に終了した場合は、再入テストに入らずに素直に終了する。
        if (m_Exit)
        {
            DebugUtility::DisableReentryTest(true);
        }

        return m_Exit;
    }


    //---------------------------------------------------------------------------
    //! @brief        ビューポートを設定します。
    //!
    //! @param[in]    x        ビューポートの左上原点 X 座標です。
    //! @param[in]    y        ビューポートの左上原点 Y 座標です。
    //! @param[in]    width    ビューポートの幅です。
    //! @param[in]    height   ビューポートの高さです。
    //! @param[in]    nearZ    ビューポートの near 値です。
    //! @param[in]    farZ     ビューポートの far 値です。
    //---------------------------------------------------------------------------
    void SetViewport( f32 x, f32 y, f32 width, f32 height, f32 nearZ, f32 farZ );

    //---------------------------------------------------------------------------
    //! @brief        シザーを設定します。
    //!
    //! @param[in]    x        シザーの左上原点 X 座標です。
    //! @param[in]    y        シザーの左上原点 Y 座標です。
    //! @param[in]    width    シザーの幅です。
    //! @param[in]    height   シザーの高さです。
    //---------------------------------------------------------------------------
    void SetScissor( f32 x, f32 y, f32 width, f32 height );

    //---------------------------------------------------------------------------
    //! @brief        画面クリアを行います。
    //---------------------------------------------------------------------------
    void ClearFrameBuffers();

    //---------------------------------------------------------------------------
    //! @brief        画面クリアを行います。
    //!
    //! @param[in]    flag     クリアする要素をビットマスクで指定します。
    //! @param[in]    color    この色でクリアします。 CLEAR_FLAG_COLOR が指定されていない場合は無視されます。
    //! @param[in]    depth    この値でZバッファを埋めます。 CLEAR_FLAG_DEPTH が指定されていない場合は無視されます。
    //! @param[in]    stencil  この値でステンシルバッファを埋めます。 CLEAR_FLAG_STENCIL が指定されていない場合は無視されます。
    //---------------------------------------------------------------------------
    void ClearFrameBuffersDetail( u8 flag, const nw::ut::Color4u8& color, f32 depth, u32 stencil );

    //---------------------------------------------------------------------------
    //! @brief        画面更新処理を行います。
    //---------------------------------------------------------------------------
    void SwapBuffer();

    //---------------------------------------------------------------------------
    //! @brief        TV の描画を開始します。
    //---------------------------------------------------------------------------
    void BeginDrawTV();

    //---------------------------------------------------------------------------
    //! @brief        TV の描画を終了します。
    //---------------------------------------------------------------------------
    void EndDrawTV();

    //---------------------------------------------------------------------------
    //! @brief        DRC の描画を開始します。
    //---------------------------------------------------------------------------
    void BeginDrawDRC();
    //---------------------------------------------------------------------------
    //! @brief        RDC の描画を終了します。
    //---------------------------------------------------------------------------
    void EndDrawDRC();

    //---------------------------------------------------------------------------
    //! @brief        VSync 待ち処理を行います。
    //---------------------------------------------------------------------------
    void WaitForVBlank();

    //---------------------------------------------------------------------------
    //! @brief        フレーム開始時処理を行います。
    //!
    //!               ここでは、以下の処理を行います。
    //!               - DramMeter のフレーム開始処理
    //---------------------------------------------------------------------------
    void BeginFrame();

    //---------------------------------------------------------------------------
    //! @brief        フレーム終了時処理を行います。
    //!
    //!               ここでは、以下の処理を行います。
    //!               - デバッグ用文字描画のための DevTextWriter のバッファのクリア
    //!               - DramMeter のフレーム終了処理
    //---------------------------------------------------------------------------
    void EndFrame();


    //---------------------------------------------------------------------------
    //! @brief        Pad の更新を行います。
    //---------------------------------------------------------------------------
    void UpdatePad();

    //---------------------------------------------------------------------------
    //! @brief        Pad 管理クラスのインスタンスへのポインタを取得します。
    //!
    //! @return       Pad 管理クラスのインスタンスへのポインタを返します。
    //---------------------------------------------------------------------------
    nw::demo::Pad* GetPad() const { return m_Pad; }

    //---------------------------------------------------------------------------
    //! @brief        WPad 管理クラスのインスタンスへのポインタを取得します。
    //!
    //! @return       WPad 管理クラスのインスタンスへのポインタを返します。
    //---------------------------------------------------------------------------
    nw::demo::Pad* GetWPad() const { return m_WPad; }

    //---------------------------------------------------------------------------
    //! @brief        VPad 管理クラスのインスタンスへのポインタを取得します。
    //!
    //! @return       VPad 管理クラスのインスタンスへのポインタを返します。
    //---------------------------------------------------------------------------
    nw::demo::Pad* GetVPad() const { return m_VPad; }

    //---------------------------------------------------------------------------
    //! @brief        マウス管理クラスのインスタンスへのポインタを取得します。
    //!
    //! @return       マウス管理クラスのインスタンスへのポインタを返します。
    //---------------------------------------------------------------------------
    nw::demo::Mouse* GetMouse() const { return m_Mouse; }

    //---------------------------------------------------------------------------
    //! @brief        キーボード・マウスデバイスへのポインタを取得します。
    //!
    //! @return       キーボード・マウスデバイスへのポインタを返します。
    //---------------------------------------------------------------------------
    nw::demo::KeyboardMouseDevice* GetKeyboardMouseDevice() const { return nw::demo::KeyboardMouseDeviceWin::GetInstance(); }


    //---------------------------------------------------------------------------
    //! @brief        1 フレームを測定する計測メーターを取得します。
    //!
    //! @return       計測メーターを返します。
    //---------------------------------------------------------------------------
    nw::dev::CPUMeter& GetMeterFrame() { return m_MeterFrame; }

    //---------------------------------------------------------------------------
    //! @brief        Calc 計測メーターを取得します。
    //!
    //! @return       計測メーターを返します。
    //---------------------------------------------------------------------------
    nw::dev::MultiCPUMeter<2>& GetMeterCalc() { return m_MeterCalc; }

    //---------------------------------------------------------------------------
    //! @brief        Draw 計測メーターを取得します。
    //!
    //! @return       計測メーターを返します。
    //---------------------------------------------------------------------------
    nw::dev::MultiCPUMeter<2>& GetMeterDraw() { return m_MeterDraw; }

    //---------------------------------------------------------------------------
    //! @brief        GPU 計測メーターを取得します。
    //!
    //! @return       計測メーターを返します。
    //---------------------------------------------------------------------------
    nw::dev::MultiGPUMeter<2>& GetMeterGPU() { return m_MeterGPU; }

    //---------------------------------------------------------------------------
    //! @brief        MeterDrawer を取得します。
    //!
    //! @return       MeterDrawer を返します。
    //---------------------------------------------------------------------------
    nw::dev::MeterDrawer& GetMeterDrawer() { return m_MeterDrawer; }

    //---------------------------------------------------------------------------
    //! @brief        負荷メータを描画します。
    //---------------------------------------------------------------------------
    void DrawLoadMeters();

    //---------------------------------------------------------------------------
    //! @brief        負荷計測結果をログに出力します。
    //---------------------------------------------------------------------------
    void DumpLoadMeters() { m_MeterDrawer.Dump(); }


    //---------------------------------------------------------------------------
    //! @brief        デバッグ用文字列描画に用いることができる DevTextWriter を取得します。
    //!
    //! @return       DevTextWriter を返します。
    //---------------------------------------------------------------------------
    nw::dev::DevTextWriter* GetTextWriter() { return &m_DevTextWriter; }

    //---------------------------------------------------------------------------
    //! @brief        デバッグ用文字列描画のための DevTextWriter をクリアします。
    //---------------------------------------------------------------------------
    void ClearTextWriter();

    //---------------------------------------------------------------------------
    //! @brief        ポインタカーソルを描画します。
    //!
    //!               ポインタの座標系はウィンドウの左上が(-1,-1)、右下が(1,1)です。
    //!
    //! @param[in]    posX     ポインタカーソルの x 座標位置です。
    //! @param[in]    posY     ポインタカーソルの y 座標位置です。
    //! @param[in]    width    ウィンドウの幅です。
    //! @param[in]    height   ウィンドウの高さです。
    //---------------------------------------------------------------------------
    void DrawPointerCursor( f32 posX, f32 posY, f32 width, f32 height );

    //---------------------------------------------------------------------------
    //! @brief        フォントバイナリのポインタを取得します。
    //---------------------------------------------------------------------------
    const u8* GetFontBinary() const { return m_Arg.fontBinary; }

    //---------------------------------------------------------------------------
    //! @brief        フォントバイナリのポインタを取得します。
    //---------------------------------------------------------------------------
    u8*       GetFontBinary()  { return m_Arg.fontBinary; }

    //---------------------------------------------------------------------------
    //! @brief        フォントバイナリのサイズを取得します。
    //---------------------------------------------------------------------------
    u32       GetFontBinarySize() const { return m_Arg.fontBinarySize; }

    //---------------------------------------------------------------------------
    //! @brief        DRC表示の回転を設定します。
    //! @param [in] rotateState 回転方法です。
    //---------------------------------------------------------------------------
    void SetRotateDRC( DisplayRotateState rotateState);


protected:
    //! ウィンドウのセットアップを行います。
    void SetupWindow();

    //! メッセージプロシージャの実装です。
    _W64 long MsgProcImpl( void* hWnd, unsigned int msg, _W64 unsigned int wParam, _W64 long lParam, BOOL *handled );

    //! メッセージプロシージャの外側部分です。
    static _W64 long MsgProc( void* hWnd, unsigned int msg, _W64 unsigned wParam, _W64 long lParam, BOOL *handled );

    //! フレームバッファを作成します。
    TextureFrameBufferGL* CreateFrameBuffer( nw::gfnd::TextureGL* color, nw::gfnd::TextureGL* depth, const nw::math::VEC2& frameBufferSize );


    static SystemWinGLDRC*     s_Instance;                  //!< 唯一のインスタンスです。

    CreateArg                  m_Arg;                       //!< 生成パラメータです。
    void*                      m_HWnd;                      //!< ウインドウハンドルです。
    void*                      m_HWndDRC;                   //!< ウインドウハンドルです。
    void*                      m_HGLRCForSwap;              //!< Swap専用GL コンテキストです。特定のGPU(ATI等)でContextを指定しておかないとSwapできないときがあるため。
    void*                      m_MsgProcCallback;           //!< ユーザー設定のメッセージプロシージャです。

    bool                       m_Exit;                      //!< 無限ループ抜けフラグです。
    void*                      m_HGLRC;                     //!< TV 用GL コンテキストです。
    void*                      m_HDC;                       //!< TV 用ウインドウコンテキストです。
    void*                      m_HGLRCDRC;                  //!< DRC 用 GL コンテキストです。
    void*                      m_HDCDRC;                    //!< DRC 用 ウインドウコンテキストです。
    TextureFrameBufferGL*      m_DefaultFrameBuffer;        //!< TV 用フレームバッファです。
    DisplayBufferGL            m_DisplayBuffer;             //!< TV 用ディスプレイバッファです。
    TextureFrameBufferGL*      m_DefaultFrameBufferDRC;     //!< DRC 用フレームバッファです。
    DisplayBufferGL            m_DisplayBufferDRC;          //!< DRC 用ディスプレイバッファです。

    nw::gfnd::TextureGL        m_Color;                     //!< TV 用カラーバッファ用テクスチャです。
    nw::gfnd::TextureGL        m_Depth;                     //!< TV 用デプスバッファ用テクスチャです。
    nw::gfnd::TextureGL        m_ColorDRC;                  //!< DRC 用カラーバッファ用テクスチャです。
    nw::gfnd::TextureGL        m_DepthDRC;                  //!< DRC 用デプスバッファ用テクスチャです。

    nw::demo::PadWin*          m_Pad;                       //!< デバッグ用パッドです。
    nw::demo::WPadWin*         m_WPad;                      //!< WPAD を PC で再現します。
    nw::demo::VPadWin*         m_VPad;                      //!< VPAD を PC で再現します。
    nw::demo::MouseWin*        m_Mouse;                     //!< マウス入力を扱います。

    nw::dev::CPUMeter          m_MeterFrame;                //!< 1 フレームの処理負荷メーターです。
    nw::dev::MultiCPUMeter<2>  m_MeterCalc;                 //!< 計算処理負荷メーターです。
    nw::dev::MultiCPUMeter<2>  m_MeterDraw;                 //!< 描画処理負荷メーターです。
    nw::dev::MultiGPUMeter<2>  m_MeterGPU;                  //!< GPU 負荷メーターです。
    nw::dev::MeterDrawer       m_MeterDrawer;               //!< 負荷メーターを描画します。
    nw::math::MTX44            m_TextWriterProjMtx;         //!< DevTextWriter 用プロジェクション行列です。
    nw::math::MTX34            m_TextWriterViewMtx;         //!< DevTextWriter 用ビュー行列です。
    nw::dev::DevTextWriter     m_DevTextWriter;             //!< デバッグ用文字描画のための DevTextWriter です。
    bool                       m_TextWriterInitialized;     //!< DevTextWriter が初期化されているかを表します。

    nw::gfnd::TextureGL        m_PointerTexture;            //!< ポインタカーソル用テクスチャです。
    u8*                        m_PointerTextureImage;       //!< ポインタカーソル用画像データです。

    nw::ut::Tick               m_LastUpdateTime;            //!< 前回のUpdateを行った時間です。
    nw::ut::TimeSpan           m_FrameTime;                 //!< 1フレームに相当する時間を格納します。
    nw::ut::TimeSpan           m_LastDiffTime;              //!< 前回のフレーム更新にかかった時間です。

    DisplayState               m_DisplayState;              //!< 画面表示を行ってよいかを判定するステート変数です。
};

} // namespace nw::demo
} // namespace nw

#endif // NW_DEMO_SYSTEM_WIN_GL_DRC_H_
