﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

/**
 * @file mm_AudioOutRenderer.h
 * @brief Audio playback nn::audio::AudioOut
 */
#include "mm_MoviePlayerUtils.h"
#include <nn/audio.h>
#include <nn/audio/audio_Resampler.h>

/**
 * @brief
 * AudioOutRenderer class for rendering audio using nn::audio::AudioOut
 *
 */
class AudioOutRenderer
{
public:
   /**
    * @brief
    * AudioOutRenderer constructor.
    */
    AudioOutRenderer();

   /**
    * @brief
    * AudioOutRenderer destructor.
    *
    */
    ~AudioOutRenderer();

   /**
    * @brief
    * Initialize AudioOutRenderer.
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    *
    */
    movie::Status Initialize();

   /**
    * @brief Open AudioOutRenderer.
    *
    * @param[in] sampleRate                     Audio sample rate in hertz.
    * @param[in] channelCount                   Number of audio channels.
    * @param[out] resampleAudio                 Does client need resampling of audio data.
    * @param[in] audioFrameSize                 Input audio frame size.
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    * @retval ::Status_ErrorBadValue            Invalid input parameters.
    * @retval ::Status_OutOfMemory              Memory allocation failed.
    *
    * @details
    * An instance of nn::audio::AudioOut is created and is initialized. If the input sample rate is not
    * supported by AudioOut, resampler needs to be used. Audio buffers are allocated and submitted to AudioOut.
    * Output variable will be updated to indicate resampler need. Client needs to call resampler APIs
    * before submitting PCM data for rendering.
    */
    movie::Status Open(uint32_t sampleRate, int channelCount, bool *resampleAudio, int audioFrameSize);

   /**
    * @brief Start rendering audio.
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    * @retval ::Status_UnknownError             Failed to start playback.
    *
    * @details
    * nn::audio::AudioOut will be started. This will result in playback of audio buffers submitted during
    * initialization.
    *
    */
    movie::Status Start();

   /**
    * @brief Get empty buffer from AudioOut
    *
    * @param[out] audioOutBuffer                 Empty buffer from audio out
    * @param[out] dataBuffer                     Data buffer pointer.
    * @param[out] outSize                        Data buffer size in bytes.
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    * @retval ::Status_OutputBufferNotAvailable  AudioOut buffers not available.
    *
    * @details
    * A client can use this API to get empty AudioOut buffer. This AudioOut buffer can be filled with PCM
    * data using the data buffer pointer returned. It will also return the size of empty buffer.
    */
    movie::Status GetEmptyAudioOutBuffer(nn::audio::AudioOutBuffer** audioOutBuffer, void **dataBuffer, size_t *outSize);

   /**
    * @brief Append filled audio buffer to AudioOut
    *
    * @param[in] audioOutBuffer                 Filled buffer to send.
    * @param[in] dataBuffer                     Filled data buffer pointer.
    * @param[in] size                           Data buffer size in bytes.
    * @param[in] FilledSize                     Filled data size in bytes.
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    *
    * @details
    * A client can use this API to send filled AudioOut buffer for rendering. The empty buffer needs to be obtained
    * from AudioOutRenderer. Client needs to provide filled buffer along with buffer size and filled data size.
    */
    movie::Status AppendFilledAudioOutBuffer(nn::audio::AudioOutBuffer* audioOutBuffer, void *dataBuffer, size_t size, size_t FilledSize);

   /**
    * @brief Get resampled output buffer size
    *
    * @param[out] size                          Buffer size needed for resampling.
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    * @retval ::Status_ErrorBadValue            Invalid input parameters.
    * @retval ::Status_UnknownError             Resampler is not needed.
    *
    * @details
    * A client can use this API to get resampled audio buffer size. This information is needed for submitting the
    * buffer for rendering.
    */
    movie::Status GetResampledOutBufferSize(size_t *size);

   /**
    * @brief Resample audio
    *
    * @param[in] resamplerBuffer                Buffer (output) for resampling.
    * @param[in] inputDataBuffer                Input data buffer.
    * @param[in] inputDataSize                  Input buffer data size.
    * @param[out] outputDataSize                Resampled data size.
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    * @retval ::Status_ErrorBadValue            Invalid input parameters.
    * @retval ::Status_UnknownError             Resampler is not needed.
    *
    * @details
    * A client can use this API to resample PCM data. If resampling is needed, it has to be done before submitting PCM
    * data for rendering. If sample rate is not supported by AudioOut, resampling is needed. Resampler need is indicated
    * to the client when AudioOutRenderer Open() API is called.
    */
    movie::Status ResampleAudio(void *resamplerBuffer, void *inputDataBuffer, size_t inputDataSize, size_t *outputDataSize);

   /**
    * @brief Stop rendering audio.
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    *
    * @details
    * A client can use this API to stop rendering of PCM data. AudioOutRenderer will stop AudioOut, no data will be
    * played out.
    *
    */
    movie::Status Stop();

   /**
    * @brief Close AudioOutRenderer.
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    *
    * @details
    * A client can use this API to close AudioOutRenderer. All resources allocated are released.
    *
    */
    movie::Status Close();

   /**
    * @brief Get played out audio duration.
    *
    * @param[in] nowUs                          Current time.
    * @return duaration in microseconds.
    *
    * @details
    * A client can use this API to find the audio sample played out by AudioOutRenderer. The play out duration
    * is queried from AudioOut and returned to client.
    *
    */
    int64_t GetPlayedOutAudioDurationUs(int64_t nowUs);

private:
    int m_Channels;
    int m_InputSampleRate;
    int m_InputChannels;
    int m_FrameSize;
    static const int s_NumBuffers = 4;
    bool m_AudioOutStarted;
    bool m_AudioOutOpened;
    int64_t m_PresentationTimeUs;
    size_t m_SampleSize;
    size_t m_BufferSize;
    nn::audio::AudioOutBuffer m_AudioOutBuffer[ s_NumBuffers ];
    nn::audio::AudioOut m_AudioOut;
    nn::audio::SampleFormat m_SampleFormat;

    nn::audio::ResamplerType m_Resampler;
    void *m_InBuffers[ s_NumBuffers ];
    void *m_ResamplerBuffer;
    void *m_ResamplerOutBuffers[ s_NumBuffers ];
    bool m_ResampleAudio;
    int m_OutputSampleRate;
    int m_OutputChannels;
    size_t m_ResamplerOutBufferSize;
};
