﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       コントローラに関する API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nns/hid/hid_Attribute.h>
#include <nns/hid/hid_Controller.h>
#include <nns/hid/hid_ControllerId.h>
#include <nns/hid/hid_ControllerManager.h>
#include <nns/hid/hid_GamePadNxAsset.h>

namespace nns { namespace hid {

/**
 * @brief       コントローラのクラスです。
 *
 * @attention   Nx 向けゲームパッドのクラスです。GamePad クラスとは同時に使用しないでください
 */
class GamePadNx : public nns::hid::Controller
{
    NN_DISALLOW_COPY(GamePadNx);
    NN_DISALLOW_MOVE(GamePadNx);

public:
    /**
     * @brief       GamePadNx のコンストラクタです。
     *
     * @param[in]   pManager                    コントローラの管理者です。
     *
     * @pre
     *              - pManager != NULL
     */
    explicit GamePadNx(nns::hid::ControllerManager* pManager) NN_NOEXCEPT;

    /**
     * @brief       GamePadNx のデストラクタです。
     */
    virtual ~GamePadNx() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief       コントローラ番号を返します。
     *
     * @return      コントローラ番号です。
     */
    int GetControllerNumber() const NN_NOEXCEPT
    {
        return m_ControllerNumber;
    }

    /**
     * @brief       コントローラ番号を設定します。
     *
     * @param[in]   controllerNumber            コントローラ番号です。
     */
    void SetControllerNumber(int controllerNumber) NN_NOEXCEPT
    {
        m_ControllerNumber = controllerNumber;
    }

    /**
     * @brief       コントローラの状態を更新します。
     *
     * @details     更新はフレーム毎に行う必要があります。
     */
    virtual void Update() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief       コントローラのコントローラ識別子を返します。
     *
     * @return      コントローラのコントローラ識別子です。
     */
    virtual nns::hid::ControllerId GetControllerId() NN_NOEXCEPT NN_OVERRIDE
    {
        return nns::hid::ControllerId_GamePad;
    }

    /**
     * @brief       コントローラが接続状態にあるか否かを返します。
     *
     * @return      コントローラが接続状態にあるか否かを表す値です。
     */
    virtual bool IsConnected() const NN_NOEXCEPT NN_OVERRIDE
    {
        return m_IsConnected;
    }

    /**
     * @brief       コントローラの状態を表す属性値の集合を返します。
     *
     * @return      コントローラの状態を表す属性値の集合です。
     */
    virtual const AttributesSet GetAttributesSet() const NN_NOEXCEPT
    {
        return m_Attributes;
    }

private:
    GamePadNxAsset* m_pGamePadNxAsset;              //!< 対応するコントローラのデバイスアセットです。

    int m_ControllerNumber;                         //!< コントローラのコントローラ番号です。

    bool m_IsConnected;                             //!< コントローラが接続状態にあるか否かを表す値です。

    AttributesSet m_Attributes;                     //!< コントローラの状態を表す属性値です。

};

}} // namespace nns::hid
