﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <nn/gfx.h>
#include <nn/util/util_Color.h>
#include <nns/gfx/gfx_GpuBuffer.h>
#include <nns/gfx/gfx_PrimitiveRendererGfxRes.h>
#include <nn/nn_Allocator.h>

namespace nns
{
namespace gfx
{
namespace PrimitiveRenderer
{

//------------------------------------------------------------------------------
//! @breif 前方宣言
//------------------------------------------------------------------------------
struct RendererInfo;
class Renderer;


//------------------------------------------------------------------------------
//! @brief 公開APIです。
//------------------------------------------------------------------------------

//! @brief レンダラクラスです。
//! @details １つのインスタンスが、１つのスレッドで処理します。
class Renderer
{
    NN_DISALLOW_COPY( Renderer );

public:

    // TODO: ユーザが直接指定できる描画インターフェイスを用意します。

    //! @brief モデルごとの情報を与える UniformBlock
    struct Model
    {
        nn::util::FloatT4x4 u_userMatrix;
        nn::util::Float4 u_color;
        nn::util::Float2 u_uv_src;
        nn::util::Float2 u_uv_size;
        nn::util::Float4 u_layer;       // xは、テクスチャではLODです。テクスチャ配列ではレイヤーです。
        float rate;
    };

    //! @brief mvp行列を与える UniformBlock
    struct View
    {
        nn::util::FloatT4x4         u_mvp;
        nn::util::FloatT4x4         u_view;
        nn::util::FloatT4x4         u_model;

    };

    //! @brief シェーダで利用するパラメータを管理する構造体
    struct Param
    {
        nn::util::Matrix4x4fType    projMatrix;
        nn::util::Matrix4x4fType    viewMatrix;
        nn::util::Matrix4x4fType    modelMatrix;

        float lineWidth;

        void SetDefault()
        {
            nn::util::MatrixIdentity( &projMatrix );
            nn::util::MatrixIdentity( &viewMatrix );
            nn::util::MatrixIdentity( &modelMatrix );
            lineWidth = 1.0f;
        }
    };

public:
    //! @brief デフォルトシェーダのパラメータを初期値にします。
    //! @param[in]    pModel モデルデータです。
    static void SetDefault( Model* pModel );

    //! @brief デフォルトシェーダのパラメータを初期値にします。
    //! @param[in]    pView ビューデータです。
    static void SetDefault( View* pView );

    //! @brief 必要なMemoryPoolを返します。
    //! @param[in]    pGfxDevice  gfxデバイスです。
    //! @param[in]    info        レンダラ初期化に使うパラメータ情報です。
    static size_t GetRequiredMemoryPoolSize( nn::gfx::Device* pGfxDevice,
                                             const RendererInfo& info );

    //! @brief MemoryPoolのアライメントを返します。
    //! @param[in]    pGfxDevice  gfxデバイスです。
    //! @param[in]    info        レンダラ初期化に使うパラメータ情報です。
    //! @return                   MemoryPoolのアライメント
    static size_t GetMemoryPoolAlignment( nn::gfx::Device* pGfxDevice,
                                           const RendererInfo& info );

public:
    //----------------------------------------
    //! @name コンストラクタ／デストラクタ
    //@{

    //! @brief コンストラクタです。
    //!
    Renderer();

    //! @brief デストラクタです。
    //!
    virtual ~Renderer();

    //@}


    //----------------------------------------
    //! @name 初期化／後処理
    //@{

    //---------------------------------------------------------------------------
    //! @brief 初期化します
    //!
    //! @param[in]    pGfxDevice         gfxデバイスです。
    //! @param[in]    info               レンダラ初期化に使うパラメータ情報です。
    //! @param[in]    pGraphicsResource  レンダラが利用するリソースです。
    //! @param[in]    pMemoryPool        Renderer内で利用するメモリプールのポインタです。
    //! @param[in]    memoryPoolOffset   pMemoryPoolのオフセット値です。
    //! @param[in]    memoryPoolSize     pMmoeryPoolのサイズです。
    //! @return                          初期化に成功したらtrueを返します。
    //---------------------------------------------------------------------------
    bool Initialize( nn::gfx::Device* pGfxDevice,
                     const RendererInfo& info,
                     GraphicsResource* pGraphicsResource,
                     nn::gfx::MemoryPool *pMemoryPool,
                     ptrdiff_t memoryPoolOffset,
                     size_t memoryPoolSize );

    //---------------------------------------------------------------------------
    //! @brief 終了処理です。
    //! @param[in]    pGfxDevice              Initialize時に使用したgfxデバイスです。
    //---------------------------------------------------------------------------
    void Finalize( nn::gfx::Device* pGfxDevice );

    //@}

    //----------------------------------------
    //! @name 設定／取得（パラメータ）
    //! @{

    //---------------------------------------------------------------------------
    //! @brief 設定パラメータ初期状態にリセットします。
    //---------------------------------------------------------------------------
    void SetDefaultParameters();

    //---------------------------------------------------------------------------
    //! @brief 投影行列を設定します。
    //! @param[in]    pProjectionMtx 投影行列です。
    //---------------------------------------------------------------------------
    void SetProjectionMatrix( const nn::util::Matrix4x4fType* pProjectionMtx );

    //---------------------------------------------------------------------------
    //! @brief 視野行列を設定します。
    //! @param[in]    pViewMtx 視野行列です。
    //---------------------------------------------------------------------------
    void SetViewMatrix( const nn::util::Matrix4x3fType* pViewMtx );

    //---------------------------------------------------------------------------
    //! @brief モデル行列を設定します。
    //! @param[in]    pModelMatrix モデル行列です。
    //---------------------------------------------------------------------------
    void SetModelMatrix(const nn::util::Matrix4x3fType* pModelMatrix) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief モデル行列を設定します。
    //! @param[in]    pModelMatrix モデル行列です。
    //---------------------------------------------------------------------------
    void SetModelMatrix(const nn::util::Matrix4x4fType* pModelMatrix) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief ユーザ定数バッファを更新します。
    //! @param[out]   pGpuAddress 定数バッファのgpuAddressです。
    //! @param[in]    pPodData    定数バッファに設定するPODデータです。
    //! @param[in]    size        定数バッファに設定するPODデータのサイズです。
    //---------------------------------------------------------------------------
    void SetUserConstantBuffer( nn::gfx::GpuAddress* pGpuAddress, void* pPodData, size_t size ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief 線幅を設定します。
    //! @details TODO：ユーザが設定できるようにします。
    //! @param[in]    width 線幅です。
    //---------------------------------------------------------------------------
    void SetLineWidth(float width);

    //---------------------------------------------------------------------------
    //! @brief カラーを設定します。
    //! @details プリミティブを描画するときのカラーを設定します。カラー0,1をまとめて設定します。
    //! @param[in]    color カラー
    //---------------------------------------------------------------------------
    void SetColor( const nn::util::Uint8x4& color ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief カラーを設定します。
    //! @details プリミティブを描画するときのカラーを設定します。カラー0,1をまとめて設定します。
    //! @param[in]    color カラー
    //---------------------------------------------------------------------------
    void SetColor( const nn::util::Color4u8& color ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief ユーザ独自のピクセルシェーダを設定します。
    //! @details プリミティブを描画するときに、このメソッドで渡したピクセルシェーダを利用するようになります。
    //! @param[in]    pPixelShader ピクセルシェーダのポインタ
    //---------------------------------------------------------------------------
    void SetUserPixelShader( nn::gfx::Shader* pPixelShader ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief スクリーンの幅を設定します。
    //! @param[in]    screenWidth スクリーンの幅
    //---------------------------------------------------------------------------
    void SetScreenWidth(const int screenWidth)
    {
        m_ScreenWidth = screenWidth;
    }

    //---------------------------------------------------------------------------
    //! @brief スクリーンの高さを設定します。
    //! @param[in]    screenHeight スクリーンの高さ
    //---------------------------------------------------------------------------
    void SetScreenHeight(const int screenHeight)
    {
        m_ScreenHeight = screenHeight;
    }

    //---------------------------------------------------------------------------
    //! @brief システムのプリセット ブレンドステートを取得します。
    //! @param[in]    type ブレンドの種類です。
    //! @return       ブレンドステートを返します。
    //---------------------------------------------------------------------------
    nn::gfx::BlendState* GetBlendState( BlendType type )
    {
        return m_pGraphicsResource->GetBlendState( type );
    }

    //---------------------------------------------------------------------------
    //! @brief システムのプリセット 深度ステンシルステートを取得します。
    //! @param[in]    type     深度ステンシルの種類です。
    //! @return                深度ステンシルステートを返します。
    //---------------------------------------------------------------------------
    nn::gfx::DepthStencilState* GetDepthStencilState( DepthStencilType type )
    {
        return m_pGraphicsResource->GetDepthStencilState( type );
    }

    //---------------------------------------------------------------------------
    //! @brief システムのプリセット ラスタライザステートを取得します。
    //! @param[in] topologyType     プリミティブトポロジの種類です。
    //! @param[in] cullMode         カリングモードです。
    //! @param[in] fillMode         フィルモードです。
    //! @return                     ラスタライザステートを返します。
    //---------------------------------------------------------------------------
    nn::gfx::RasterizerState* GetRasterizerState(
        const nn::gfx::PrimitiveTopologyType topologyType,
        const nn::gfx::CullMode cullMode,
        const nn::gfx::FillMode fillMode )
    {
        return m_pGraphicsResource->GetRasterizerState( topologyType, cullMode, fillMode);
    }

    //---------------------------------------------------------------------------
    //! @brief PrimitiveRendererが保持しているGpuBufferを取得します。
    //! @return                nns::gfx::GpuBufferのポインタを返します。
    //---------------------------------------------------------------------------
    nns::gfx::GpuBuffer* GetGpuBuffer()
    {
        return &m_GpuBuffer;
    }

    //! @}

    //----------------------------------------
    //! @name 更新
    //! @{

    //---------------------------------------------------------------------------
    //! @brief 更新します。描画コール数が0になり、定数バッファがクリアされます。
    //! @param[in]    bufferIndex   利用する定数バッファのインデックス
    //---------------------------------------------------------------------------
    void Update( int bufferIndex = 0 ) NN_NOEXCEPT;

    //! @}

    //----------------------------------------
    //! @name 描画
    //! @{

    //---------------------------------------------------------------------------
    //! @brief        点を描画します。
    //!
    //! @param[in]    pCommandBuffer コマンドバッファです。
    //! @param[in]    center        中心位置です。
    //---------------------------------------------------------------------------
    void DrawPoint( nn::gfx::CommandBuffer* pCommandBuffer,
                    const nn::util::Vector3fType& center ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief        線分を描画します。
    //!
    //! @param[in]    pCommandBuffer コマンドバッファです。
    //! @param[in]    begin          開始位置です。
    //! @param[in]    end            停止位置です。
    //---------------------------------------------------------------------------
    void DrawLine( nn::gfx::CommandBuffer* pCommandBuffer,
                   const nn::util::Vector3fType& begin,
                   const nn::util::Vector3fType& end ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief        三角形を描画します。TODO：実装中です。
    //!
    //! @param[in]    pCommandBuffer コマンドバッファです。
    //! @param[in]    surface        サーフェスの種類です。
    //! @param[in]    center         中心位置です。
    //! @param[in]    size           サイズです。
    //---------------------------------------------------------------------------
    void DrawTriangle( nn::gfx::CommandBuffer* pCommandBuffer,
                       const Surface surface,
                       const nn::util::Vector3fType& center,
                       const nn::util::Vector3fType& size ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief        テクスチャ付きの三角形を描画します。
    //!
    //! @param[in]    pCommandBuffer コマンドバッファです。
    //! @param[in]    center         中心位置です。
    //! @param[in]    size           サイズです。
    //! @param[in]    textureViewSlot  テクスチャデスクリプタスロットです。
    //! @param[in]    samplerSlot      サンプラデスクリプタスロットです。
    //---------------------------------------------------------------------------
    void DrawTriangle( nn::gfx::CommandBuffer* pCommandBuffer,
                       const nn::util::Vector3fType& center,
                       const nn::util::Vector3fType& size,
                       const nn::gfx::DescriptorSlot& textureViewSlot,
                       const nn::gfx::DescriptorSlot& samplerSlot ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief        ユーザメッシュを描画します。
    //!
    //! @param[in]    pCommandBuffer  コマンドバッファです。
    //! @param[in]    topology        トポロジーですです。
    //! @param[in]    pUserMesh       ユーザメッシュです。
    //---------------------------------------------------------------------------
    void DrawUserMesh( nn::gfx::CommandBuffer* pCommandBuffer,
                       const nn::gfx::PrimitiveTopology topology,
                       const PrimitiveMesh* pUserMesh ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief        テクスチャ付きユーザメッシュを描画します。
    //!
    //! @param[in]    pCommandBuffer  コマンドバッファです。
    //! @param[in]    topology        トポロジーですです。
    //! @param[in]    pUserMesh       ユーザメッシュです。
    //! @param[in]    textureViewSlot  テクスチャデスクリプタスロットです。
    //! @param[in]    samplerSlot      サンプラデスクリプタスロットです。
    //---------------------------------------------------------------------------
    void DrawUserMesh( nn::gfx::CommandBuffer* pCommandBuffer,
                       const nn::gfx::PrimitiveTopology topology,
                       const PrimitiveMesh* pUserMesh,
                       const nn::gfx::DescriptorSlot& textureViewSlot,
                       const nn::gfx::DescriptorSlot& samplerSlot ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief        ユーザメッシュを描画します。
    //!
    //! @param[in]    pCommandBuffer  コマンドバッファです。
    //! @param[in]    topology        トポロジーですです。
    //! @param[in]    pUserMesh       ユーザメッシュです。
    //! @param[in]    indexBase       pUserMesh内インデックスバッファ配列の基点。
    //! @param[in]    indexCount      インデックス数。
    //! @param[in]    vertexBase      pUserMesh内頂点バッファ配列の基点。
    //---------------------------------------------------------------------------
    void DrawUserMesh( nn::gfx::CommandBuffer* pCommandBuffer,
                       const nn::gfx::PrimitiveTopology topology,
                       const PrimitiveMesh* pUserMesh,
                       const int indexBase,
                       const int indexCount,
                       const int vertexBase );

    //---------------------------------------------------------------------------
    //! @brief        ユーザメッシュを描画します。
    //!
    //! @param[in]    pCommandBuffer  コマンドバッファです。
    //! @param[in]    topology        トポロジーですです。
    //! @param[in]    pUserMesh       ユーザメッシュです。
    //! @param[in]    indexBase       pUserMesh内インデックスバッファ配列の基点。
    //! @param[in]    indexCount      インデックス数。
    //! @param[in]    vertexBase      pUserMesh内頂点バッファ配列の基点。
    //! @param[in]    textureViewSlot  テクスチャデスクリプタスロットです。
    //! @param[in]    samplerSlot      サンプラデスクリプタスロットです。
    //---------------------------------------------------------------------------
    void DrawUserMesh( nn::gfx::CommandBuffer* pCommandBuffer,
                       const nn::gfx::PrimitiveTopology topology,
                       const PrimitiveMesh* pUserMesh,
                       const int indexBase,
                       const int indexCount,
                       const int vertexBase,
                       const nn::gfx::DescriptorSlot& textureViewSlot,
                       const nn::gfx::DescriptorSlot& samplerSlot ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief        球を描画します。
    //!
    //! @param[in]    pCommandBuffer コマンドバッファです。
    //! @param[in]    surface        サーフェスの種類です。
    //! @param[in]    subdiv         サーフェスの細分化レベルです。Normal は 8 x 16 分割、Coarseは、4 x 8分割です。
    //! @param[in]    center         中心位置です。
    //! @param[in]    diameter       直径です。
    //---------------------------------------------------------------------------
    void DrawSphere( nn::gfx::CommandBuffer* pCommandBuffer,
                     const Surface surface,
                     const Subdiv subdiv,
                     const nn::util::Vector3fType& center,
                     const float diameter ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief        テクスチャ付きの球を描画します。
    //!
    //! @param[in]    pCommandBuffer コマンドバッファです。
    //! @param[in]    subdiv         サーフェスの細分化レベルです。Normal は 8 x 16 分割、Coarseは、4 x 8分割です。
    //! @param[in]    center         中心位置です。
    //! @param[in]    diameter       直径です。
    //! @param[in]    textureViewSlot  テクスチャデスクリプタスロットです。
    //! @param[in]    samplerSlot      サンプラデスクリプタスロットです。
    //---------------------------------------------------------------------------
    void DrawSphere( nn::gfx::CommandBuffer* pCommandBuffer,
                     const Subdiv subdiv,
                     const nn::util::Vector3fType& center,
                     const float diameter,
                     const nn::gfx::DescriptorSlot& textureViewSlot,
                     const nn::gfx::DescriptorSlot& samplerSlot ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief        指定した位置に円を描画します。
    //!
    //! @param[in]    pCommandBuffer コマンドバッファです。
    //! @param[in]    surface     サーフェスの種類です。
    //! @param[in]    subdiv      サーフェスの細分化レベルです。Coarseは、16分割です。Normal は 32分割、Coarseは、16分割です。
    //! @param[in]    center      中心点です。
    //! @param[in]    diameter    直径です。
    //---------------------------------------------------------------------------
    void DrawCircle( nn::gfx::CommandBuffer* pCommandBuffer,
                     const Surface surface,
                     const Subdiv subdiv,
                     const nn::util::Vector3fType& center,
                     float diameter) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief        指定した位置にテクスチャ付きの円を描画します。
    //!
    //! @param[in]    pCommandBuffer コマンドバッファです。
    //! @param[in]    subdiv           サーフェスの細分化レベルです。Normal は 32分割、Coarseは、16分割です。
    //! @param[in]    center           中心点です。
    //! @param[in]    diameter         直径です。
    //! @param[in]    textureViewSlot  テクスチャデスクリプタスロットです。
    //! @param[in]    samplerSlot      サンプラデスクリプタスロットです。
    //---------------------------------------------------------------------------
    void DrawCircle( nn::gfx::CommandBuffer* pCommandBuffer,
                     const Subdiv subdiv,
                     const nn::util::Vector3fType& center,
                     float diameter,
                     const nn::gfx::DescriptorSlot& textureViewSlot,
                     const nn::gfx::DescriptorSlot& samplerSlot ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief        直方体を描画します。
    //!
    //! @param[in]    pCommandBuffer コマンドバッファです。
    //! @param[in]    surface     サーフェスの種類です。
    //! @param[in]    center      中心位置です。
    //! @param[in]    size      　サイズです。
    //---------------------------------------------------------------------------
    void DrawCube( nn::gfx::CommandBuffer* pCommandBuffer,
                   const Surface surface,
                   const nn::util::Vector3fType& center,
                   const nn::util::Vector3fType& size ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief        テクスチャ付きの直方体を描画します。
    //!
    //! @param[in]    pCommandBuffer コマンドバッファです。
    //! @param[in]    center      中心位置です。
    //! @param[in]    size      　サイズです。
    //! @param[in]    textureViewSlot  テクスチャデスクリプタスロットです。
    //! @param[in]    samplerSlot      サンプラデスクリプタスロットです。
    //---------------------------------------------------------------------------
    void DrawCube( nn::gfx::CommandBuffer* pCommandBuffer,
                   const nn::util::Vector3fType& center,
                   const nn::util::Vector3fType& size,
                   const nn::gfx::DescriptorSlot& textureViewSlot,
                   const nn::gfx::DescriptorSlot& samplerSlot ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief        四角形を描画します。
    //!
    //! @param[in]    pCommandBuffer コマンドバッファです。
    //! @param[in]    center      中心位置です。
    //! @param[in]    size      　サイズです。
    //---------------------------------------------------------------------------
    void DrawQuad( nn::gfx::CommandBuffer* pCommandBuffer,
                   const nn::util::Vector3fType& center,
                   const nn::util::Vector3fType& size ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief        四角形にテクスチャを描画します。
    //!
    //! @param[in]    pCommandBuffer   コマンドバッファです。
    //! @param[in]    center           中心位置です。
    //! @param[in]    size             サイズです。
    //! @param[in]    textureViewSlot  テクスチャデスクリプタスロットです。
    //! @param[in]    samplerSlot      サンプラデスクリプタスロットです。
    //---------------------------------------------------------------------------
    void DrawQuad( nn::gfx::CommandBuffer* pCommandBuffer,
                   const nn::util::Vector3fType& center,
                   const nn::util::Vector3fType& size,
                   const nn::gfx::DescriptorSlot& textureViewSlot,
                   const nn::gfx::DescriptorSlot& samplerSlot ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief        四角形に、2Dテクスチャ配列で、インデックス指定されたテクスチャを描画します。
    //!
    //! @param[in]    pCommandBuffer   コマンドバッファです。
    //! @param[in]    center           中心位置です。
    //! @param[in]    size             サイズです。
    //! @param[in]    textureViewSlot  テクスチャデスクリプタスロットです。
    //! @param[in]    samplerSlot      サンプラデスクリプタスロットです。
    //! @param[in]    lod              テクスチャLODです。配列のインデックスです。
    //---------------------------------------------------------------------------
    void DrawQuadTextureArray( nn::gfx::CommandBuffer* pCommandBuffer,
                               const nn::util::Vector3fType& center,
                               const nn::util::Vector3fType& size,
                               const nn::gfx::DescriptorSlot& textureViewSlot,
                               const nn::gfx::DescriptorSlot& samplerSlot,
                               const float lod ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief        四角形にキューブマップを描画します。
    //!
    //! @param[in]    pCommandBuffer   コマンドバッファです。
    //! @param[in]    center           中心位置です。
    //! @param[in]    size             サイズです。
    //! @param[in]    textureViewSlot  テクスチャデスクリプタスロットです。
    //! @param[in]    samplerSlot      サンプラデスクリプタスロットです。
    //---------------------------------------------------------------------------
    void DrawQuadCubemap( nn::gfx::CommandBuffer* pCommandBuffer,
                          const nn::util::Vector3fType& center,
                          const nn::util::Vector3fType& size,
                          const nn::gfx::DescriptorSlot& textureViewSlot,
                          const nn::gfx::DescriptorSlot& samplerSlot ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief        スクリーンにテクスチャを描画します。
    //!
    //! @param[in]    pCommandBuffer   コマンドバッファです。
    //! @param[in]    textureViewSlot  テクスチャデスクリプタスロットです。
    //! @param[in]    samplerSlot      サンプラデスクリプタスロットです。
    //---------------------------------------------------------------------------
    void DrawScreenQuad( nn::gfx::CommandBuffer* pCommandBuffer,
                         const nn::gfx::DescriptorSlot& textureViewSlot,
                         const nn::gfx::DescriptorSlot& samplerSlot ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief        スクリーンにテクスチャを上下逆向きに描画します。
    //!
    //! @param[in]    pCommandBuffer   コマンドバッファです。
    //! @param[in]    textureViewSlot  テクスチャデスクリプタスロットです。
    //! @param[in]    samplerSlot      サンプラデスクリプタスロットです。
    //---------------------------------------------------------------------------
    void DrawScreenQuadYFlip( nn::gfx::CommandBuffer* pCommandBuffer,
                              const nn::gfx::DescriptorSlot& textureViewSlot,
                              const nn::gfx::DescriptorSlot& samplerSlot ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief        スクリーンに四角形を描画します。
    //!
    //! @param[in]    pCommandBuffer コマンドバッファです。
    //! @param[in]    x              四角形の左上頂点のx座標です。
    //! @param[in]    y              四角形の左上頂点のy座標です。
    //! @param[in]    width          四角形の幅です。
    //! @param[in]    height         四角形の高さです。
    //---------------------------------------------------------------------------
    void Draw2DRect( nn::gfx::CommandBuffer* pCommandBuffer,
                     const float x,
                     const float y,
                     const float width,
                     const float height ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief        スクリーンに四角形を描画します。
    //!
    //! @param[in]    pCommandBuffer   コマンドバッファです。
    //! @param[in]    textureViewSlot  テクスチャデスクリプタスロットです。
    //! @param[in]    samplerSlot      サンプラデスクリプタスロットです。
    //! @param[in]    x                四角形の左上頂点のx座標です。
    //! @param[in]    y                四角形の左上頂点のy座標です。
    //! @param[in]    width            四角形の幅です。
    //! @param[in]    height           四角形の高さです。
    //---------------------------------------------------------------------------
    void Draw2DRect( nn::gfx::CommandBuffer* pCommandBuffer,
                     const float x,
                     const float y,
                     const float width,
                     const float height,
                     const nn::gfx::DescriptorSlot& textureViewSlot,
                     const nn::gfx::DescriptorSlot& samplerSlot ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief        スクリーンに2D枠を描画します。
    //!
    //! @param[in]    pCommandBuffer   コマンドバッファです。
    //! @param[in]    x                枠の左上頂点のx座標です。
    //! @param[in]    y                枠の左上頂点のy座標です。
    //! @param[in]    width            枠の幅です。
    //! @param[in]    height           枠の高さです。
    //---------------------------------------------------------------------------
    void Draw2DFrame( nn::gfx::CommandBuffer* pCommandBuffer,
                      const float x,
                      const float y,
                      const float width,
                      const float height) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief        スクリーンに線分を描画します。
    //!
    //! @param[in]    pCommandBuffer   コマンドバッファです。
    //! @param[in]    beginX           開始位置のX座標です。
    //! @param[in]    beginY           開始位置のY座標です。
    //! @param[in]    endX             停止位置のX座標です。
    //! @param[in]    endY             停止位置のY座標です。
    //---------------------------------------------------------------------------
    void Draw2DLine( nn::gfx::CommandBuffer* pCommandBuffer,
                     const float beginX,
                     const float beginY,
                     const float endX,
                     const float endY ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief        円錐形を描画します。
    //!
    //! @param[in]    pCommandBuffer コマンドバッファです。
    //! @param[in]    surface   　   サーフェスの種類です。
    //! @param[in]    center         中心位置です。
    //! @param[in]    radius         半径です。
    //! @param[in]    height         円錐形の高さです。
    //---------------------------------------------------------------------------
    void DrawCone( nn::gfx::CommandBuffer* pCommandBuffer,
                   const Surface surface,
                   const nn::util::Vector3fType& center,
                   float radius,
                   float height ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief        テクスチャ付きの円錐形を描画します。
    //!
    //! @param[in]    pCommandBuffer コマンドバッファです。
    //! @param[in]    center         中心位置です。
    //! @param[in]    radius         半径です。
    //! @param[in]    height         円錐形の高さです。
    //! @param[in]    textureViewSlot  テクスチャデスクリプタスロットです。
    //! @param[in]    samplerSlot      サンプラデスクリプタスロットです。
    //---------------------------------------------------------------------------
    void DrawCone( nn::gfx::CommandBuffer* pCommandBuffer,
                   const nn::util::Vector3fType& center,
                   float radius,
                   float height,
                   const nn::gfx::DescriptorSlot& textureViewSlot,
                   const nn::gfx::DescriptorSlot& samplerSlot ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief        円錐形を描画します。
    //!
    //! @param[in]    pCommandBuffer コマンドバッファです。
    //! @param[in]    surface   　   サーフェスの種類です。
    //! @param[in]    center         中心位置です。
    //! @param[in]    size      　   サイズです。
    //---------------------------------------------------------------------------
    void DrawCone( nn::gfx::CommandBuffer* pCommandBuffer,
                   const Surface surface,
                   const nn::util::Vector3fType& center,
                   const nn::util::Vector3fType& size ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief        テクスチャ付きの円錐形を描画します。
    //!
    //! @param[in]    pCommandBuffer コマンドバッファです。
    //! @param[in]    center         中心位置です。
    //! @param[in]    size      　   サイズです。
    //! @param[in]    textureViewSlot  テクスチャデスクリプタスロットです。
    //! @param[in]    samplerSlot      サンプラデスクリプタスロットです。
    //---------------------------------------------------------------------------
    void DrawCone( nn::gfx::CommandBuffer* pCommandBuffer,
                   const nn::util::Vector3fType& center,
                   const nn::util::Vector3fType& size,
                   const nn::gfx::DescriptorSlot& textureViewSlot,
                   const nn::gfx::DescriptorSlot& samplerSlot ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief        カプセル形を描画します。
    //!
    //! @param[in]    pCommandBuffer コマンドバッファです。
    //! @param[in]    surface   　   サーフェスの種類です。
    //! @param[in]    center         中心位置です。
    //! @param[in]    radius         半径です。
    //! @param[in]    height         カプセル形の高さです。
    //---------------------------------------------------------------------------
    void DrawCapsule( nn::gfx::CommandBuffer* pCommandBuffer,
                      const Surface surface,
                      const nn::util::Vector3fType& center,
                      float radius,
                      float height ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief        テクスチャ付きのカプセル形を描画します。
    //!
    //! @param[in]    pCommandBuffer コマンドバッファです。
    //! @param[in]    center         中心位置です。
    //! @param[in]    radius         半径です。
    //! @param[in]    height         カプセル形の高さです。
    //! @param[in]    textureViewSlot  テクスチャデスクリプタスロットです。
    //! @param[in]    samplerSlot      サンプラデスクリプタスロットです。
    //---------------------------------------------------------------------------
    void DrawCapsule( nn::gfx::CommandBuffer* pCommandBuffer,
                      const nn::util::Vector3fType& center,
                      float radius,
                      float height,
                      const nn::gfx::DescriptorSlot& textureViewSlot,
                      const nn::gfx::DescriptorSlot& samplerSlot ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief        カプセル形を描画します。
    //!
    //! @param[in]    pCommandBuffer コマンドバッファです。
    //! @param[in]    surface   　   サーフェスの種類です。
    //! @param[in]    center         中心位置です。
    //! @param[in]    size      　   サイズです。
    //---------------------------------------------------------------------------
    void DrawCapsule( nn::gfx::CommandBuffer* pCommandBuffer,
                  const Surface surface,
                  const nn::util::Vector3fType& center,
                  const nn::util::Vector3fType& size ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief        テクスチャ付きのカプセル形を描画します。
    //!
    //! @param[in]    pCommandBuffer コマンドバッファです。
    //! @param[in]    center         中心位置です。
    //! @param[in]    size      　   サイズです。
    //! @param[in]    textureViewSlot  テクスチャデスクリプタスロットです。
    //! @param[in]    samplerSlot      サンプラデスクリプタスロットです。
    //---------------------------------------------------------------------------
    void DrawCapsule( nn::gfx::CommandBuffer* pCommandBuffer,
                      const nn::util::Vector3fType& center,
                      const nn::util::Vector3fType& size,
                      const nn::gfx::DescriptorSlot& textureViewSlot,
                      const nn::gfx::DescriptorSlot& samplerSlot ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief        パイプ形を描画します。
    //!
    //! @param[in]    pCommandBuffer コマンドバッファです。
    //! @param[in]    surface   　   サーフェスの種類です。
    //! @param[in]    center         中心位置です。
    //! @param[in]    radius         半径です。
    //! @param[in]    height         パイプ形の高さです。
    //---------------------------------------------------------------------------
    void DrawPipe( nn::gfx::CommandBuffer* pCommandBuffer,
                   const Surface surface,
                   const nn::util::Vector3fType& center,
                   float radius,
                   float height ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief        テクスチャ付きのパイプ形を描画します。
    //!
    //! @param[in]    pCommandBuffer コマンドバッファです。
    //! @param[in]    center         中心位置です。
    //! @param[in]    radius         半径です。
    //! @param[in]    height         パイプ形の高さです。
    //! @param[in]    textureViewSlot  テクスチャデスクリプタスロットです。
    //! @param[in]    samplerSlot      サンプラデスクリプタスロットです。
    //---------------------------------------------------------------------------
    void DrawPipe( nn::gfx::CommandBuffer* pCommandBuffer,
                   const nn::util::Vector3fType& center,
                   float radius,
                   float height,
                   const nn::gfx::DescriptorSlot& textureViewSlot,
                   const nn::gfx::DescriptorSlot& samplerSlot ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief        パイプ形を描画します。
    //!
    //! @param[in]    pCommandBuffer コマンドバッファです。
    //! @param[in]    surface   　   サーフェスの種類です。
    //! @param[in]    center         中心位置です。
    //! @param[in]    size      　   サイズです。
    //---------------------------------------------------------------------------
    void DrawPipe( nn::gfx::CommandBuffer* pCommandBuffer,
                   const Surface surface,
                   const nn::util::Vector3fType& center,
                   const nn::util::Vector3fType& size ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief        テクスチャ付きのパイプ形を描画します。
    //!
    //! @param[in]    pCommandBuffer コマンドバッファです。
    //! @param[in]    center         中心位置です。
    //! @param[in]    size      　   サイズです。
    //! @param[in]    textureViewSlot  テクスチャデスクリプタスロットです。
    //! @param[in]    samplerSlot      サンプラデスクリプタスロットです。
    //---------------------------------------------------------------------------
    void DrawPipe( nn::gfx::CommandBuffer* pCommandBuffer,
                   const nn::util::Vector3fType& center,
                   const nn::util::Vector3fType& size,
                   const nn::gfx::DescriptorSlot& textureViewSlot,
                   const nn::gfx::DescriptorSlot& samplerSlot ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief        シリンダー形を描画します。
    //!
    //! @param[in]    pCommandBuffer コマンドバッファです。
    //! @param[in]    surface   　   サーフェスの種類です。
    //! @param[in]    center         中心位置です。
    //! @param[in]    radius         半径です。
    //! @param[in]    height         シリンダー形の高さです。
    //---------------------------------------------------------------------------
    void DrawCylinder( nn::gfx::CommandBuffer* pCommandBuffer,
                       const Surface surface,
                       const nn::util::Vector3fType& center,
                       float radius,
                       float height ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief        テクスチャ付きのシリンダー形を描画します。
    //!
    //! @param[in]    pCommandBuffer コマンドバッファです。
    //! @param[in]    center         中心位置です。
    //! @param[in]    radius         半径です。
    //! @param[in]    height         シリンダー形の高さです。
    //! @param[in]    textureViewSlot  テクスチャデスクリプタスロットです。
    //! @param[in]    samplerSlot      サンプラデスクリプタスロットです。
    //---------------------------------------------------------------------------
    void DrawCylinder( nn::gfx::CommandBuffer* pCommandBuffer,
                       const nn::util::Vector3fType& center,
                       float radius,
                       float height,
                       const nn::gfx::DescriptorSlot& textureViewSlot,
                       const nn::gfx::DescriptorSlot& samplerSlot ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief        シリンダー形を描画します。
    //!
    //! @param[in]    pCommandBuffer コマンドバッファです。
    //! @param[in]    surface   　   サーフェスの種類です。
    //! @param[in]    center         中心位置です。
    //! @param[in]    size      　   サイズです。
    //---------------------------------------------------------------------------
    void DrawCylinder( nn::gfx::CommandBuffer* pCommandBuffer,
                       const Surface surface,
                       const nn::util::Vector3fType& center,
                       const nn::util::Vector3fType& size ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief       テクスチャ付きのシリンダー形を描画します。
    //!
    //! @param[in]    pCommandBuffer コマンドバッファです。
    //! @param[in]    center         中心位置です。
    //! @param[in]    size      　   サイズです。
    //! @param[in]    textureViewSlot  テクスチャデスクリプタスロットです。
    //! @param[in]    samplerSlot      サンプラデスクリプタスロットです。
    //---------------------------------------------------------------------------
    void DrawCylinder( nn::gfx::CommandBuffer* pCommandBuffer,
                       const nn::util::Vector3fType& center,
                       const nn::util::Vector3fType& size,
                       const nn::gfx::DescriptorSlot& textureViewSlot,
                       const nn::gfx::DescriptorSlot& samplerSlot ) NN_NOEXCEPT;

    //! @}

    //----------------------------------------
    //! @name ユーティリティー
    //! @{

    //---------------------------------------------------------------------------
    //! @brief システムのプリセット ブレンドステートを設定します。
    //! @param[in]    pCommandBuffer コマンドバッファです。
    //! @param[in]    type ブレンドの種類です。
    //---------------------------------------------------------------------------
    void SetBlendState( nn::gfx::CommandBuffer* pCommandBuffer, BlendType type )
    {
        return pCommandBuffer->SetBlendState( m_pGraphicsResource->GetBlendState( type ) );
    }

    //---------------------------------------------------------------------------
    //! @brief システムのプリセット 深度ステンシルステートを設定します。
    //! @param[in]    pCommandBuffer コマンドバッファです。
    //! @param[in]    type     深度ステンシルの種類です。
    //---------------------------------------------------------------------------
    void SetDepthStencilState( nn::gfx::CommandBuffer* pCommandBuffer, DepthStencilType type )
    {
        return pCommandBuffer->SetDepthStencilState( m_pGraphicsResource->GetDepthStencilState( type ) );
    }

    //! @}


private:
    //---------------------------------------------------------------------------
    //! @briefprivate デフォルトシェーダの定数バッファを更新します。
    //! @param[in]    pPodData 定数バッファに設定するPODデータです。
    //---------------------------------------------------------------------------
    void SetViewConstantBuffer( View* pPodData ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @briefprivate デフォルトシェーダの定数バッファを更新します。
    //! @details      ユーザが設定できるようにします。現行版は呼び出しても反映されません。
    //! @param[in]    pPodData 定数バッファに設定するPODデータです。
    //---------------------------------------------------------------------------
    void SetModelConstantBuffer( Model* pPodData ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @briefprivate ビュー依存の定数バッファを更新します。
    //---------------------------------------------------------------------------
    void UpdateGpuView() NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @briefprivate  Triangle 描画の内部処理共通化のための実装です。
    //!
    //! @param[in]    pCommandBuffer   コマンドバッファです。
    //! @param[in]    surface          サーフェスの種類です。
    //! @param[in]    center           中心位置です。
    //! @param[in]    size             サイズです。
    //! @param[in]    pTextureViewSlot  テクスチャデスクリプタスロットです。
    //! @param[in]    pSamplerSlot      サンプラデスクリプタスロットです。
    //---------------------------------------------------------------------------
    void DrawTriangleImpl( nn::gfx::CommandBuffer* pCommandBuffer,
                           const Surface surface,
                           const nn::util::Vector3fType& center,
                           const nn::util::Vector3fType& size,
                           const nn::gfx::DescriptorSlot* pTextureViewSlot,
                           const nn::gfx::DescriptorSlot* pSamplerSlot ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @briefprivate  Sphere 描画の内部処理共通化のための実装です。
    //!
    //! @param[in]    pCommandBuffer   コマンドバッファです。
    //! @param[in]    surface          サーフェスの種類です。
    //! @param[in]    subdiv           サーフェスの細分化レベルです。Normal は 8 x 16 分割、Coarseは、4 x 8分割です。
    //! @param[in]    center           中心位置です。
    //! @param[in]    diameter         直径です。
    //! @param[in]    pTextureViewSlot  テクスチャデスクリプタスロットです。
    //! @param[in]    pSamplerSlot      サンプラデスクリプタスロットです。
    //---------------------------------------------------------------------------
    void DrawSphereImpl( nn::gfx::CommandBuffer* pCommandBuffer,
                         const Surface surface,
                         const Subdiv subdiv,
                         const nn::util::Vector3fType& center,
                         const float diameter,
                         const nn::gfx::DescriptorSlot* pTextureViewSlot,
                         const nn::gfx::DescriptorSlot* pSamplerSlot ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @briefprivate  Circle 描画の内部処理共通化のための実装です。
    //!
    //! @param[in]    pCommandBuffer コマンドバッファです。
    //! @param[in]    surface          サーフェスの種類です。
    //! @param[in]    subdiv           サーフェスの細分化レベルです。Normal は 32分割、Coarseは、16分割です。
    //! @param[in]    center           中心点です。
    //! @param[in]    diameter         直径です。
    //! @param[in]    pTextureViewSlot  テクスチャデスクリプタスロットです。
    //! @param[in]    pSamplerSlot      サンプラデスクリプタスロットです。
    //---------------------------------------------------------------------------
    void DrawCircleImpl(nn::gfx::CommandBuffer* pCommandBuffer,
                        const Surface surface,
                        const Subdiv subdiv,
                        const nn::util::Vector3fType& center,
                        float diameter,
                        const nn::gfx::DescriptorSlot* pTextureViewSlot,
                        const nn::gfx::DescriptorSlot* pSamplerSlot ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @briefprivate  Cube 描画の内部処理共通化のための実装です。
    //!
    //! @param[in]    pCommandBuffer コマンドバッファです。
    //! @param[in]    surface          サーフェスの種類です。
    //! @param[in]    center           中心位置です。
    //! @param[in]    size      　     サイズです。
    //! @param[in]    pTextureViewSlot  テクスチャデスクリプタスロットです。
    //! @param[in]    pSamplerSlot      サンプラデスクリプタスロットです。
    //---------------------------------------------------------------------------
    void DrawCubeImpl( nn::gfx::CommandBuffer* pCommandBuffer,
                       const Surface surface,
                       const nn::util::Vector3fType& center,
                       const nn::util::Vector3fType& size,
                       const nn::gfx::DescriptorSlot* pTextureViewSlot,
                       const nn::gfx::DescriptorSlot* pSamplerSlot ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @briefprivate  Cone 描画の内部処理共通化のための実装です。
    //!
    //! @param[in]    pCommandBuffer   コマンドバッファです。
    //! @param[in]    surface          サーフェスの種類です。
    //! @param[in]    center           中心位置です。
    //! @param[in]    size      　     サイズです。
    //! @param[in]    pTextureViewSlot  テクスチャデスクリプタスロットです。
    //! @param[in]    pSamplerSlot      サンプラデスクリプタスロットです。
    //---------------------------------------------------------------------------
    void DrawConeImpl( nn::gfx::CommandBuffer* pCommandBuffer,
                       const Surface surface,
                       const nn::util::Vector3fType& center,
                       const nn::util::Vector3fType& size,
                       const nn::gfx::DescriptorSlot* pTextureViewSlot,
                       const nn::gfx::DescriptorSlot* pSamplerSlot) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @briefprivate  Capsule 描画の内部処理共通化のための実装です。
    //!
    //! @param[in]    pCommandBuffer   コマンドバッファです。
    //! @param[in]    surface          サーフェスの種類です。
    //! @param[in]    center           中心位置です。
    //! @param[in]    size      　     サイズです。
    //! @param[in]    pTextureViewSlot  テクスチャデスクリプタスロットです。
    //! @param[in]    pSamplerSlot      サンプラデスクリプタスロットです。
    //---------------------------------------------------------------------------
    void DrawCapsuleImpl( nn::gfx::CommandBuffer* pCommandBuffer,
                          const Surface surface,
                          const nn::util::Vector3fType& center,
                          const nn::util::Vector3fType& size,
                          const nn::gfx::DescriptorSlot* pTextureViewSlot,
                          const nn::gfx::DescriptorSlot* pSamplerSlot ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @briefprivate  Pipe 描画の内部処理共通化のための実装です。
    //!
    //! @param[in]    pCommandBuffer   コマンドバッファです。
    //! @param[in]    surface          サーフェスの種類です。
    //! @param[in]    center           中心位置です。
    //! @param[in]    size      　     サイズです。
    //! @param[in]    pTextureViewSlot  テクスチャデスクリプタスロットです。
    //! @param[in]    pSamplerSlot      サンプラデスクリプタスロットです。
    //---------------------------------------------------------------------------
    void DrawPipeImpl( nn::gfx::CommandBuffer* pCommandBuffer,
                       const Surface surface,
                       const nn::util::Vector3fType& center,
                       const nn::util::Vector3fType& size,
                       const nn::gfx::DescriptorSlot* pTextureViewSlot,
                       const nn::gfx::DescriptorSlot* pSamplerSlot ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @briefprivate  Cylinder 描画の内部処理共通化のための実装です。
    //!
    //! @param[in]    pCommandBuffer   コマンドバッファです。
    //! @param[in]    surface          サーフェスの種類です。
    //! @param[in]    center           中心位置です。
    //! @param[in]    size      　     サイズです。
    //! @param[in]    pTextureViewSlot  テクスチャデスクリプタスロットです。
    //! @param[in]    pSamplerSlot      サンプラデスクリプタスロットです。
    //---------------------------------------------------------------------------
    void DrawCylinderImpl( nn::gfx::CommandBuffer* pCommandBuffer,
                           const Surface surface,
                           const nn::util::Vector3fType& center,
                           const nn::util::Vector3fType& size,
                           const nn::gfx::DescriptorSlot* pTextureViewSlot,
                           const nn::gfx::DescriptorSlot* pSamplerSlot ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @briefprivate プリミティブを描画します。
    //!
    //! @param[in]    shapeType        利用するシェーダタイプです。
    //! @param[in]    shaderVariation  利用するシェーダバリエーションです。
    //! @param[in]    pCommandBuffer   コマンドバッファです。
    //! @param[in]    pModel           描画するモデルです。
    //! @param[in]    samplerType      サンプラータイプです。
    //! @param[in]    pTextureViewSlot  テクスチャデスクリプタスロットです。
    //! @param[in]    pSamplerSlot      サンプラデスクリプタスロットです。
    //---------------------------------------------------------------------------
    void DrawPrimitiveInternal( ShapeType shapeType,
                                ShaderVariation shaderVariation,
                                nn::gfx::CommandBuffer* pCommandBuffer,
                                Model* pModel,
                                const SamplerType samplerType = SamplerType_CountMax,
                                const nn::gfx::DescriptorSlot* pTextureViewSlot = nullptr,
                                const nn::gfx::DescriptorSlot* pSamplerSlot = nullptr ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @briefprivate プリミティブを描画します。
    //!
    //! @param[in]    topology         トポロジーです。
    //! @param[in]    shaderVariation  利用するシェーダバリエーションです。
    //! @param[in]    pCommandBuffer   コマンドバッファです。
    //! @param[in]    pShape           描画するシェイプです。
    //! @param[in]    pModel           描画するモデルです。
    //! @param[in]    indexBase        描画開始頂インデックスです。
    //! @param[in]    numUserIndices   描画する頂点数です。
    //! @param[in]    userBaseVertex   それぞれのインデクスに加算されるベース頂点
    //! @param[in]    samplerType      サンプラータイプです。
    //! @param[in]    pTextureViewSlot  テクスチャデスクリプタスロットです。
    //! @param[in]    pSamplerSlot      サンプラデスクリプタスロットです。
    //---------------------------------------------------------------------------
    void DrawPrimitiveInternal( nn::gfx::PrimitiveTopology topology,
                                ShaderVariation shaderVariation,
                                nn::gfx::CommandBuffer* pCommandBuffer,
                                const nn::gfx::util::PrimitiveShape* pShape,
                                Model* pModel,
                                const int indexBase,
                                const int numUserIndices,
                                const int userBaseVertex,
                                const SamplerType samplerType,
                                const nn::gfx::DescriptorSlot* pTextureViewSlot,
                                const nn::gfx::DescriptorSlot* pSamplerSlot
                                ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @briefprivate プリミティブを描画します。
    //!
    //! @param[in]    topology         トポロジーです。
    //! @param[in]    shaderVariation  利用するシェーダバリエーションです。
    //! @param[in]    pCommandBuffer   コマンドバッファです。
    //! @param[in]    pMesh            描画するプリミティブメッシュです。
    //! @param[in]    pModel           描画するモデルです。
    //! @param[in]    indexBase        描画開始頂インデックスです。
    //! @param[in]    numUserIndices   描画する頂点数です。
    //! @param[in]    userBaseVertex   それぞれのインデクスに加算されるベース頂点
    //! @param[in]    samplerType      サンプラータイプです。
    //! @param[in]    pTextureViewSlot  テクスチャデスクリプタスロットです。
    //! @param[in]    pSamplerSlot      サンプラデスクリプタスロットです。
    //---------------------------------------------------------------------------
    void DrawPrimitiveInternal(nn::gfx::PrimitiveTopology topology,
                                ShaderVariation shaderVariation,
                                nn::gfx::CommandBuffer* pCommandBuffer,
                                const PrimitiveMesh* pMesh,
                                Model* pModel,
                                const int indexBase,
                                const int numUserIndices,
                                const int userBaseVertex,
                                const SamplerType samplerType,
                                const nn::gfx::DescriptorSlot* pTextureViewSlot,
                                const nn::gfx::DescriptorSlot* pSamplerSlot
                                ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @briefprivate 定数バッファの状態をリセットします。
    void ResetConstantBufferStatus() NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @briefprivate オフセットから定数バッファのデータを取得します。
    //! @param[in]    offset      オフセット値です。
    //! @param[in]    size        サイズです。
    //---------------------------------------------------------------------------
    void* GetUniformBlock(ptrdiff_t offset, size_t size) NN_NOEXCEPT;

    template<typename T>
    inline T* GetUniformBlock(ptrdiff_t offset) NN_NOEXCEPT
    {
        return static_cast<T*>(GetUniformBlock(offset, sizeof(T)));
    }

    //---------------------------------------------------------------------------
    //! @briefprivate UniformBlock のためのメモリを確保します。
    //! @param[in]    size        サイズです。
    //---------------------------------------------------------------------------
    void* CreateUniformBlock(size_t size) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @briefprivate UniformBlock のためのメモリを確保します。
    template <typename T>
    inline T* CreateUniformBlock() NN_NOEXCEPT
    {
        return static_cast<T*>(CreateUniformBlock(sizeof(T)));
    }

    //---------------------------------------------------------------------------
    //! @briefprivate UniformBlock のためのメモリを確保します。
    void UpdateConstantBufferPointer(ConstantBufferType type, void* pointer);

    template<typename T>
    T* UpdateConstantBuffer( ConstantBufferType type, T* pPodData )
    {
        NN_SDK_ASSERT(m_ConstantUpdateCallCount[type] < m_ConstantUpdateCallCountMax[type],
                      "[type %d] UpdateCall(%d) >= UpdateCallCountMax(%d)",
                      m_ConstantUpdateCallCount[type], m_ConstantUpdateCallCountMax[type]);

        if( T* pData = CreateUniformBlock<T>() )
        {
            std::memcpy(pData, pPodData, sizeof(T));
            UpdateConstantBufferPointer(type, pData);
            m_ConstantUpdateCallCount[type]++;
            return pData;
        }
        return nullptr;
    }

    //---------------------------------------------------------------------------
    //! @briefprivate 描画情報を構築
    void SetConstantBuffer( nn::gfx::CommandBuffer* pCommandBuffer,
                            int slot,
                            nn::gfx::ShaderStage shaderStage,
                            void* pointer,
                            size_t memorySize );

    //---------------------------------------------------------------------------
    //! @briefprivate モデル構築
    //! @param[in]    pOutModel  書き込み先モデル。
    //! @param[in]    scale      スケール。
    //! @param[in]    trans      位置。
    //! @param[in]    color      カラー。
    //---------------------------------------------------------------------------
    void MakeModelConstantBuffer( Model* pOutModel,
                                  const nn::util::Vector3fType& scale,
                                  const nn::util::Vector3fType& trans,
                                  const nn::util::Float4& color );

    //---------------------------------------------------------------------------
    //! @briefprivate モデルの描画情報を構築
    //! @param[in]    pOutModel  書き込み先モデル。
    //! @param[in]    scale      スケール。
    //! @param[in]    trans      位置。
    //! @param[in]    color      カラー。
    //---------------------------------------------------------------------------
    void MakeModelConstantBufferForQuad( Model* pOutModel,
                                        const nn::util::Vector3fType& scale,
                                        const nn::util::Vector3fType& trans,
                                        const nn::util::Float4& color );


    //---------------------------------------------------------------------------
    //! @briefprivate モデルの描画情報を構築
    //! @param[in]    pOutModel  書き込み先モデル。
    //! @param[in]    scale      スケール。
    //! @param[in]    trans      位置。
    //---------------------------------------------------------------------------
    void MakeModelConstantBuffer( Model* pOutModel,
                                  const nn::util::Vector3fType& scale,
                                  const nn::util::Vector3fType& trans );

    //---------------------------------------------------------------------------
    //! @briefprivate シェーダセットアップ処理
    //! @param[in]    pCommandBuffer   コマンドバッファです。
    //! @param[in]    pDefaultShader   標準シェーダ。
    //! @param[in]    pUserPixelShader ユーザー側で生成されたシェーダ。
    //! @param[in]    interleaved      インタリーブ版かどうか。
    //---------------------------------------------------------------------------
    void SetupShader( nn::gfx::CommandBuffer* pCommandBuffer,
                      Shader*                 pDefaultShader,
                      const nn::gfx::Shader*  pUserPixelShader,
                      bool interleaved);

private:
    GraphicsResource*       m_pGraphicsResource;
    Param                   m_Param;
    View*                   m_pView;
    Model*                  m_pModel;
    nn::util::Float4        m_Color;
    nn::gfx::Shader*        m_pUserPixelShader;
    size_t                  m_UserConstatnElementSize;
    MeshSet*                m_pMeshSet;
    nns::gfx::GpuBuffer     m_GpuBuffer;
    int                     m_MultiBufferQuantity;
    int                     m_DrawCallCount;
    int                     m_DrawCallCountMax;
    void*                   m_ConstantMemoryPointer[ConstantBufferType_CountMax];
    int                     m_ConstantUpdateCallCount[ConstantBufferType_CountMax];
    int                     m_ConstantUpdateCallCountMax[ConstantBufferType_CountMax];
    int                     m_ScreenWidth;
    int                     m_ScreenHeight;
    int                     m_SetUserPixelModelSlot;
};


//! @brief レンダラ初期化情報です。
struct RendererInfo
{
public:

    //---------------------------------------------------------------------------
    //! @brief 初期値定数
    //---------------------------------------------------------------------------
    enum Constants
    {
        Constants_MultiBufferingQuantityMax     = 1,    //!< 1スレッド、1インスタンスです。
        Constants_DrawCallCountMax              = 1024, //!< 1フレームのドローコール最大数です。
        Constants_ViewFunctionCallCountMax      = 1024, //!< 1フレームのビュー設定コール最大数です。
        Constants_UserFunctionCallCountMax      = 64,   //!< 1フレームのユーザ設定コール最大数です。
        Constants_UserConstantBufferDataSizeMax = 64,   //!< ユーザ定数バッファの上限サイズ(Bytes)です。
    };

    //----------------------------------------
    //! @name コンストラクタ／デストラクタ
    //@{

    //! @brief コンストラクタです。
    //!
    RendererInfo() {}

    //@}

    //---------------------------------------------------------------------------
    //! @brief 初期値を設定します。
    //---------------------------------------------------------------------------
    void SetDefault()
    {
        m_DrawCallCountMax          = Constants_DrawCallCountMax;
        m_ViewFunctionCallCountMax  = Constants_ViewFunctionCallCountMax;
        m_UserFunctionCallCountMax  = Constants_UserFunctionCallCountMax;
        m_MultiBufferQuantity       = Constants_MultiBufferingQuantityMax;
        m_AdditionalBufferSize      = 0;
    }

    //---------------------------------------------------------------------------
    //! @brief 定数バッファのバッファリング数を設定します。
    //! @param[in]    quantity バッファリング数です。
    //---------------------------------------------------------------------------
    void SetMultiBufferQuantity( int quantity ) NN_NOEXCEPT
    {
        m_MultiBufferQuantity = quantity;
    }

    //---------------------------------------------------------------------------
    //! @brief 1フレームのドローコール最大数を設定します。
    //! @param[in]    count ドローコール最大数数です。
    //---------------------------------------------------------------------------
    void SetDrawCallCountMax( int count ) NN_NOEXCEPT
    {
        m_DrawCallCountMax = count;
    }

    //---------------------------------------------------------------------------
    //! @brief 1フレームのビュー設定コール最大数を設定します。
    //! @param[in]    count ビュー設定コール最大数数です。
    //---------------------------------------------------------------------------
    void SetViewFunctionCallCountMax( int count ) NN_NOEXCEPT
    {
        m_ViewFunctionCallCountMax = count;
    }

    //---------------------------------------------------------------------------
    //! @brief  初期化時に使用されるメモリアロケータを設定します。
    //! @param[in]  pAllocateFunction   アロケータ関数へのポインタです。
    //! @param[in]  pAllocateFunctionUserData   アロケータ関数へ渡すユーザーデータポインタです。
    //---------------------------------------------------------------------------
    void SetAllocator(
        nn::AlignedAllocateFunctionWithUserData pAllocateFunction,
        void* pAllocateFunctionUserData)
    {
        m_pAllocateFunction = pAllocateFunction;
        m_pAllocateFunctionUserData = pAllocateFunctionUserData;
    }

    //---------------------------------------------------------------------------
    //! @brief  GpuBufferの追加容量を設定します。
    //! @param[in]  size   別途追加するサイズ
    //---------------------------------------------------------------------------
    void SetAdditionalBufferSize( size_t size )
    {
        m_AdditionalBufferSize = size;
    }

    //---------------------------------------------------------------------------
    //! @brief 実際に初期化されるコンスタントバッファサイズを計算します。
    //! @param[in]    pGfxDevice         gfxデバイスです。
    //---------------------------------------------------------------------------
    size_t CalculateConstantBufferSize( nn::gfx::Device* pGfxDevice ) const NN_NOEXCEPT
    {
        nn::gfx::Buffer::InfoType constantBufferInfo;
        constantBufferInfo.SetDefault();
        constantBufferInfo.SetGpuAccessFlags(nn::gfx::GpuAccess_ConstantBuffer);
        size_t alignment = nn::gfx::Buffer::GetBufferAlignment( pGfxDevice, constantBufferInfo );

        size_t uniformViewSize  = nn::util::align_up( sizeof( Renderer::View ), alignment );
        size_t uniformModelSize = nn::util::align_up( sizeof( Renderer::Model ), alignment );
        size_t uniformUserSize  = nn::util::align_up( sizeof( Constants_UserConstantBufferDataSizeMax), alignment );

        size_t uniformModelTotalSize = uniformModelSize * m_DrawCallCountMax;
        size_t uniformViewTotalSize  = uniformViewSize * m_ViewFunctionCallCountMax;
        size_t uniformUserTotalSize  = uniformUserSize * m_UserFunctionCallCountMax;

        // １つのコンスタントバッファのサイズ
        size_t constantBufferMemorySize =
            uniformModelTotalSize + uniformViewTotalSize + uniformUserTotalSize;

        return constantBufferMemorySize;
    }

    nn::AlignedAllocateFunctionWithUserData GetAllocateFunction()
    {
        return m_pAllocateFunction;
    }

    void* GetAllocateFunctionUserData()
    {
        return m_pAllocateFunctionUserData;
    }

private:
    friend class Renderer;
    friend class GraphicsResource;

    int                                     m_DrawCallCountMax;
    int                                     m_ViewFunctionCallCountMax;
    int                                     m_UserFunctionCallCountMax;
    int                                     m_MultiBufferQuantity;
    nn::AlignedAllocateFunctionWithUserData m_pAllocateFunction;
    void*                                   m_pAllocateFunctionUserData;
    size_t                                  m_AdditionalBufferSize;
};

//---------------------------------------------------------------------------
//! @brief PrimitiveRendererのインスタンスを生成します。
//! @param[in]    pGfxDevice   gfxデバイスです。
//! @param[in]    info         レンダラ初期化に使うパラメータ情報です。
//! @return                    PrimitiveRendererのインスタンスを返します。
//---------------------------------------------------------------------------
Renderer* CreateRenderer( nn::gfx::Device*                        pGfxDevice,
                          RendererInfo&                           info );

//---------------------------------------------------------------------------
//! @brief PrimitiveRendererを破棄します。
//! @param[in]    pRenderer               破棄するPrimitiveRendererです。
//! @param[in]    pGfxDevice              初期化時に利用したgfxデバイスです。
//! @param[in]    pFreeFunction           CreateRenderer で確保した内部データ構造のメモリを開放するための関数へのポインタです。
//! @param[in]    pFreeFunctionUserData   pFreeFunction へ渡すユーザーデータへのポインタです。
//---------------------------------------------------------------------------
void DestroyRenderer( Renderer* pRenderer,
                      nn::gfx::Device* pGfxDevice,
                      nn::FreeFunctionWithUserData pFreeFunction,
                      void* pFreeFunctionUserData );

//---------------------------------------------------------------------------
//! @brief    CreateRenderer関数で初期化したグラフィックスリソースを返します。
//! @return   グラフィックスリソースを返します。
//---------------------------------------------------------------------------
GraphicsResource* GetGraphicsResource();

} // namespace PrimitiveRenderer
} // namespace gfx
} // namespace nns
