﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/gfx.h>
#include <nn/nn_Assert.h>

namespace nns
{
namespace gfx
{
//------------------------------------------------------------------------------
//! @brief 深度ステンシルバッファを管理します。
//------------------------------------------------------------------------------
class DepthStencilBuffer
{
public:
    class InfoType;

    //! @brief 不正なオフセットです。
    static const int InvalidOffset = -1;

    //! @brief デフォルトコンストラクタです。
    DepthStencilBuffer() NN_NOEXCEPT
        : m_Width(0),
          m_Height(0),
          m_Texture(),
          m_TextureView(),
          m_DepthStencilView(),
          m_ViewCount(1),
          m_MemoryPoolOffset(InvalidOffset)
    {
    }

    //! @brief 初期化します。
    //! @param[in]    pDevice               gfxデバイスです。
    //! @param[in]    pInfoType             カラーバッファを構築するためのパラメータです。
    //! @param[in]    pMemoryPool           メモリプールのポインタです。
    //! @param[in]    memoryPoolOffset      メモリプール上のオフセットです。
    //! @param[in]    memorySize            メモリプール上のサイズです。
    void Initialize(nn::gfx::Device* pDevice,
        const nn::gfx::TextureInfo& infoType,
        nn::gfx::MemoryPool* pMemoryPool,
        ptrdiff_t memoryPoolOffset,
        size_t memorySize) NN_NOEXCEPT;

    //! @brief 終了処理です。
    //! @param[in]    pDevice       gfxデバイスです。
    //! @param[in]    FreeMemory    メモリ解放関数のポインタです。
    void Finalize(nn::gfx::Device* pDevice) NN_NOEXCEPT;

    //! @brief  テクスチャーの幅を取得します。
    //!
    //! @return テクスチャーの幅
    int GetWidth() const NN_NOEXCEPT
    {
        return m_Width;
    }

    //! @brief  テクスチャーの高さを取得します。
    //!
    //! @return テクスチャーの高さ
    int GetHeight() const NN_NOEXCEPT
    {
        return m_Height;
    }

    //! @brief テクスチャを取得します。
    //! @return テクスチャを返します。
    nn::gfx::Texture* GetTexture() NN_NOEXCEPT
    {
        return &m_Texture;
    }

    //! @brief テクスチャを取得します。
    //! @return テクスチャを返します。
    const nn::gfx::Texture* GetTexture() const NN_NOEXCEPT
    {
        return &m_Texture;
    }

    //! @brief テクスチャビューを取得します
    //! @return テクスチャビューを返します。
    nn::gfx::TextureView* GetTextureView() NN_NOEXCEPT
    {
        return &m_TextureView;
    }

    //! @brief テクスチャビューを取得します
    //! @return テクスチャビューを返します。
    const nn::gfx::TextureView* GetTextureView() const NN_NOEXCEPT
    {
        return &m_TextureView;
    }
    //! @brief ビューの個数を取得します。
    //! @return ビューの個数を返します。
    int GetViewCount() const NN_NOEXCEPT
    {
        return m_ViewCount;
    }

    //! @brief 深度ステンシルビューを取得します。
    //! @return 深度ステンシルビューを返します。
    nn::gfx::DepthStencilView* GetDepthStencilView() NN_NOEXCEPT
    {
        return &m_DepthStencilView[0];
    }

    //! @brief 指定したインデックスの深度ステンシルビューを取得します。
    //! @param[in]    index     取得する深度ステンシルビューのインデックスです。
    //! @return 指定されたインデックスの深度ステンシルビューを返します。
    //!
    //! @details
    //! イメージの次元が nn::gfx::ImageDimension_2dArray の場合、インデックス の 0 番目が 2DArray としてのビューです。
    //! それ以降のインデックスは 2DArray 内の1枚づつに対応するビューです。
    //!
    nn::gfx::DepthStencilView* GetDepthStencilView(int index) NN_NOEXCEPT
    {
        NN_ASSERT_RANGE(index, 0, GetViewCount());
        return &m_DepthStencilView[index];
    }

    //! @brief 深度ステンシルビューを取得します。
    //! @return 深度ステンシルビューを返します。
    const nn::gfx::DepthStencilView* GetDepthStencilView() const NN_NOEXCEPT
    {
        return &m_DepthStencilView[0];
    }

    //! @brief 指定したインデックスの深度ステンシルビューを取得します。
    //! @param[in]    index     取得する深度ステンシルビューのインデックスです。
    //! @return 指定されたインデックスの深度ステンシルビューを返します。
    //!
    //! @details
    //! イメージの次元が nn::gfx::ImageDimension_2dArray の場合、インデックス の 0 番目が 2DArray としてのビューです。
    //! それ以降のインデックスは 2DArray 内の1枚づつに対応するビューです。
    //!
    const nn::gfx::DepthStencilView* GetDepthStencilView(int index) const NN_NOEXCEPT
    {
        NN_ASSERT_RANGE(index, 0, GetViewCount());
        return &m_DepthStencilView[index];
    }

    //! @brief メモリプール上のオフセットを取得します。
    //! @return メモリプール上のオフセットを返します。
    ptrdiff_t GetMemoryPoolOffset()
    {
        return m_MemoryPoolOffset;
    }

    enum Max
    {
        MaxViewCount = 8,
    };

private:
    int m_Width;
    int m_Height;
    nn::gfx::Texture m_Texture;
    nn::gfx::TextureView m_TextureView;
    nn::gfx::DepthStencilView m_DepthStencilView[MaxViewCount + 1];
    int m_ViewCount;
    ptrdiff_t m_MemoryPoolOffset;
};



} // namespace gfx
} // namespace nns
