﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nns/gfx/gfx_GraphicsFramework.h>
#include <nns/gfx/gfx_PrimitiveRenderer.h>
#include <nns/gfx/gfx_PrimitiveRendererMeterDrawer.h>

namespace nns {
namespace gfx {

class DebugGraphicsFramework;

//------------------------------------------------------------------------------
//! @brief デバッグ用のグラフィックスフレームワーククラスです。
//! @details GraphicsFramework を継承し、以下の機能が追加されます。
//!          nns::gfx::PrimitiveRenderer::Renderer の取得
//!          nns::gfx::PrimitiveRenderer::Renderer の初期化ユーティリティー関数
//!          nn::gfx::util::DebugFontTextWriter の取得
//!          nn::perf::LoadMeterCenter による自動パフォーマンス計測とパフォーマンスメーターの描画
//------------------------------------------------------------------------------
class DebugGraphicsFramework : public GraphicsFramework
{
public:
    //! @brief デフォルトコンストラクターです。
    DebugGraphicsFramework() NN_NOEXCEPT
        : m_IsDebugInitialized(false)
        , m_DebugPerf()
    {
        m_DebugPerf.isInitialized = false;
    }

    //! @brief フレームワークで管理する PrimitiveRenderer::Renderer のラッパーです。
    struct PrimitiveRenderer
    {
        nns::gfx::PrimitiveRenderer::Renderer object;
        ptrdiff_t memoryPoolOffset;
    };

    //! @brief 初期化します。
    //! @param[in]    info                      初期化用の情報です。
    //! @param[in]    pAllocateFunction         メモリー確保用コールバック関数です。
    //! @param[in]    pFreeFunction             メモリー解放用コールバック関数です。
    //! @param[in]    pAllocateFunctionUserData メモリー確保、解放時に呼ばれるユーザー定義のパラメーターです。
    //! @details  GraphicsFramework::Initialize() に加え、GraphicsFramework::InitializeDebugFontTextWriter() と nns::gfx::PrimitiveRenderer::CreateRenderer() を呼びます。
    //! @post     nns::gfx::PrimitiveRenderer::GetGraphicsResource() が有効になる。
    virtual void Initialize(
        const FrameworkInfo& info,
        nn::AlignedAllocateFunctionWithUserData pAllocateFunction,
        nn::FreeFunctionWithUserData pFreeFunction,
        void* pAllocateFunctionUserData
    ) NN_NOEXCEPT NN_OVERRIDE;

    //! @brief 初期化します。
    //! @param[in]    info  初期化用の情報です。
    //! @details  GraphicsFramework::Initialize() に加え、GraphicsFramework::InitializeDebugFontTextWriter() と nns::gfx::PrimitiveRenderer::CreateRenderer() を呼びます。
    //! @post     nns::gfx::PrimitiveRenderer::GetGraphicsResource() が有効になる。
    virtual void Initialize(const FrameworkInfo& info) NN_NOEXCEPT NN_OVERRIDE;

    //! @brief 終了処理です。
    //! @details  GraphicsFramework::Finalize() に加え、GraphicsFramework::FinalizeDebugFontTextWriter() と nns::gfx::PrimitiveRenderer::DestroyRenderer() を呼びます。
    //! @post     nns::gfx::PrimitiveRenderer::GetGraphicsResource() が無効になる。
    virtual void Finalize() NN_NOEXCEPT NN_OVERRIDE;

    //! @brief PrimitiveRenderer を初期化します。
    //! @param[in]    pPrimitiveRenderere 初期化する DebugGraphicsFramework::PrimitiveRenderer のポインターです。
    //! @param[in]    renderInfo 初期化に使用する RenderInfo です。
    void InitializePrimitiveRenderer(DebugGraphicsFramework::PrimitiveRenderer* pOutPrimitiveRenderer, const nns::gfx::PrimitiveRenderer::RendererInfo& rendererInfo) NN_NOEXCEPT;

    //! @brief PrimitiveRenderer を破棄します。
    //! @param[in]    pPrimitiveRenderere 破棄する DebugGraphicsFramework::PrimitiveRenderer のポインターです。
    void FinalizePrimitiveRenderer(DebugGraphicsFramework::PrimitiveRenderer* pOutPrimitiveRenderer) NN_NOEXCEPT;

    //! @brief 計測機能を初期化します。
    //! @param[in]   nn::perf::LoadMeterCenterInfo の初期化に使用する LoadMeterCenterInfo です。
    //! @details     シングルトン nn::perf::LoadMeterCenter を初期化するため、アプリ側で nn::perf::LoadMeterCenter を初期化するとエラーとなります。
    //!              本関数を呼ぶことで、CPU と GPU 負荷が ProcessFrame() で自動計測され、パフォーマンスメーターが EndFrame() で自動描画されます。
    void InitializePerf( const nn::perf::LoadMeterCenterInfo& info) NN_NOEXCEPT;

    //! @brief 計測機能を初期化します。
    //! @details     CPU と GPU 負荷が ProcessFrame() で自動計測され、パフォーマンスメーターが EndFrame() で自動描画されます。
    //!              計測区間は、nn::os::SetThreadCoreMask() を用いて、計測を行うスレッドを特定のコアに割り当てる必要があります。
    //!              シングルトン nn::perf::LoadMeterCenter を初期化するため、アプリ側で nn::perf::LoadMeterCenter を初期化するとエラーとなります。
    //!              SetCoreCount(1),SetCpuSectionCountMax(256),SetGpuSectionCountMax(256) と FrameworkMode に最適な BufferCount を設定した LoadMeterCenterInfo で初期化します。
    //!              FrameworkMode を変更した場合は、FinalizePerf() と InitializePerf() を行い、計測機能を再構築してください。
    void InitializePerf() NN_NOEXCEPT;

    //! @brief 計測機能を破棄します。
    void FinalizePerf() NN_NOEXCEPT;

    //! @brief ルートコマンドバッファへのコマンドの追加を開始します。
    //! @param[in]    bufferIndex バッファのインデックスです。
    //! @details    この関数の内部で nns::gfx::PrimitiveRenderer::Renderer::Update() 実行されます。
    //!             アプリ内で nns::gfx::PrimitiveRenderer::Renderer::Update() を呼ぶ必要はありません。
    virtual void BeginFrame(int bufferIndex) NN_NOEXCEPT NN_OVERRIDE;

    //! @brief ルートコマンドバッファーへのコマンドの追加を終了します。
    //! @param[in]    bufferIndex バッファーのインデックスです。
    //! @details      この関数は EndFrame(bufferIndex, true) を呼びます。
    virtual void EndFrame(int bufferIndex) NN_NOEXCEPT NN_OVERRIDE;

    //! @brief ルートコマンドバッファーへのコマンドの追加を終了します。
    //! @param[in]    bufferIndex バッファーのインデックスです。
    //! @param[in]    enabledCopyImage カラーバッファーの内容をスキャンバッファーにコピーする場合は true を指定します。
    //! @details      この関数では以下の内容を行います。
    //! - enabledCopyImage が true の場合、カラーバッファーの内容をスキャンバッファーにコピーするコマンドを追加
    //! - InitializePerf() で計測機能が有効の場合、パフォーマンスメーターの描画
    //! - フレームワークが保持する nn::gfx::util::DebugFontTextWriter の Draw()
    //! - フレームワークが保持する nn::gfx::CommandBuffer の End()
    virtual void EndFrame(int bufferIndex, bool enabledCopyImage) NN_NOEXCEPT NN_OVERRIDE;

    //! @brief 1 フレームの処理を実行します。
    //! @details InitializePerf() で計測機能が有効になっている場合は、フレーム全体の CPU と GPU の負荷を自動で計測します。
    virtual void ProcessFrame() NN_NOEXCEPT NN_OVERRIDE;


    //! @brief このオブジェクトが初期化済かを返します。
    //! @return 初期化済の場合 true を返します。それ以外の場合 false を返します。
    //! @details
    //! Initialize() により初期化済になります。
    //! Finalize() により未初期化になります。
    bool IsInitialized() const NN_NOEXCEPT
    {
        return m_IsDebugInitialized;
    }

    //! @brief InitializePerf() で計測機能が初期化されているかどうかを取得します。
    //! @return 初期化されている場合は ture を返します。
    bool IsPerfInitialized() const NN_NOEXCEPT
    {
        return m_DebugPerf.isInitialized;
    }

    //! @brief nn::gfx::util::DebugFontTextWriter を取得します。
    //! @return DebugFontTextWriter のポインターを返します。
    //! @details  EndFrame() 内で内部の設定が変更されるため、フレームの先頭で再設定してください。
    //!           EndFrame() 内で nn::gfx::util::DebugFontTextWriter::Draw() を行うため、アプリ内で nn::gfx::util::DebugFontTextWriter::Draw() を呼ぶ必要はありません。
    nns::gfx::GraphicsFramework::DebugFontTextWriter* GetDebugFont() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        return pFontWriter;
    }

    //! @brief nns::gfx::PrimitiveRenderer::Renderer を取得します。
    //! @return Renderer のポインターを返します。
    //! @details  EndFrame() 内で内部の設定が変更されるため、フレームの先頭で再設定してください。
    //!
    nns::gfx::PrimitiveRenderer::Renderer* GetPrimitiveRenderer() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        return m_pRenderer;
    }

private:
    //! @brief ユーザーコマンドとデバッグ用計測コマンドの発行、PresentTexture を行います。
    //! @param[in]    bufferIndex バッファーのインデックスです。
    void ExecuteDebugCommand(int bufferIndex) NN_NOEXCEPT;

    //! @brief デバッグ計測コマンドの作成を行います。
    //! @param[in]    bufferIndex バッファーのインデックスです。
    void MakeDebugCommand(int bufferIndex) NN_NOEXCEPT;

    // DebugGraphicsFramework が初期化されているか
    bool m_IsDebugInitialized;

    // デバッグフォント
    nns::gfx::GraphicsFramework::DebugFontTextWriter* pFontWriter;

    // プリミティブレンダラー
    nns::gfx::PrimitiveRenderer::Renderer* m_pRenderer;

    // 計測機能
    struct DebugPerf
    {
        nn::perf::LoadMeterCenterInfo info;
        nns::gfx::PrimitiveRenderer::MeterDrawer* pMeterDrawer;
        nns::gfx::GraphicsFramework::CommandBuffer* pCommandBufferBegin;
        nns::gfx::GraphicsFramework::CommandBuffer* pCommandBufferEnd;
        void*     pMemory;
        ptrdiff_t memoryPoolOffset;
        bool isInitialized;
    } m_DebugPerf;
};

} // namespace gfx
} // namespace nns
