﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Allocator.h>
#include <nn/nn_Macro.h>
#include <nn/gfx/gfx_Types.h>
#include <nn/gfx/gfx_StateInfo.h>

#include "dbgui_Types.h"

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
#define NNS_DBGUI_ENABLE_SOFTWARE_KEYBOARD
#endif

namespace nns { namespace dbgui {


/**
* @brief ライブラリの初期化の情報
*/
class InterfaceInfo
{
    NN_DISALLOW_COPY(InterfaceInfo);

private:
    nn::gfx::DescriptorPool*                    m_pSamplerDescriptorPool;
    int                                         m_SamplerDescriptorPoolIndex;
    nn::gfx::DescriptorPool*                    m_pTextureViewDescriptorPool;
    int                                         m_TextureViewDescriptorPoolIndex;
    nn::gfx::DescriptorPool*                    m_pConstantBufferDescriptorPool;
    int                                         m_ConstantBufferDescriptorPoolIndex;

    int                                         m_ImguiDisplayWidth;
    int                                         m_ImguiDisplayHeight;

    UpdateScissorCallback                       m_pUpdateScissorCallback;
    void*                                       m_pUpdateScissorCallbackUserData;

    ViewportYAxisDirection                      m_ViewportYAxisDirection;

    int                                         m_BufferCount;

    nn::AlignedAllocateFunctionWithUserData     m_pMemoryCallbackAllocate;
    nn::FreeFunctionWithUserData                m_pMemoryCallbackFree;
    void*                                       m_pMemoryCallbackUserData;

    const void*                                 m_pCustomTtfBuffer;
    size_t                                      m_CustomTtfBufferSize;
    bool                                        m_IsCustomTtfCompressed;
    float                                       m_CustomTtfCharacterSize;

    const char*                                 m_IniFilePath;

    MouseMoveFunction                           m_pMouseMoveFunction;
    void*                                       m_pMouseMoveFunctionUserData;

    int                                         m_NpadNavigationKeyBindings[NavigationKeyBinding_Max];
    int                                         m_DebugPadNavigationKeyBindings[NavigationKeyBinding_Max];

    float                                       m_MouseWheelScrollRatio;

public:
    /**
    * @brief コンストラクター
    *
    * @post
    * - 各パラメーターがゼロ初期化されている
    */
                                InterfaceInfo() NN_NOEXCEPT;

    /**
    * @brief デストラクター
    */
                                ~InterfaceInfo() NN_NOEXCEPT;

    /**
    * @brief 各パラメーターを既定値に設定するためのヘルパー関数です
    */
    void                        SetDefault() NN_NOEXCEPT;

    /**
    * @brief パッドキーアサインを既定の設定にします
    *
    * @details
    * キーアサインは以下の通りです：\n
    * \n
    * Npad:\n
    * NavigationKeyBinding_Activate    NpadButton::A\n
    * NavigationKeyBinding_Cancel      NpadButton::B\n
    * NavigationKeyBinding_Input       NpadButton::X\n
    * NavigationKeyBinding_Menu        NpadButton::Y\n
    * NavigationKeyBinding_Up          NpadButton::Up\n
    * NavigationKeyBinding_Down        NpadButton::Down\n
    * NavigationKeyBinding_Left        NpadButton::Left\n
    * NavigationKeyBinding_Right       NpadButton::Right\n
    * NavigationKeyBinding_FocusPrev   NpadButton::ZL\n
    * NavigationKeyBinding_FocusNext   NpadButton::ZR\n
    * NavigationKeyBinding_TweakSlow   NpadButton::L\n
    * NavigationKeyBinding_TweakFast   NpadButton::R\n
    * NavigationKeyBinding_AccessMenu  NpadButton::Minus\n
    * \n
    * DebugPad:\n
    * NavigationKeyBinding_Activate    DebugPadButton::A\n
    * NavigationKeyBinding_Cancel      DebugPadButton::B\n
    * NavigationKeyBinding_Input       DebugPadButton::X\n
    * NavigationKeyBinding_Menu        DebugPadButton::Y\n
    * NavigationKeyBinding_Up          DebugPadButton::Up\n
    * NavigationKeyBinding_Down        DebugPadButton::Down\n
    * NavigationKeyBinding_Left        DebugPadButton::Left\n
    * NavigationKeyBinding_Right       DebugPadButton::Right\n
    * NavigationKeyBinding_FocusPrev   DebugPadButton::ZL\n
    * NavigationKeyBinding_FocusNext   DebugPadButton::ZR\n
    * NavigationKeyBinding_TweakSlow   DebugPadButton::L\n
    * NavigationKeyBinding_TweakFast   DebugPadButton::R\n
    * NavigationKeyBinding_AccessMenu  DebugPadButton::Select\n
    */
    void                        SetDefaultKeyBindings() NN_NOEXCEPT;


    /**
    * @brief 有効な値が設定されているかを確認します
    *
    * @return 有効な値が設定されている場合 true を返します
    *
    * @details
    * 以下のポイントを確認します：\n
    * 各デスクリプタープールとデスクリプタープールのインデックスは有効な値が設定されています\n
    * バッファの数が 1 以上です\n
    * メモリコールバックが設定されている\n
    * ttfバッファのプロパティーは有効な値が設定されています\n
    * キーボード入力の方式は有効な値が設定されています\n
    * Navigation のキーアサインは全部設定されています\n
    */
    bool                        IsValid() const NN_NOEXCEPT;

    /**
    * @brief バッファ数を設定する
    *
    * @param[in] count              必要なバッファの数
    *
    */
    void                        SetBufferCount(int count) NN_NOEXCEPT;

    /**
    * @brief バッファの数を取得します
    *
    * @return バッファの数を返します
    */
    int                         GetBufferCount() const NN_NOEXCEPT;

    /**
    * @brief ImGui が描画する領域の基準サイズの設定
    *
    * @param[in] imguiDisplayWidth      解像度の横幅
    * @param[in] imguiDisplayHeight     解像度の縦幅
    *
    * @details
    * 頂点シェーダーで基準サイズから (-1, 1) へ射影した後、GPU でビューポート変換を行うため、
    * 実際に描画されるサイズは設定されたビューポートの解像度によって変化します。
    */
    void                        SetImguiDisplaySize(int imguiDisplayWidth, int imguiDisplayHeight) NN_NOEXCEPT;

    /**
    * @brief ImGui が描画する領域の基準サイズの横幅を取得します
    *
    * @return ImGui が描画する領域の基準サイズの横幅を返します
    */
    int                         GetImguiDisplayWidth() const NN_NOEXCEPT;

    /**
    * @brief ImGui が描画する領域の基準サイズの縦幅を取得します
    *
    * @return ImGui が描画する領域の基準サイズの縦幅を返します
    */
    int                         GetImguiDisplayHeight() const NN_NOEXCEPT;

    /**
    * @brief ライブラリのメモリアロケータを設定します
    *
    * @param[in] pMemoryCallbackAllocate        メモリ確保のコールバック
    * @param[in] pMemoryCallbackFree            メモリ解放のコールバック
    * @param[in] pMemoryCallbackUserData        コールバックのユーザーデータ
    */
    void                        SetAllocator(
                                    nn::AlignedAllocateFunctionWithUserData pMemoryCallbackAllocate,
                                    nn::FreeFunctionWithUserData pMemoryCallbackFree,
                                    void* pMemoryCallbackUserData) NN_NOEXCEPT;
    /**
    * @brief メモリ確保のコールバックを取得します
    *
    * @return メモリ確保のコールバックを返します
    */
    nn::AlignedAllocateFunctionWithUserData
                                GetMemoryCallbackAllocate() const NN_NOEXCEPT;
    /**
    * @brief メモリ解放のコールバックを取得します
    *
    * @return メモリ解放のコールバックを返します
    */
    nn::FreeFunctionWithUserData
                                GetMemoryCallbackFree() const NN_NOEXCEPT;
    /**
    * @brief コールバックのユーザーデータを取得します
    *
    * @return コールバックのユーザーデータを返します
    */
    void*                       GetMemoryCallbackUserData() const NN_NOEXCEPT;

    /**
    * @brief デスクリプタープールと対象のスロット番号を設定します
    *
    * @param[in] pSamplerDescriptorPool                 テクスチャサンプラのデスクリプタープール
    * @param[in] samplerDescriptorPoolIndex             テクスチャサンプラのスロット番号
    * @param[in] pTextureViewDescriptorPool             テクスチャビューのデスクリプタープール
    * @param[in] textureViewDescriptorPoolIndex         テクスチャビューのスロット番号
    * @param[in] pConstantBufferDescriptorPool          定数バッファーのデスクリプタープール
    * @param[in] constantBufferDescriptorPoolIndex      定数バッファーのスロット番号
    */
    void                        SetDescriptorPoolData(
                                    nn::gfx::DescriptorPool* pSamplerDescriptorPool,
                                    int samplerDescriptorPoolIndex,
                                    nn::gfx::DescriptorPool* pTextureViewDescriptorPool,
                                    int textureViewDescriptorPoolIndex,
                                    nn::gfx::DescriptorPool* pConstantBufferDescriptorPool,
                                    int constantBufferDescriptorPoolIndex) NN_NOEXCEPT;
    /**
    * @brief テクスチャサンプラのデスクリプタープールを取得します
    *
    * @return テクスチャサンプラのデスクリプタープールを返します
    */
    nn::gfx::DescriptorPool*    GetSamplerDescriptorPool() const NN_NOEXCEPT;
    /**
    * @brief テクスチャサンプラのスロット番号を取得します
    *
    * @return テクスチャサンプラのスロット番号を返します
    */
    int                         GetSamplerDescriptorPoolIndex() const NN_NOEXCEPT;

    /**
    * @brief テクスチャビューのデスクリプタープールを取得します
    *
    * @return テクスチャビューのデスクリプタープールを返します
    */
    nn::gfx::DescriptorPool*    GetTextureViewDescriptorPool() const NN_NOEXCEPT;

    /**
    * @brief テクスチャビューのスロット番号を取得します
    *
    * @return テクスチャビューのスロット番号を返します
    */
    int                         GetTextureViewDescriptorPoolIndex() const NN_NOEXCEPT;

    /**
    * @brief 定数バッファーのデスクリプタープールを取得します
    *
    * @return 定数バッファーのデスクリプタープールを返します
    */
    nn::gfx::DescriptorPool*    GetConstantBufferDescriptorPool() const NN_NOEXCEPT;

    /**
    * @brief 定数バッファーのスロット番号を取得します
    *
    * @return 定数バッファーのスロット番号を返します
    */
    int                         GetConstantBufferDescriptorPoolIndex() const NN_NOEXCEPT;

    /**
    * @brief カスタム True Type Font (TTF) を設定します
    *
    * @param[in] pCustomTtfBuffer           TTFファイルのバッファ
    * @param[in] customTtfBufferSize        バッファのサイズ
    * @param[in] isCustomTtfCompressed      バッファの内容が圧縮されたか。stb_decompressを呼び出すことで解凍します。
    * @param[in] customTtfCharacterSize     文字の縦幅
    */
    void                        SetCustomTtf(
                                    const void* pCustomTtfBuffer, size_t customTtfBufferSize,
                                    bool isCustomTtfCompressed, float customTtfCharacterSize) NN_NOEXCEPT;
    /**
    * @brief TTFファイルのバッファの取得
    *
    * @return TTFファイルのバッファを返します
    */
    const void*                 GetCustomTtfBuffer() const NN_NOEXCEPT;

    /**
    * @brief バッファのサイズを取得します
    *
    * @return バッファのサイズを返します
    */
    size_t                      GetCustomTtfBufferSize() const NN_NOEXCEPT;

    /**
    * @brief バッファの内容が圧縮されたかの取得
    *
    * @return バッファの内容が圧縮されたかを返します
    */
    bool                        IsCustomTtfCompressed() const NN_NOEXCEPT;
    /**
    * @brief 文字の縦幅の取得
    *
    * @return 文字の縦幅を返します
    */
    float                       GetCustomTtfCharacterSize() const NN_NOEXCEPT;

    /**
    * @brief ImGui の初期化ファイルのパスを設定します
    *
    * @param[in] filePath ImGui の初期化ファイルのパス
    */
    void                        SetIniFilePath(const char* filePath) NN_NOEXCEPT;

    /**
    * @brief ImGui の初期化ファイルのパスを取得します
    *
    * @return ImGui の初期化ファイルのパスを返します
    */
    const char*                 GetIniFilePath() const NN_NOEXCEPT;

    /**
    * @brief シザーを変換するコールバックの設定
    *
    * @param[in] pUpdateScissorCallback             アプリコールバック
    * @param[in] pUserData                          コールバックのユーザーデータ
    */
    void                        SetUpdateScissorCallback(UpdateScissorCallback pUpdateScissorCallback, void* pUserData) NN_NOEXCEPT;

    /**
    * @brief シザーを変換するコールバックの取得
    *
    * @return シザーを変換するコールバックを返します
    */
    UpdateScissorCallback       GetUpdateScissorCallback() const NN_NOEXCEPT;

    /**
    * @brief シザーを変換するコールバックのユーザーデータの取得
    *
    * @return シザーを変換するコールバックのユーザーデータを返します
    */
    void*                       GetUpdateScissorCallbackUserData() const NN_NOEXCEPT;

    /**
    * @brief ハードウェアが描画するビューポートのY軸の方向を設定します
    *
    * @param[in] viewportYAxisDirection ハードウェアが描画するビューポートのY軸の方向
    */
    void                        SetViewportYAxisDirection(ViewportYAxisDirection viewportYAxisDirection) NN_NOEXCEPT;

    /**
    * @brief レンダリングのハードウェアがビューポートのY軸の方向の取得
    *
    * @return レンダリングのハードウェアがビューポートのY軸の方向を返します
    */
    ViewportYAxisDirection      GetViewportYAxisDirection() const NN_NOEXCEPT;


    /**
    * @brief Navigation 機能で ImGui からマウス座標の変更を依頼された際に
    * アプリへ転送するためのコールバックを設定します
    *
    * @param[in] pMouseMoveFunction         アプリのコールバック
    * @param[in] pUserData                  コールバックのユーザデータ
    */
    void                        SetMouseMoveFunction(
                                    MouseMoveFunction pMouseMoveFunction,
                                    void* pUserData) NN_NOEXCEPT;

    /**
    * @brief マウス座標変更のコールバックの取得
    *
    * @return マウス座標変更のコールバックを返します
    */
    MouseMoveFunction           GetMouseMoveFunction() const NN_NOEXCEPT;

    /**
    * @brief マウス座標変更のコールバックのユーザデータの取得
    *
    * @return マウス座標変更のコールバックのユーザデータを返します
    */
    void*                       GetMouseMoveFunctionUserData() const NN_NOEXCEPT;

    /**
    * @brief Npad のキーアサインを設定します
    *
    * @param[in] navigationKeyBinding       NavigationKeyBinding の一つの値
    * @param[in] buttonIndex                NpadButton の索引
    */
    void                        SetNpadNavigationKeyBinding(NavigationKeyBinding navigationKeyBinding, int buttonIndex) NN_NOEXCEPT;

    /**
    * @brief Npad のキーアサインの取得
    *
    * @param[in] navigationKeyBinding       NavigationKeyBindingの一つの値
    *
    * @return 設定された NpadButton の索引を返します
    */
    int                         GetNpadNavigationKeyBinding(NavigationKeyBinding navigationKeyBinding) const NN_NOEXCEPT;

    /**
    * @brief Npad のキーアサインの配列の取得
    *
    * @return 設定されたNpadButton の配列を返します
    */
    const int*                  GetNpadNavigationKeyBindings() const NN_NOEXCEPT;

    /**
    * @brief DebugPad のキーアサインを設定します
    *
    * @param[in] navigationKeyBinding       NavigationKeyBinding の一つの値
    * @param[in] buttonIndex                DebugPadButton の索引
    */
    void                        SetDebugPadNavigationKeyBinding(NavigationKeyBinding navigationKeyBinding, int buttonIndex) NN_NOEXCEPT;

    /**
    * @brief DebugPad のキーアサインの取得
    *
    * @param[in] navigationKeyBinding       NavigationKeyBindingの一つの値
    *
    * @return 設定された NpadButton の索引を返します
    */
    int                         GetDebugPadNavigationKeyBinding(NavigationKeyBinding navigationKeyBinding) const NN_NOEXCEPT;

    /**
    * @brief DebugPad のキーアサインの配列の取得
    *
    * @return 設定されたNpadButton の配列を返します
    */
    const int*                  GetDebugPadNavigationKeyBindings() const NN_NOEXCEPT;


    /**
    * @brief マウスホイールの回転差分の倍率を設定します
    *
    * @param[in] mouseWheelScrollRatio      マウスホイールの回転差分の倍率
    *
    * @details
    * 渡した値でマウスホイールの回転差分を乗算する。
    * ImGui では 1.0f = ５つの行列、アプリに適当な倍率を調節ことができます。
    */
    void                        SetMouseWheelScrollRatio(float mouseWheelScrollRatio) NN_NOEXCEPT;

    /**
    * @brief マウスホイールの回転差分の倍率をの取得
    *
    * @return マウスホイールの回転差分の倍率を返します
    */
    float                       GetMouseWheelScrollRatio() const NN_NOEXCEPT;
};


inline void InterfaceInfo::SetBufferCount(int count) NN_NOEXCEPT
{
    m_BufferCount = count;
}

inline int InterfaceInfo::GetBufferCount() const NN_NOEXCEPT
{
    return m_BufferCount;
}

inline void InterfaceInfo::SetImguiDisplaySize(int imguiDisplayWidth, int imguiDisplayHeight) NN_NOEXCEPT
{
    m_ImguiDisplayWidth = imguiDisplayWidth;
    m_ImguiDisplayHeight = imguiDisplayHeight;
}

inline int InterfaceInfo::GetImguiDisplayWidth() const NN_NOEXCEPT
{
    return m_ImguiDisplayWidth;
}

inline int InterfaceInfo::GetImguiDisplayHeight() const NN_NOEXCEPT
{
    return m_ImguiDisplayHeight;
}

inline void InterfaceInfo::SetAllocator(
    nn::AlignedAllocateFunctionWithUserData pMemoryCallbackAllocate,
    nn::FreeFunctionWithUserData pMemoryCallbackFree,
    void* pMemoryCallbackUserData) NN_NOEXCEPT
{
    m_pMemoryCallbackAllocate = pMemoryCallbackAllocate;
    m_pMemoryCallbackFree = pMemoryCallbackFree;
    m_pMemoryCallbackUserData = pMemoryCallbackUserData;
}

inline nn::AlignedAllocateFunctionWithUserData InterfaceInfo::GetMemoryCallbackAllocate() const NN_NOEXCEPT
{
    return m_pMemoryCallbackAllocate;
}

inline nn::FreeFunctionWithUserData InterfaceInfo::GetMemoryCallbackFree() const NN_NOEXCEPT
{
    return m_pMemoryCallbackFree;
}

inline void* InterfaceInfo::GetMemoryCallbackUserData() const NN_NOEXCEPT
{
    return m_pMemoryCallbackUserData;
}

inline void InterfaceInfo::SetDescriptorPoolData(
    nn::gfx::DescriptorPool* pSamplerDescriptorPool, int samplerDescriptorPoolIndex,
    nn::gfx::DescriptorPool* pTextureViewDescriptorPool, int textureViewDescriptorPoolIndex,
    nn::gfx::DescriptorPool* pConstantBufferDescriptorPool, int constantBufferDescriptorPoolIndex) NN_NOEXCEPT
{
    m_pSamplerDescriptorPool = pSamplerDescriptorPool;
    m_SamplerDescriptorPoolIndex = samplerDescriptorPoolIndex;
    m_pTextureViewDescriptorPool = pTextureViewDescriptorPool;
    m_TextureViewDescriptorPoolIndex = textureViewDescriptorPoolIndex;
    m_pConstantBufferDescriptorPool = pConstantBufferDescriptorPool;
    m_ConstantBufferDescriptorPoolIndex = constantBufferDescriptorPoolIndex;
}

inline nn::gfx::DescriptorPool* InterfaceInfo::GetSamplerDescriptorPool() const NN_NOEXCEPT
{
    return m_pSamplerDescriptorPool;
}

inline int InterfaceInfo::GetSamplerDescriptorPoolIndex() const NN_NOEXCEPT
{
    return m_SamplerDescriptorPoolIndex;
}

inline nn::gfx::DescriptorPool* InterfaceInfo::GetTextureViewDescriptorPool() const NN_NOEXCEPT
{
    return m_pTextureViewDescriptorPool;
}

inline int InterfaceInfo::GetTextureViewDescriptorPoolIndex() const NN_NOEXCEPT
{
    return m_TextureViewDescriptorPoolIndex;
}

inline nn::gfx::DescriptorPool* InterfaceInfo::GetConstantBufferDescriptorPool() const NN_NOEXCEPT
{
    return m_pConstantBufferDescriptorPool;
}

inline int InterfaceInfo::GetConstantBufferDescriptorPoolIndex() const NN_NOEXCEPT
{
    return m_ConstantBufferDescriptorPoolIndex;
}

inline void InterfaceInfo::SetCustomTtf(
    const void* pCustomTtfBuffer, size_t customTtfBufferSize,
    bool isCustomTtfCompressed, float customTtfCharacterSize) NN_NOEXCEPT
{
    m_pCustomTtfBuffer = pCustomTtfBuffer;
    m_CustomTtfBufferSize = customTtfBufferSize;
    m_IsCustomTtfCompressed = isCustomTtfCompressed;
    m_CustomTtfCharacterSize = customTtfCharacterSize;
}

inline const void* InterfaceInfo::GetCustomTtfBuffer() const NN_NOEXCEPT
{
    return m_pCustomTtfBuffer;
}

inline size_t InterfaceInfo::GetCustomTtfBufferSize() const NN_NOEXCEPT
{
    return m_CustomTtfBufferSize;
}

inline bool InterfaceInfo::IsCustomTtfCompressed() const NN_NOEXCEPT
{
    return m_IsCustomTtfCompressed;
}

inline float InterfaceInfo::GetCustomTtfCharacterSize() const NN_NOEXCEPT
{
    return m_CustomTtfCharacterSize;
}

inline void InterfaceInfo::SetIniFilePath(const char* filePath) NN_NOEXCEPT
{
    m_IniFilePath = filePath;
}

inline const char* InterfaceInfo::GetIniFilePath() const NN_NOEXCEPT
{
    return m_IniFilePath;
}

inline void InterfaceInfo::SetUpdateScissorCallback(UpdateScissorCallback pUpdateScissorCallback, void* pUserData) NN_NOEXCEPT
{
    m_pUpdateScissorCallback = pUpdateScissorCallback;
    m_pUpdateScissorCallbackUserData = pUserData;
}

inline UpdateScissorCallback InterfaceInfo::GetUpdateScissorCallback() const NN_NOEXCEPT
{
    return m_pUpdateScissorCallback;
}

inline void* InterfaceInfo::GetUpdateScissorCallbackUserData() const NN_NOEXCEPT
{
    return m_pUpdateScissorCallbackUserData;
}

inline void InterfaceInfo::SetViewportYAxisDirection(ViewportYAxisDirection viewportYAxisDirection) NN_NOEXCEPT
{
    m_ViewportYAxisDirection = viewportYAxisDirection;
}

inline ViewportYAxisDirection InterfaceInfo::GetViewportYAxisDirection() const NN_NOEXCEPT
{
    return m_ViewportYAxisDirection;
}

inline void InterfaceInfo::SetMouseMoveFunction(
    MouseMoveFunction pMouseMoveFunction,
    void* pUserData) NN_NOEXCEPT
{
    m_pMouseMoveFunction = pMouseMoveFunction;
    m_pMouseMoveFunctionUserData = pUserData;
}

inline MouseMoveFunction InterfaceInfo::GetMouseMoveFunction() const NN_NOEXCEPT
{
    return m_pMouseMoveFunction;
}

inline void* InterfaceInfo::GetMouseMoveFunctionUserData() const NN_NOEXCEPT
{
    return m_pMouseMoveFunctionUserData;
}

inline void InterfaceInfo::SetNpadNavigationKeyBinding(NavigationKeyBinding navigationKeyBinding, int buttonIndex) NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(navigationKeyBinding, 0, NavigationKeyBinding_Max);
    m_NpadNavigationKeyBindings[navigationKeyBinding] = buttonIndex;
}

inline int InterfaceInfo::GetNpadNavigationKeyBinding(NavigationKeyBinding navigationKeyBinding) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(navigationKeyBinding, 0, NavigationKeyBinding_Max);
    return m_NpadNavigationKeyBindings[navigationKeyBinding];
}

inline const int* InterfaceInfo::GetNpadNavigationKeyBindings() const NN_NOEXCEPT
{
    return m_NpadNavigationKeyBindings;
}


inline void InterfaceInfo::SetDebugPadNavigationKeyBinding(NavigationKeyBinding navigationKeyBinding, int buttonIndex) NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(navigationKeyBinding, 0, NavigationKeyBinding_Max);
    m_DebugPadNavigationKeyBindings[navigationKeyBinding] = buttonIndex;
}

inline int InterfaceInfo::GetDebugPadNavigationKeyBinding(NavigationKeyBinding navigationKeyBinding) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(navigationKeyBinding, 0, NavigationKeyBinding_Max);
    return m_DebugPadNavigationKeyBindings[navigationKeyBinding];
}

inline const int* InterfaceInfo::GetDebugPadNavigationKeyBindings() const NN_NOEXCEPT
{
    return m_DebugPadNavigationKeyBindings;
}

inline void InterfaceInfo::SetMouseWheelScrollRatio(float mouseWheelScrollRatio) NN_NOEXCEPT
{
    m_MouseWheelScrollRatio = mouseWheelScrollRatio;
}


inline float InterfaceInfo::GetMouseWheelScrollRatio() const NN_NOEXCEPT
{
    return m_MouseWheelScrollRatio;
}



} } // namespace nns { namespace dbgui {
