﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/gfx/gfx_Types.h>
#include <nn/hid/hid_NpadCommonTypes.h>
#include <nn/hid/hid_DebugPad.h>

#include "dbgui_InterfaceInfo.h"

namespace nn { namespace hid {
struct DebugPadState;
struct KeyboardState;
struct MouseState;
struct TouchState;
struct AnalogStickState;
} }

namespace nns { namespace dbgui {

/**
* @name 初期化／後処理
* @{
*/

/**
* @brief ライブラリの初期化
*
* @param[in]          pDevice         gfx::Device のインスタンス
* @param[in]          info            ライブラリの設定
*
* @pre
* - !IsInitialized()
*/
void Initialize(nn::gfx::Device* pDevice, const InterfaceInfo& info) NN_NOEXCEPT;


/**
* @brief ライブラリの終了
*
* @param[in]          pDevice         gfx::Device のインスタンス。初期化の時もらったインスタンスと同じです
*
* @pre
* - IsInitialized()
*/
void Finalize(nn::gfx::Device* pDevice) NN_NOEXCEPT;

/**
* @brief 初期化済みかどうかを取得します。
* @return 初期化済みかどうかを返します。
*/
bool IsInitialized() NN_NOEXCEPT;


/**
* @}
*/

/**
* @name 文字入力
* @{
*/

/**
* @brief Keyboard の入力状態から文字入力を行います。
*
* @param[in] keyboardState      hid からもらったキーボードの入力状態

* @details
* 文字を入力するために使われています。
* 通常の ASCII の文字しか対応していないので、他の文字の対応が必要だったら
* AddInputCharacter　か　AddInputCharactersUtf8 を直接に使ってください。
*
* @pre
* - IsInitialized()
*
* @see AddInputCharacter
* @see AddInputCharactersUtf8
*/
void AddInputCharactersFromKeyboardState(const nn::hid::KeyboardState& keyboardState) NN_NOEXCEPT;

/**
* @brief ImGui へ文字入力を転送します。
*
* @param[in] c      ImGui へ送る文字
*
* @pre
* - IsInitialized()
*/
void AddInputCharacter(char c) NN_NOEXCEPT;


/**
* @brief ImGui へUTF8文字入力を転送します。
*
* @param[in] c      ImGui へ送るUTF8文字
*
* @pre
* - IsInitialized()
*/
void AddInputCharactersUtf8(const char* c) NN_NOEXCEPT;



#if defined(NNS_DBGUI_ENABLE_SOFTWARE_KEYBOARD)
/**
* @brief ソフトキーボードの管理
*
* @details
* この関数を呼び出すことで、文字入力が要ったらソフトキーボードを表示します。
*
* @pre
* - IsInitialized()
*/
void UpdateSoftwareKeyboard() NN_NOEXCEPT;
#endif


/**
* @}
*/


/**
* @name メニューナビゲーション入力
* @{
*/

/**
* @brief キーボードの入力状態を ImGui へ転送
*
* @param[in] keyboardState      hid からもらったキーボードの入力状態
*
* @pre
* - IsInitialized()
*
* @details
* この関数は、メニューナビゲーション向けの有力しか対応しています。\n
* 文字入力は AddInputCharactersFromKeyboardState か AddInputCharacter か
* AddInputCharactersUtf8 という関数を呼び出すことで文字入力を ImGui へ転送できます。
*
* @pre
* - IsInitialized()
*
* @see AddInputCharactersFromKeyboardState
* @see AddInputCharacter
* @see AddInputCharactersUtf8
*
*/
void SetKeyboardState(const nn::hid::KeyboardState& keyboardState) NN_NOEXCEPT;



/**
* @brief デバッグパッドの入力状態を ImGui へ転送
*
* @param[in] debugPadButtonSet      hid からもらったボタンの状態
* @param[in] analogStickState       hid からもらったアナログスティックの入力状態
*
* @pre
* - IsInitialized()
*/
void SetDebugPadState(
    const nn::hid::DebugPadButtonSet& debugPadButtonSet,
    const nn::hid::AnalogStickState& analogStickState) NN_NOEXCEPT;

/**
* @brief Npadの入力状態を ImGui へ転送
*
* @param[in] npadButtonSet      hid からもらったボタンの状態
* @param[in] analogStickState   hid からもらったアナログスティックの入力状態
*
* @pre
* - IsInitialized()
*/
void SetNpadState(
    const nn::hid::NpadButtonSet& npadButtonSet,
    const nn::hid::AnalogStickState& analogStickState) NN_NOEXCEPT;

/**
* @brief マウスの入力状態を ImGui へ転送
*
* @param[in] mouseState         hid からもらったマウスの入力状態
*
* @pre
* - IsInitialized()
*/
void SetMouseState(const nn::hid::MouseState& mouseState) NN_NOEXCEPT;

/**
* @brief タッチの入力状態を ImGui へ転送
*
* @param[in] touchState      hid からもらったッチの入力状態
*
* @pre
* - IsInitialized()
*/
void SetTouchState(const nn::hid::TouchState& touchState) NN_NOEXCEPT;

/**
* @}
*/

/**
* @name ImGui の実行
* @{
*/

/**
* @brief 次のフレームを用意します。
*
*
* @details
* ImGui::NewFrame の代わりにこの関数を呼び出すことで、
* ImGui と本ライブラリは次のフレームを用意します。
*
* @pre
* - IsInitialized()
*/
void NewFrame() NN_NOEXCEPT;

/**
* @brief フレームを描画します。
*
* @param[in] pCommandBuffer         レンダリングコマンドを追加へのコマンドバッファ
*
* @details ImGui::Render の代わりにこの関数を呼び出すことで、
* ImGui のフレームを描画します。
*
* @pre
* - NewFrame() が呼び出されました。
* - IsInitialized()
*/
void Render(nn::gfx::CommandBuffer* pCommandBuffer) NN_NOEXCEPT;

/**
* @}
*/

} } // namespace nns { namespace dbgui {
