﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//------------------------------------------------------------------
// デモ: RuntimeTypeInfo
//
// 概要
//   クラスに実行時型情報を埋め込み安全なダウンキャストをおこなうサンプルです。
//   Baseクラスのポインタが指しているAAAクラス、BBBクラス、CCC_AAAクラスの
//   オブジェクトへのインスタンスを安全に派生クラスにキャストします。
//
//   シリアル経由でのデバッグ出力のみのサンプルです。
//
//------------------------------------------------------------------

#define NW_CONSOLE_ENABLE

#include <nw/ut/ut_RuntimeTypeInfo.h>

#if defined(NW_PLATFORM_CAFE)
#include <cafe.h>
#endif

// クラスBase
class Base
{
public:
    NW_UT_RUNTIME_TYPEINFO_ROOT()

    virtual void    Print()   { NW_LOG("I'm Base\n"); }
};

// クラスAAA
class AAA : public Base
{
public:
    NW_UT_RUNTIME_TYPEINFO(Base)

    virtual void    Print()   { NW_LOG("I'm AAA\n");    }
    void            FuncAAA() { NW_LOG("do FuncAAA\n"); }
};

// クラスBBB
class BBB : public Base
{
public:
    NW_UT_RUNTIME_TYPEINFO(Base)

    virtual void    Print()   { NW_LOG("I'm BBB\n");    }
    void            FuncBBB() { NW_LOG("do FuncBBB\n"); }
};

// クラスCCC
class CCC_AAA : public AAA
{
public:
    NW_UT_RUNTIME_TYPEINFO(AAA)

    virtual void    Print()   { NW_LOG("I'm CCC\n");    }
    void            FuncCCC() { NW_LOG("do FuncCCC\n"); }

};




/*---------------------------------------------------------------------------*
  Name:         ObjTest

  Description:  オブジェクトのダウンキャストのテスト関数。
                クラスAAA, BBB, CCC_AAAへのダウンキャストを試みて
                キャストに成功した場合には派生クラスのメソッドを実行します。

  Arguments:    obj     Baseの派生オブジェクトへのポインタ

  Returns:      None.
 *---------------------------------------------------------------------------*/
static void
ObjTest( Base* obj )
{
    AAA*     pAAA;
    BBB*     pBBB;
    CCC_AAA* pCCC;

    NW_LOG("-----------------------------\n");
    obj->Print();

    // クラスAAAにキャスト
    pAAA = nw::ut::DynamicCast<AAA*>( obj );
    if ( pAAA )
    {
        pAAA->FuncAAA();
    }
    else
    {
        NW_LOG("Can't cast to AAA\n");
    }

    // クラスBBBにキャスト
    pBBB = nw::ut::DynamicCast<BBB*>( obj );
    if ( pBBB )
    {
        pBBB->FuncBBB();
    }
    else
    {
        NW_LOG("Can't cast to BBB\n");
    }

    // クラスCCC_AAAにキャスト
    pCCC = nw::ut::DynamicCast<CCC_AAA*>( obj );
    if ( pCCC )
    {
        pCCC->FuncCCC();
    }
    else
    {
        NW_LOG("Can't cast to CCC\n");
    }

    // リリースビルドでは高速に動作する、ASSERT 付きキャスト
    // CCC_AAA* pCCC2 = nw::ut::StaticCast<CCC_AAA*>( obj );
}



/*---------------------------------------------------------------------------*
  Name:         main

  Description:  サンプルのメイン関数です。

  Arguments:    なし。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
int
main(int /* argc */, char ** /* argv */)
{
    Base    obj1;
    AAA     obj2;
    BBB     obj3;
    CCC_AAA obj4;

    ObjTest( &obj1 );
    ObjTest( &obj2 );
    ObjTest( &obj3 );
    ObjTest( &obj4 );

    NW_LOG( "End of demo.\n" );
    return 0;
}



