﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#define NW_CONSOLE_ENABLE // Release 版でも NW_LOG を有効にするため

#include <cafe.h>
#include <cafe/pads/kpad/kpad.h>
#include <nw/snd.h>
#include <nw/demo/demo_Main.h>
#include "common.fsid"

namespace
{
const char CONTENT_DIR[] = "/vol/content";
const char SOUND_ARC_PATH[] = "snd/common/common.bfsar";
const s32 SOUND_HEAP_SIZE = 4 * 1024 * 1024;

FSClient* s_pFsClient;

nw::snd::FsSoundArchive     s_SoundArchive;
nw::snd::SoundArchivePlayer s_SoundArchivePlayer;
nw::snd::SoundDataManager   s_SoundDataManager;
nw::snd::SoundHeap          s_SoundHeap;
nw::snd::SoundHandle        s_SoundHandle;

void* s_pMemoryForSoundSystem;
void* s_pMemoryForFsSoundArchive;
void* s_pMemoryForInfoBlock;
void* s_pMemoryForSoundDataManager;
void* s_pMemoryForSoundArchivePlayer;
void* s_pMemoryForSoundHeap;
void* s_pMemoryForStreamBuffer;

f32 s_Volume = 1.0f;
f32 s_Pitch = 1.0f;
const f32 PARAM_DELTA = 0.2f;
const f32 PARAM_MIN = 0.2f;
const f32 PARAM_MAX = 4.0f;

const int REMOTE_OUTPUT_LINE_FLAG[ WPAD_MAX_CONTROLLERS ] =
{
    nw::snd::OUTPUT_LINE_REMOTE0,
    nw::snd::OUTPUT_LINE_REMOTE1,
    nw::snd::OUTPUT_LINE_REMOTE2,
    nw::snd::OUTPUT_LINE_REMOTE3
};

void SpeakerSetupCallback( s32 channel, s32 result )
{
    NW_LOG( "SpeakerSetupCallback    [channel] %d, [result] %d \n", channel, result );
}

void SpeakerShutdownCallback( s32 channel, s32 result )
{
    NW_LOG( "SpeakerShutdownCallback [channel] %d, [result] %d \n", channel, result );
}

void ConnectCallback( s32 channel, s32 reason )
{
    if ( reason == WPAD_ERR_NONE )
    {
        nw::snd::SoundSystem::GetRemoteSpeaker( channel ).Initialize( SpeakerSetupCallback );
    }
    else if ( reason == WPAD_ERR_NO_CONTROLLER )
    {
        nw::snd::SoundSystem::GetRemoteSpeaker( channel ).Finalize( SpeakerShutdownCallback );
    }
}

void StartSound( u32 soundId, int remoteIndex )
{
    s_SoundHandle.Stop( 0 );
    bool result = s_SoundArchivePlayer.StartSound( &s_SoundHandle, soundId ).IsSuccess();
    if ( result )
    {
        s_SoundHandle.SetOutputLine( REMOTE_OUTPUT_LINE_FLAG[ remoteIndex ] );
        s_SoundHandle.SetRemoteOutVolume( remoteIndex, s_Volume );
        s_SoundHandle.SetPitch( s_Pitch );
        NW_LOG("StartSound(%08x) remoteIndex(%d) volume(%.2f) pitch(%.2f)\n",
            soundId, remoteIndex, s_Volume, s_Pitch );
    }
    else
    {
        NW_LOG("StartSound(%08x) remoteIndex(%d) failed.\n", soundId, remoteIndex );
    }
}

bool KpadProc()
{
    static KPADStatus kpads[ WPAD_MAX_CONTROLLERS ][ 1 ];

    s32 error[ WPAD_MAX_CONTROLLERS ];

    for ( int i=0; i<WPAD_MAX_CONTROLLERS; i++ )
    {
        (void)KPADReadEx( i, &kpads[i][0], 1, &error[i] );
        KPADStatus& kpad = kpads[i][0];

        // サウンド再生・停止
        if ( kpad.trig & KPAD_BUTTON_A )
        {
            StartSound( WSD_HIHAT_CLOSE, i );
        }
        if ( kpad.trig & KPAD_BUTTON_PLUS )
        {
            StartSound( WSD_SNARE, i );
        }
        if ( kpad.trig & KPAD_BUTTON_MINUS )
        {
            StartSound( WSD_HIHAT_CLOSE_REMOTE_FILTER, i );
        }
        if ( kpad.trig & KPAD_BUTTON_B )
        {
            s_SoundHandle.Stop( 0 );
        }

        // パラメータ変更
        if ( kpad.trig & KPAD_BUTTON_LEFT )
        {
            s_Volume -= PARAM_DELTA;
            if ( s_Volume < PARAM_MIN ) { s_Volume = PARAM_MIN; }
            s_SoundHandle.SetRemoteOutVolume( i, s_Volume );
            NW_LOG("RemoteOutVolume(%.2f)\n", s_Volume);
        }
        if ( kpad.trig & KPAD_BUTTON_RIGHT )
        {
            s_Volume += PARAM_DELTA;
            if ( s_Volume > PARAM_MAX ) { s_Volume = PARAM_MAX; }
            s_SoundHandle.SetRemoteOutVolume( i, s_Volume );
            NW_LOG("RemoteOutVolume(%.2f)\n", s_Volume);
        }
        if ( kpad.trig & KPAD_BUTTON_UP )
        {
            s_Pitch += PARAM_DELTA;
            if ( s_Pitch > PARAM_MAX ) { s_Pitch = PARAM_MAX; }
            s_SoundHandle.SetPitch( s_Pitch );
            NW_LOG("Pitch(%.2f)\n", s_Pitch);
        }
        if ( kpad.trig & KPAD_BUTTON_DOWN )
        {
            s_Pitch -= PARAM_DELTA;
            if ( s_Pitch < PARAM_MIN ) { s_Pitch = PARAM_MIN; }
            s_SoundHandle.SetPitch( s_Pitch );
            NW_LOG("Pitch(%.2f)\n", s_Pitch);
        }

        if ( kpad.trig & KPAD_BUTTON_1 )
        {
            nw::snd::SoundSystem::GetRemoteSpeaker( i ).Initialize( SpeakerSetupCallback );
        }
        if ( kpad.trig & KPAD_BUTTON_2 )
        {
            nw::snd::SoundSystem::GetRemoteSpeaker( i ).Finalize( SpeakerShutdownCallback );
        }

        if ( kpad.trig & KPAD_BUTTON_HOME )
        {
            s_SoundHandle.Stop( 0 );
            return false;
        }
    }

    return true;
}

void InitializeSdk()
{
    PADInit();

    // FS 初期化
    {
        FSInit();
        s_pFsClient = reinterpret_cast<FSClient*>(MEMAllocFromDefaultHeap(sizeof(FSClient)));
        NW_ASSERT_NOT_NULL(s_pFsClient);
        std::memset(s_pFsClient, 0x00, sizeof(FSClient));

        FSAddClient(s_pFsClient, FS_RET_NO_ERROR);
    }

    // ディレクトリ移動
    {
        FSCmdBlock* block = reinterpret_cast<FSCmdBlock*>(MEMAllocFromDefaultHeap(sizeof(FSCmdBlock)));
        NW_ASSERT_NOT_NULL(block);
        FSInitCmdBlock(block);

        FSStatus status = FSChangeDir(s_pFsClient, block, CONTENT_DIR, FS_RET_NO_ERROR);
        NW_ASSERT(status == FS_STATUS_OK);

        MEMFreeToDefaultHeap(block);
    }

    // パッド関連の初期化
    KPADInit();

    for ( int i = 0; i < WPAD_MAX_CONTROLLERS; i++ )
    {
        KPADSetConnectCallback( i, ConnectCallback );
    }

    // サウンド関連の初期化
    AXInit();
    AXSetDefaultMixerSelect( AX_PB_MIXER_SELECT_DSP );
}

void InitializeNwSound()
{
    // サウンドシステムの初期化
    nw::snd::SoundSystem::SoundSystemParam param;
    size_t workMemSize = nw::snd::SoundSystem::GetRequiredMemSize( param );
    s_pMemoryForSoundSystem = MEMAllocFromDefaultHeapEx( workMemSize, 4 );

    nw::snd::SoundSystem::Initialize(
            param,
            reinterpret_cast<uptr>( s_pMemoryForSoundSystem ),
            workMemSize );
}

void OpenArchiveAndLoadData()
{
    // サウンドアーカイブの初期化
    {
        size_t size = s_SoundArchive.GetRequiredMemSize();
        s_pMemoryForFsSoundArchive = MEMAllocFromDefaultHeap(size);


        if ( ! s_SoundArchive.Open(
                s_pFsClient, SOUND_ARC_PATH, s_pMemoryForFsSoundArchive, size) )
        {
            NW_ASSERTMSG( 0, "cannot open bfsar(%s)\n", SOUND_ARC_PATH );
        }
    }

    // INFO ブロックのロード
    {
        size_t infoBlockSize = s_SoundArchive.GetHeaderSize();
        s_pMemoryForInfoBlock = MEMAllocFromDefaultHeapEx( infoBlockSize, nw::snd::FsSoundArchive::BUFFER_ALIGN_SIZE );
        if ( ! s_SoundArchive.LoadHeader( s_pMemoryForInfoBlock, infoBlockSize ) )
        {
            NW_ASSERTMSG( 0, "cannot load infoBlock(%s)", SOUND_ARC_PATH );
        }
    }

    // サウンドデータマネージャーの初期化
    {
        size_t setupSize = s_SoundDataManager.GetRequiredMemSize( &s_SoundArchive );
        s_pMemoryForSoundDataManager = MEMAllocFromDefaultHeapEx( setupSize, 4 );
        s_SoundDataManager.Initialize( &s_SoundArchive, s_pMemoryForSoundDataManager, setupSize );
    }

    // サウンドアーカイブプレイヤーの初期化
    {
        size_t setupSize = s_SoundArchivePlayer.GetRequiredMemSize( &s_SoundArchive );
        s_pMemoryForSoundArchivePlayer = MEMAllocFromDefaultHeapEx( setupSize, 32 );
        size_t setupStrmBufferSize =
            s_SoundArchivePlayer.GetRequiredStreamBufferSize( &s_SoundArchive );
        s_pMemoryForStreamBuffer = MEMAllocFromDefaultHeapEx( setupStrmBufferSize, 8 );
        bool result = s_SoundArchivePlayer.Initialize(
                &s_SoundArchive,
                &s_SoundDataManager,
                s_pMemoryForSoundArchivePlayer, setupSize,
                s_pMemoryForStreamBuffer, setupStrmBufferSize );
        NW_ASSERT( result );
    }

    // サウンドヒープの構築
    {
        s_pMemoryForSoundHeap = MEMAllocFromDefaultHeap( SOUND_HEAP_SIZE );
        bool result = s_SoundHeap.Create( s_pMemoryForSoundHeap, SOUND_HEAP_SIZE );
        NW_ASSERT( result );
    }

    if ( ! s_SoundDataManager.LoadData( WSD_HIHAT_CLOSE, &s_SoundHeap ) )
    {
        NW_ASSERTMSG( false, "LoadData(WSD_HIHAT_CLOSE) failed." );
    }
}

} // namespace

int
NwDemoMain(int argc, char **argv)
{
    u32 gx2InitAttribs[] =
    {
        GX2_INIT_ATTRIB_ARGC,    (u32)argc,  // number of args
        GX2_INIT_ATTRIB_ARGV,    (u32)argv,  // command-line args
        GX2_INIT_ATTRIB_NULL                 // terminates the list
    };
    GX2Init(gx2InitAttribs);

    InitializeSdk();
    InitializeNwSound();

    OpenArchiveAndLoadData();


    NW_LOG("---------------------------------------------\n");
    NW_LOG("NW4F Demo/snd/remote\n");
    NW_LOG("---------------------------------------------\n");
    NW_LOG("[A] Start WSD  (WSD_HIHAT_CLOSE)\n");
    NW_LOG("[+] Start WSD  (WSD_SNARE)\n");
    NW_LOG("[-] Start WSD  (WSD_HIHAT_CLOSE_REMOTE_FILTER)\n");
    NW_LOG("[B] Stop Sound\n");
    NW_LOG("[LEFT/RIGHT] Change RemoteOutVolume (0.2-4.0)\n");
    NW_LOG("[UP/DOWN]    Change Pitch (0.2-4.0)\n");
    NW_LOG("[1/2]  RemoteSpeaker ON/OFF\n");
    NW_LOG("[HOME] Exit Application\n");
    NW_LOG("---------------------------------------------\n");

    while( true ) {
        GX2WaitForVsync();

        bool result = KpadProc();
        s_SoundArchivePlayer.Update();

        if ( result == false )
        {
            break;
        }
    }

    s_SoundArchivePlayer.Finalize();
    s_SoundDataManager.Finalize();
    s_SoundArchive.Close();
    s_SoundHeap.Destroy();
    nw::snd::SoundSystem::Finalize();

    MEMFreeToDefaultHeap( s_pMemoryForSoundHeap );
    MEMFreeToDefaultHeap( s_pMemoryForSoundArchivePlayer );
    MEMFreeToDefaultHeap( s_pMemoryForStreamBuffer );
    MEMFreeToDefaultHeap( s_pMemoryForSoundDataManager );
    MEMFreeToDefaultHeap( s_pMemoryForInfoBlock );
    MEMFreeToDefaultHeap( s_pMemoryForFsSoundArchive );
    MEMFreeToDefaultHeap( s_pMemoryForSoundSystem );

    FSDelClient(s_pFsClient, FS_RET_NO_ERROR);
    MEMFreeToDefaultHeap(s_pFsClient);
    s_pFsClient = NULL;
    FSShutdown();

    AXQuit();

    GX2Shutdown();

    return 0;
}
