﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#define NW_CONSOLE_ENABLE // Release 版でも NW_LOG を有効にするため

#include <nw/snd.h>
#include <nw/demo.h>
#include "common.fsid"

#if defined(NW_PLATFORM_CAFE)
#include <cafe.h>
#elif defined( NW_USE_NINTENDO_SDK )
#define WIN32_LEAN_AND_MEAN
#define NOMINMAX
#include <nn/nn_Windows.h>
#include <nn/fs.h>
#else
#include <winext/cafe.h>
#include <Windows.h>
#endif

namespace
{
#if defined(NW_USE_NINTENDO_SDK)
const char MOUNT_NAME[] = "content";
const char SOUND_ARC_PATH[] = "content:snd/common/common.bfsar";
#else
const char CONTENT_DIR[] = "/vol/content";
const char SOUND_ARC_PATH[] = "snd/common/common.bfsar";
#endif
const s32 SOUND_HEAP_SIZE = 4 * 1024 * 1024;
const f32 CONTROL_PAD_STEP = 0.2f;

nw::snd::FsSoundArchive     s_SoundArchive;
nw::snd::SoundArchivePlayer s_SoundArchivePlayer;
nw::snd::SoundDataManager   s_SoundDataManager;
nw::snd::SoundHeap          s_SoundHeap;
nw::snd::SoundHandle        s_SoundHandle;
nw::snd::SoundHandle        s_SoundHandleHold;
nw::snd::Sound3DManager     s_Sound3DManager;
nw::snd::Sound3DActor       s_Sound3DActor;

const int LISTENER_COUNT = 2;
nw::snd::Sound3DListener    s_Sound3DListener[LISTENER_COUNT];

nw::math::VEC3              s_ActorPos;

int s_FilterType;

void* s_pMemoryForSoundSystem;
void* s_pMemoryForInfoBlock;
void* s_pMemoryForSoundDataManager;
void* s_pMemoryForSoundArchivePlayer;
void* s_pMemoryForSoundHeap;
void* s_pMemoryForStreamBuffer;
void* s_pMemoryFor3dManager;

#if defined(NW_PLATFORM_CAFE)
FSClient* s_pFsClient;
void* s_pMemoryForFsSoundArchive;
#endif

#if defined(NW_USE_NINTENDO_SDK)
void* Allocate(size_t size)
{
    nw::demo::DefaultAllocator allocator;
    return allocator.Alloc(size);
}

void Deallocate(void* memory, size_t size)
{
    NW_UNUSED_VARIABLE(size);

    nw::demo::DefaultAllocator allocator;
    allocator.Free(memory);
}
#endif

void CalcListenerMatrix( nw::math::MTX34& mtx, const nw::math::VEC3& pos  )
{
    // const nw::math::VEC3 pos( 0.0f, 0.0f, -3.0f );   // リスナーの位置
    const nw::math::VEC3 upVec( 0.0f, 1.0f, 0.0f );  // リスナーのUp方向ベクトル
    const f32 degree = 0.0f;    // リスナーの向き

    // リスナーの方向ベクトル
    const nw::math::VEC3 direction(
        -nw::math::SinDeg( degree ), 0.0f, -nw::math::CosDeg( degree )
    );
    nw::math::VEC3 target = pos + direction;

    // リスナー行列生成
    nw::math::MTX34LookAt( &mtx, &pos, &upVec, &target );
}

void InitializeNwSound()
{
    nw::demo::DefaultAllocator allocator;

    // サウンドシステムの初期化
    nw::snd::SoundSystem::SoundSystemParam param;
    {
        size_t workMemSize = nw::snd::SoundSystem::GetRequiredMemSize( param );
        s_pMemoryForSoundSystem = allocator.Alloc( workMemSize, 4 );

        nw::snd::SoundSystem::Initialize(
                param,
                reinterpret_cast<uptr>( s_pMemoryForSoundSystem ),
                workMemSize );

        nw::snd::SoundSystem::SetOutputMode( nw::snd::OUTPUT_MODE_SURROUND );
    }

    // サウンドアーカイブの初期化
    {
#if defined(NW_PLATFORM_CAFE)
        size_t size = s_SoundArchive.GetRequiredMemSize();
        s_pMemoryForFsSoundArchive = MEMAllocFromDefaultHeap(size);
        if ( ! s_SoundArchive.Open(
                    s_pFsClient, SOUND_ARC_PATH, s_pMemoryForFsSoundArchive, size) )
        {
            NW_ASSERTMSG( 0, "cannot open bfsar(%s)\n", SOUND_ARC_PATH );
        }
#else
        if ( ! s_SoundArchive.Open(SOUND_ARC_PATH) )
        {
            NW_ASSERTMSG( 0, "cannot open bfsar(%s)\n", SOUND_ARC_PATH );
        }
#endif
    }

    // INFO ブロックのロード
    {
        size_t infoBlockSize = s_SoundArchive.GetHeaderSize();
        s_pMemoryForInfoBlock = allocator.Alloc( infoBlockSize, nw::snd::FsSoundArchive::BUFFER_ALIGN_SIZE );
        if ( ! s_SoundArchive.LoadHeader( s_pMemoryForInfoBlock, infoBlockSize ) )
        {
            NW_ASSERTMSG( 0, "cannot load infoBlock(%s)", SOUND_ARC_PATH );
        }
    }

    // サウンドデータマネージャーの初期化
    {
        size_t setupSize = s_SoundDataManager.GetRequiredMemSize( &s_SoundArchive );
        s_pMemoryForSoundDataManager = allocator.Alloc( setupSize, 4 );
        s_SoundDataManager.Initialize( &s_SoundArchive, s_pMemoryForSoundDataManager, setupSize );
    }

    // サウンドアーカイブプレイヤーの初期化
    {
        size_t setupSize = s_SoundArchivePlayer.GetRequiredMemSize( &s_SoundArchive );
        s_pMemoryForSoundArchivePlayer = allocator.Alloc( setupSize, 32 );
        size_t setupStrmBufferSize =
            s_SoundArchivePlayer.GetRequiredStreamBufferSize( &s_SoundArchive );
        s_pMemoryForStreamBuffer = allocator.Alloc( setupStrmBufferSize, 8 );
        bool result = s_SoundArchivePlayer.Initialize(
                &s_SoundArchive,
                &s_SoundDataManager,
                s_pMemoryForSoundArchivePlayer, setupSize,
                s_pMemoryForStreamBuffer, setupStrmBufferSize );
        NW_ASSERT( result );
    }

    // 3D サウンドマネージャーの初期化
    {
        size_t setupSize = s_Sound3DManager.GetRequiredMemSize( &s_SoundArchive );
        s_pMemoryFor3dManager = allocator.Alloc( setupSize, 4 );
        NW_LOG("SOund3DManager %p\n", s_pMemoryFor3dManager);
        s_Sound3DManager.Initialize( &s_SoundArchive, s_pMemoryFor3dManager, setupSize );
        s_Sound3DManager.SetMaxPriorityReduction( 32 );
        s_Sound3DManager.SetSonicVelocity( 340.0f / 60 );
    }

    // 3D サウンドリスナーの初期化
    for ( int i = 0; i < LISTENER_COUNT; i++ )
    {
        s_Sound3DManager.AddListener( &s_Sound3DListener[i] );

        // リスナー 2 つを、x = -3.0f (TV 出力用)  および
        //                  x =  3.0f (DRC 出力用) の位置に配置する
        nw::math::MTX34 listenerMtx;
        nw::math::VEC3 listenerPos( 0.0f, 0.0f, -3.0f );
        if ( i == 0 ) { listenerPos.x = -3.0f; }
        else { listenerPos.x = 3.0f; }

        CalcListenerMatrix( listenerMtx, listenerPos );
        s_Sound3DListener[i].SetMatrix( listenerMtx );
        s_Sound3DListener[i].SetMaxVolumeDistance( 2.0f );
        s_Sound3DListener[i].SetUnitDistance( 5.0f );
        s_Sound3DListener[i].SetInteriorSize( 5.0f );

        // 出力先属性を設定
        if ( i == 0 )
        {
            s_Sound3DListener[i].SetOutputTypeFlag( nw::snd::Sound3DListener::OUTPUT_TYPE_TV );
        }
        else
        {
            s_Sound3DListener[i].SetOutputTypeFlag( nw::snd::Sound3DListener::OUTPUT_TYPE_DRC0 );
        }
    }

    // 3D サウンドアクターの初期化
    {
        s_Sound3DActor.Initialize( s_SoundArchivePlayer, s_Sound3DManager );
        s_ActorPos = nw::math::VEC3::Zero();
        s_Sound3DActor.SetPosition( s_ActorPos );
    }

    // サウンドヒープの構築
    {
        s_pMemoryForSoundHeap = allocator.Alloc( SOUND_HEAP_SIZE );
        bool result = s_SoundHeap.Create( s_pMemoryForSoundHeap, SOUND_HEAP_SIZE );
        NW_ASSERT( result );
    }

    // データのロード
    if ( ! s_SoundDataManager.LoadData( SE_IDLE, &s_SoundHeap ) )
    {
        NW_ASSERTMSG( false, "LoadData(SE_IDLE) failed." );
    }

    // 3D 関連パラメータの初期化
    {
        s_FilterType = 0;
        s_Sound3DManager.SetBiquadFilterType( s_FilterType );
    }
}

const char* GetFilterTypeString( int type )
{
    const char* FILTER_TYPE_STRING[] =
    {
        "NONE",
        "BIQUAD_LPF",
        "BIQUAD_HPF",
        "BIQUAD_BPF_512",
        "BIQUAD_BPF_1024",
        "BIQUAD_BPF_2048"
    };
    if ( type > nw::snd::BIQUAD_FILTER_TYPE_BPF2048 || type < 0 )
    {
        return "INVALID";
    }
    return FILTER_TYPE_STRING[ type ];
}


void PrintUsage()
{
    NW_LOG("----------------------------------------\n");
    NW_LOG("NW4F Demo/snd/multiListener\n");
    NW_LOG("----------------------------------------\n");
    NW_LOG("[A] HoldSound (Filter OFF)\n");
    NW_LOG("[X] HoldSound (Filter ON)\n");
    NW_LOG("[B] Reset Actor Position\n");
    NW_LOG("[L] Change Filter Type\n");
    NW_LOG("[LEFT/RIGHT] Move Actor Position.x\n");
    NW_LOG("[UP/DOWN]    Move Actor Position.z\n");
#if defined( NW_PLATFORM_CAFE )
    NW_LOG("[HOME]       Exit Application\n");
#elif defined( NW_PLATFORM_WIN32 ) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
    NW_LOG("[S]          Exit Application\n");
#endif
    NW_LOG("[R]          Print Usage (this)\n");
    NW_LOG("---------------------------------------\n");
}

void PrintPosition()
{
    NW_LOG("x(% 2.2f) z(% 2.2f)\n", s_ActorPos.x, s_ActorPos.z );
}

} // namespace

int
NwDemoMain(int argc, char **argv)
{
#if defined( NW_PLATFORM_CAFE )
    AXInit();
    AXSetDefaultMixerSelect( AX_PB_MIXER_SELECT_DSP );
#else
    NW_UNUSED_VARIABLE(argc);
    NW_UNUSED_VARIABLE(argv);
#endif
    nw::demo::DefaultAllocator allocator;
    nw::demo::DemoSystem::CreateArg demoSystemArg;
    demoSystemArg.allocator = &allocator;
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
    demoSystemArg.waitVBlank = 0; // DemoSystem の SwapBuffer は呼ばないため、VBlank 待ちはしない。
#endif
    nw::demo::DemoSystem* pDemo =
        new( allocator.Alloc( sizeof( nw::demo::DemoSystem ) ) )
        nw::demo::DemoSystem( demoSystemArg );
    pDemo->Initialize();
    pDemo->InitializeGraphicsSystem();

#if defined(NW_PLATFORM_CAFE)
    // FS 初期化
    {
        FSInit();
        s_pFsClient = reinterpret_cast<FSClient*>(MEMAllocFromDefaultHeap(sizeof(FSClient)));
        NW_ASSERT_NOT_NULL(s_pFsClient);
        std::memset(s_pFsClient, 0x00, sizeof(FSClient));

        FSAddClient(s_pFsClient, FS_RET_NO_ERROR);
    }

    // ディレクトリ移動
    {
        FSCmdBlock* block = reinterpret_cast<FSCmdBlock*>(MEMAllocFromDefaultHeap(sizeof(FSCmdBlock)));
        NW_ASSERT_NOT_NULL(block);
        FSInitCmdBlock(block);

        FSStatus status = FSChangeDir(s_pFsClient, block, CONTENT_DIR, FS_RET_NO_ERROR);
        NW_ASSERT(status == FS_STATUS_OK);

        MEMFreeToDefaultHeap(block);
    }
#elif defined( NW_USE_NINTENDO_SDK )
    // nn::fs の初期化
    nn::fs::SetAllocator(Allocate, Deallocate);
    // CONTENT_DIRのマウント
    {
        static const int cFileNameLen = 512;
        char buf[cFileNameLen];
        if ( GetEnvironmentVariableA("CAFE_CONTENT_DIR", buf, cFileNameLen) > 0 )
        {
            nn::fs::MountHost(MOUNT_NAME, buf);
        }
        else
        {
            NW_LOG("cannot read CAFE_CONTENT_DIR\n");
            return -1;
        }
    }
#else
    // ディレクトリ移動
    {
        static const int cFileNameLen = 512;
        char buf[cFileNameLen];
        if ( GetEnvironmentVariableA("CAFE_CONTENT_DIR", buf, cFileNameLen) > 0 )
        {
            NW_LOG("buf(%s)\n", buf);
            SetCurrentDirectoryA(buf);
        }
        else
        {
            NW_LOG("cannot read CAFE_CONTENT_DIR\n");
            return -1;
        }
    }
#endif

    InitializeNwSound();

    PrintUsage();

    while ( !pDemo->IsExiting() )
    {
#if defined( NW_PLATFORM_CAFE )
        GX2WaitForVsync();
#else
        pDemo->WaitForVBlank();
#endif

        pDemo->UpdatePad();
        nw::demo::Pad* pad = pDemo->GetPad();

        // ドップラー効果・フィルター設定なしのサウンドをホールド再生
        if ( pad->IsHold( nw::demo::Pad::MASK_A ) ) {
            s_Sound3DActor.HoldSound( &s_SoundHandle, SE_IDLE ).IsSuccess();
            s_SoundHandle.SetOutputLine( nw::snd::OUTPUT_LINE_MAIN | nw::snd::OUTPUT_LINE_DRC0 );
        }

        // ドップラー効果・フィルター設定ありのサウンドをホールド再生
        // ただし、リスナーが複数あるので、ドップラー効果は無効
        if ( pad->IsHold( nw::demo::Pad::MASK_X ) ) {
            s_Sound3DActor.HoldSound( &s_SoundHandle, SE_IDLE_DOPPLER ).IsSuccess();
            s_SoundHandle.SetOutputLine( nw::snd::OUTPUT_LINE_MAIN | nw::snd::OUTPUT_LINE_DRC0 );
        }

        // 3D サウンドアクター座標をリセット
        if ( pad->IsTrig( nw::demo::Pad::MASK_B ) ) {
            s_ActorPos.x = 0.0f;
            s_ActorPos.z = 0.0f;

            // 座標が飛ぶときは、ResetPosition 関数を呼び出す
            s_Sound3DActor.ResetPosition();
            PrintPosition();
        }

        // 3D サウンドアクターの座標変更
        if ( pad->IsHold( nw::demo::Pad::MASK_UP ) )
        {
            s_ActorPos.z += -CONTROL_PAD_STEP;
            PrintPosition();
        }
        if ( pad->IsHold( nw::demo::Pad::MASK_DOWN ) )
        {
            s_ActorPos.z += CONTROL_PAD_STEP;
            PrintPosition();
        }
        if ( pad->IsHold( nw::demo::Pad::MASK_LEFT ) )
        {
            s_ActorPos.x += -CONTROL_PAD_STEP;
            PrintPosition();
        }
        if ( pad->IsHold( nw::demo::Pad::MASK_RIGHT ) )
        {
            s_ActorPos.x += CONTROL_PAD_STEP;
            PrintPosition();
        }

        // フィルタタイプの変更
        if ( pad->IsTrig( nw::demo::Pad::MASK_L ) )
        {
            s_FilterType += 1;
            if ( s_FilterType > nw::snd::BIQUAD_FILTER_TYPE_BPF2048 )
            {
                s_FilterType = 0;
            }
            s_Sound3DManager.SetBiquadFilterType( s_FilterType );
            NW_LOG("FilterType(%s)\n",
                GetFilterTypeString( s_Sound3DManager.GetBiquadFilterType() ) );
        }

        // Usage 表示
        if ( pad->IsTrig( nw::demo::Pad::MASK_R ) ) {
            PrintUsage();
        }

        // Exit
        if ( pad->IsTrig( nw::demo::Pad::MASK_START ) ) {
            break;
        }

        s_Sound3DActor.SetPosition( s_ActorPos );
        s_SoundArchivePlayer.Update();
    }

    s_Sound3DActor.Finalize();
    s_Sound3DManager.Finalize();
    s_SoundHandle.Stop( 0 );
    s_SoundArchivePlayer.Finalize();
    s_SoundDataManager.Finalize();
    s_SoundArchive.Close();
    s_SoundHeap.Destroy();
    nw::snd::SoundSystem::Finalize();

    allocator.Free( s_pMemoryFor3dManager );
    allocator.Free( s_pMemoryForSoundHeap );
    allocator.Free( s_pMemoryForSoundArchivePlayer );
    allocator.Free( s_pMemoryForStreamBuffer );
    allocator.Free( s_pMemoryForSoundDataManager );
    allocator.Free( s_pMemoryForInfoBlock );
    allocator.Free( s_pMemoryForSoundSystem );

#if defined( NW_PLATFORM_CAFE )
    allocator.Free( s_pMemoryForFsSoundArchive );
    FSDelClient(s_pFsClient, FS_RET_NO_ERROR);
    MEMFreeToDefaultHeap(s_pFsClient);
    s_pFsClient = NULL;
    FSShutdown();

    AXQuit();
#endif

    pDemo->FinalizeGraphicsSystem();
    pDemo->Finalize();
    allocator.Free( pDemo );

#if defined ( NW_USE_NINTENDO_SDK )
    nn::fs::Unmount(MOUNT_NAME);
#endif

    return 0;
}
