﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#define NW_CONSOLE_ENABLE // Release 版でも NW_LOG を有効にするため

#include <nw/snd.h>
#include <nw/demo.h>
#include "common.fsid"

#if defined( NW_PLATFORM_CAFE )
#include <cafe.h>
#elif defined( NW_USE_NINTENDO_SDK )
#define WIN32_LEAN_AND_MEAN
#define NOMINMAX
#include <nn/nn_Windows.h>
#include <nn/fs.h>
#else
#include <winext/cafe.h>
#include <Windows.h>
#endif

namespace
{
#if defined(NW_USE_NINTENDO_SDK)
const char MOUNT_NAME[] = "content";
const char SOUND_ARC_PATH[] = "content:snd/common/common.bfsar";
#else
const char CONTENT_DIR[] = "/vol/content";
const char SOUND_ARC_PATH[] = "snd/common/common.bfsar";
#endif
const s32 SOUND_HEAP_SIZE = 4 * 1024 * 1024;

nw::snd::FsSoundArchive     s_SoundArchive;
nw::snd::SoundArchivePlayer s_SoundArchivePlayer;
nw::snd::SoundDataManager   s_SoundDataManager;
nw::snd::SoundHeap          s_SoundHeap;
nw::snd::SoundHandle        s_SoundHandle;

void* s_pMemoryForSoundSystem;
void* s_pMemoryForInfoBlock;
void* s_pMemoryForSoundDataManager;
void* s_pMemoryForSoundArchivePlayer;
void* s_pMemoryForSoundHeap;
void* s_pMemoryForStreamBuffer;

#if defined(NW_PLATFORM_CAFE)
FSClient* s_pFsClient;
void* s_pMemoryForFsSoundArchive;
#endif

int s_FilterType;
f32 s_LpfValue;
f32 s_BqfValue;
const f32 FILTER_DELTA = 0.1f;
const f32 FILTER_VALUE_EPSILON = 0.5f * FILTER_DELTA;
const f32 LPF_VALUE_MIN = -1.0f;
const f32 LPF_VALUE_MAX = 0.0f;
const f32 BQF_VALUE_MIN = 0.0f;
const f32 BQF_VALUE_MAX = 1.0f;

enum FilterType
{
    FILTER_TYPE_NONE,
    FILTER_TYPE_HANDLE_LPF,            // 単極フィルタ
    FILTER_TYPE_HANDLE_BIQUAD_LPF,     // 以下、双極フィルタ
    FILTER_TYPE_HANDLE_BIQUAD_HPF,
    FILTER_TYPE_HANDLE_BIQUAD_BPF_512,
    FILTER_TYPE_HANDLE_BIQUAD_BPF_1024,
    FILTER_TYPE_HANDLE_BIQUAD_BPF_2048,
    FILTER_TYPE_PLAYER_LPF,            // 単極フィルタ
    FILTER_TYPE_PLAYER_BIQUAD_LPF,     // 以下、双極フィルタ
    FILTER_TYPE_PLAYER_BIQUAD_HPF,
    FILTER_TYPE_PLAYER_BIQUAD_BPF_512,
    FILTER_TYPE_PLAYER_BIQUAD_BPF_1024,
    FILTER_TYPE_PLAYER_BIQUAD_BPF_2048,
    FILTER_TYPE_MAX = FILTER_TYPE_PLAYER_BIQUAD_BPF_2048
};

const char* GetFilterTypeString( int type )
{
    static const char* FILTER_TYPE_STRING[] =
    {
        "NONE",
        "HANDLE_LPF",
        "HANDLE_BIQUAD_LPF",
        "HANDLE_BIQUAD_HPF",
        "HANDLE_BIQUAD_BPF_512",
        "HANDLE_BIQUAD_BPF_1024",
        "HANDLE_BIQUAD_BPF_2048",
        "PLAYER_LPF",
        "PLAYER_BIQUAD_LPF",
        "PLAYER_BIQUAD_HPF",
        "PLAYER_BIQUAD_BPF_512",
        "PLAYER_BIQUAD_BPF_1024",
        "PLAYER_BIQUAD_BPF_2048",
    };
    if ( type > FILTER_TYPE_MAX || type < 0 )
    {
        return "INVALID";
    }
    return FILTER_TYPE_STRING[ type ];
}

void SetFilter()
{
    nw::snd::SoundPlayer& player = s_SoundArchivePlayer.GetSoundPlayer(PLAYER_DEFAULT);

    switch ( s_FilterType )
    {
    case FILTER_TYPE_NONE:
        // ハンドルもプレイヤーもデフォルト値
        s_SoundHandle.SetLpfFreq( LPF_VALUE_MAX );
        s_SoundHandle.SetBiquadFilter( nw::snd::BIQUAD_FILTER_TYPE_INHERIT, BQF_VALUE_MIN );
        player.SetLpfFreq( LPF_VALUE_MAX );
        player.SetBiquadFilter( nw::snd::BIQUAD_FILTER_TYPE_INHERIT, BQF_VALUE_MIN );
        break;
    case FILTER_TYPE_HANDLE_LPF:
        // ハンドルの LPF のみ設定
        s_SoundHandle.SetLpfFreq( s_LpfValue );
        // 他はデフォルト値
        s_SoundHandle.SetBiquadFilter( nw::snd::BIQUAD_FILTER_TYPE_INHERIT, BQF_VALUE_MIN );
        player.SetLpfFreq( LPF_VALUE_MAX );
        player.SetBiquadFilter( nw::snd::BIQUAD_FILTER_TYPE_INHERIT, BQF_VALUE_MIN );
        break;
    case FILTER_TYPE_HANDLE_BIQUAD_LPF:
    case FILTER_TYPE_HANDLE_BIQUAD_HPF:
    case FILTER_TYPE_HANDLE_BIQUAD_BPF_512:
    case FILTER_TYPE_HANDLE_BIQUAD_BPF_1024:
    case FILTER_TYPE_HANDLE_BIQUAD_BPF_2048:
        // ハンドルの Biquad Filter のみ設定
        s_SoundHandle.SetBiquadFilter( s_FilterType - FILTER_TYPE_HANDLE_LPF, s_BqfValue );
        // 他はデフォルト値
        s_SoundHandle.SetLpfFreq( LPF_VALUE_MAX );
        player.SetLpfFreq( LPF_VALUE_MAX );
        player.SetBiquadFilter( nw::snd::BIQUAD_FILTER_TYPE_INHERIT, BQF_VALUE_MIN );
        break;
    case FILTER_TYPE_PLAYER_LPF:
        // プレイヤーの LPF のみ設定
        player.SetLpfFreq( s_LpfValue );
        // 他はデフォルト値
        s_SoundHandle.SetLpfFreq( LPF_VALUE_MAX );
        s_SoundHandle.SetBiquadFilter( nw::snd::BIQUAD_FILTER_TYPE_INHERIT, BQF_VALUE_MIN );
        player.SetBiquadFilter( nw::snd::BIQUAD_FILTER_TYPE_INHERIT, BQF_VALUE_MIN );
        break;
    case FILTER_TYPE_PLAYER_BIQUAD_LPF:
    case FILTER_TYPE_PLAYER_BIQUAD_HPF:
    case FILTER_TYPE_PLAYER_BIQUAD_BPF_512:
    case FILTER_TYPE_PLAYER_BIQUAD_BPF_1024:
    case FILTER_TYPE_PLAYER_BIQUAD_BPF_2048:
        // プレイヤーの Biquad Filter のみ設定
        player.SetBiquadFilter( s_FilterType - FILTER_TYPE_PLAYER_LPF, s_BqfValue );
        // 他はデフォルト値
        s_SoundHandle.SetLpfFreq( LPF_VALUE_MAX );
        s_SoundHandle.SetBiquadFilter( nw::snd::BIQUAD_FILTER_TYPE_INHERIT, BQF_VALUE_MIN );
        player.SetLpfFreq( LPF_VALUE_MAX );
        break;
    }

    NW_LOG("filter( %-22s )   ", GetFilterTypeString( s_FilterType ));
    if ( s_FilterType == FILTER_TYPE_NONE )
    {
        NW_LOG("\n");
    }
    else if ( s_FilterType == FILTER_TYPE_HANDLE_LPF ||
              s_FilterType == FILTER_TYPE_PLAYER_LPF )
    {
        NW_LOG("value(% .1f)\n", s_LpfValue );
    }
    else
    {
        NW_LOG("value(% .1f)\n", s_BqfValue );
    }
}

#if defined(NW_USE_NINTENDO_SDK)
void* Allocate(size_t size)
{
    nw::demo::DefaultAllocator allocator;
    return allocator.Alloc(size);
}

void Deallocate(void* memory, size_t size)
{
    NW_UNUSED_VARIABLE(size);

    nw::demo::DefaultAllocator allocator;
    allocator.Free(memory);
}
#endif

void InitializeSound()
{
    nw::demo::DefaultAllocator allocator;

    // サウンドシステムの初期化
    {
        nw::snd::SoundSystem::SoundSystemParam param;
        size_t workMemSize = nw::snd::SoundSystem::GetRequiredMemSize( param );
        s_pMemoryForSoundSystem = allocator.Alloc( workMemSize, 4 );

        nw::snd::SoundSystem::Initialize(
            param,
            reinterpret_cast<uptr>( s_pMemoryForSoundSystem ),
            workMemSize );
    }

    // サウンドアーカイブの初期化
    {
#if defined(NW_PLATFORM_CAFE)
        size_t size = s_SoundArchive.GetRequiredMemSize();
        s_pMemoryForFsSoundArchive = MEMAllocFromDefaultHeap(size);
        if ( ! s_SoundArchive.Open(
                    s_pFsClient, SOUND_ARC_PATH, s_pMemoryForFsSoundArchive, size) )
        {
            NW_ASSERTMSG( 0, "cannot open bfsar(%s)\n", SOUND_ARC_PATH );
        }
#else
        if ( ! s_SoundArchive.Open(SOUND_ARC_PATH) )
        {
            NW_ASSERTMSG( 0, "cannot open bfsar(%s)\n", SOUND_ARC_PATH );
        }
#endif
    }

    // INFO ブロックのロード
    {
        size_t infoBlockSize = s_SoundArchive.GetHeaderSize();
        s_pMemoryForInfoBlock = allocator.Alloc( infoBlockSize, nw::snd::FsSoundArchive::BUFFER_ALIGN_SIZE );
        if ( ! s_SoundArchive.LoadHeader( s_pMemoryForInfoBlock, infoBlockSize ) )
        {
            NW_ASSERTMSG( 0, "cannot load infoBlock(%s)", SOUND_ARC_PATH );
        }
    }

    // サウンドデータマネージャーの初期化
    {
        size_t setupSize = s_SoundDataManager.GetRequiredMemSize( &s_SoundArchive );
        s_pMemoryForSoundDataManager = allocator.Alloc( setupSize, 4 );
        s_SoundDataManager.Initialize( &s_SoundArchive, s_pMemoryForSoundDataManager, setupSize );
    }

    // サウンドアーカイブプレイヤーの初期化
    {
        size_t setupSize = s_SoundArchivePlayer.GetRequiredMemSize( &s_SoundArchive );
        s_pMemoryForSoundArchivePlayer = allocator.Alloc( setupSize, 32 );
        size_t setupStrmBufferSize =
            s_SoundArchivePlayer.GetRequiredStreamBufferSize( &s_SoundArchive );
        s_pMemoryForStreamBuffer = allocator.Alloc( setupStrmBufferSize, 8 );
        bool result = s_SoundArchivePlayer.Initialize(
                &s_SoundArchive,
                &s_SoundDataManager,
                s_pMemoryForSoundArchivePlayer, setupSize,
                s_pMemoryForStreamBuffer, setupStrmBufferSize );
        NW_ASSERT( result );
    }

    // サウンドヒープの構築
    {
        s_pMemoryForSoundHeap = allocator.Alloc( SOUND_HEAP_SIZE );
        bool result = s_SoundHeap.Create( s_pMemoryForSoundHeap, SOUND_HEAP_SIZE );
        NW_ASSERT( result );
    }

    if ( ! s_SoundDataManager.LoadData( SEQ_MARIOKART, &s_SoundHeap ) )
    {
        NW_ASSERTMSG( false, "LoadData(SEQ_MARIOKART) failed." );
    }
    if ( ! s_SoundDataManager.LoadData( SE_YOSHI, &s_SoundHeap ) )
    {
        NW_ASSERTMSG( false, "LoadData(SE_YOSHI) failed." );
    }

    s_FilterType = 0;
    s_BqfValue = 0.8f;
    s_LpfValue = -0.4f;
    SetFilter();    // 初期状態表示のため
}

} // namespace

int
NwDemoMain(int argc, char **argv)
{
#if defined( NW_PLATFORM_CAFE )
    AXInit();
#else
    NW_UNUSED_VARIABLE(argc);
    NW_UNUSED_VARIABLE(argv);
#endif

    nw::demo::DefaultAllocator allocator;
    nw::demo::DemoSystem::CreateArg demoSystemArg;
    demoSystemArg.allocator = &allocator;
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
    demoSystemArg.waitVBlank = 0; // DemoSystem の SwapBuffer は呼ばないため、VBlank 待ちはしない。
#endif
    nw::demo::DemoSystem* pDemo =
        new( allocator.Alloc( sizeof( nw::demo::DemoSystem ) ) )
        nw::demo::DemoSystem( demoSystemArg );
    pDemo->Initialize();
    pDemo->InitializeGraphicsSystem();

#if defined(NW_PLATFORM_CAFE)
    // FS 初期化
    {
        FSInit();
        s_pFsClient = reinterpret_cast<FSClient*>(MEMAllocFromDefaultHeap(sizeof(FSClient)));
        NW_ASSERT_NOT_NULL(s_pFsClient);
        std::memset(s_pFsClient, 0x00, sizeof(FSClient));

        FSAddClient(s_pFsClient, FS_RET_NO_ERROR);
    }

    // ディレクトリ移動
    {
        FSCmdBlock* block = reinterpret_cast<FSCmdBlock*>(MEMAllocFromDefaultHeap(sizeof(FSCmdBlock)));
        NW_ASSERT_NOT_NULL(block);
        FSInitCmdBlock(block);

        FSStatus status = FSChangeDir(s_pFsClient, block, CONTENT_DIR, FS_RET_NO_ERROR);
        NW_ASSERT(status == FS_STATUS_OK);

        MEMFreeToDefaultHeap(block);
    }
#elif defined( NW_USE_NINTENDO_SDK )
    // nn::fs の初期化
    nn::fs::SetAllocator(Allocate, Deallocate);
    // CONTENT_DIRのマウント
    {
        static const int cFileNameLen = 512;
        char buf[cFileNameLen];
        if ( GetEnvironmentVariableA("CAFE_CONTENT_DIR", buf, cFileNameLen) > 0 )
        {
            nn::fs::MountHost(MOUNT_NAME, buf);
        }
        else
        {
            NW_LOG("cannot read CAFE_CONTENT_DIR\n");
            return -1;
        }
    }
#else
    // ディレクトリ移動
    {
        static const int cFileNameLen = 512;
        char buf[cFileNameLen];
        if ( GetEnvironmentVariableA("CAFE_CONTENT_DIR", buf, cFileNameLen) > 0 )
        {
            NW_LOG("buf(%s)\n", buf);
            SetCurrentDirectoryA(buf);
        }
        else
        {
            NW_LOG("cannot read CAFE_CONTENT_DIR\n");
            return -1;
        }
    }
#endif

    InitializeSound();

    NW_LOG("----------------------------------------\n");
    NW_LOG("NW4F Demo/snd/filter\n");
    NW_LOG("----------------------------------------\n");
    NW_LOG("[A]          Start SEQ  (SEQ_MARIOKART)\n");
    NW_LOG("[X]          Start WSD  (SE_YOSHI)\n");
    NW_LOG("[Y]          Start STRM (STRM_MARIOKART)\n");
    NW_LOG("[B]          Stop Sound\n");
    NW_LOG("[LEFT/RIGHT] Change FilterType\n");
    NW_LOG("[UP/DOWN]    Change FilterValue\n");
#if defined( NW_PLATFORM_CAFE )
    NW_LOG("[HOME]       Exit Application\n");
#elif defined( NW_PLATFORM_WIN32 ) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
    NW_LOG("[S]          Exit Application\n");
#endif
    NW_LOG("----------------------------------------\n");

    while ( !pDemo->IsExiting() )
    {
#if defined( NW_PLATFORM_CAFE )
        GX2WaitForVsync();
#else
        pDemo->WaitForVBlank();
#endif

        pDemo->UpdatePad();
        nw::demo::Pad* pad = pDemo->GetPad();

        // サウンドの再生
        if ( pad->IsTrig( nw::demo::Pad::MASK_A ) )
        {
            s_SoundHandle.Stop( 0 );
            bool result = s_SoundArchivePlayer
                .StartSound( &s_SoundHandle, SEQ_MARIOKART )
                .IsSuccess();
            SetFilter();
            NW_LOG("[SEQ] SEQ_MARIOKART ... (%d)\n", result);
        }
        if ( pad->IsTrig( nw::demo::Pad::MASK_X ) )
        {
            s_SoundHandle.Stop( 0 );
            bool result = s_SoundArchivePlayer
                .StartSound( &s_SoundHandle, SE_YOSHI )
                .IsSuccess();
            SetFilter();
            NW_LOG("[WSD] SE_YOSHI ... (%d)\n", result);
        }
        if ( pad->IsTrig( nw::demo::Pad::MASK_Y ) )
        {
            s_SoundHandle.Stop( 0 );
            bool result = s_SoundArchivePlayer
                .StartSound( &s_SoundHandle, STRM_MARIOKART )
                .IsSuccess();
            SetFilter();
            NW_LOG("[STRM] STRM_MARIOKART ... (%d)\n", result);
        }
        if ( pad->IsTrig( nw::demo::Pad::MASK_B ) )
        {
            s_SoundHandle.Stop( 0 );
        }

        // フィルタの調整
        if ( pad->IsTrig( nw::demo::Pad::MASK_LEFT ) )
        {
            s_FilterType -= 1;
            if ( s_FilterType < 0 )
            {
                s_FilterType = FILTER_TYPE_MAX;
            }
            SetFilter();
        }
        if ( pad->IsTrig( nw::demo::Pad::MASK_RIGHT ) )
        {
            s_FilterType += 1;
            if ( s_FilterType > FILTER_TYPE_MAX )
            {
                s_FilterType = 0;
            }
            SetFilter();
        }
        if ( s_FilterType == FILTER_TYPE_NONE )
        {
            // do nothing
        }
        else if ( s_FilterType == FILTER_TYPE_HANDLE_LPF ||
                  s_FilterType == FILTER_TYPE_PLAYER_LPF )
        {
            if ( pad->IsTrig( nw::demo::Pad::MASK_UP ) )
            {
                s_LpfValue += FILTER_DELTA;
                if ( s_LpfValue > LPF_VALUE_MAX - FILTER_VALUE_EPSILON )
                {
                    s_LpfValue = LPF_VALUE_MAX;
                }
                SetFilter();
            }
            if ( pad->IsTrig( nw::demo::Pad::MASK_DOWN ) )
            {
                s_LpfValue -= FILTER_DELTA;
                if ( s_LpfValue < LPF_VALUE_MIN + FILTER_VALUE_EPSILON )
                {
                    s_LpfValue = LPF_VALUE_MIN;
                }
                SetFilter();
            }
        }
        else    // biquad filter
        {
            if ( pad->IsTrig( nw::demo::Pad::MASK_UP ) )
            {
                s_BqfValue += FILTER_DELTA;
                if ( s_BqfValue > BQF_VALUE_MAX - FILTER_VALUE_EPSILON )
                {
                    s_BqfValue = BQF_VALUE_MAX;
                }
                SetFilter();
            }
            if ( pad->IsTrig( nw::demo::Pad::MASK_DOWN ) )
            {
                s_BqfValue -= FILTER_DELTA;
                if ( s_BqfValue < BQF_VALUE_MIN + FILTER_VALUE_EPSILON )
                {
                    s_BqfValue = BQF_VALUE_MIN;
                }
                SetFilter();
            }
        }

        if ( pad->IsTrig( nw::demo::Pad::MASK_START ) ) {
            break;
        }

        s_SoundArchivePlayer.Update();
    }

    s_SoundArchivePlayer.Finalize();
    s_SoundDataManager.Finalize();
    s_SoundArchive.Close();
    s_SoundHeap.Destroy();
    nw::snd::SoundSystem::Finalize();

    allocator.Free( s_pMemoryForSoundHeap );
    allocator.Free( s_pMemoryForSoundArchivePlayer );
    allocator.Free( s_pMemoryForStreamBuffer );
    allocator.Free( s_pMemoryForSoundDataManager );
    allocator.Free( s_pMemoryForInfoBlock );
    allocator.Free( s_pMemoryForSoundSystem );

#if defined( NW_PLATFORM_CAFE )
    allocator.Free( s_pMemoryForFsSoundArchive );
    FSDelClient(s_pFsClient, FS_RET_NO_ERROR);
    MEMFreeToDefaultHeap(s_pFsClient);
    s_pFsClient = NULL;
    FSShutdown();

    AXQuit();
#endif

    pDemo->FinalizeGraphicsSystem();
    pDemo->Finalize();
    allocator.Free( pDemo );

#if defined ( NW_USE_NINTENDO_SDK )
    nn::fs::Unmount(MOUNT_NAME);
#endif

    return 0;
}
